/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.ApplicationInfo;

import java.util.Map;
import java.util.HashSet;
import java.util.HashMap;

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 * QUERYの第１カラムは、選択キーになります。第２カラムはラベルです。ここまでは必須です。
 * 第３カラムが存在する場合は、短縮カラムとして認識されます。存在しない場合は、
 * 短縮ラベルは使用しません。
 *
 * メニュー作成用に、SELECT文を与えます。
 * SELECT 値,ラベル[,グループ][,クラス] FROM XXXX で指定され、値、ラベルまでは必須、
 * グループは、optgroup に対して指定するラベルです。クラスは、そのオプションに
 * 色づけなどを行う為の指定です。
 * なお、グループ、クラス は、NULL(または、ゼロ文字列)の場合は、適用されません。)
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class Selection_DB implements Selection {
public class Selection_DB extends Selection_NULL {
	// 3.5.4.8 (2004/02/23) USE_MULTI_KEY_SELECT を定義しておきます。
	private final boolean USE_MULTI_KEY_SELECT	= HybsSystem.sysBool( "USE_MULTI_KEY_SELECT" ) ;
	private final long	  DB_CACHE_TIME			= (long)HybsSystem.sysInt( "DB_CACHE_TIME" ) ;

	private final boolean	isMultiSelect ;
	private final boolean	isShortLavel ;		// 短縮ラベルを使用できるかどうか
	private final long		createTime ;		// キャッシュの破棄タイミングを計るための作成時間

	private final int[]		ADRS  ;
	private final String	CACHE ;
	private final int		LEN	 ;
	private final int[]		LADRS  ; // 5.1.3.0 (2010/02/01)
	private final String	LCACHE ; // 5.1.3.0 (2010/02/01)
	private final int		LLEN   ; // 5.1.3.0 (2010/02/01)
	private final Map<String,Integer>	map  ;

	private final String[]	value ;		// 値
	private final String[]	label ;		// ラベル
	private final String[]	slabel ;	// 短縮ラベル

	private static final int VAL  = 0;
	private static final int LBL  = 1;
	private static final int SLBL = 2;
	private static final int GRP  = 3;
	private static final int CLS  = 4;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	private static final ApplicationInfo appInfo;
	static {
		/** コネクションにアプリケーション情報を追記するかどうか指定 */
		boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			String SYSTEM_ID = HybsSystem.sys( "SYSTEM_ID" );
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "Selection_DB",null,null );
		}
		else {
			appInfo = null;
		}
	}

	/**
	 * コンストラクター
	 *
	 * ＤＢ検索用のＳＱＬ文を与えて、初期化します。
	 * SQL文は、KEY,LNAME [,SNAME] で、第３項がなければ、LNAME を使用します。
	 * LNAME は、通常の値を返す場合に、SNAME は、一覧表示の値を返す場合に使用します。
	 * 特別に、KEY のみの場合は、lang に基づく ResourceManager からﾗﾍﾞﾙを取得します。
	 * ただし、その場合は、オーナー(SYSTEM_ID)は選べません。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) コンストラクター 新規追加
	 * @og.rev 3.6.0.9 (2004/12/03) isMultiSelect の判定をラベル部のユニーク度で判定します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 3.8.9.2 (2007/07/28) グループと、クラスを追加。Select文の第３、第４引数として指定。
	 * @og.rev 4.0.0.0 (2006/11/15) lang 属性を追加します。
	 * @og.rev 4.3.8.0 (2009/08/01) ツールチップ表示機能追加
	 * @og.rev 5.1.3.0 (2010/02/01) ラベル(短)がnullの場合でも、ラベル(短)で表示されてしまうバグを修正
	 * @og.rev 5.1.3.0 (2010/02/01) 一覧表示以外は、ツールチップ表示しない
	 *
	 * @param newQuery	DB検索(SQL)文字列
	 * @param dbid		データベース接続先ID
	 * @param lang		リソースを使用する場合の言語
	 */
	public Selection_DB( final String newQuery,final String dbid,final String lang ) {

		String[][] cols = DBUtil.dbExecute( newQuery,null,appInfo,dbid );	// 3.8.7.0 (2006/12/15)
		int count = cols.length;

		value	= new String[count];
		label	= new String[count];
		slabel	= new String[count];
		ADRS	= new int[count];
		map		= new HashMap<String,Integer>(count);

		int len = ( count > 0 ) ? cols[0].length : 0 ;
		isShortLavel  = len > SLBL ;	// >= 3 と同意;
		boolean isGrp = len > GRP ;		// >= 4 と同意;
		boolean isCls = len > CLS ;		// >= 5 と同意;

		boolean useLabelData = false ;
		ResourceManager resource = null;
		if( len == 1 ) {
			useLabelData = true;
			resource = ResourceFactory.newInstance( lang );
		}

		// 3.6.0.9 (2004/12/03)
		HashSet<String> set = new HashSet<String>();
		boolean flag = USE_MULTI_KEY_SELECT;	// 判定処理を行う。false なら判定処理も行わない。

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_LARGE );
		String bkGroupKey = "";
		for( int i=0; i<count; i++ ) {
			value[i] = cols[i][VAL];
			if( useLabelData ) {
				label[i] = resource.getLabel( value[i] );
			}
			else {
				label[i] = cols[i][LBL];
				if( isShortLavel ) { slabel[i] = cols[i][SLBL]; }
			}
			map.put( value[i], Integer.valueOf( i ) );

			// 3.8.9.2 (2007/07/28)
			if( isGrp ) {
				String groupKey = cols[i][GRP];
				if( !groupKey.equals( bkGroupKey ) ) {	// キーブレイク
					if( ! "".equals( bkGroupKey ) ) {
						buf.append( "</optgroup>" );
					}
					if( ! "".equals( groupKey ) ) {
						buf.append( "<optgroup label=\"" + groupKey + "\">" );
					}
					bkGroupKey = groupKey;
				}
			}

			buf.append( "<option value=\"" ).append( value[i] ).append( "\"" );
			ADRS[i] = buf.length() ;
			if( isCls ) {
				String classKey = cols[i][CLS];
				if( ! "".equals( classKey ) ) {
					buf.append( " class=\"" ).append( classKey ).append( "\"" );
				}
			}
			// 4.3.8.0 (2009/08/01) slabel利用の場合はlabelをtitle属性にセット
			//buf.append( ">" ).append( label[i] ).append( "</option>" );
			if( isShortLavel && slabel[i] != null && slabel[i].length() > 0 ){ // 5.1.3.0 (2010/02/01)
				if( !label[i].equals( slabel[i] ) ){ // slabelとlabelが異なる場合のみ
					buf.append( " title=\"" ).append( label[i] ).append( "\"" );
				}
				buf.append( ">" ).append( slabel[i] );
			}
			else{
				buf.append( ">" ).append( label[i] );
			}
			buf.append( "</option>" );

			// 3.6.0.9 (2004/12/03)
			if( flag && label[i].length() > 0 ) {
				flag = set.add( label[i].substring(0,1) );	// 重複時は false
			}
		}
		if( isGrp && ! "".equals( bkGroupKey ) ) {
			buf.append( "</optgroup>" );
		}

		CACHE = buf.toString();
		LEN   = CACHE.length() + 30;

		// 5.1.3.0 (2010/02/01) ツールチップ表示が適用されている場合のみ、ツールチップなしの状態のoptionをキャッシュする。
		if( CACHE.indexOf( "title=\"" ) < 0 ) {
			LADRS  = null;
			LCACHE = null;
			LLEN   = 0;
		}
		else {
			LADRS  = new int[count];
			StringBuilder lbuf = new StringBuilder( HybsSystem.BUFFER_LARGE );

			bkGroupKey = "";
			for( int i=0; i<count; i++ ) {
				if( isGrp ) {
					String groupKey = cols[i][GRP];
					if( !groupKey.equals( bkGroupKey ) ) {
						if( ! "".equals( bkGroupKey ) )	{ lbuf.append( "</optgroup>" ); }
						if( ! "".equals( groupKey ) )	{ lbuf.append( "<optgroup label=\"" + groupKey + "\">" ); }
						bkGroupKey = groupKey;
					}
				}
				lbuf.append( "<option value=\"" ).append( value[i] ).append( "\"" );
				LADRS[i] = lbuf.length() ;
				if( isCls && ! "".equals( cols[i][CLS] ) ) {
					lbuf.append( " class=\"" ).append( cols[i][CLS] ).append( "\"" );
				}
				lbuf.append( ">" ).append( label[i] ).append( "</option>" );
			}
			if( isGrp && ! "".equals( bkGroupKey ) ) {
				lbuf.append( "</optgroup>" );
			}
			LCACHE = lbuf.toString();
			LLEN   = LCACHE.length() + 30;
		}

		isMultiSelect = USE_MULTI_KEY_SELECT && ! flag ;	// flag の反転に注意
		createTime = System.currentTimeMillis() ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、ラベル(短)が設定されている場合でも、これを使用せずに必ずラベル(長)を使用します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
	 * @og.rev 4.0.0.0 (2005/01/31) selectValue が、null/ゼロ文字列でない場合に、選択肢にない場合は、エラー
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 *
	 * @return  オプションタグ
	 * @see     #getOption( String, boolean, boolean )
	 */
//	public String getOption( final String selectValue,final boolean seqFlag ) {
//		return getOption( selectValue, seqFlag, false );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * このメソッドでは、引数のuseShortLabelがtrueに指定された場合に、ラベル(短)をベースとした
	 * ツールチップ表示を行います。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 追加
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return  オプションタグ
	 * @see     #getOption( String, boolean )
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		int[] adrs = null;
		String cache = null;
		int len = 0;
		if( !useShortLabel && LCACHE != null && LCACHE.length() > 0 ) {
			adrs = LADRS;
			cache = LCACHE;
			len = LLEN;
		}
		else {
			adrs = ADRS;
			cache = CACHE;
			len = LEN;
		}

		// マッチするアドレスを探す。
		Integer sel = map.get( selectValue );

		if( sel == null ) {
			// 4.0.0 (2005/01/31)
			if( selectValue != null && selectValue.length() > 0 ) {
				String errMsg = "DBコードに存在しない値が指定されました。"
							+ " value=[" + selectValue + "]"
							+ HybsSystem.CR ;
				LogWriter.log( errMsg );
			}
			return cache;
		}
		else {
			int selected = sel.intValue();
			StringBuilder buf = new StringBuilder( len );
			// 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
			if( seqFlag ) {
				buf.append( "<option value=\"" ).append( value[selected] ).append( "\"" );
			}
			else {
				buf.append( cache.substring( 0,adrs[selected] ) );
			}
			buf.append( " selected=\"selected\"" );
			buf.append( cache.substring( adrs[selected] ) );
			return buf.toString() ;
		}
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 2.1.0.1 (2002/10/17) 選択リストを、正方向にしか選べないようにする sequenceFlag を導入する
	 * @og.rev 3.8.6.0 (2006/09/29) useLabel 属性 追加
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 */
//	public String getRadio( final String name,final String selectValue,final boolean useLabel ) {
//		String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue  選択されている値
	 *
	 * @return  オプションタグ
	 */
//	public String getRadioLabel( final String selectValue ) {
//		String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String,boolean )
	 */
//	public String getValueLabel( final String selectValue ) {
//		return getValueLabel( selectValue,false );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 * @og.rev 5.3.5.0 (2011/05/01) 名称(短)表示時に名称(長)をツールチップで表示する。
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを使用する [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// マッチするアドレスを探す。
		Integer sel = map.get( selectValue );

		if( sel == null ) {
			return selectValue;
		}
		else {
			if( isShortLavel && flag ) {
				return "<span title=\""+label[sel.intValue()]+"\">"+slabel[sel.intValue()]+"</span>";
			}
			else {
				return label[sel.intValue()] ;
			}
		}
	}

	/**
	 * マルチ・キーセレクトを使用するかどうかを返します。
	 * true：使用する。false:使用しない です。
	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
	 * true を返します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規作成
	 *
	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する)
	 */
	@Override
	public boolean useMultiSelect() {
		return isMultiSelect;
	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	@Override
	public boolean isTimeOver() {
		return ( System.currentTimeMillis() - createTime ) > DB_CACHE_TIME ;
	}
}
