/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
// import java.text.DateFormat;
// import java.text.SimpleDateFormat;
// import java.util.Date;
// import java.util.Locale;

/**
 * Logを書き込む為の PrintWriter を管理するクラスです。
 *
 * 実際の Log の書き込みには， LogSender を利用して下さい。
 *
 * @og.group エラー処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class LogWriter {
	private static PrintWriter writer = null;

	private static String logFileUrl = null; // 4.1.0.1 (2008/01/23)

	/**
	 * デフォルトコンストラクター
	 * private にして，コンストラクターの作成をさせない様にしています。
	 *
	 */
	private LogWriter() {
	}

	/**
	 * Logファイルの出力先を設定します。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) 新規作成
	 *
	 * @param   url 出力先
	 */
	public static synchronized void init( final String url ) {
		logFileUrl = url;
	}

	/**
	 * Logを書き出します。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) 出力時間を出力する。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param   message メッセージ
	 */
	public static synchronized void log( final String message ) {
		if( writer == null ) {
			writer = getPrintWriter();
		}
//		writer.println( message );
//		DateFormat formatter = new SimpleDateFormat( "yyyy/MM/dd HH:mm:ss.SSS",Locale.JAPAN );
//		writer.println( "[WriteTime= " + formatter.format( new Date() ) + "] " + message ); // 4.1.0.1 (2008/01/23)
		writer.println( "[WriteTime= " + HybsDateUtil.getDate( "yyyy/MM/dd HH:mm:ss.SSS" ) + "] " + message );	// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
		writer.flush();
	}

	/**
	 * 例外のスタックトレースをLogWriterのPrintWriterに書き出します。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) 新規作成
	 * @og.rev 4.3.4.5 (2009/01/08) nullチェック追加
	 *
	 * @param   th スタックトレースの取得元Throwableオブジェクト
	 */
	public static synchronized void log( final Throwable th ) {
		if( writer == null ) {
			writer = getPrintWriter();
		}
		th.printStackTrace( writer );
	}

	/**
	 * PrintWriter を close() します。
	 *
	 */
	public static synchronized void close() {
		if( writer != null ) { writer.close(); }
		writer = null;
	}

	/**
	 * 追加モードで作成した，PrintWriter を取得します。
	 * PrintWriter は，シングルトーンとして，唯一存在させています。
	 *
	 * @og.rev 4.1.0.1 (2008/01/23) ログファイル出力先を外部から指定する。
	 *
	 * @return 追加モードで作成したPrintWriter
	 */
	private static synchronized PrintWriter getPrintWriter() {
//		String logFileName = HybsSystem.url2dir( HybsSystem.sys( "SYS_LOG_URL" ) );

		if( logFileUrl == null || logFileUrl.length() == 0 ) {
			return new PrintWriter( System.err );
		}
		else {
			// 日付フォームのファイル名を変換します。
			DateSet dateSet = new DateSet();
			logFileUrl = dateSet.changeString(logFileUrl );

			try {
				File logFile = new File( logFileUrl );
				return new PrintWriter( new BufferedWriter( new FileWriter( logFile, true ) ) );
			}
			catch( IOException ex ) {
				String errMsg = "ログライターが作成できません。[" + logFileUrl + "]";
//				throw new HybsSystemException( errMsg, e ); // 3.5.5.4 (2004/04/15) 引数の並び順変更
				throw new RuntimeException( errMsg, ex );
			}
		}
	}
}
