/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
// import java.util.Map;

import org.opengion.fukurou.system.OgBuilder ;				// 6.4.4.1 (2016/03/18)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_SEQUENCE は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、シーケンス一覧の検索結果より、GF09 のシーケンス定義テーブルから
 * 必要な情報を取得し、シーケンス作成スクリプトを作成します。
 *
 * この処理を実行するには、DBTableModelのカラムとして、
 *  SYSTEM_ID,TBLSYU,SEQNAME,INCREBY,STARTVAL,MINVAL,MAXVAL,FGCYCLE,SUCACHE
 * が必要です。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/install/03_SEQ ;    出力ファイルの基準フォルダ(必須)
 *       XML : false ;                                  XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,SEQNAME,INCREBY,STARTVAL,MINVAL,MAXVAL,FGCYCLE,SUCACHE from GF09
 * 
 *      ① &lt;og:tableFilter classId="SEQUENCE" keys="DIR,XML" vals='"{&#064;BASE_DIR}/sql/install/03_SEQ,"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="SEQUENCE" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/install/03_SEQ ;
 *                   XML : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.1.9.0 (2010/08/01) DB定義DB・シーケンス定義追加
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2010/08/01
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_SEQUENCE extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.1 (2016/10/21)" ;

	private static final String[] DBKEY = {"SYSTEM_ID","TBLSYU","SEQNAME","INCREBY","STARTVAL","MINVAL","MAXVAL","FGCYCLE","SUCACHE" };

	/** データのアクセス用の配列番号 {@value} */
	protected static final int SYSTEM_ID		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TBLSYU			= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SEQNAME			= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INCREBY			= 3;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int STARTVAL			= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int MINVAL			= 5;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int MAXVAL			= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int FGCYCLE			= 7;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SUCACHE			= 8;

	private static final String ENCODE = "UTF-8" ; // 4.3.6.6 (2009/05/15)

	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;
	private static final int K = FixLengthData.K ;

	/** 各種定数 */
	// 6.0.2.3 (2014/10/10) AbstractTableFilter へ移動

	/** XML形式かどうか  */

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
	 */
	public TableFilter_SEQUENCE() {
		super();
		initSet( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		initSet( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

//	/**
//	 * keys の整合性チェックを行うための初期設定を行います。
//	 *
//	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
//	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるため、廃止。
//	 *
//	 * @param	keysMap keys の整合性チェックを行うための Map
//	 */
//	@Override
//	protected void init( final Map<String,String> keysMap ) {
//		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
//		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
//	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG 自体にCRを追加。キャッシュします。
	 * @og.rev 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
//		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		isXml = StringUtil.nval( getValue( "XML" ), false );
		execEndTag = isXml ? CR + EXEC_END_TAG : ";" ;	// 6.0.2.3 (2014/10/10)

		final File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//		String[] data  = null;
//		PrintWriter writer = null;
		final int[] clmNo = getTableColumnNo( DBKEY );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		final int rowCnt  = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
//			data  = table.getValues( row );
			final String[] data = table.getValues( row );			// 6.3.9.1 (2015/11/27)
			final String seqName = data[clmNo[SEQNAME]];
			// 6.3.7.0 (2015/09/04) AutoCloseableを使用したtry-with-resources構築に対応。
//			try {
//				writer = FileUtil.getPrintWriter( new File( dir,seqName + ( isXml ? ".xml" : ".sql" ) ),ENCODE );
			try( final PrintWriter writer = FileUtil.getPrintWriter( new File( dir,seqName + ( isXml ? ".xml" : ".sql" ) ),ENCODE ) ) {
				if( isXml ) { writer.println( XML_START_TAG ); }
				writer.print( makeHeadLine( clmNo,data ) );
				writer.println( makeLineList( clmNo,data ) );
				if( isXml ) { writer.println( XML_END_TAG ); }
			}
			catch( RuntimeException ex ) {		// catch は、close() されてから呼ばれます。
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_SEQNENCE Error",ErrorMessage.NG )
					.addMessage( row+1,ErrorMessage.NG,"SEQUENCE"
						, "SEQ=[" + seqName + "]" 
						, StringUtil.array2csv( data )
					)
					.addMessage( ex );
				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
				System.out.println( errMessage );
//				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_SEQNENCE Error",ErrorMessage.NG );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"SEQUENCE",ex.getMessage() );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"SEQUENCE",StringUtil.array2csv( data ) );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"SEQUENCE","SEQ=[" + seqName + "]" );
//				System.out.println( errMessage );
			}
//			if( isXml ) { writer.println( XML_END_TAG ); }
//			writer.close();		// 6.0.2.3 (2014/10/10) nullチェック不要。その前でエラーになるはず。
		}

		return table;
	}

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 * @og.rtnNotNull
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		final String seqName = data[clmNo[SEQNAME]];
		final String LINE1 = seqName;
		final String LINE2 = "Created : " + HybsSystem.getDate() ;

		// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
		final int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		final String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		fixData.addAllListData( outData );

		return fixData.getAllFixData();
	}

	/**
	 * シーケンス作成の処理を実行します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュ(execEndTag)を利用します。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	シーケンス作成
	 * @og.rtnNotNull
	 */
	protected String makeLineList( final int[] clmNo,final String[] data ) {
		return new OgBuilder()
				.appendIfCR( isXml , EXEC_START_TAG )
				.appendCR( "CREATE SEQUENCE "	, data[clmNo[SEQNAME]] )
				.append( "  START WITH "		, data[clmNo[STARTVAL]] )
				.append( " INCREMENT BY "		, data[clmNo[INCREBY]] )
				.append( " MINVALUE "			, data[clmNo[MINVAL]] )
				.append( " MAXVALUE "			, data[clmNo[MAXVAL]] )
				.appendCase( "1".equals( data[clmNo[FGCYCLE]] )			// flagで選択されます。
								, " CYCLE" , " NOCYCLE" )				// true , false
				.appendCase( Integer.parseInt( data[clmNo[SUCACHE]] ) <= 0	// 注意：判定条件を反転してます。
								, " NOCACHE"							// true
								, " CACHE " , data[clmNo[SUCACHE]] )	// false
				.append( execEndTag )
				.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//
//		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }
//
//		buf.append( "CREATE SEQUENCE " ).append( data[clmNo[SEQNAME]] ).append( CR )
//			.append( "  START WITH " ).append( data[clmNo[STARTVAL]] )
//			.append( " INCREMENT BY " ).append( data[clmNo[INCREBY]] )
//			.append( " MINVALUE " ).append( data[clmNo[MINVAL]] )
//			.append( " MAXVALUE " ).append( data[clmNo[MAXVAL]] );
//
//		if( "1".equals( data[clmNo[FGCYCLE]] ) ) {
//			buf.append( " CYCLE" );
//		}
//		else {
//			buf.append( " NOCYCLE" );
//		}
//
//		if( Integer.parseInt( data[clmNo[SUCACHE]] ) > 0 ) {		// 6.0.2.4 (2014/10/17) メソッド間違い
//			buf.append( " CACHE " ).append( data[clmNo[SUCACHE]] );
//		}
//		else {
//			buf.append( " NOCACHE" );
//		}
//
//		// 6.0.2.3 (2014/10/10) EXEC_END_TAG の置き換え＋"/" の判定が必要。
//		buf.append( execEndTag );
//
//		return buf.toString();
	}
}
