/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
// import java.util.Map;

import org.opengion.fukurou.system.OgBuilder ;				// 6.4.4.1 (2016/03/18)
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;			// 5.5.2.6 (2012/05/25)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_INDEX は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、インデックス一覧の検索結果より、GF07 のインデックスカラム定義テーブルから
 * 必要な情報を取得し、インデックス作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"C.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,TABLE_LABEL,INDEX_NAME,NAME_JA,INDTYPE,TABLESPACE_NAME,INITIAL_EXTENT)
 * の項目を取得する必要があります。
 *
 * 6.1.0.0 (2014/12/26) より、NEXT_EXTENT は使用しなくなりました。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/install/02_INDEX ;    出力ファイルの基準フォルダ(必須)
 *       XML : false ;                                    XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *       DROP: false ;                                    INDEX構文の前に、DROP構文を出力するかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,TABLE_NAME,TABLE_LABEL,INDEX_NAME,NAME_JA,INDTYPE,TABLESPACE_NAME,INITIAL_EXTENT from GF07
 *      ① &lt;og:tableFilter classId="INDEX" keys="DIR,XML" vals="{&#064;BASE_DIR}/sql/install/02_INDEX,false" /&gt;
 *
 *      ② &lt;og:tableFilter classId="INDEX" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/install/02_INDEX ;
 *                   XML : false ;
 *                   DROP: false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_INDEX extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.1 (2016/10/21)" ;

	// 6.1.0.0 (2014/12/26) NEXT_EXTENT は、使いません。
	private static final String[] DBKEY = {"SYSTEM_ID","TBLSYU","TABLE_NAME","TABLE_LABEL","INDEX_NAME","NAME_JA","INDTYPE",
							"TABLESPACE_NAME","INITIAL_EXTENT" };

	// 5.1.1.0 (2009/12/01) データのアクセス用の配列番号のIDを private ⇒ protected にします。
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SYSTEM_ID		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TBLSYU			= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_NAME		= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_LABEL		= 3;		// GF02 の NAME_JA より JOIN
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INDEX_NAME		= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INDTYPE			= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLESPACE_NAME	= 7;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INITIAL_EXTENT	= 8;

	// 5.1.1.2 (2009/12/10)
	private static final String GF07_SEL = "select A.CLM, B.USE_LENGTH"
											+ " from GF07 A left outer join GF05 B"
											+ " on    A.SYSTEM_ID  = B.SYSTEM_ID"
											+ " and   A.TBLSYU     = B.TBLSYU"
											+ " and   A.TABLE_NAME = B.TABLE_NAME"
											+ " and   A.CLM        = B.CLM"
											+ " and   B.FGJ        = '1'"
											+ " where A.SYSTEM_ID=? and A.TBLSYU=? and A.TABLE_NAME=? and A.INDEX_NAME=?"
											+ " and   A.FGJ='1'"
											+ " order by A.SEQNO" ;

 //	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ; // 4.3.6.6 (2009/05/15)

	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;
	private static final int K = FixLengthData.K ;

	/** 各種定数 */
	// 6.0.2.3 (2014/10/10) AbstractTableFilter へ移動

	/** XML形式かどうか  */

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるように変更。
	 */
	public TableFilter_INDEX() {
		super();
		initSet( "DIR"	, "出力ファイルの基準フォルダ(必須)"							);
		initSet( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"		);
		initSet( "DROP"	, "INDEX構文の前に、DROP構文を出力するかどうか(初期値:false)"	);
	}

//	/**
//	 * keys の整合性チェックを行うための初期設定を行います。
//	 *
//	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
//	 * @og.rev 6.4.1.1 (2016/01/16) keysMap を、サブクラスから設定させるため、廃止。
//	 *
//	 * @param	keysMap keys の整合性チェックを行うための Map
//	 */
//	@Override
//	protected void init( final Map<String,String> keysMap ) {
//		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"							);
//		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"		);
//		keysMap.put( "DROP"	, "INDEX構文の前に、DROP構文を出力するかどうか(初期値:false)"	);
//	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.3.7.0 (2009/06/01) XML出力機能追加
	 * @og.rev 5.1.1.0 (2009/12/01) XML_START_TAG に、tableName をセットします。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.6.9.2 (2013/10/18) INDEXを作成する前に、削除構文を入れるかどうかを指定。
	 * @og.rev 6.0.2.3 (2014/10/10) EXEC_END_TAG を追加。キャッシュします。
	 * @og.rev 6.0.2.3 (2014/10/10) isDrop=false の場合、まとめDropファイル(AllDropC.xml)を作成します。
	 * @og.rev 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
//		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		isXml  = StringUtil.nval( getValue( "XML" ), false );
		execEndTag = isXml ? CR + EXEC_END_TAG : ";" ;	// 6.0.2.3 (2014/10/10)

		final File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		// 6.0.2.3 (2014/10/10) isDrop=false の場合、まとめDropファイル(AllDropC.xml)を作成します。
		// 5.6.9.2 (2013/10/18) DROP構文を出力するかどうか 
		final boolean isDrop = StringUtil.nval( getValue( "DROP" ), false );
		PrintWriter dropWriter = null;
		if( !isDrop ) {					// isDrop == false の場合に、まとめDropする。
			dropWriter = FileUtil.getPrintWriter( new File( dir,"AllDrop" + ( isXml ? "C.xml" : "C.sql" ) ),ENCODE );
			if( isXml ) { dropWriter.println( XML_START_TAG ); }
		}

		String[] data  = null;
		String bkTableName = null;
		PrintWriter writer = null;
		final Transaction tran = getTransaction();				// 5.5.2.6 (2012/05/25)
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE ) ;	// 6.1.0.0 (2014/12/26) refactoring
		final int[] clmNo = getTableColumnNo( DBKEY );
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		final int rowCnt = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
			String tableName = null;
//			String tableLbl  = null;
			String indexName = null;
			try {
				data  = table.getValues( row );
				final String systemId  = data[clmNo[SYSTEM_ID]];
				final String tblsyu    = data[clmNo[TBLSYU]];
				tableName = data[clmNo[TABLE_NAME]];
//				tableLbl  = data[clmNo[TABLE_LABEL]];
				indexName = data[clmNo[INDEX_NAME]];

				// テーブルがキーブレイクすると、セーブファイルを切り替える。
				if( ! tableName.equals( bkTableName ) ) {
					if( writer != null ) {
						if( isXml ) { writer.println( XML_END_TAG ); }
						writer.close();
					}
					bkTableName = tableName;
					writer = FileUtil.getPrintWriter( new File( dir,tableName + ( isXml ? "C.xml" : "C.sql" ) ),ENCODE );
					if( isXml ) { writer.println( XML_START_TAG.replace( "xxx",tableName ) ); }		// 5.1.1.0 (2009/12/01) tableName をセット
					writer.print( makeHeadLine( clmNo,data ) );
				}

				final String[] vals = new String[] { systemId,tblsyu,tableName,indexName };
				final String[][] gf07 = DBUtil.dbExecute( GF07_SEL,vals,tran );						// 5.1.9.0 (2010/08/01) Transaction 対応
				if( gf07.length == 0 ) {
					System.out.println( "TABLE=[" + tableName + "],INDEX=[" + indexName + "] is Not Found!" );
					continue;
				}
				// テーブルに対するカラム列
				buf.setLength(0);
				for( int j=0; j<gf07.length; j++ ) {
					// 5.1.1.2 (2009/12/10)
					buf.append( makeIndexClmStr( gf07[j][0], gf07[j][1] ) ).append( ',' );		// 6.0.2.5 (2014/10/31) char を append する。
				}
				buf.deleteCharAt( buf.length()-1 );			// 最後の "," を取り除く処理

				// 5.6.9.2 (2013/10/18) DROP構文を出力する。
				if( isDrop ) {
					writer.print( makeDropLine( clmNo,data ) );
				}
				// 6.0.2.3 (2014/10/10) isDrop=false の場合、まとめDropファイルを作成します。
				else {
					dropWriter.print( makeDropLine( clmNo,data ) );
				}

				final String clms = buf.toString();
				writer.print( makeLineList( clmNo,data,clms ) );
				writer.println( makeEndLine( clmNo,data ) );
			}
			catch( RuntimeException ex ) {
				// 6.5.0.1 (2016/10/21) ErrorMessage をまとめるのと、直接 Throwable を渡します。
				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_INDEX Error",ErrorMessage.NG )
					.addMessage( row+1,ErrorMessage.NG,"INDEX"
						, "TABLE=[" + tableName + "],INDEX=[" + indexName + "]"
						, StringUtil.array2csv( data )
					)
					.addMessage( ex );
				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
				System.out.println( errMessage );
//				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_INDEX Error",ErrorMessage.NG );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"INDEX",ex.getMessage() );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"INDEX",StringUtil.array2csv( data ) );
//				errMessage.addMessage( row+1,ErrorMessage.NG,"INDEX","TABLE=[" + tableName + "],INDEX=[" + indexName + "]" );
//				// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
//				System.out.println( errMessage );
			}
		}
		if( isXml ) { writer.println( XML_END_TAG ); }
		writer.close();					// 6.0.2.3 (2014/10/10) nullチェック不要。その前でエラーになるはず。
		if( dropWriter != null ) {		// こちらは使わないケースがあるので、nullチェックは必要。
			if( isXml ) { dropWriter.println( XML_END_TAG ); }
			dropWriter.close();
		}

		return table;
	}

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 * @og.rtnNotNull
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		final String tableName = data[clmNo[TABLE_NAME]];
		final String LINE1 = tableName + " ( " + data[clmNo[TABLE_LABEL]] + " )" ;
		final String LINE2 = "Created : " + HybsSystem.getDate() ;

		// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
		final int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		final int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		final FixLengthData fixData = new FixLengthData( addLen,type );

		final String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		fixData.addAllListData( outData );

		return fixData.getAllFixData();
	}

	/**
	 * インデックス作成の処理を実行します。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) プライマリキー対応
	 * @og.rev 5.6.9.2 (2013/10/18) INDTYPE で、その他ではなく、2:通常 で判断する。
	 * @og.rev 6.4.0.5 (2016/01/09) INDTYPE が 9:未使用 の場合の処理を追加。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 * @param   clms	カラム名(CSV形式)
	 *
	 * @return	作成された１行分の文字列
	 * @og.rtnNotNull
	 */
	protected String makeLineList( final int[] clmNo,final String[] data,final String clms ) {
		// 6.4.0.5 (2016/01/09) INDTYPE が 9:未使用 の場合の処理を追加。
		final String idxtype   = data[clmNo[INDTYPE]];

		if( "9".equals( idxtype ) ) { return ""; }		// 6.4.0.5 (2016/01/09)

		final String tableName = data[clmNo[TABLE_NAME]];
		final String indexName = data[clmNo[INDEX_NAME]];
//		final String idxtype   = data[clmNo[INDTYPE]];

		return new OgBuilder()
				.appendCR()												// 先頭に、改行を入れておきます。
				.appendIfCR( isXml , EXEC_START_TAG )
				.appendIf( "0".equals( idxtype )							// 0:プライマリキー
							, "ALTER TABLE "		, tableName
							, " ADD CONSTRAINT "	, indexName
							, " PRIMARY KEY ( " , clms , " )" )
				.appendIf( "1".equals( idxtype )							// 1:ユニークキー
							, "ALTER TABLE "		, tableName
							, " ADD CONSTRAINT "	, indexName
							, " UNIQUE( " , clms , " )" )
				.appendIf( "2".equals( idxtype )							// 2:通常
							, "CREATE INDEX "		, indexName
							, " ON "				, tableName
							, "( " , clms , " )" )
				.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
//			.append( CR );				// 先頭に、改行を入れておきます。
//		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }
//
//		// 5.3.8.0 (2011/08/01) プライマリキー対応
//		if( "0".equals( idxtype ) ) {	// 0:プライマリキー
//			buf.append( "ALTER TABLE " ).append( tableName ).append( " ADD CONSTRAINT " );
//			buf.append( indexName ).append( " PRIMARY KEY ( " ).append( clms );
//			buf.append( " )" );
//		}
//		else if( "1".equals( idxtype ) ) {	// 1:ユニークキー
//			buf.append( "ALTER TABLE " ).append( tableName ).append( " ADD CONSTRAINT " );
//			buf.append( indexName ).append( " UNIQUE( " ).append( clms );
//			buf.append( " )" );
//		}
//		// 5.6.9.2 (2013/10/18) INDTYPE で、その他ではなく、2:通常 で判断する。
//		else if( "2".equals( idxtype ) ) {	// 2:通常
//			buf.append( "CREATE INDEX " ).append( indexName ).append( " ON " );
//			buf.append( tableName ).append( "( " ).append( clms );
//			buf.append( " )" );
//		}
//		else {
//			final String errMsg = "INDTYPE が、0,1,2 以外の値が使われています。INDTYPE=[" + idxtype + "]"
//							+ " TABLE_NAME=[" + tableName + "] INDEX_NAME=[" + indexName + "]" ;
//			System.out.println( errMsg );
//		}
//
//		return buf.toString();
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 *
	 * 6.1.0.0 (2014/12/26) より、
	 *   １．TABLESPACE_NAME を指定しない場合は、TABLESPACE 句を出力しません。
	 *   ２．INITIAL_EXTENT を 0 で指定した場合は、STORAGE 句を出力しません。
	 *   ３．NEXT と PCTINCREASE は、出力しません。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) プライマリキー対応２
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュ(execEndTag)を利用します。
	 * @og.rev 6.1.0.0 (2014/12/26) TABLESPACE_NAME,INITIAL_EXTENT が未設定の場合、設定しません。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 * @og.rtnNotNull
	 */
	protected String makeEndLine( final int[] clmNo,final String[] data ) {

	 	// 6.1.0.0 (2014/12/26) TABLESPACE_NAME,INITIAL_EXTENT が未設定の場合、設定しません。
		final String tblSpcse = data[clmNo[TABLESPACE_NAME]] ;
		final String initExt  = data[clmNo[INITIAL_EXTENT]] ;

		final OgBuilder buf = new OgBuilder()
				.appendCR();												// 先頭に、改行を入れておきます。

		if( !StringUtil.isNull( tblSpcse ) || !StringUtil.isNull( initExt ) ) {
			final String idxtype   = data[clmNo[INDTYPE]];
			buf.appendIf( "0".equals( idxtype ) || "1".equals( idxtype )	// 0:プライマリキー , 1:ユニークキー
							, "USING INDEX " )
				.appendIfCR( !StringUtil.isNull( tblSpcse )
							, "TABLESPACE " , tblSpcse )
				.appendIfCR( !StringUtil.isNull( initExt ) && initExt.charAt(0) != '0'
							, "STORAGE( INITIAL " , initExt , "K )" );
		}

		return buf.append( execEndTag ).toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
//			.append( CR );		// 先頭に、改行を入れておきます。
//
//		if( !StringUtil.isNull( tblSpcse ) || !StringUtil.isNull( initExt ) ) {
//			final String idxtype   = data[clmNo[INDTYPE]];
//			if( "0".equals( idxtype ) || "1".equals( idxtype ) ) {	// 0:プライマリキー , 1:ユニークキー
//				buf.append( "USING INDEX " );
//			}
//
//			if( !StringUtil.isNull( tblSpcse ) ) {
//				buf.append( "TABLESPACE " ).append( tblSpcse ).append( CR );
//			}
//
//			if( !StringUtil.isNull( initExt ) && initExt.charAt(0) != '0' ) {
//				buf.append( "STORAGE( INITIAL " ).append( initExt ).append( "K )" );
//			}
//		}
//		buf.append( execEndTag );
//
//		return buf.toString();
	}

	/**
	 * インデックス削除の構文を、作成します。
	 *
	 * @og.rev 5.6.9.2 (2013/10/18) 新規作成
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュ(execEndTag)を利用します。
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	作成された１行分の文字列
	 * @og.rtnNotNull
	 */
	protected String makeDropLine( final int[] clmNo,final String[] data ) {
		final String tableName = data[clmNo[TABLE_NAME]];
		final String indexName = data[clmNo[INDEX_NAME]];
		final String idxtype   = data[clmNo[INDTYPE]];

		return new OgBuilder()
				.appendCR()													// 先頭に、改行を入れておきます。
				.appendIfCR( isXml , EXEC_START_TAG )
				.appendIf( "0".equals( idxtype ) || "1".equals( idxtype )		// 0:プライマリキー、1:ユニークキー
							, "ALTER TABLE "		, tableName
							, " DROP CONSTRAINT "	, indexName )
				.appendIf( "2".equals( idxtype )								// 2:通常
							, "DROP INDEX "			, indexName )
				.append( execEndTag )
				.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
//			.append( CR );				// 先頭に、改行を入れておきます。
//		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }
//
//		// 0:プライマリキー , 1:ユニークキー
//		if( "0".equals( idxtype ) || "1".equals( idxtype ) ) {
//			buf.append( "ALTER TABLE " ).append( tableName ).append( " DROP CONSTRAINT " );
//			buf.append( indexName );
//		}
//		// 5.6.9.2 (2013/10/18) INDTYPE で、その他ではなく、2:通常 で判断する。
//		else if( "2".equals( idxtype ) ) {	// 2:通常
//			buf.append( "DROP INDEX " ).append( indexName );
//		}
//	//  一連の処理で、makeLineList ですでにエラーが出ているハズなので、ここでは出しません。
//	//	else {
//	//	final String errMsg = "INDTYPE が、0,1,2 以外の値が使われています。INDTYPE=[" + idxtype + "]"
//	//						+ " TABLE_NAME=[" + tableName + "] INDEX_NAME=[" + indexName + "]" ;
//	//		System.out.println( errMsg );
//	//	}
//
//		// 6.0.2.3 (2014/10/10) EXEC_END_TAG の置き換え＋"/" の判定が必要。
//		buf.append( execEndTag );
//
//		return buf.toString();
	}

	/**
	 * インデックスを作成するための文字列を返します。
	 * 通常、カラム名をそのまま返します。
	 * 但し、唯一、MySQLの場合、500バイト以上のカラムについては、TEXTで定義しており、
	 * この場合、インデックス化するバイト数(最大255)を指定する必要があります。
	 * このケースに対応するため、カラム名とバイト数を元に判定し、部分インデックスを
	 * 作成するための文字列を作成します。
	 *
	 * @param	clm		カラム名
	 * @param	useLen	カラムのバイト数
	 *
	 * @return	インデックスカラムの文字列
	 * @see TableFilter_INDEX_MYSQL
	 */
	protected String makeIndexClmStr( final String clm, final String useLen ) {
		return clm;
	}
}
