/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
// import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.util.HybsDateUtil;			// 5.5.8.5 (2012/11/27)
import org.opengion.fukurou.model.Formatter;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.SQLException;

// import java.util.Locale;								// 6.4.4.2 (2016/04/01)

/**
 * 引数引き当て(PreparedStatement) を利用した登録系Queryです。
 *
 * java.sql.PreparedStatement を用いて、データベース検索処理を行います。
 * 引数の指定方法は、DBTableModele のカラム名に対応する名称を、SQL文の[カラム名]形式で
 * 記述します。これを解析して、実際に実行する PreparedStatement に対応する文字列を
 * 作成します。
 * たとえば、INSERT INTO GEXX (CLM,NAME_JA,LABEL_NAME) VALUES ([CLM],[NAME_JA],[LABEL_NAME] )
 * と記述すれば、内部で、DBTableModele のカラム名に対応する値を取り出し、SQL文として、
 * INSERT INTO GEXX (CLM,NAME_JA,LABEL_NAME) VALUES (?,?,? ) を実行します。
 *
 * @og.formSample
 * ●使用例
 *
 *    ・QUERYを直接書く場合
 *    【entry.jsp】
 *        &lt;og:tableUpdate
 *            command   = &quot;{&#064;command}&quot;
 *            queryType = &quot;JDBCTableUpdate&quot;
 *        &gt;
 *            INSERT INTO GE41
 *                (CLM,NAME_JA,LABEL_NAME,KBSAKU,SYSTEM_ID,LANG,
 *                 FGJ,DYSET,DYUPD,USRSET,USRUPD,PGUPD)
 *            VALUES
 *                ([CLM],[NAME_JA],[LABEL_NAME],[KBSAKU],[SYSTEM_ID],[LANG],
 *                 &#39;1&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.YMDH}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;USER.ID}&#39;,&#39;{&#064;GUI.KEY}&#39;)
 *        &lt;/og:tableUpdate&gt;
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCTableUpdate extends AbstractQuery {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.6.0 (2016/05/27)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Query_JDBCTableUpdate() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * [カラム名] 部分の引数を、DBTableModelから順番にセットしていきます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 4.0.0.0 (2007/05/09) ParameterMetaData を使用したパラメータ設定追加。
	 * @og.rev 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 * @og.rev 5.5.5.4 (2012/08/18) useParamMetaData 処理を、ループの外に出す。(PostgreSQL対応)
	 * @og.rev 5.5.5.4 (2012/08/18) DATE オブジェクトを登録できるようにする。
	 * @og.rev 5.5.8.5 (2012/11/27) TIMESTAMP型でも処理できるようにします。
	 * @og.rev 5.6.9.4 (2013/10/31) エラーメッセージに１行前の情報も出力します。
	 * @og.rev 6.3.6.1 (2015/08/28) close(),realClose() 廃止。Queryはキャッシュしません。
	 * @og.rev 6.4.2.1 (2016/02/05) try-with-resources 文で記述。
	 * @og.rev 6.4.3.4 (2016/03/11) Formatterに新しいコンストラクターを追加する。
	 * @og.rev 6.4.4.2 (2016/04/01) contains 判定を行う新しいメソッドを使用します。
	 * @og.rev 6.4.6.0 (2016/05/27) isNumber , isDate 追加。
	 *
	 * @param   rowNo 選択された行番号配列(登録する対象行)
	 * @param   table DBTableModelオブジェクト(登録する元データ)
	 */
	@Override
	public void execute( final int[] rowNo, final DBTableModel table ) {
//		PreparedStatement pstmt = null ;

		int row = 0;			// エラー時に表示するエラー行番号
//		try {
		int executeCount = 0;	// 処理件数
		final Formatter form = new Formatter( table,getStatement() );	// 6.4.3.4 (2016/03/11)
//		final Formatter form = new Formatter( table );
//		form.setFormat( getStatement() );
		final int[] clmNos = form.getClmNos();		// 引数の個数分の配列。カラム番号を保存
		final String query = form.getQueryFormatString();
		final int    cnt   = clmNos.length;			// 引数の個数(カラムの個数ではありません。)

		// 5.5.5.4 (2012/08/18) Timestamp オブジェクトを登録できるようにする。
		boolean useTimeStamp = false;
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//		boolean[] isTime = new boolean[cnt];
		final boolean[] isTime = new boolean[cnt];
		for( int j=0; j<cnt; j++ ) {
			// 5.5.8.5 (2012/11/27) TIMESTAMP型でも処理できるようにします。
	//		final String clsName = table.getDBColumn( clmNos[j] ).getClassName();
			// 6.4.4.2 (2016/04/01) contains 判定を行う新しいメソッドを使用します。
//			isTime[j] = "DATE".equalsIgnoreCase( clsName ) || "TIMESTAMP".equalsIgnoreCase( clsName );
//			isTime[j] = clsName != null && StringUtil.contains( clsName.toUpperCase( Locale.JAPAN ) , "DATE" , "TIMESTAMP" );
			isTime[j] = table.getDBColumn( clmNos[j] ).isDateType();	// 6.4.6.0 (2016/05/27)
			if( !useTimeStamp && isTime[j] ) { useTimeStamp = true; }	// isTime[j] == true 時に、一度だけ実行される。
		}

		final Connection conn = getConnection();
//		pstmt = conn.prepareStatement( query );

		// 6.4.2.1 (2016/02/05) try-with-resources 文
		try( final PreparedStatement pstmt = conn.prepareStatement( query ) ) {
			pstmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
		//	((oracle.jdbc.OraclePreparedStatement)pstmt).setExecuteBatch(50);
			// 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
			final boolean useParamMetaData = useParameterMetaData();	// 5.3.8.0 (2011/08/01)

			// 5.5.5.4 (2012/08/18) 以下、useParamMetaData、useTimeStamp、通常の３種類を、行列ループの外に出す。
			// 5.5.5.4 (2012/08/18) useParamMetaData 処理を、ループの外に出す。(PostgreSQL対応)
			if( useParamMetaData ) {
				// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//				int[] types = new int[cnt];
				final int[] types = new int[cnt];
				final ParameterMetaData pMeta = pstmt.getParameterMetaData();
				for( int j=0; j<cnt; j++ ) {
					types[j] = pMeta.getParameterType( j+1 );	// ややこしいが配列の個数と添え字の関係から、j と j+1 での処理となる。
				}

				for( int i=0; i<rowNo.length; i++ ) {
					row = rowNo[i];
					for( int j=0; j<cnt; j++ ) {
						final String val = StringUtil.rTrim( table.getValue( row,clmNos[j] ) );
						if( val == null || val.isEmpty() ) {
							pstmt.setNull( j+1, types[j] );
						}
						else {
							pstmt.setObject( j+1, val, types[j] );
						}
					}
					executeCount += pstmt.executeUpdate();
				}
			}
			// 5.5.5.4 (2012/08/18) PostgreSQL対応 以外のDBの場合
			else {
				// 5.5.5.4 (2012/08/18) Timestamp オブジェクトを登録する場合
				if( useTimeStamp ) {
					for( int i=0; i<rowNo.length; i++ ) {
						row = rowNo[i];
						for( int j=0; j<cnt; j++ ) {
							final String val = StringUtil.rTrim( table.getValue( row,clmNos[j] ) );
							if( isTime[j] && val != null && !val.isEmpty() ) {
								// 5.5.8.5 (2012/11/27) val は、yyyy-mm-dd hh:mm:ss[.f...] 形式でなければならない。
								final java.sql.Timestamp time = java.sql.Timestamp.valueOf( HybsDateUtil.parseTimestamp( val ) );
								pstmt.setObject( j+1,time );
							}
							else {
								pstmt.setObject( j+1,val );
							}
						}
						executeCount += pstmt.executeUpdate();
					}
				}
				// 5.5.5.4 (2012/08/18) その他：つまり、これが通常の処理
				else {
					for( int i=0; i<rowNo.length; i++ ) {
						row = rowNo[i];
						for( int j=0; j<cnt; j++ ) {
							final String val = StringUtil.rTrim( table.getValue( row,clmNos[j] ) );
							pstmt.setObject( j+1,val );
						}
						executeCount += pstmt.executeUpdate();
					}
				}
			}
			setExecuteCount( executeCount );
			setErrorCode( ErrorMessage.OK );
//			commit();										// 6.3.6.1 (2015/08/28)
		}
		catch (SQLException ex) {		// catch は、close() されてから呼ばれます。
			setErrorCode( ErrorMessage.EXCEPTION );
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
				.append( ex.getMessage() ).append( ':' ).append( ex.getSQLState() ).append( CR )
				.append( "  QUERY=" ).append( getStatement() ).append( CR )
				.append( "  ROW =[" ).append( (row+1) ).append( ']' ).append( CR )
				.append( "  VALS=[" ).append( StringUtil.array2csv( table.getValues(row) )).append( ']' )
				.append( CR ) ;
			// 5.6.9.4 (2013/10/31)
			if( row > 0 ) {
//				errMsg.append("  ROW(-1) =[" ).append( (row) ).append( ']' ).append( CR )
				errMsg.append("  ROW(-1) =[" ).append( row ).append( ']' ).append( CR )		// 6.3.9.0 (2015/11/06) This statement may have some unnecessary parentheses(PMD)
					.append(  "  VALS(-1)=[" ).append( StringUtil.array2csv( table.getValues(row-1) )).append( ']' )
					.append( CR ) ;
			}
//			rollback();
//			realClose();									// 6.3.6.1 (2015/08/28) close(),realClose() 廃止。
			throw new HybsSystemException( errMsg.toString(),ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
//		finally {
//			Closer.stmtClose( pstmt );
//		}
	}
}
