/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.XHTMLTag;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * 入力フィールドやボタンなどを作成するHTML拡張タグです。
 *
 * 簡易的な入力用タグとして、columnタグがありますので、通常は、columnタグをお使い下さい。
 * name 属性に ラベルリソース のキーを与えることで、ロケールにあわせたリソースを
 * 使用して、画面に表示します。
 * ロケールは、session 登録項目の HybsSystem#LANG_KEY を初期値で使用し、
 * language 属性で定義されている場合は、そちらを優先します。
 *
 * @og.formSample
 * ●形式：&lt;og:input name=… value=… /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:input
 *       type               【HTML】表示形式[text/password/checkbox/radio/submit/reset/button/image/file]を指定します(初期値:text)
 *                                  HTML5追加分[search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
 *       name               【HTML】名前を指定します
 *       value              【HTML】このフィールドに設定する値を指定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       lbls               【TAG】ラベルをCSV形式で複数指定します
 *       td                 【TAG】テーブル形式の &lt;td&gt; タグを使用するかどうか[yes/no/false]を指定します(初期値:yes)
 *       colspan            【TAG】フィールド部分の colspan を指定します
 *       nextForm           【TAG】入力カーソルを指定されたname属性のフィールドへ自動的に移動します
 *       optionAttributes   【TAG】JavaScript などの HTML基本タグ以外の属性を、そのままタグとして使用します
 *       must               【TAG】必須入力を表す色に変えるかどうか[true/false]を指定します(初期値:false)
 *       mustAny            【TAG】選択必須入力(どれかひとつ必須)を表す色[true/mustAny/その他]を指定します(初期値:無指定)
 *       aimai              【TAG】曖昧検索可能フィールドとして、曖昧検索方法を指定します(初期値:null)
 *       size               【HTML】横幅を指定します(typeがtextまたはpasswordの場合は文字数で、それ以外はピクセル)
 *       maxlength          【HTML】最大入力文字数を指定します(typeがtextまたはpasswordの場合)
 *       checked            【HTML】type が checkbox か radio のとき、初めから選択された状態で表示します("checked"のみ指定可)
 *       src                【HTML】type が image の場合、送信ボタンとして利用する画像のURLをセットします
 *       alt                【HTML】type が image の場合で、画像が表示できない場合の代替テキストをセットします
 *       accept             【HTML】type が file の場合に、MIMEタイプをCSV形式で複数指定出来ます
 *       usemap             【HTML】type が image の場合、クライアントサイド・イメージマップのURLを指定します
 *       ismap              【HTML】type が image の場合に、サーバーサイド・イメージマップを使用する場合に指定します
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       readonly           【TAG】その部品に対して変更が出来ないように(readonly)指定します(サーバーに送信される)
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】アクセスキー(alt+キーで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onChange           【HTML】JavaScriptのイベント onChange を設定します(例:onChange="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       onSelect           【HTML】JavaScriptのイベント onSelect を設定します
 *       onKeydown          【HTML】JavaScriptのイベント onKeydown を設定します
 *       onKeypress         【HTML】JavaScriptのイベント onKeypress を設定します
 *       onKeyup            【HTML】JavaScriptのイベント onKeyup を設定します
 *       autocomplete       【HTML5】入力候補を提示して入力内容を自動補完する[on/off]（初期値はon）。
 *       autofocus          【HTML5】指定した入力欄にカーソルが当たって自動的にフォーカスされます。
 *       pattern            【HTML5】正規表現で入力値のパターンを指定します。
 *       placeholder        【HTML5】入力欄に初期表示する内容を指定します。
 *       required           【HTML5】入力必須を指定します
 *       list               【HTML5】ユーザーに入力候補として提案するデータリストの要素のid属性の値を指定します。
 *       min                【HTML5】入力欄で入力できる最小値を指定します。
 *       max                【HTML5】入力欄で入力できる最大値を指定します。
 *       step               【HTML5】入力欄で刻むステップ値を指定する。
 *       roles              【TAG】ロールをセットします
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:input&gt;
 *
 * ●使用例
 *    &lt;og:input name="OYA" /&gt;                             lbl属性が指定されていない場合は、name属性をキーにしてLabelResourceを読み取ります。
 *    &lt;og:input name="PN" lbl="KO" /&gt;                     lbl属性を指定することで、ラベルだけを付替えることができます。
 *    &lt;og:input name="PN" size="13" maxlength="11" /&gt;     強制的にサイズの変更をするときは、size属性、maxlength属性を使用できます。
 *    &lt;og:input name="OYA" td="false" /&gt;                  自動作成されるtdタグやラベルを外すときはtd属性を使います。
 *    &lt;og:input name="OYA" td="no" /&gt;                     自動作成されるtdタグを外し、ラベルとフィールドは残します。
 *    &lt;og:input name="PN" optionAttributes="onBlur='blr()'
 *                            onFocus='fcs()'" /&gt;
 *                                                               JavaScriptのイベントをコーディングするときは、optionAttributes属性を使います。
 *    &lt;og:input type="radio" name="OYA" /&gt;                ラベル部分と入力フィールド部分がテーブルタグの&lt;td&gt;により左右に分割されます。
 *
 *    &lt;table&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;PN&quot; value=&quot;{&#064;PN}&quot; /&gt;&lt;/tr&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;CD&quot; value=&quot;{&#064;CD}&quot; /&gt;&lt;/tr&gt;
 *    &lt;/table&gt;
 *
 *    &lt;table&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;PN&quot; value=&quot;{&#064;PN}&quot; &gt;部品入力フィールド&lt;/og:input&gt;&lt;/tr&gt;
 *        &lt;tr&gt;&lt;og:input name=&quot;CD&quot; value=&quot;{&#064;CD}&quot; &gt;コードフィールド&lt;/og:input&gt;&lt;/tr&gt;
 *    &lt;/table&gt;
 *                                                               HTML 表示時は、前後に&lt;tr&gt;タグで囲って,整形できます。
 *
 *    &lt;og:input name="A"  nextForm="B" /&gt;                最大桁数入力後、フォーム B にフォーカスが移動します。
 *    &lt;og:input name="B" /&gt;
 *
 *    BODY 部分に記述した値は、入力フィールドの直後にセットされます。
 *    &lt;og:input name="PN" &gt;
 *      &lt;button type="button" onclick="popup(･･･);return false;"&gt;
 *          &lt;og:message lbl="POPUP" comment="▼" /&gt;
 *      &lt;/button&gt;
 *    &lt;/og:input&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class InputTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.7.1.0 (2013/12/06)" ;

	private static final long serialVersionUID = 571020131206L ;

	// 5.7.1.0 (2013/12/06) HTML5 で新たに追加された、type を追加
//	private static final String CHECK_TYPE = "|text|password|checkbox|radio|submit|reset|button|image|file|hidden|" ;
	private static final String CHECK_TYPE = "|text|password|checkbox|radio|submit|reset|button|image|file|hidden|" 
									+ "search|tel|url|email|datetime|date|month|week|time|datetime-local|number|range|color|" ;

	// 4.0.0 (2005/01/31) HTML_LABEL_SEPARATOR を boolean 変数として取得します。
	private final String CLM = ( HybsSystem.sysBool( "HTML_LABEL_SEPARATOR" ) ) ? ":" : "" ;

	private String  td_flag		= "yes";	// yes:<td> タグを使用 / false:フィールドのみ / no:ラベルとフィールド
	// 3.0.1.4 (2003/03/17) colspan 属性を追加。
	private String  colspan     = "";
	// 3.5.4.2 (2003/12/15) nextForm 属性を追加。
	private String  nextForm    = null;

	// 4.0.0 (2005/01/31) COLUMNS_MAXSIZE を定義しておきます。
	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" ) ;   // 表示フィールドの大きさ

	// 4.0.0 (2007/05/30) BODY 部分に記述した値は、入力フィールドの直後にセットされます。
	private String  body		= null;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/05/30) 新規追加(BODY部の評価)
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
		}
		return ( SKIP_BODY );				// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2007/05/30) 新規追加(BODY部の評価)
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = getBodyString();
		return(SKIP_BODY);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性を追加。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) nextForm 属性を追加。
	 * @og.rev 5.8.7.1 (2015/05/22) body追加(6.2.4.0対応)
	 */
	@Override
	protected void release2() {
		super.release2();
		td_flag		= "yes";	// table形式の <td> タグを使用するかどうか
		colspan 	= "";
		nextForm	= null;		// 3.5.4.2 (2003/12/15)
		body		= null;		// 5.8.7.1 (2015/05/22)
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 * @og.rev 2.0.1.0 (2002/10/10) ラベルとフィールドのセパレーターとして、コロン(：)を使用するかどうかを指定できる
	 * @og.rev 3.0.1.2 (2003/03/07) forward.jsp の代替用 CommonForwardTag を新規作成
	 * @og.rev 3.1.0.1 (2003/03/26) キャッシュエントリーは、type="submit" の場合のみ、登録する。
	 * @og.rev 3.1.1.0 (2003/03/28) forward.jsp 関係の処理を削除する。
	 * @og.rev 3.1.1.0 (2003/03/28) radio ボタン等で、ラベルをクリックしても値をセットできるようにする。
	 * @og.rev 3.5.4.2 (2003/12/15) 入力カーソルを自動的に次のフィールドへ移動する機能を追加する。
	 * @og.rev 4.0.0.0 (2007/05/30) BODY 部分に記述した値は、入力フィールドの直後にセットされます。
	 * @og.rev 4.3.7.1 (2009/06/08) id=labelのclass化
	 * @og.rev 5.2.1.0 (2010/10/01) must , mustAny 属性を自動化します。
	 * @og.rev 5.6.2.2 (2013/03/15) 自動must処理の出力位置を変更
	 * @og.rev 5.8.4.1 (2015/02/10) aimai対応 6.2.0.0逆移植
	 *
	 * @return  入力用のテキストフィールドタグ
	 */
	@Override
	protected String makeTag() {
		if( get( "value" ) == null ) { set( "value","" ); }
		if( getMsglbl() == null ) { setLbl( get( "name" ) ); }

		// 3.5.4.2 (2003/12/15) 入力カーソルを自動的に次のフィールドへ移動する機能
		if( nextForm != null ) {
			String onKeyup = "nextForm(this,'" + nextForm + "'," + get( "maxlength" ) + ");" ;
			set( "onKeyup",onKeyup );
		}

		String type = get( "type" );
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		// もう少しましな、別の方法を考える必要があります。
		if( "yes".equals( td_flag ) ) {
			if( type != null && ( "radio".equals( type ) || "checkbox".equals( type ) )) {
				if( get( "id" ) == null ) { set( "id",TaglibUtil.getTagId() ); }

				rtn.append( "<td>" );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
				rtn.append( "</td>" );
				rtn.append( "<td " );
				rtn.append( colspan );
				rtn.append( ">" );
				rtn.append( "<label for=\"" );
				rtn.append( get( "id" ) );
				rtn.append( "\">" );
				rtn.append( getLongLabel() );		// 4.0.0 (2005/01/31)
				rtn.append( "</label>" );
//				rtn.append( "</td>" );
			}
			else if( type != null && ( "submit".equals( type ) || "reset".equals( type ) || "button".equals( type ) ) ) {
				set( "value",getMsglbl() );
				rtn.append( "<td" );
				rtn.append( colspan );
				rtn.append( ">" );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
				rtn.append( "</td>" );
//				rtn.append( "<td></td>" );
				rtn.append( "<td>" );
			}
			else {
				// 4.3.7.1 (2009/06/08) id=labelのclass化
				// rtn.append( "<td id=\"label\">" );
				rtn.append( "<td class=\"label\">" );
				rtn.append( getLongLabel() );		// 4.0.0 (2005/01/31)
				rtn.append( CLM );
				rtn.append( "</td>" );
				rtn.append( "<td" );
				rtn.append( colspan );
				rtn.append( ">" );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
//				rtn.append( "</td>" );
			}
			rtn.append( makeAimaiPicker( get("name") ) ); // 5.8.4.2(2015/02/20)
			if( getMustType() != null ) { rtn.append( makeMustHidden( get( "name" ) ) ); } // 5.6.2.2 (2013/03/15)
			rtn.append( "</td>" );
		}
		else if( "no".equals( td_flag ) ) {
			if( type != null && ( "radio".equals( type ) || "checkbox".equals( type ) )) {
				rtn.append( "<label>" );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
				// 4.3.7.1 (2009/06/08) id=labelのclass化
				// rtn.append( "<span id=\"label\">" );
				rtn.append( "<span class=\"label\">" );
				rtn.append( getLongLabel() );		// 4.0.0 (2005/01/31)
				rtn.append( "</span>" );
				rtn.append( "</label>" );
			}
			else if( type != null && ( "submit".equals( type ) || "reset".equals( type ) || "button".equals( type ) ) ) {
				set( "value",getMsglbl() );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
			}
			else {
				// 4.3.7.1 (2009/06/08) id=labelのclass化
				// rtn.append( "<span id=\"label\">" );
				rtn.append( "<span class=\"label\">" );
				rtn.append( getLongLabel() );		// 4.0.0 (2005/01/31)
				rtn.append( CLM );
				rtn.append( "</span>" );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
			}
			rtn.append( makeAimaiPicker( get("name") ) ); // 5.8.4.2(2015/02/20)
			if( getMustType() != null ) { rtn.append( makeMustHidden( get( "name" ) ) ); } // 5.6.2.2 (2013/03/15)
		}
		else {
			if( type != null && ( "submit".equals( type ) || "reset".equals( type ) || "button".equals( type ) ) ) {
				set( "value",getMsglbl() );
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
			}
			else {
				rtn.append( XHTMLTag.input( getAttributes() ) );
				if( body != null ) { rtn.append( body ); }	// 4.0.0 (2007/05/30) BODY部の評価
			}
			rtn.append( makeAimaiPicker( get("name") ) ); // 5.8.4.2(2015/02/20)
			if( getMustType() != null ) { rtn.append( makeMustHidden( get( "name" ) ) ); } // 5.6.2.2 (2013/03/15)
		}

	 	// 5.2.1.0 (2010/10/01) must , mustAny 属性を自動化します。
		// 5.6.2.2 (2013/03/15) tdの外に出力される不正形を正します
//		if( getMustType() != null ) { rtn.append( makeMustHidden( get( "name" ) ) ); }

		return rtn.toString() ;
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag
	 *  &lt;og:input name="PN" /&gt;
	 *
	 * @param   name 名前
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】表示形式を指定します(初期値:text)。
	 *
	 * @og.tag
	 * 通常は、表示形式[text/password/checkbox/radio/submit/reset/button/image/file]を指定します。
	 * HTML5 で追加された形式[search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 * も、設定できます。
	 *
	 * HTML5 の機能を有効にするには、ネイティブモードで動作させる必要があります。
	 *   ① USE_IE7_HEADER = "false" に設定する。
	 *   ② USE_HTML5_HEADER = "true" に設定する。
	 *   ③ IEのツール⇒互換表示設定で、互換表示に追加したWebサイトから削除する。
	 *   ④ 同上の設定で、イントラサイトを互換表示で表示するのチェックを外す。
	 * 必要があります。
	 *
	 * &lt;og:input type="text"     /&gt; テキストボックスを表示
	 * &lt;og:input type="password" /&gt; パスワード専用テキストボックスを表示
	 * &lt;og:input type="checkbox" /&gt; チェックボックスを表示
	 * &lt;og:input type="radio"    /&gt; ラジオボタンを表示
	 * &lt;og:input type="submit"   /&gt; 送信実行ボタンを表示
	 * &lt;og:input type="reset"    /&gt; 入力したフィールド内容を全てリセット
	 * &lt;og:input type="button"   /&gt; ボタンを表示
	 * &lt;og:input type="image"    /&gt; イメージを表示してクリックした座標を返す
	 * &lt;og:input type="file"     /&gt; ファイル名入力ボックスを表示
	 * &lt;og:input type="hidden"   /&gt; 表示させずにサーバーに送信するデータ
	 *
	 * HTML5 で追加されたタイプ
	 * &lt;og:input type="search"			/&gt; 検索テキストの入力欄を作成する
	 * &lt;og:input type="tel"				/&gt; 電話番号の入力欄を作成する
	 * &lt;og:input type="url"				/&gt; URLの入力欄を作成する
	 * &lt;og:input type="email"			/&gt; メールアドレスの入力欄を作成する
	 * &lt;og:input type="datetime"			/&gt; UTC（協定世界時）による日時の入力欄を作成する
	 * &lt;og:input type="date"				/&gt; 日付の入力欄を作成する
	 * &lt;og:input type="month"			/&gt; 月の入力欄を作成する
	 * &lt;og:input type="week"				/&gt; 週の入力欄を作成する
	 * &lt;og:input type="time"				/&gt; 時間の入力欄を作成する
	 * &lt;og:input type="datetime-local"	/&gt; UTC（協定世界時）によらないローカル日時の入力欄を作成する
	 * &lt;og:input type="number"			/&gt; 数値の入力欄を作成する
	 * &lt;og:input type="range"			/&gt; レンジの入力欄を作成する
	 * &lt;og:input type="color"			/&gt; 色の入力欄を作成する
	 *
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5 対応
	 *
	 * @param   type 表示形式[text/password/checkbox/radio/submit/reset/button/image/file]、HTML5用 [search/tel/url/email/datetime/date/month/week/time/datetime-local/number/range/color]
	 */
	public void setType( final String type ) {
		String tp = getRequestParameter( type );

		if( check( "|" + tp + "|" , CHECK_TYPE ) ) {
			set( "type",tp );
		}
		else {
			String errMsg = "type 属性は、下記の中から選択してください。type=["
						 + tp + "  in [" + CHECK_TYPE + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【HTML】このフィールドに設定する値を指定します。
	 *
	 * @og.tag
	 * type属性の値が"radio"か"checkbox"以外の場合、この属性は省略可能です。
 	 *&lt;og:input name="PN" value="syokichi" /&gt;
	 *
	 * @param   value 初期値
	 */
	public void setValue( final String value ) {
		set( "value",getRequestParameter( value ) );
	}

	/**
	 * 【HTML】横幅を指定します(typeがtextまたはpasswordの場合は文字数で、それ以外はピクセル)。
	 *
	 * @og.tag
	 *  &lt;og:input name="PN" size="20" /&gt;
	 *
	 * @param   size 横幅
	 */
	public void setSize( final String size ) {
		set( "size",getRequestParameter( size ) );
	}

	/**
	 * 【HTML】最大入力文字数を指定します(typeがtextまたはpasswordの場合)。
	 *
	 * @og.tag
	 * 最大文字数はsize指定を超え得るが、超えた場合ユーザエージェントはスクロール機構を提供する必要があります。
	 *この属性の初期値は「無制限」です。
	 *&lt;og:input name="PN" maxlength="20" /&gt;
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 最大入力制限数に応じたサイズ自動生成を入れます。
	 * @og.rev 3.5.4.5 (2004/01/23) CSSで処理する場合のクラス属性は、maxlength が ５ 以下の場合とする。
	 * @og.rev 4.0.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
	 *
	 * @param   maxlength 最大入力文字数
	 */
	public void setMaxlength( final String maxlength ) {
		String mlen = nval( getRequestParameter( maxlength ),null ) ;
		set( "maxlength",mlen );
		if( get( "size" ) == null && mlen != null ) {
			// 4.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
			int size = Integer.parseInt( mlen );
			if( size > COLUMNS_MAXSIZE ) { size = COLUMNS_MAXSIZE; }
			set( "size",String.valueOf( size ) );
		}

		// 3.5.4.5 (2004/01/23) 追加
		if( mlen != null && mlen.length() == 1 ) {
			char ch = mlen.charAt(0);
			if( ch >= '1' && ch <= '5' ) {
				add( "class", "S0" + ch );	// 3.5.5.9 (2004/06/07) セパレータ引数付きのメソッドに変更
			}
		}
	}

	/**
	 * 【HTML】type が checkbox か radio のとき、初めから選択された状態で表示します("checked"のみ指定可)。
	 *
	 * @og.tag
	 *
	 * &lt;og:input name="PN" type="checkbox" checked="checked" /&gt;
	 *
	 * @param   ch 選択された状態にするかどうか[checked:選択状態/そのた:なにもしない]
	 */
	public void setChecked( final String ch ) {
		String checked = getRequestParameter( ch );
		if( "checked".equals( checked )) {
			set( "checked","checked" );
		}
	}

	/**
	 * 【HTML】type が image の場合、送信ボタンとして利用する画像のURLをセットします。
	 *
	 * @og.tag src セット
	 *
	 * @param   src 画像のURL
	 */
	public void setSrc( final String src ) {
		set( "src",getRequestParameter( src ) );
	}

	/**
	 * 【HTML】type が image の場合で、画像が表示できない場合の代替テキストをセットします。
	 *
	 * @og.tag alt セット
	 *
	 * @param   alt 代替テキスト
	 */
	public void setAlt( final String alt ) {
		set( "alt",getRequestParameter( alt ) );
	}

	/**
	 * 【HTML】type が file の場合に、MIMEタイプをCSV形式で複数指定出来ます。
	 *
	 * @og.tag accept セット
	 *
	 * @param   accept MIMEタイプ(CSV形式)
	 */
	public void setAccept( final String accept ) {
		set( "accept",getRequestParameter( accept ) );
	}

	/**
	 * 【HTML】type が image の場合、クライアントサイド・イメージマップのURLを指定します。
	 *
	 * @og.tag usemap セット
	 *
	 * @param   usemap クライアントサイド・イメージマップのURL
	 */
	public void setUsemap( final String usemap ) {
		set( "usemap",getRequestParameter( usemap ) );
	}

	/**
	 * 【HTML】type が image の場合に、サーバーサイド・イメージマップを使用する場合に指定します。
	 *
	 * @og.tag ismap セット
	 *
	 * @param   ismap サーバーサイド・イメージマップを使用する場合に指定
	 */
	public void setIsmap( final String ismap ) {
		set( "ismap",getRequestParameter( ismap ) );
	}

	/**
	 * 【TAG】テーブル形式の &lt;td&gt; タグを使用するかどうか[yes/no/false]を指定します(初期値:yes)。
	 *
	 * @og.tag
	 * 通常(初期値:yes)は、テーブル形式の&lt;td&gt; タグを使用して、ラベルとフィールドを整列させます。
	 * no の場合は、tdタグを使用せず、ラベルとフィールドを連続して出力します。
	 * false は、ラベルを出力せず、フィールドのみ出力します。
	 *  &lt;og:input name="PN" td="false" /&gt;
	 *
	 * @og.rev 2.0.0.8 (2002/10/09) yes/no/false で指定するように変更
	 *
	 * @param   flag [yes:tdタグを使用/no:ラベルとフィールド/false:フィールドのみ]
	 */
	public void setTd( final String flag ) {
		String td = nval( getRequestParameter( flag ),td_flag );

		if( check( "|" + td + "|" , "|yes|no|false|" ) ) {
			td_flag = td;
		}
		else {
			String errMsg = "td 属性は、下記の中から選択してください。td=["
						 + td  + "  in [|true|false|label|]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ラベルをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * シングルラベルの lbl 属性との違いは,ここではラベルを複数 カンマ区切りで
	 * 渡すことが可能であることです。これにより、"A,B,C" という値に対して、
	 * "Aのラベル表示,Bのラベル表示,Cのラベル表示" という具合に文字列を
	 * 再合成します。
	 * これは、SQL文のOrder By 句で、ソート順を指定する場合などに便利です。
	 *
	 *  &lt;og:orderBy columns="MKNMJP,MKCD,MKNMEN" lbls="MKNMJP,MKCD,MKNMEN" /&gt;
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	 *
	 * @param   lbls 複数ラベルID(カンマ区切り)
	 */
	public void setLbls( final String lbls ) {

		String[] array = getCSVParameter( lbls );

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append( getLabel( array[0] ) );
		for( int i=1; i<array.length; i++ ) {
			buf.append( "," );
			buf.append( getLabel( array[i] ) );
		}
//		setMsglbl( buf.toString() );
		setLbl( buf.toString() );		// 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	}

	/**
	 * 【TAG】フィールド部分の colspan を指定します。
	 *
	 * @og.tag
	 * 通常は、ラベルとフィールドが、別のtdで囲われます。しかし、場合によっては、
	 * フィールド部の長いカラムがある場合、上下のカラムとの位置関係がおかしくなり、
	 * ラベルとフィールドが離れて表示されるケースがあります。
	 * この場合、長いフィールド部を持つカラムに、colspan 属性を指定すると、
	 * その分のカラムを割り当てることにより、上下のカラムとの位置関係を
	 * 調節することが可能になります。
	 * 通常は、3 を指定して、フィールド、ラベル、フィールドの
	 * 組み合わせ部分に、長いフィールドが適用されるように使用します。
	 *
	 *  &lt;og:input name="PN" colspan="3" /&gt;
	 *
	 * @og.rev 3.0.1.4 (2003/03/17) colspan 属性を追加。
	 *
	 * @param   sp フィールド部分のcolspan
	 */
	public void setColspan( final String sp ) {
		colspan = nval( getRequestParameter( sp ),colspan );
		if( colspan.length() > 0 ) {
			colspan = " colspan=\"" + colspan + "\" ";
		}
	}

	/**
	 * 【TAG】入力カーソルを指定されたname属性のフィールドへ自動的に移動します。
	 *
	 * @og.tag
	 * JavaScript の onKeyup イベントを利用して、最大桁数まで入力すると、
	 * 次のフィールドに、自動的に飛ぶように設定します。
	 * 引数は、次に飛ばすフィールドの name 属性です。
	 * 実行するには、JavaScript の nextForm ファンクションが必要です。(default.js で指定)
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   name 次にフォーカスを移すタグのname属性
	 */
	public void setNextForm( final String name ) {
		nextForm = nval( getRequestParameter( name ),nextForm );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"					,VERSION			)
				.println( "td_flag"					,td_flag			)
				.println( "colspan"					,colspan			)
				.println( "nextForm"				,nextForm			)
				.println( "CHECK_TYPE"				,CHECK_TYPE			)
				.println( "HTML_LABEL_SEPARATOR"	,CLM				)
				.println( "HTML_COLUMNS_MAXSIZE"	,COLUMNS_MAXSIZE	)
				.println( "Other..."				,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
