/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.util.ErrMsg;

import java.sql.SQLData;
import java.sql.SQLInput;
import java.sql.SQLOutput;
import java.sql.SQLException;

/**
 * SQLData インターフェースを継承した システム変数の受け渡し用オブジェクトです。
 * 行番号情報と改廃コード[A:追加/C:変更/D:削除]を持っています。
 *
 * @og.group エラー処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBErrMsg implements SQLData {
	private String sql_type ;
	private ErrMsg errMsg   ;

	/**
	 *  デフォルトコンストラクター
	 */
	public DBErrMsg() {
		sql_type = null;
		errMsg   = null;
	}

	/**
	 *  すべての属性情報を指定して、新しい DBErrMsg オブジェクトを作成します。
	 *
	 * @param    type データベースタイプ文字列
	 * @param    no 行番号
	 * @param    keka 結果 0:正常 1:警告 2:異常
	 * @param    errCD メッセージID
	 * @param    arg0 メッセージの引数0
	 * @param    arg1 メッセージの引数1
	 * @param    arg2 メッセージの引数2
	 * @param    arg3 メッセージの引数3
	 * @param    arg4 メッセージの引数4
	 * @param    pg PG名
	 * @param    step ステップ名
	 */
//	public DBErrMsg( final String type,final int no,final int keka,final String errCD,
//						final String arg0,final String arg1,final String arg2,
//						final String arg3,final String arg4 ) {
	public DBErrMsg( final String type,final int no,final int keka,final String errCD,
			final String arg0,final String arg1,final String arg2,final String arg3,final String arg4,
			final String pg,final String step) {
		sql_type = type;
//		errMsg   = new ErrMsg( no,keka,errCD,arg0,arg1,arg2,arg3,arg4 );
		errMsg   = new ErrMsg( no,keka,pg,step,errCD,arg0,arg1,arg2,arg3,arg4 ); // 3.8.9.5 (2007/09/12)
	}

	/**
	 *  内部のエラーメッセージオブジェクトを返します。
	 *
	 * @return   エラーメッセージ
	 */
	public ErrMsg getErrMsg() {
		return errMsg;
	}

	// ============================================================
	// implements SQLData
	// ============================================================

	/**
	 *  ＳＱＬタイプの文字列を返します。
	 *
	 * @return    ＳＱＬタイプの文字列
	 * @throws SQLException ※ この実装からは SQLException は、throw されません。
	 */
	public String getSQLTypeName() throws SQLException {
		return sql_type;
	}

	/**
	 *  データベース内部より内部属性を取得し、オブジェクトを構築します。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) エラーメッセージ表示の、行番号がおかしい件の修正。
	 *
	 * @param	stream	ストリーム
	 * @param    typeName ＳＱＬタイプの文字列
	 * @throws SQLException データベースアクセスエラー
	 */
	public void readSQL( final SQLInput stream, final String typeName ) throws SQLException {
		sql_type = typeName;

		int    no    = stream.readInt()+1;		// PLSQLでは、引数そのままの番号をセットする。
		int    kekka = stream.readInt();
		String id    = stream.readString();
		String arg0 = stream.readString();
		String arg1 = stream.readString();
		String arg2 = stream.readString();
		String arg3 = stream.readString();
		String arg4 = stream.readString();
		String pg    = stream.readString(); // 3.8.9.5 (2007/09/12)
		String step  = stream.readString(); // 3.8.9.5 (2007/09/12)

//		errMsg   = new ErrMsg( no,kekka,id,arg0,arg1,arg2,arg3,arg4 );
		errMsg   = new ErrMsg( no,kekka,pg,step,id,arg0,arg1,arg2,arg3,arg4 );	 // 3.8.9.5 (2007/09/12)
	}

	/**
	 *  データベース内部に内部属性を設定します。
	 *
	 * @param	stream	ストリーム
	 * @throws SQLException データベースアクセスエラー
	 */
	public void writeSQL( final SQLOutput stream ) throws SQLException {
		stream.writeInt(    errMsg.getNo()    );
		stream.writeInt(    errMsg.getKekka() );
		stream.writeString( errMsg.getId()    );
		stream.writeString( errMsg.getArg(0) );
		stream.writeString( errMsg.getArg(1) );
		stream.writeString( errMsg.getArg(2) );
		stream.writeString( errMsg.getArg(3) );
		stream.writeString( errMsg.getArg(4) );
		stream.writeString( errMsg.getPg()    );	 // 3.8.9.5 (2007/09/12)
		stream.writeString( errMsg.getStep()  );	 // 3.8.9.5 (2007/09/12)
	}
}
