/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystem;

/**
 * JavaScript のツリー階層を持ったテーブル表示を行う、ツリーテーブル表示クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLTreeBOM extends ViewForm_HTMLTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0.0 (2005/08/31)" ;

	public static final String COLUMN_LEVEL_KEY = "COLUMN_LEVEL";

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.2.1 (2003/10/27) JavaScript 内のダブルコーテーションをシングルコーテーションに変更する。
	 * @og.rev 3.9.0.1 (2007/12/18) DBクラスを出力する。
	 *
	 * @param  stNo     表示開始位置
	 * @param  pgSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int stNo, final int pgSize )  {
		// このクラスでは、テーブル全データを使用します。
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		int startNo = 0;
		int pageSize = getRowCount();

		int lastNo = getLastNo( startNo, pageSize );

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		out.append( getHeader() );

		int level;
		boolean isFld;
		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for( int row=startNo; row<lastNo; row++ ) {
			// カラム==０は、レベルを指定する。
			level = Integer.parseInt( getValueLabel(row,0) );
			isFld = false;
			if( row+1<lastNo ) {
				int nextLevel = Integer.parseInt( getValueLabel(row+1,0) );
				isFld = level < nextLevel ;
			}
			out.append( getLevelScript( level,isFld ) );

			// カラム==０は、レベルを指定するので表示しない。
			for(int column = 1; column < clmCnt; column++) {
				if( isColumnDisplay( column ) ) {
					out.append("<td class=\""); // 3.9.0.1 (2007/12/18)
					out.append( getColumnDbType(column) );
					out.append("\"> ");
					out.append( getValueLabel(row,column) );
					out.append("</td>");
				}
			}
			out.append( "', '', 'gold')" );
			if( level != 0 ) {
				out.append( ")" );
			}
			out.append( HybsSystem.CR );
		}
		out.append( getFutter() );

		return out.toString();
	}

	/**
	 * DBTableModel から テーブルのヘッダータグ文字列を作成して返します。
	 * JavaScript の TreeBody では、JavaScriptに関連する定義もこのヘッダーに
	 * 含めます。
	 *
	 * @og.rev 3.5.2.1 (2003/10/27) JavaScript 内のダブルコーテーションをシングルコーテーションに変更する。
	 * @og.rev 3.9.0.1 (2007/12/18) 文字サイズ変更スクリプト対応のため、id="viewTable"を出力
	 *
	 * @return  テーブルのヘッダータグ文字列
	 */
	@Override
	protected String getHeader() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append("<table border=\"0\" cellspacing=\"2\" cellpadding=\"0\"  summary=\"bomTable\" id=\"viewTable\">"); // 3.9.0.1 (2007/12/18)
		buf.append( HybsSystem.CR );
		buf.append("<script type=\"text/javascript\">");
		buf.append( HybsSystem.CR );
		buf.append("<!--");
		buf.append( HybsSystem.CR );
		buf.append("aux0 = gFld('");
		// カラム==０は、レベルを指定するので表示しない。
		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for(int column = 1; column < clmCnt; column++) {
			if( isColumnDisplay( column ) ) {
				buf.append("<th>");
				buf.append( getColumnLabel(column) );
				buf.append("</th>");
			}
		}
		buf.append("', '', 'gold')");
		buf.append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * DBTableModel から テーブルのフッタータグ文字列を作成して返します。
	 * JavaScript の TreeBody では、JavaScriptに関連する定義もこのフッターに
	 * 含めます。
	 *
	 * @return  テーブルのフッタータグ文字列
	 */
	protected String getFutter() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append("initializeDocument()").append( HybsSystem.CR );
		buf.append("//-->").append( HybsSystem.CR );
		buf.append("</script>").append( HybsSystem.CR );
		buf.append("</table>").append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * 行のレベルに応じた JavaScript関数のヘッダー部分を返します。
	 *
	 * @og.rev 3.5.2.1 (2003/10/27) JavaScript 内のダブルコーテーションをシングルコーテーションに変更する。
	 *
	 * @param	lvl		ツリーのレベル
	 * @param	isFld	フォルダかどうか[true:フォルダ/false:最下層]
	 *
	 * @return	JavaScript関数のヘッダー部分
	 */
	private String getLevelScript( final int lvl,final boolean isFld ) {

		String auxX = "\taux" + ( lvl );
		String auxY = "aux" + ( lvl-1 );

		final String rtn ;
		if( isFld ) {
			rtn = auxX + " = insFld(" + auxY + ", gFld('";
		}
		else {
			rtn = "\tinsFld(" + auxY + ", gLnk('CONTENTS','";
		}

		return rtn;
	}
}
