/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.util.List;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.html.FormatterType;
import org.opengion.hayabusa.html.TableFormatter;
import org.opengion.hayabusa.html.ViewAjaxTreeTableParam;

/**
 * JavaScript のツリー階層を持ったテーブル表示を行う、ツリーテーブル表示クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ViewForm_HTMLAjaxTreeTable extends ViewForm_HTMLCustomTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.3.7.4 (2009/07/01)" ;

	private static final String JSP = HybsSystem.sys( "JSP" );

	private int[]			childSearchKeys = null;
	private String			childSearchJsp	= null;
	private String			levelClm		= null;
	private int				levelClmPos		= -1;
	private String			imgCollapsed    = null;
	private String			imgExpanded     = null;
	private String			imgNoSub        = null;
	private boolean			expandAll		= false; // 4.3.3.0 (2008/10/01)
	private int				childViewStartNo= -1; // 4.3.3.0 (2008/10/01)
	private int				expCtrlClmPos	= -1; // 4.3.5.0 (2008/02/01)

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition属性,childViewStartNo属性対応
	 * @og.rev 4.3.7.4 (2009/07/01) tbodyタグの入れ子を解消(FireFox対応)
	 *
	 * @param  strNo     表示開始位置
	 * @param  pageSize  表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int strNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		initParam();

		// 4.3.3.0 (2008/10/01) 子データ差分取得用
		int startNo = 0;
		if( childViewStartNo >= 0 )	{ startNo = childViewStartNo; }
		else						{ startNo = strNo; }

		int lastNo = getLastNo( startNo, pageSize );

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		if( headerFormat == null ) {
			makeDefaultFormat();
		}

		headerFormat.makeFormat( getDBTableModel() );

		out.append( getCountForm( startNo,pageSize ) );
		out.append( getHeader() );

		if( bodyFormatsCount == 0 ) {
			bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];
			bodyFormats[0] = headerFormat ;
			bodyFormatsCount ++ ;
		}
		else {
			for( int i=0; i<bodyFormatsCount; i++ ) {
				bodyFormats[i].makeFormat( getDBTableModel() );
			}
		}

		int bgClrCnt = 0;
		for( int row=startNo; row<lastNo; row++ ) {
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			for( int i=0; i<bodyFormatsCount; i++ ) {
				TableFormatter bodyFormat = bodyFormats[i];
				if( ! bodyFormat.isUse( row,getDBTableModel() ) ) { continue; }		// 3.5.4.0 (2003/11/25)
				out.append("<tbody").append( getBgColorCycleClass( bgClrCnt++,row ) );
				if( isNoTransition() ) { // 4.3.3.0 (2008/10/01)
					out.append( getHiddenRowValue( row ) );
				}
				out.append(">");	// 3.7.0.3 (2005/03/01)
				out.append( bodyFormat.getTrTag() );

				if( isNumberDisplay() ) {
					String ckboxTD = "<td" + bodyFormat.getRowspan() + ">";
					out.append( makeCheckbox( ckboxTD,row,0 ) );
				}

				int cl = 0;
				for( ; cl < bodyFormat.getLocationSize(); cl++ ) {
					String fmt = bodyFormat.getFormat(cl);
					int loc = bodyFormat.getLocation(cl);
					if( ! bodyFormat.isNoClass() && loc >= 0 ) {
						StringBuilder newtg = new StringBuilder( HybsSystem.BUFFER_LARGE );
						newtg.append("<td class=\"");
						newtg.append( getColumnDbType(loc) );
						newtg.append("\" ");
						String tdclass = newtg.toString();
						fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
					}
					out.append( fmt );
					if( loc >= 0 ) {
						if( levelClm != null && levelClm.equals( getDBColumn( loc ).getName() ) ) {
							out.append( getLvlClmTag( row ) );
						}
						else {
							switch( bodyFormat.getType(cl) ) {
								case '#' : out.append( getColumnLabel(loc) );		break;
								case '$' : out.append( getRendererValue(row,loc) );	break;
								case '!' : out.append( getValue(row,loc) );			break;
								default  : out.append( getValueLabel(row,loc) );	break;
							}
						}
					}
					else {
						out.append( bodyFormat.getSystemFormat(row,loc) );
					}
				}
				out.append( bodyFormat.getFormat(cl) );
				out.append("</tbody>").append( HybsSystem.CR );
			}
		}

		if( footerFormat != null ) {
			out.append( getTableFoot() );
		}

		out.append("</table>").append( HybsSystem.CR );

		out.append( getScrollBarEndDiv() );

		out.append( getParameterTag() );

		return out.toString();
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];

		bodyFormatsCount = 0;
		for( int i=0; i<list.size(); i++ ) {
			TableFormatter format = list.get( i );		// 4.3.3.6 (2008/11/15) Generics警告対応
			switch( format.getFormatType() ) {
				case TYPE_HEAD : headerFormat = format; break;
				case TYPE_BODY : bodyFormats[bodyFormatsCount++] = format; break;
				case TYPE_FOOT : footerFormat = format; break;
				default : String errMsg = "FormatterType の定義外の値が指定されました。";
				// 4.3.4.4 (2009/01/01)
				  throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * フォーマッターが設定されていない場合は、DBTableModelの情報からデフォルトの
	 * フォーマッターを作成します。
	 *
	 * @og.rev 4.3.3.6 (2008/11/15) columnDisplay,noDisplay対応
	 * @og.rev 4.3.5.0 (2008/02/01) 全展開コントロール用カラムへの対応
	 */
	private void makeDefaultFormat() {
		StringBuilder buf = new StringBuilder();
		String[] clms = getDBTableModel().getNames();
		buf.append( "<tr>" );
		for ( int i = 0; i < clms.length; i++ ) {
			if( isColumnDisplay( i ) && i != expCtrlClmPos ) { // 4.3.3.6 (2008/11/15) // 4.3.5.0 (2008/02/01)
				buf.append( "<td>[" + clms[i] + "]</td>" );
			}
		}
		buf.append( "</tr>" );

		TableFormatter formatter = new TableFormatter();
		formatter.setFormat( buf.toString() );
		formatter.setFormatType( FormatterType.TYPE_HEAD );

		headerFormat = formatter;
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return	フォーマットメソッドを使用できるか
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * 初期パラメーターを設定します。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 初期全展開の属性追加
	 * @og.rev 4.3.5.0 (2008/02/01) 全展開時の状態をコントロールするためのフラグを追加
	 */
	private void initParam() {
		String[] tmp	= StringUtil.csv2Array( getParam( ViewAjaxTreeTableParam.CHILD_SEARCH_KEYS, "" ) );
		childSearchKeys	= new int[tmp.length];
		for( int i=0; i<tmp.length; i++ ) {
			childSearchKeys[i] = getDBTableModel().getColumnNo( tmp[i] );
		}
		childSearchJsp	= getParam( ViewAjaxTreeTableParam.CHILD_SEARCH_JSP, "getChildTag.jsp" );
		levelClm		= getParam( ViewAjaxTreeTableParam.LVL_CLM_KEY, "LVL" );
		levelClmPos		= getDBTableModel().getColumnNo( levelClm );
		imgCollapsed    = getParam( ViewAjaxTreeTableParam.IMG_COLLAPSED, "collapsed.gif" );
		imgExpanded     = getParam( ViewAjaxTreeTableParam.IMG_EXPANDED, "expanded.gif" );
		imgNoSub        = getParam( ViewAjaxTreeTableParam.IMG_NO_SUB, "nosub.gif" );
		expandAll		= Boolean.valueOf( getParam( ViewAjaxTreeTableParam.EXPAND_ALL, "false" ) ); // 4.3.2.0 (2008/09/11)
		childViewStartNo= Integer.valueOf( getParam( ViewAjaxTreeTableParam.CHILD_VIEW_START_NO, "-1" ) ); // 4.3.3.0 (2008/10/01)
		String expCtrlClm	= getParam( ViewAjaxTreeTableParam.EXPAND_CONTROL_CLM_KEY, "EXPAND_CONTROL" ); // 4.3.5.0 (2008/02/01)
		expCtrlClmPos	= getDBTableModel().getColumnNo( expCtrlClm, false );
	}

	/**
	 * JavaScriptに渡すためのパラメータをhiddenタグで出力します。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 初期全展開対応
	 * @og.rev 4.3.5.0 (2008/02/01) 全展開時の状態をコントロールするためのフラグを追加
	 *
	 * @param row 行番号
	 *
	 * @return HTMLタグ
	 */
	private String getLvlClmTag( final int row ) {
		StringBuilder keys = new StringBuilder();
		keys.append( "command," + levelClm );
		for( int i=0; i<childSearchKeys.length; i++ ) {
			keys.append( "," ).append( getColumnName( childSearchKeys[i] ) );
		}

		StringBuilder vals = new StringBuilder();
		vals.append( "NEW," + getValue( row, levelClmPos ) );
		for( int i=0; i<childSearchKeys.length; i++ ) {
			vals.append( "," ).append( getValue( row, childSearchKeys[i] ) );
		}

		String imgsrc = null;
		StringBuilder clazz = new StringBuilder();
		clazz.append( "lvlctl unreplaceable" );
		if( expandAll ) { // 4.3.3.0 (2008/10/01)
			if( row == getRowCount() - 1
					|| Integer.parseInt( getValue( row, levelClmPos ) ) >= Integer.parseInt( getValue( row+1, levelClmPos ) ) ) {
				boolean isExp = expCtrlClmPos > -1 && StringUtil.nval( getValue( row, expCtrlClmPos ), false ) ;
				if( isExp ) {
					imgsrc = JSP + "/image/" + imgCollapsed;
				}
				else {
					imgsrc = JSP + "/image/" + imgNoSub;
					clazz.append( " fetched nosub" );
				}
			}
			else {
				imgsrc = JSP + "/image/" + imgExpanded;
				clazz.append( " fetched expanded" );
			}
		}
		else {
			imgsrc = JSP + "/image/" + imgCollapsed;
		}

		TagBuffer tag = new TagBuffer( "img" );
		tag.add( "class", clazz.toString() );
		tag.add( "src" , imgsrc );
		tag.add( "alt" , "Level " + getValue( row, levelClmPos ) );
		tag.add( "lvl" , getValue( row, levelClmPos ) );
		tag.add( "keys" , keys.toString() );
		tag.add( "vals" , vals.toString() );

		return getRendererValue( row, levelClmPos ) + tag.makeTag();
	}

	/**
	 * JavaScriptに渡すためのパラメーターをhiddenタグをして出力します。
	 *
	 * @return hiddenタグ
	 */
	private String getParameterTag() {
		StringBuilder buf = new StringBuilder();
		buf.append( XHTMLTag.hidden( ViewAjaxTreeTableParam.CHILD_SEARCH_JSP, childSearchJsp ) );
		buf.append( XHTMLTag.hidden( ViewAjaxTreeTableParam.IMG_COLLAPSED, JSP + "/image/" + imgCollapsed ) );
		buf.append( XHTMLTag.hidden( ViewAjaxTreeTableParam.IMG_EXPANDED, JSP + "/image/" + imgExpanded ) );
		buf.append( XHTMLTag.hidden( ViewAjaxTreeTableParam.IMG_NO_SUB, JSP + "/image/" + imgNoSub ) );
		return buf.toString();
	}
}
