/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;
import java.util.List;
import java.util.Locale;
import java.util.Map ;
import java.util.LinkedHashMap ;

import org.opengion.fukurou.util.HybsEntry;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableWriter をJSON形式で出力する為の実装クラスです。
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，JSON形式ファイルの出力機能を実現しています。
 *
 * 出力のJSON形式(JavaScript Object Notation)は、JavaScriptにおける
 * オブジェクトの表記法をベースとした軽量なデータ記述言語です。
 * このクラスでは、基本的にはすべてを文字列として処理しますので、
 * 数字や、true,false も、ダブルコーテーションでくくることになります。
 * ただし、null の場合は、null 表記になりますが、通常 空文字 "" になる場合もあります。
 * 1レコードのみの場合でも、配列要素として取り扱います。
 * UTF-8 でエンコードし、MIMEタイプはapplication/json、拡張子はjson とするのが一般的です。
 *
 * [
 *   { "カラム1":"値1"  ,  "カラム2":"値2"  , … } ,			１レコード目
 *   { "カラム1":"値1"  ,  "カラム2":"値2"  , … } ,			２レコード目
 *    ・・・・ 
 *   { "カラム1":"値1"  ,  "カラム2":"値2"  , … }				Ｎレコード目
 * ]
 *
 * writeTableParamタグで、key="JsonName" value="パラメータ名" を指定すると、
 * JSON形式で、配列をオブジェクトとしてまとめるパラメータを指定する事が可能です。
 *
 * {
 * パラメータ名:[
 *   { "カラム1":"値1"  ,  "カラム2":"値2"  , … } ,			１レコード目
 *   { "カラム1":"値1"  ,  "カラム2":"値2"  , … } ,			２レコード目
 *    ・・・・ 
 *   { "カラム1":"値1"  ,  "カラム2":"値2"  , … }				Ｎレコード目
 * ]
 * }
 *
 * writeTableParamタグで、key="LowerCase" value="true" を指定すると、
 * すべてのカラムを小文字で出力します。（データ受信側の都合を配慮）
 * 初期値は、false なので、基本的に大文字になります。
 *
 * @og.rev 5.6.0.3 (2012/01/24) 新規作成
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_JSON extends TableWriter_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	// 5.6.6.1 (2013/07/12) keys の整合性チェックを行います。
	private static final Map<String,String> keysMap = new LinkedHashMap<String,String>();

	private String  jsonName    = null;		// JSON形式で、配列をオブジェクトとしてまとめるパラメータを指定する場合に使う。
	private boolean toLowerCase = false;	// カラム名(=パラメータ名)を小文字にする場合、true にセットします。初期値は大文字のまま。

	static {
		keysMap.put( "JSONNAME"		, "JSON形式で、配列をオブジェクトとしてまとめる場合に使う"					);
		keysMap.put( "LOWERCASE"	, "カラム名(=パラメータ名)を小文字にする場合、true をセット(初期値:false)"	);
	}

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	public void writeDBTable( final PrintWriter writer )  {
		super.setHeaderSequence( "D" );
		super.writeDBTable( writer );
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * 出力のXML形式は、ORACLE XDK での出力ファイルと同じ形式ですので、直接データベースに
	 * 登録することができます。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) JSON形式のフォーマットを作成します。
	 * @og.rev 5.6.1.2 (2013/02/22) jsonName もダブルクオートで囲う。
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		int numberOfRows =  table.getRowCount();

		String[] names = table.getNames();
		if( toLowerCase ) {
			for( int i=0; i<names.length; i++ ) {
				names[i] = names[i].toLowerCase(Locale.JAPAN);
			}
		}

		if( jsonName != null ) {
			writer.println( "{" );
			writer.print( "\"" );			// 5.6.1.2 (2013/02/22) jsonName もダブルクオートで囲う。
			writer.print( jsonName );
			writer.print( "\":" );			// 5.6.1.2 (2013/02/22) jsonName もダブルクオートで囲う。
		}

		writer.println( "[" );

		for( int row=0; row<numberOfRows; row++ ) {
			if( row > 0 ) { writer.print( "," ); }			// 最初以後の連結は、, を使う
			writer.print( "{" );
			for( int i=0; i<numberOfColumns; i++ ) {
				int clm = clmNo[i];
				String val = table.getValue(row,clm);
				// \(バックスラッシュ) と "(ダブルクオート)のエスケープ
				if( val.contains( "\\" ) || val.contains( "\"" ) ) {
					StringBuilder sb = new StringBuilder();
					for( int j=0; j<val.length(); j++ ) {
						char ch = val.charAt( j );
						if( ch == '\\' || ch == '\"' ) { sb.append( '\\' ); }
						sb.append( ch );
					}
					val = sb.toString();
				}

				if( i > 0 ) { writer.print( "," ); }		// 最初以後の連結は、, を使う
				writer.print( '"' );
				writer.print( names[clm] );
				writer.print( "\":\"" );
				writer.print( val );
				writer.print( '"' );
			}
			writer.println( "}" );
		}

		writer.println( "]" );

		if( jsonName != null ) {
			writer.println( "}" );
		}
	}

	/**
	 * パラメーターリストをセットします。
	 * JSONのパラメータ名を指定します。
	 * 引数が、null の場合は、何もしません。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) JSONのパラメータ名を指定します。
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェックを行います。
	 *
	 * @param	listParam	HybsEntryリスト
	 */
	@Override
	public void setParam( final List<HybsEntry> listParam ) {
		if( listParam != null && ! listParam.isEmpty() ) {
			for( HybsEntry entry : listParam ) {
				String key = entry.getKey() ;		// 5.6.6.1 (2013/07/12) keys の整合性チェック
				checkParam( key , keysMap );

				String val = entry.getValue() ;		// 5.6.6.1 (2013/07/12) val を先に取得
				if( val != null && val.length() > 0 ) {
					// JSON形式で、配列をオブジェクトとしてまとめるパラメータを指定する場合に使う。
					if( "JsonName".equalsIgnoreCase( key ) ) {
						jsonName = val ;
					}
					// カラム名(=パラメータ名)を小文字にする場合、true にセットします。初期値はfalse:大文字のまま。
					else if( "LowerCase".equalsIgnoreCase( key ) ) {
						toLowerCase = Boolean.parseBoolean( val );
					}
				}
			}
		}
	}
}
