/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * HTMLAREA エディターは、カラムのデータをテキストエリアで編集し、
 * HTML文字をエスケープして登録する場合に使用するクラスです。
 *
 * エスケープする以外はTEXTAREAと同様です
 *
 * @og.rev 5.5.8.0 (2012/11/01) 新規作成
 * @og.group データ編集
 *
 * @version  5.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class Editor_HTMLAREA extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.8.0 (2012/11/01)" ;

	private  String		  rows1		 ;
	private  String		  rows2		 ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_HTMLAREA() {}

	/**
	 * コンストラクター。
	 *
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_HTMLAREA( final DBColumn clm ) {
		super( clm );
		String  disabled = clm.isWritable() ? null : "disabled" ;

		int r1 = clm.getTotalSize()/Integer.parseInt(size1) + 1;	// // 4.0.0 (2005/01/31) メソッド名変更
		if( r1 > 5 ) { rows1 = "5"; }
		else { rows1 = String.valueOf( r1 ); }

		int r2 = clm.getTotalSize()/Integer.parseInt(size2) + 1;	// // 4.0.0 (2005/01/31) メソッド名変更
		if( r2 > 5 ) { rows2 = "5"; }
		else { rows2 = String.valueOf( r2 ); }

		// 3.7.0.4 (2005/03/14) size に、"rows,cols" を指定できるように変更
		String param = StringUtil.nval( clm.getEditorParam(),clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			int st = param.indexOf( ',' );
			if( st > 0 ) {
				rows1 = param.substring( 0,st );
				rows2 = rows1 ;
				size1 = param.substring( st+1 );
				size2 = size1;
			}
		}

		attributes = new Attributes();
		attributes.set( "disabled" ,disabled );

		attributes.addAttributes( clm.getEditorAttributes() );
		attributes.add( "class" ,clm.getDbType() );		// 4.0.0 (2005/01/31)

		optAttr = attributes.get( "optionAttributes" );
		tagBuffer.add( XHTMLTag.textareaAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_HTMLAREA( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの編集用文字列
	 */
	@Override
	public String getValue( final String value ) {

		TagBuffer tag = new TagBuffer( "textarea" );
		tag.add( "name"    , name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , name ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( "cols"    , size1 );
		tag.add( "rows"    , rows1 );
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)
		tag.setBody( StringUtil.htmlFilter(value) );

		return tag.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 名前と行番号の区切り記号を "^" から "__" に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value ) {

		TagBuffer tag = new TagBuffer( "textarea" );
		String newName = name + HybsSystem.JOINT_STRING + row;
		// tag.add( "name"    , name + HybsSystem.JOINT_STRING + row );
		tag.add( "name"    , newName );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , newName );
		}
		tag.add( "cols"    , size2 );
		tag.add( "rows"    , rows2 );
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)
		tag.setBody( StringUtil.htmlFilter(value) );

		return tag.makeTag( row,StringUtil.htmlFilter(value) );
	}
}
