/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.XHTMLTag;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * ラジオボタンで表示順(Order By 句)を指定するタグです。
 *
 * name 属性値は、 "ORDER_BY" 固定です。
 * columns 属性は、カンマ区切りのカラム属性を設定します。
 * ロケールは、ユーザー情報の lang 属性を初期値で使用し、 セットされていない場合は、
 * リクエスト情報のロケールから取得します。
 *
 * @og.formSample
 * ●形式：&lt;og:orderBy columns="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:orderBy
 *       columns          ○【TAG】表示順対象のカラムをCSV形式で複数指定します(必須)。
 *       checked            【TAG】初めからチェックされた状態(checked)で表示します
 *       lbls               【TAG】ラベルをCSV形式で複数指定します
 *       checkedValue       【TAG】初めからチェックされた状態にする値を指定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:orderBy columns="PN,YOBI" checked="checked" /&gt;
 *    &lt;og:orderBy columns="YOBI,PN"   /&gt;
 *    &lt;og:orderBy columns="MKNM,YOBI" /&gt;
 *
 * @og.rev 3.1.4.0 (2003/04/18) 継承元を、CommonTagSupport に変更。
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class OrderByTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.2.0 (2010/11/01)" ;

	private static final long serialVersionUID = 522020101101L ;

	private String columns      = null;
	private String labels       = null;
	private String checked      = null;
	private String checkedValue = null;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.4.1 (2003/04/21) HTMLTagSupport の継承を廃止した為、doEndTag() を追加。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			jspPrint( makeTag() );
		}
		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		columns      = null;
		labels       = null;
		checked      = null;
		checkedValue = null;
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) radio ボタン等で、ラベルをクリックしても値をセットできるようにする。
	 *
	 * @return  入力用のテキストフィールドタグ
	 */
	protected String makeTag() {
		set( "type","radio" );
		set( "name","ORDER_BY" );
		set( "value",columns );
		if( labels == null ) { setLbls( columns ); }
		String lbls = getMsglbl();

		if( checkedValue != null && checkedValue.length() > 0 ) {
			if( checkedValue.equals( columns ) ) {
				set( "checked","checked" );
			}
		}
		else {
			set( "checked",checked );
		}

		if( get( "id" ) == null ) { set( "id",TaglibUtil.getTagId() ); }

		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		rtn.append( "<td>" );
		rtn.append( XHTMLTag.input( getAttributes() ) );
		rtn.append( "</td>" );
		rtn.append( "<td>" );
		rtn.append( "<label for=\"" );
		rtn.append( get( "id" ) );
		rtn.append( "\">" );
		rtn.append( lbls );
		rtn.append( "</label>" );
		rtn.append( "</td>" );

		return rtn.toString() ;
	}

	/**
	 * 【TAG】表示順対象のカラムをCSV形式で複数指定します。
	 *
	 * @og.tag 表示順対象のカラムを指定します。
	 *
	 * @param   columns カラム(CSV形式)
	 */
	public void setColumns( final String columns ) {
		this.columns = getRequestParameter( columns );
	}

	/**
	 * 【TAG】初めからチェックされた状態(checked)で表示します。
	 *
	 * @og.tag
	 * check 値が "checked" の場合のみ、実行されます。
	 *
	 * @param   ch "checked" 値
	 */
	public void setChecked( final String ch ) {
		checked = getRequestParameter( ch );
		if( "checked".equals( checked )) {
			checked = "checked";
		}
		else {
			checked = null;
		}
	}

	/**
	 * 【TAG】初めからチェックされた状態にする値を指定します。
	 *
	 * @og.tag
	 * チェックバリューが、自分自身のカラム名と同じであれば、チェックがつきます。
	 * 通常、checkedValue = "{&#064;ORDER_BY}" で、自動的にチェックされます。
	 *
	 * @param   ch チェックされた状態にする値
	 */
	public void setCheckedValue( final String ch ) {
		checkedValue = nval( getRequestParameter( ch ),checkedValue );
	}

	/**
	 * 【TAG】ラベルをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * ラベル(複数)を変更するときに、lbls属性を使います。
	 *
	 * シングルラベルの lbl 属性との違いは,ここではラベルを複数 カンマ区切りで
	 * 渡すことが可能であることです。これにより、"A,B,C" という値に対して、
	 * "Aのラベル表示,Bのラベル表示,Cのラベル表示" という具合に文字列を
	 * 再合成します。
	 * これは、SQL文のOrder By 句で、ソート順を指定する場合などに便利です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	 *
	 * @param   lbls 複数ラベルID(カンマ区切り)
	 */
	public void setLbls( final String lbls ) {
		labels = getRequestParameter( lbls ) ;

		String[] array = getCSVParameter( lbls );

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		buf.append( getLabel( array[0] ) );
		for( int i=1; i<array.length; i++ ) {
			buf.append( "," );
			buf.append( getLabel( array[i] ) );
		}
		setLbl( buf.toString() );		// 5.2.2.0 (2010/11/01) setMsglbl 廃止 ⇒ setLbl に置換え
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "columns"			,columns		)
				.println( "labels"			,labels			)
				.println( "checked"			,checked		)
				.println( "checkedValue"	,checkedValue	)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
