/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Attributes;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * 検索結果のカラム表示に対して様々な属性を付加(マーキング)するタグです(参照:viewMarker)。
 *
 * このタグは、カラム毎にマーキングするタグです。親タグとして、viewMarker を
 * 使用する必要があります。
 * このタグでは、BODY部に指定した値を、レンデラーとして使用します。通常の文字は、
 * そのままで、{&#064;XXXX}で指定した変数は、リクエスト値を設定します。
 * [XXXX]で指定した値は、検索結果のDBTableModelの値を行毎に割り当てます。
 * マーカー指定の有無(マーカーするかしないか)は、onMark属性と、markList属性で
 * 指定します。markList属性に指定の値に、onMark属性に設定した値が存在する場合、
 * マーカーされます。 このmarkList属性には、"1" と "true"が初期設定
 * されているため、onMark属性に"1" または "true"を指定すれば、全行マークされます。
 * また、どちらの属性も、{&#064;XXXX} や、[XXXX]変数が使用できます。[XXXX]変数では、
 * 行毎に、onMark属性や、markList属性を設定できる為、(通常はどちらか固定)
 * 行毎の マーカー指定の有無を指定できます。
 * [XXXX]変数でカラム名の先頭に$を付加した場合(例:[$XXXX])は、元の値がURLエンコード
 * されて返されます。
 * ※ <b>現段階では、viewMarker がカラム名をキーにcolumnMarkerオブジェクトを
 * 管理している為、行毎にマークの有無は指定できますが、マークを切り替える
 * 使い方が出来ません。</b>
 *
 * @og.formSample
 * ●形式：&lt;og:columnMarker column="･･･" ･･･ &gt; ･･･ &lt;/og:columnMarker &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:columnMarker
 *       column             【TAG】マーク処理を行うカラム名を指定します
 *       columns            【TAG】マーク処理を行うカラム名をカンマ区切り(CSV形式)で複数指定します
 *       onMark             【TAG】マークを処理する(true or 1)/処理しない(false or 0)の設定を指定します(初期値:true)
 *       markList           【TAG】処理するマークを含むような文字列を、"|"区切りの文字列で指定します(初期値:"true|TRUE|1")
 *       instrVals          【TAG】スペースで区切られた複数の値について、マークします
 *       useFormatDeco      【TAG】[$XXXX],[#XXXX]機能を有効にします(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:columnMarker&gt;
 *
 * ●使用例
 *     商品CD２(CDSYHN02)は赤字で表示する。
 *     商品CD３(CDSYHN03)は-----と表示する。
 *     &lt;og:viewMarker command="{&#064;command}"&gt;
 *         &lt;og:columnMarker column="CDSYHN02" onMark="true" &gt;
 *             &lt;font color='red'&gt;[VCDSYHN02]&lt;/font&gt;
 *         &lt;/og:columnMarker&gt;
 *         &lt;og:columnMarker column="CDSYHN03" onMark="[XXXX]" markList="[YYYY]" &gt;
 *             &lt;CENTER&gt;&lt;-----&lt;/CENTER&gt;
 *         &lt;/og:columnMarker&gt;
 *     &lt;/og:viewMarker&gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ColumnMarkerTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.3.0 (2013/04/01)" ;

	private static final long serialVersionUID = 563020130401L ;

	private String		column		= null;
	private String[]	columns		= null;
	private String		onMark		= "true";			// true または 1
	// 3.5.2.0 (2003/10/20)
	private String		markList	= "true|TRUE|1";	// true または 1
	private String		instrVals	= null;				// 3.8.8.1 (2007/01/06)

	private String		useFormatDeco = "false";		// 5.6.3.0 (2013/04/01) [$XXXX],[#XXXX]機能を有効にするかどうか

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.5.2.0 (2003/10/20) markList属性を追加
	 * @og.rev 3.8.8.1 (2007/01/06) instrVals属性を追加
	 * @og.rev 4.0.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
	 * @og.rev 5.6.3.0 (2013/04/01) useFormatDeco属性を追加（[$XXXX],[#XXXX]機能を有効にするかどうか）
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		ViewMarkerTag viewMarker = (ViewMarkerTag)findAncestorWithClass( this,ViewMarkerTag.class );
		if( viewMarker != null ) {
			String marker = getBodyString();

			set( "body"		,marker );
			set( "onMark"	,onMark );
			set( "markList"	,markList );
			set( "instrVals",instrVals );			// 3.8.8.1 (2007/01/06)
			set( "useFormatDeco",useFormatDeco );	// 5.6.3.0 (2013/04/01) [$XXXX],[#XXXX]機能を有効にするかどうか
			boolean flag = false;
			if( column != null && column.length() > 0 ) {
				// 4.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
				set( "column",column );
				viewMarker.addAttribute( getAttributes() );
				flag = true;
			}

			if( columns != null && columns.length > 0 ) {
				flag = true;
				for( int i=0; i<columns.length; i++ ) {
					String newMarker = StringUtil.replace( marker,"[$1]","[" + columns[i] + "]" );
					set( "body"  ,newMarker );
					// 4.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
					set( "column",columns[i] );
					viewMarker.addAttribute( new Attributes( getAttributes() ) );
				}
			}

			if( ! flag ) {
				String errMsg = "colum か、columns は、どちらか必ず必要です。";
				throw new HybsSystemException( errMsg );
			}
		}
		return SKIP_BODY ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 *
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.2.0 (2003/10/20) markList属性を追加
	 * @og.rev 3.8.8.1 (2007/01/06) instrVals属性を追加
	 * @og.rev 5.6.3.0 (2013/04/01) useFormatDeco属性を追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		column		= null;
		columns		= null;
		onMark		= "true";			// true または 1
		markList	= "true|TRUE|1";
		instrVals	= null;				// 3.8.8.1 (2007/01/06)
		useFormatDeco = "false";		// 5.6.3.0 (2013/04/01) [$XXXX],[#XXXX]機能を有効にするかどうか
	}

	/**
	 * 【TAG】マーク処理を行うカラム名を指定します。
	 *
	 * @og.tag
	 * このカラム名のTableModelに対して、マークを処理します。
	 *
	 * @param	clm カラム名
	 */
	public void setColumn( final String clm ) {
		column = getRequestParameter( clm );
	}

	/**
	 * 【TAG】マーク処理を行うカラム名をカンマ区切り(CSV形式)で複数指定します。
	 *
	 * @og.tag
	 * この複数のカラム名のTableModelに対して、 マークを処理します。
	 * カラム名は、カンマ区切りで複数指定することができます。その場合は、
	 * 指定のカラムに対して、すべて同一の処理を行います。
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 *
	 * @param	clms マーク処理を行うカラム名(CSV形式)
	 */
	public void setColumns( final String clms ) {
		columns = StringUtil.csv2Array( getRequestParameter( clms ) );
		if( columns.length == 0 ) { columns = null; }
	}

	/**
	 * 【TAG】マークを処理する(true or 1)/処理しない(false or 0)の設定を指定します(初期値:true)。
	 *
	 * @og.tag
	 * マークを処理する場合は、"true"(または "1")
	 * 処理しない場合は, "true以外"(または "0")をセットします。
	 * 初期値は、 "true"(マークを処理する)です。
	 * さらに、[カラム名] で、動的にカラムの値で、マークをする、しないを
	 * 選ぶ事が可能になります。値は、"true"(または "1") で、マークします。
	 * 追記 3.5.2.0 (2003/10/20)：markList属性に、処理対象文字列郡を指定できます。
	 * これにより、マークを処理するかどうかの判断が、true,1 以外にも使用できるようになりました。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) onMark に、[カラム名] の値をセットできるように修正。
	 *
	 * @param	flag マークの処理 [true,1:処理する/それ以外:処理しない]
	 */
	public void setOnMark( final String flag ) {
		onMark = nval( getRequestParameter( flag ),onMark );
	}

	/**
	 * 【TAG】処理するマークを含むような文字列を、"|"区切りの文字列で指定します(初期値:"true|TRUE|1")。
	 *
	 * @og.tag
	 * markListで与えられた文字列に、onMark文字列(大文字/小文字の区別あり)が
	 * 含まれていれば、処理します。
	 * 例えば、"A","B","C" という文字列が、onMark で指定された
	 * 場合に処理するようにしたい場合は、"A|B|C" をセットします。
	 * markList には、[カラム名]指定が可能です。
	 * 初期値は、 "true|TRUE|1"です。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規追加
	 *
	 * @param	list 処理するマーク(indexOf による含む/含まない判定)
	 */
	public void setMarkList( final String list ) {
		markList = nval( getRequestParameter( list ),markList );
	}

	/**
	 * 【TAG】スペースで区切られた複数の値について、マークします。
	 *
	 * @og.tag
	 * 最終的に作成された表示結果に対して、引数の文字列を含む箇所に、
	 * <span class="instr0">引数文字列</span> 文字列と置き換えます。
	 * 0 の部分は、引数文字列の現れた順番を指定します。
	 * これにより、部分検索された箇所のみにマークすることが可能です。
	 * 部分文字列は、スペースで区切り、それぞれ独立した形でマーク
	 * されますので、検索エンジン等で検索したキャッシュ表示のような
	 * 効果を持たすことが可能になります。
	 *
	 * @param	val スペースで区切られた複数の値
	 * @see		SqlAndTag#setInstrVals( String )
	 */
	public void setInstrVals( final String val ) {
		instrVals = nval( getRequestParameter( val ),instrVals );
	}

	/**
	 * 【TAG】[$XXXX],[#XXXX]機能を有効にするかどうか指定します(初期値:"false")。
	 *
	 * @og.tag
	 * カラムのフォーマット処理で、[$XXXX],[#XXXX]などの就職記号を付けると、
	 * データでない値を出力できます。
	 * [$XXXX] は、レンデラー処理された文字列を、[#XXXX]は、ラベルリソースを出力します。
	 * この、columnMarker のデフォルトでは、[$XXXX]は、URLエンコード処理された結果が
	 * 返されます。他の処理(viewタグのボディ部等)とは異なる処理ですが、互換性のため、
	 * [$XXXX]のレンデラー処理は、行われません。
	 * この、属性で、"true" を設定すれば、他のフォーマット処理と同じ処理が実行されます。
	 * 初期値は、互換性を考慮し、"false" になっています。
	 *
	 * @og.rev 5.6.3.0 (2013/04/01) 新規追加
	 *
	 * @param	val [$XXXX],[#XXXX]機能を有効にするかどうか(true:有効/false:無効)
	 */
	public void setUseFormatDeco( final String val ) {
		useFormatDeco = nval( getRequestParameter( val ),useFormatDeco );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "column"		,column		)
				.println( "columns"		,columns	)
				.println( "onMark"		,onMark		)
				.println( "markList"	,markList	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
