/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.FileUtil;

/**
 * 伝送要求に対して、ファイルまたはディレクトリをスキャンし、それに含まれる
 * ファイルの絶対パスのリストを取得します。
 *
 * 伝送定義マスタの読取対象は、スキャン対象のファイルまたはディレクトリを指定します。
 * 処理実行後は、正常終了した場合は、スキャンしたファイルを削除します。
 * 但し、読取パラメーターに"NODEL"を指定した場合、ファイルの削除は行われません。
 * また、エラーが発生した場合は読取パラメーターの設定に関わらずファイルの削除は
 * 行われません。
 *
 * 読取対象にディレクトリを指定した場合は、再起的にサブフォルダもスキャンされます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferRead_FILELIST implements TransferRead {

	// 更新(削除)対象のファイル名(配列)
	private String[] fileNames = null;

	/**
	 * ファイルまたはディレクトリをスキャンしファイルの絶対パスのリストを取得します。
	 *
	 * @og.rev 5.4.3.2 (2011/12/06) コピー中のファイル判定追加
	 * @og.rev 5.5.2.4 (2012/05/16) 配列を返す場合は、内部表現を暴露しないように、clone を返します。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 *
	 * @return ファイル一覧(配列)
	 */
	@Override
	public String[] read( final TransferConfig config, final Transaction tran ) {
		File file = new File( config.getReadObj() );
		if( !file.exists() ) {
			String errMsg = "スキャン対象のファイル/ディレクトリ。[FILE=" + file.getAbsolutePath() + "]";
			throw new RuntimeException( errMsg );
		}

		List<String> list = new ArrayList<String>();
		FileUtil.getFileList( file, false, list , false); // 5.4.3.2 コピー判定追加

		fileNames = list.toArray( new String[list.size()] );

		return fileNames.clone();
	}

	/**
	 * 更新(削除)対象のファイル名(配列)を返します。
	 *
	 * @og.rev 5.5.2.4 (2012/05/16) 配列を返す場合は、内部表現を暴露しないように、clone を返します。
	 *
	 * @return ファイル名(配列)
	 */
	public String[] getKeys() {
		String[] rtn = null ;
		if( fileNames != null ) { rtn = fileNames.clone(); }
		return rtn ;
	}

	/**
	 * 更新(削除)対象のファイル名(配列)をセットします。
	 *
	 * @og.rev 5.5.2.4 (2012/05/16) 参照の格納には、System.arraycopy を使います。
	 *
	 * @param keys ファイル名(配列)
	 */
	public void setKeys( final String[] keys ) {
		if( keys != null ) {
			int size = keys.length ;
			fileNames = new String[size];
			System.arraycopy( keys,0,fileNames,0,size );
		}
		else {
			fileNames = null;
		}
	}

	/**
	 * 読取した伝送データのヘッダーデータの状況を'2'(抜出済み)に更新します。
	 * 更新対象の通番NOについては、{@link #setKeys(String[])}で外部からセットすることもできます。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 * @see #setKeys(String[])
	 */
	@Override
	public void complete( final TransferConfig config, final Transaction tran ) {
		if( fileNames == null || fileNames.length == 0 ) { return; }
		// 読取パラメーターに"NODEL"が指定されている場合は、スキャンしたファイルを削除しない。
		if( "NODEL".equalsIgnoreCase( config.getReadPrm() ) ) { return; }

		for( String fileName : fileNames ) {
			File file = new File( fileName );
			if( !file.exists() ) {
				String errMsg = "ファイルが存在しません。[FILE=" + file.getAbsolutePath() + "]";
				throw new RuntimeException( errMsg );
			}

			boolean rtn = file.delete();
			if( !rtn ) {
				String errMsg = "ファイルの削除に失敗しました。[FILE=" + file.getAbsolutePath() + "]";
				throw new RuntimeException( errMsg );
			}
		}
	}

	/**
	 * (ここでは何もしません)
	 *
	 * @param config 伝送設定オブジェクト
	 * @param appInfo DB接続情報
	 */
	@Override
	public void error( final TransferConfig config, final ApplicationInfo appInfo ) {}
}
