/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.ChartDataset;
import org.opengion.hayabusa.db.DBTableModel;

import java.sql.SQLException;
import java.sql.Connection;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * BODY部に指定のSQLの検索結果をグラフ(円、棒、線)で表示するタグです。
 *
 * グラフ化には、JFreeChart (http://www.jfree.org/jfreechart/) を使用しています。
 * chartType 属性には、ChartFactoryクラスで定義された変換表に基づく、チャートの
 * 種類を指定します。このキーから、レンデラーやデータセットクラスを求めます。
 * それぞれのクラス毎に細かい設定値がありますが、初期レベルとしましては、主要チャートの
 * 主要属性のみサポートします。また、指定の chartType では使用しない属性を指定した場合でも
 * 単に無視するだけで、警告等はありませんので、ご注意ください。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。
 *
 * useTableData="false"(初期値)の場合は、BODY部に書かれた SQL 文を実行してデータを取得します。
 * useTableData="true" にすると、その前の query タグ等で作成した DBTableModel を利用して
 * グラフ化します。その場合は、BODY部の SQL 文は、無視されます。
 * 利用する DBTableModel は、tableId と scope 属性から取得します。
 *
 * @og.formSample
 * ●形式：&lt;og:chartDataset renderer="…" … /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:chartDataset
 *       chartType        ○【TAG】ChartDataset のデータタイプを指定します(必須)。
 *       dbid               【TAG】(通常は使いません)Datasetオブジェクトを作成する時のDB接続IDを指定します
 *       useTableData       【TAG】グラフ化するデータを DBTableModel から作成するかどうか(初期値:false)
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID (初期値:HybsSystem#TBL_MDL_KEY[=h_tblmdl])
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       baseLegend         【TAG】シリーズ単位の凡例 の表示可否を設定します
 *       valueLabel         【TAG】縦軸の表示名称を指定します
 *       markValues         【TAG】マーカーラインの設定値をカンマ区切り形式で複数指定します
 *       markColors         【TAG】マーカーラインの色をカンマ区切り形式で複数指定します
 *       markOverColors     【TAG】マーカーラインの超過時のShape色をカンマ区切り形式で複数指定します
 *       dynamicOCNo        【TAG】動的なマーカーラインの基準シリーズ番号を設定します(初期値:null)
 *       useMarkAnchor      【TAG】マーカーライン描画時に、その設定値を表示します
 *       lowerBound         【TAG】チャートの縦軸の最小値をセットします(初期値:自動計算)
 *       upperBound         【TAG】チャートの縦軸の最大値をセットします(初期値:自動計算)
 *       tickSize           【TAG】チャートの縦軸の目盛の幅をセットします(初期値:自動計算)
 *       useGradient        【TAG】バーチャートのグラデーション処理を行うかどうか[true/false]をセットします(初期値:false)
 *       shapesVisible      【TAG】ラインチャートのポイントを四角表示するかどうかを指定します(初期値:true)
 *       shapeColors        【TAG】データ毎にShapeを切り替える時の色の繰返しパターンを文字列配列で指定します
 *       shapeScale         【TAG】shapeの大きさを倍率指定で変更します(初期値:null)
 *       useDottedLine      【TAG】ラインチャートの線をドットラインにするかどうか[true/false]を指定します(初期値:false 線分)
 *       seriesColors       【TAG】複数チャート描画時のチャート色の繰返しパターンをCVS形式で複数指定します
 *       valueLabelsVisible 【TAG】Value(縦軸)のラベルを表示するかどうかを指定します(初期値:true[表示する])
 *       valueMarksVisible  【TAG】Value(縦軸)のマーカーを表示するかどうかを指定します(初期値:true[表示する])
 *       itemLabelVisible   【TAG】データの値(itemText)を表示するかどうか[true/false/last]を指定します(初期値:false[表示しない])
 *       useItemLabelSep    【TAG】データの値(itemText)の表示に、桁区切り文字を使用するかどうか[true/false]を指定します(初期値:true[使用する])
 *       valueInset         【TAG】縦軸の表示領域INSET値(double)を指定します
 *       barMaxWidth        【TAG】BOXチャートのバー幅の最大値(0.0～1.0)を指定します
 *       barItemMargin      【TAG】BOXチャートのバーアイテムのマージン(0.0～1.0)を指定します
 *       barWidth           【TAG】BOXチャートのバー幅(double)を指定します
 *       visibleLimit       【TAG】表示下限値(これ以下のデータは未表示)の値(double)を指定します
 *       domainMargin       【TAG】グラフの書き出し位置の調整比率を指定します
 *       timeFormatType     【TAG】時刻を表す場合の表現の仕方[E1/E2/E3/H1/H2/H3]を指定します(初期値:null)
 *       useVerticalLabels  【TAG】ラベルの表示向きを縦にするかどうか[false/true]を指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:chartDataset&gt;
 *
 * ●使用例
 *      &lt;og:chartCreate
 *          command     = "{&#064;command}"
 *          title       = "{&#064;title}"
 *          domainLabel = "{&#064;domainLabel}"
 *          width       = "{&#064;width}"
 *          height      = "{&#064;height}"
 *          rectangleEdge   = "{&#064;rectangleEdge}"
 *          plotOrientation = "{&#064;plotOrientation}"
 *          chartBackColor  = "{&#064;chartBackColor}"
 *          plotBackColor   = "{&#064;plotBackColor}"
 *          domainMarker    = "{&#064;domainMarker}"
 *          useMarkerLabel  = "{&#064;useMarkerLabel}" &gt;
 *              &lt;og:chartDataset
 *                  chartType   = "[Bar|LineAndShape|･･･]"      チャートの種類
 *                  baseLegend  = "[true|false]"                シリーズ単位の凡例 の表示可否
 *                  valueLabel  = "縦軸"                        縦軸のラベル
 *                  lowerBound  = "0"                           下限値
 *                  upperBound  = "5000"                        上限値
 *                  tickSize    = "1000"                        縦軸の目盛の幅
 *                  markValues  = "2500,2800"                   縦軸設定値(横棒)
 *                  markColors  = "RED,GREEN"                   縦軸設定ライン色
 *                  markOverColors = "BLUE,YELLOW,RED"          マーカーラインの超過時のShape色
 *                  dynamicOCNo   = "2"                         動的なマーカーラインの基準シリーズ番号
 *                  useMarkAnchor = "[true|false]"              縦軸設定値(横棒)を表示するかどうか
 *                  useGradient   = "[false|true]"              バーチャートのグラデーション処理
 *                  shapesVisible = "[true|false]"              ラインの四角表示
 *                  shapeColors   = "RED,GREEN,BLUE"            データ毎のShapeの色
 *                  shapeScale    = "0.8"                       shapeの大きさの倍率(double)
 *                  useDottedLine = "[false|true]"              点線使用有無
 *                  seriesColors  = "RED,GREEN,BLUE"            シリーズの色サイクル指定
 *                  valueLabelsVisible = "[true|false]"         縦軸のラベルの表示有無
 *                  valueMarksVisible  = "[true|false]"         縦軸のマーカーの表示有無
 *                  itemLabelVisible   = "[false|true|last]"    データの値の表示有無
 *                  useItemLabelSep    = "[true|false]"         データの桁区切り文字の使用有無
 *                  valueInset         = "15"                   縦軸の表示領域INSET値
 *                  barMaxWidth        = "[0.0～1.0]"           バー幅の最大値(0.0～1.0)
 *                  barItemMargin      = "[0.0～1.0]"           バーアイテムのマージン(0.0～1.0)
 *                  barWidth           = "10"                   バー幅の実幅(double)
 *                  visibleLimit       = "0"                    表示下限値(これ以下のデータは未表示))
 *                  domainMargin       = "1.0"                  グラフの書き出し位置の調整比率
 *                  useTableData       = "[false|true]"         SQLではなく、DBTableModelからグラフを作成するかどうか
 *                  timeFormatType     = "[1|2]"                時間軸の設定(00:00表示)
 *                  useVerticalLabels  = "[false|true]"         時間軸の設定(ラベルの縦書き)
 *                  tableId            = "AAA"                  useTableData="true"の場合のDBTableModelの取得先ID
 *                  scope              = "[session|request]"    useTableData="true"の場合のDBTableModelの取得先scope
 *              &gt;
 *                         {&#064;SQL}
 *              &lt;/og:chartDataset&gt;
 *      &lt;/og:chartCreate&gt;
 *
 *    複数のグラフを重ね合わせる場合は、chartDataset タグを chartCreate のBODY部に
 *    複数記述します。
 *      &lt;og:chartCreate
 *          title       = "{&#064;title}"
 *          domainLabel = "{&#064;domainLabel}"
 *          width       = "{&#064;width}"
 *          height      = "{&#064;height}" &gt;
 *          rectangleEdge   = "{&#064;rectangleEdge}" &gt;
 *          plotOrientation = "{&#064;plotOrientation}" &gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType1}"
 *                  valueLabel  = "{&#064;valueLabel1}"
 *                  lowerBound  = "{&#064;lowerBound1}"
 *                  upperBound  = "{&#064;upperBound1}"
 *                  markValues  = "{&#064;markValues1}"
 *                  markColors  = "{&#064;markColors1}"
 *              &gt;
 *                         {&#064;SQL1}
 *              &lt;/og:chartDataset&gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType2}"
 *                  valueLabel  = "{&#064;valueLabel2}"
 *                  lowerBound  = "{&#064;lowerBound2}"
 *                  upperBound  = "{&#064;upperBound2}"
 *                  markValues  = "{&#064;markValues2}"
 *                  markColors  = "{&#064;markColors2}"
 *              &gt;
 *                         {&#064;SQL2}
 *              &lt;/og:chartDataset&gt;
 *      &lt;/og:chartCreate&gt;
 *
 *    chartType属性 は、チャートの種類を指定する、固有IDです。
 *    [chartType属性]
 *      Area            , Bar               , Bar3D          , BoxAndWhisker    , CategoryStep  ,
 *      Gantt           , GroupedStackedBar , IntervalBar    , LayeredBar       , Level         ,
 *      LineAndShape    , Line3D            , MinMax         , StackedArea      , StackedBar    ,
 *      StackedBar3D    , WaterfallBar      , CyclicXYItem   , HighLow          , StackedXYArea ,
 *      StackedXYArea2  , StandardXYItem    , XYBubble       , XYDifference     , XYDot         ,
 *      XYError         , XYLine3D          , XYLineAndShape , XYStepArea       , XYStep        ,
 *      Meter           , MultiplePie       , Pie            , Pie3D            , Ring          ,
 *      SpiderWeb       , Thermometer
 *
 *    markColors属性 は、マーカーラインの色を指定します
 *    java.awt.Color クラスのstatic フィールド名の文字列で指定します。
 *    [renderer属性]
 *      BLACK   , BLUE   , CYAN , DARK_GRAY , GRAY  , GREEN , LIGHT_GRAY ,
 *      MAGENTA , ORANGE , PINK , RED       , WHITE , YELLOW
 *      #XXXXXX形式の16bitRGB表記
 *
 * @og.group 画面表示
 *
 * @version  0.9.0	2007/06/19
 * @author	 Nakamura
 * @since	 JDK1.4,
 */
public class ChartDatasetTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.2.1 (2012/05/07)" ;

	private static final long serialVersionUID = 552120120507L ;

	private transient ChartDataset chDataset = new ChartDataset();

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	dbid	= "DEFAULT";
	private String	dbid	= null;
	private String	sql		= null;

	// 5.3.0.0 (2010/12/01) DBTableModel を利用して、グラフを作成します。
	private boolean	useTableData	= false;		// 互換性の関係から
	private String	tableId			= HybsSystem.TBL_MDL_KEY;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) useTableDataによるデータ取得方法の機能追加
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.3.0.0 (2010/12/01) useTableDataによるデータ取得方法の機能追加
//		return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)

		if( useTableData ) {	// DBTableModel から取得する。
			DBTableModel table = (DBTableModel)getObject( HybsSystem.TBL_MDL_KEY );
			chDataset.makeDataset( table );

			return( SKIP_BODY );			// Body を評価しない
		}
		else {
			return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
		}
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) データベース検索処理を、doEndTag() メソッドから移動
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		sql = getBodyString();
		if( sql != null && sql.length() > 0 ) {
			sql = sql.trim();
		}

		// 5.3.0.0 (2010/12/01) doEndTag() メソッドから移動
		boolean errFlag = true;
//		Connection conn = ConnectionFactory.connection( dbid,getApplicationInfo() );
		Connection conn = null;
		try {
			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );
			chDataset.makeDataset( conn,sql );
			errFlag = false;		// エラーではない
		}
		catch( SQLException ex ) {
			String errMsg = "データセット作成時にエラーが発生しました。" + HybsSystem.CR
						+ " SQL=" + sql + HybsSystem.CR
						+ ex.getMessage();			// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
		}

		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.2.1.0 (2010/10/01) debugPrint() メソッド追加
	 * @og.rev 5.3.0.0 (2010/12/01) データベース検索処理を、doAfterBody() へ移動
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 5.2.1.0 (2010/10/01) debugPrint() メソッド追加

		// 理屈上、ChartCreateTag の存在チェックがここで行われるが、DB検索処理は、先に行っている。
		// つまり、無駄なQUERYが実行される可能性が出てきたが、開発段階のみの話であるのでここに残しておく。
		ChartCreateTag chartTag = (ChartCreateTag)findAncestorWithClass( this,ChartCreateTag.class );

		if( chartTag == null ) {
			String errMsg = "chartCreate タグが見つかりませんでした。";
			throw new HybsSystemException( errMsg );
		}

	 	// 5.3.0.0 (2010/12/01) doAfterBody() へ移動
//		boolean errFlag = true;
////		Connection conn = ConnectionFactory.connection( dbid,getApplicationInfo() );
//		Connection conn = null;
//		try {
//			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );
//			chDataset.makeDataset( conn,sql );
//			errFlag = false;		// エラーではない
//		}
//		catch( SQLException ex ) {
//			String errMsg = "データセット作成時にエラーが発生しました。" + HybsSystem.CR
//						+ " SQL=" + sql + HybsSystem.CR
//						+ ex.getMessage();			// 5.1.8.0 (2010/07/01) errMsg 修正
//			throw new HybsSystemException( errMsg,ex );
//		}
//		finally {
//			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
//			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
//		}

		chartTag.addChartDataset( chDataset );

		return EVAL_PAGE;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.3.0.0 (2010/12/01) useTableData,tableId 属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		chDataset	= new ChartDataset();
//		dbid		= "DEFAULT";
		dbid		= null;
		sql			= null;
		useTableData	= false;					// 5.3.0.0 (2010/12/01)
		tableId			= HybsSystem.TBL_MDL_KEY;	// 5.3.0.0 (2010/12/01)
	}

	/**
	 * 【TAG】(通常は使いません)Datasetオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag
	 * Datasetオブジェクトを作成する時のDB接続IDを指定します。
	 * これは、システムリソースで、DEFAULT_DB_URL 等で指定している データベース接続先
	 * 情報に、XX_DB_URL を定義することで、 dbid="XX" とすると、この 接続先を使用して
	 * データベースにアクセスできます。
	 *
	 * @param	id	データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = StringUtil.nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】ChartDataset のデータタイプを指定します。
	 *
	 * @og.tag
	 * チャートタイプ は、外部からチャートを指定するのに便利なように、キー化
	 * されています。このキーに基づいて、ChartFactory クラスの
	 * チャートタイプ変換表に基づいて、レンデラーや、データセットを作成します。
	 * 基本的には、チャートタイプ は、この、レンデラー名称から決定しています。
	 * <pre>
	 *  Area            , Bar               , Bar3D          , BoxAndWhisker    , CategoryStep  ,
	 *  Gantt           , GroupedStackedBar , IntervalBar    , LayeredBar       , Level         ,
	 *  LineAndShape    , Line3D            , MinMax         , StackedArea      , StackedBar    ,
	 *  StackedBar3D    , WaterfallBar      , CyclicXYItem   , HighLow          , StackedXYArea ,
	 *  StackedXYArea2  , StandardXYItem    , XYBubble       , XYDifference     , XYDot         ,
	 *  XYError         , XYLine3D          , XYLineAndShape , XYStepArea       , XYStep        ,
	 *  Meter           , MultiplePie       , Pie            , Pie3D            , Ring          ,
	 *  SpiderWeb       , Thermometer
	 * </pre>
	 *
	 * @param	chartType	チャートタイプ
	 * @see		org.jfree.chart.renderer.category.CategoryItemRenderer
	 * @see		org.jfree.chart.renderer.xy.XYItemRenderer
	 */
	public void setChartType( final String chartType ) {
		chDataset.setChartType( getRequestParameter( chartType ) );
	}

	/**
	 * 【TAG】グラフ化するデータを DBTableModel から作成するかどうか(初期値:false)。
	 *
	 * @og.tag
	 * useTableData="false"(初期値)の場合は、BODY部に書かれた SQL 文を実行してデータを取得します。
	 * useTableData="true" にすると、その前の query タグ等で作成した DBTableModel を利用して
	 * グラフ化します。その場合は、BODY部の SQL 文は、無視されます。
	 * 利用する DBTableModel は、tableId と scope 属性から取得します。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) 新規追加
	 *
	 * @param	flag	データをDBTableModelを利用する場合、true
	 */
	public void setUseTableData( final String flag ) {
		useTableData = StringUtil.nval( getRequestParameter( flag ),useTableData );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = StringUtil.nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】シリーズ単位の凡例 の表示可否を設定します。
	 *
	 * @og.tag
	 * シリーズ単位の凡例 の表示可否を設定します。
	 *
	 * @param	baseLegend	シリーズ単位の凡例 の表示可否
	 */
	public void setBaseLegend( final String baseLegend ) {
		boolean flag = StringUtil.nval( getRequestParameter( baseLegend ),true );
		chDataset.setBaseLegend( flag );
	}

	/**
	 * 【TAG】縦軸の表示名称を指定します。
	 *
	 * @og.tag
	 * 縦軸の表示名称を指定します。
	 *
	 * @param	valueLabel	縦軸の表示名称
	 */
	public void setValueLabel( final String valueLabel ) {
		chDataset.setValueLabel( getRequestParameter( valueLabel ) );
	}

	/**
	 * 【TAG】マーカーラインの設定値をカンマ区切り形式で複数指定します。
	 *
	 * @og.tag
	 * 指定の値にマーカーラインを引きます。
	 * 色は、setMarkColors( String ) で指定します。
	 * markColors と、個数をあわせて設定する必要があります。
	 * どちらかが、指定されていない場合は、マーカー表示されません。
	 * dynamicOCNo を使用する場合は、そのValues の並びの箇所に、
	 * "G"  という文字列を設定して下さい。
	 * 例：2000,G  基準１は、2000の直線、基準２は、グラフ という意味。
	 *
	 * @param	markValues	マーカーラインの設定値(CSV形式)
	 * @see		#setMarkColors( String )
	 */
	public void setMarkValues( final String markValues ) {
		String vals = StringUtil.nval( getRequestParameter( markValues ),null );
		if( vals != null ) {
			chDataset.setMarkValues( StringUtil.csv2Array( vals ) );
		}
	}

	/**
	 * 【TAG】マーカーライン描画時に、その設定値を表示します。
	 *
	 * @og.tag
	 * マーカーラインの縦軸の設定値(markValues)に、設定値をラベル表示します。
	 * 位置は、TextAnchor.BASELINE_LEFT 固定です。
	 * 初期値は、表示する(true)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param   useAnchor true:設定値を表示する。
	 */
	public void setUseMarkAnchor( final String useAnchor ) {
		boolean flag = StringUtil.nval( getRequestParameter( useAnchor ),true );
		chDataset.setUseMarkAnchor( flag );
	}

	/**
	 * 【TAG】マーカーラインの色をカンマ区切り形式で複数指定します。
	 *
	 * @og.tag
	 * マーカーラインとは、縦軸設定のラインの事で、縦軸の設定値(markValues)に
	 * 横方向にラインを付与します。このラインの色を、カンマ区切り形式で
	 * 指定します。
	 * markValues と、あわせて設定する必要があります。
	 * 個数が異なる場合は、markValues が優先され、無指定の場合は、Color.REDで
	 * 表示されます。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * <pre>
	 *      BLACK   , BLUE   , CYAN , DARK_GRAY , GRAY  , GREEN , LIGHT_GRAY ,
	 *      MAGENTA , ORANGE , PINK , RED       , WHITE , YELLOW , (PURPLE)
	 *      #XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * </pre>
	 *
	 * @param	markColors	マーカーラインの色(CSV形式)
	 * @see		java.awt.Color#BLACK
	 * @see		#setMarkValues( String )
	 */
	public void setMarkColors( final String markColors ) {
		String clrs = StringUtil.nval( getRequestParameter( markColors ),null );
		if( clrs != null ) {
			chDataset.setMarkColors( StringUtil.csv2Array( clrs ) );
		}
	}

	/**
	 * 【TAG】マーカーラインの超過時のShape色をカンマ区切り形式で複数指定します。
	 *
	 * @og.tag
	 * HybsLine でのみ使用可能です。
	 * マーカーラインを使用する場合に、そのラインを超えない色、超える色を指定します。
	 * この色の指定は、マーカーラインの色やマーカーラインの設定値の個数＋１に
	 * なります。つまり、色１：設定値１：色２：設定値２：色３ となります。
	 * 色１は、設定値１より小さい場合に使用され、色２は、設定値１より大きく、
	 * 設定値２より小さい場合、色３は、設定値２より大きい場合になります。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * markValues と、あわせて設定する必要があります。
	 * どちらかが、指定されていない場合は、マーカー表示されません。
	 * <pre>
	 *      BLACK   , BLUE   , CYAN , DARK_GRAY , GRAY  , GREEN , LIGHT_GRAY ,
	 *      MAGENTA , ORANGE , PINK , RED       , WHITE , YELLOW , (PURPLE)
	 *      #XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * </pre>
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	overColors	マーカーラインの超過時のShape色(CSV形式)
	 * @see		java.awt.Color#BLACK
	 */
	public void setMarkOverColors( final String overColors ) {
		String clrs = StringUtil.nval( getRequestParameter( overColors ),null );
		if( clrs != null ) {
			chDataset.setMarkOverColors( StringUtil.csv2Array( clrs ) );
		}
	}

	/**
	 * 【TAG】動的なマーカーラインの基準シリーズ番号を設定します(初期値:null)。
	 *
	 * @og.tag
	 * 動的なマーカーラインを使用する場合は、基準値となるシリーズ番号を指定します。
	 *
	 * マーカーラインの最下位閾値に相当します。これは、グラフ化されますが、
	 * Shape は自動的に削除されます。
	 * この設定を使用する場合、最初のデータには、必ずShapeが付きます。それ以外の
	 * データに、Shape を付けるかどうかは、shapesVisible 属性で指定します。
	 * この線の色は、markColors で指定した、最下位の色になります。また、
	 * markValues で指定した、最下位の値は、使用されません。ただし、色指定の
	 * 関係上、設定しておく必要があります。
	 * また、isValueVisible == true で、設定値の値表示を行う場合も、最下位の
	 * 値は表示しないようにします。
	 * 初期値は、使用しない(null)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 * @og.rev 4.1.1.0 (2008/02/19) setUseDynamicOC ⇒ setDynamicOCNo
	 *
	 * @param   dynamicOCNo true:動的なマーカーラインの基準シリーズ番号
	 */
	public void setDynamicOCNo( final String dynamicOCNo ) {
		int ocNo = StringUtil.nval( getRequestParameter( dynamicOCNo ),-1 );
		chDataset.setDynamicOCNo( ocNo );
//		boolean flag = StringUtil.nval( getRequestParameter( useDynamicOC ),false );
//		chDataset.setUseDynamicOCNo( flag );
	}

	/**
	 * 【TAG】チャートの縦軸の最小値をセットします(初期値:自動計算)。
	 *
	 * @og.tag
	 * 何も指定しない場合は、データの最小値から、自動的に計算します。
	 *
	 * @param	lowerBound	縦軸の最小値
	 */
	public void setLowerBound( final String lowerBound ) {
		chDataset.setLowerBound( getRequestParameter( lowerBound ) );
	}

	/**
	 * 【TAG】チャートの縦軸の最大値をセットします(初期値:自動計算)。
	 *
	 * @og.tag
	 * 何も指定しない場合は、データの最大値から、自動的に計算します。
	 *
	 * @param	upperBound	縦軸の最大値
	 */
	public void setUpperBound( final String upperBound ) {
		chDataset.setUpperBound( getRequestParameter( upperBound ) );
	}

	/**
	 * 【TAG】チャートの縦軸の目盛の幅をセットします(初期値:自動計算)。
	 *
	 * @og.tag
	 * 何も指定しない場合は、データの範囲から、自動的に計算します。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param   size 縦軸のチック(目盛)の幅
	 */
	public void setTickSize( final String size ) {
		chDataset.setTickSize( getRequestParameter( size ) );
	}

	/**
	 * 【TAG】バーチャートのグラデーション処理を行うかどうか[true/false]をセットします(初期値:false)。
	 *
	 * @og.tag
	 * 通常のバーチャートは、単一色表示で表されますが、これにグラデーション効果を
	 * 付加するかどうかを指定します。
	 * 通常のバーチャートが、少しきれいに見えます。
	 * 初期値は、false(使用しない)です。
	 *
	 * @param	useGradient	グラデーション処理 [true:する/false:しない]
	 */
	public void setUseGradient( final String useGradient ) {
		boolean flag = StringUtil.nval( getRequestParameter( useGradient ),false );
		chDataset.setUseGradient( flag );
	}

	/**
	 * 【TAG】ラインチャートのポイントを四角表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * ラインチャートは、通常、線分で表され、各ポイントについても、線分で接続されます。
	 * shapesVisible を true に設定すると、各ポイントが、線上に四角く表示され、
	 * そのポイントの位置を、容易に判断出来るようになります。
	 * 初期値は、true( LineAndShape )です。
	 *
	 * @param	shapesVisible	ポイントを四角表示するかどうか
	 */
	public void setShapesVisible( final String shapesVisible ) {
		boolean flag = StringUtil.nval( getRequestParameter( shapesVisible ),true );
		chDataset.setShapesVisible( flag );
	}

	/**
	 * 【TAG】データ毎にShapeを切り替える時の色の繰返しパターンを文字列配列で指定します。
	 *
	 * @og.tag
	 * HybsLine でのみ使用可能です。
	 * これは、データそのものが、繰返し性のある場合に、その繰返し性に対応した
	 * 形状のShape を表示させる場合に使用します。
	 * 繰返しShapeの形状は、JFreeChart のシリーズ毎の繰返し標準形状を使用します。
	 * 現在のバージョンでは、１０個までの繰返しに対応可能です。
	 * 繰返し色を、指定した分だけ、順に使用されていきます。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	colors	データ毎の色の繰返しパターン(文字列配列)
	 * @see		java.awt.Color#BLACK
	 */
	public void setShapeColors( final String colors ) {
		String clrs = StringUtil.nval( getRequestParameter( colors ),null );
		if( clrs != null ) {
			chDataset.setShapeColors( StringUtil.csv2Array( clrs ) );
		}
	}

	/**
	 * 【TAG】繰返しshapeの開始番号を指定します(初期値:0)。
	 *
	 * @og.tag
	 * ラインチャートのShape(各グラフのポイントのマーカー)は、通常は、
	 * シリーズ毎に自動変更されます。
	 * これの変更開始順の開始番号を指定する事で、Shapeの開始形状を指定できます。
	 * 初期値(0) では、■、●、▲、◆、横長方形、▼、楕円、右三角、縦長方形、
	 * 左三角、という繰返しになります。
	 * これの、開始形状を指定することで、●から始めたり、▲から始める事が
	 * 可能になります。
	 * 初期値は、0:■です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param   startNo 繰返しshapeの開始番号
	 */
//	public void setShapeStartNo( final String startNo ) {
//		int no = StringUtil.nval( getRequestParameter( startNo ),0 );
//		chDataset.setShapeStartNo( no );
//	}

	/**
	 * 【TAG】shapeの大きさを倍率指定で変更します(初期値:null)。
	 *
	 * @og.tag
	 * ラインチャートのShape(各グラフのポイントのマーカー)の大きさは、通常は、
	 * 自動設定されます。
	 * この大きさを、倍率指定で、変更可能です。
	 * 指定は、double 型です。
	 * 初期値は、null は、スケール変更しません(自動設定のままの大きさ)
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param	scale	shapeの大きさの倍率
	 */
	public void setShapeScale( final String scale ) {
		String shapeScale = StringUtil.nval( getRequestParameter( scale ),null );
		if( shapeScale != null ) {
			chDataset.setShapeScale( shapeScale );
		}
	}

	/**
	 * 【TAG】ラインチャートの線をドットラインにするかどうか[true/false]を指定します(初期値:false[線分])。
	 *
	 * @og.tag
	 * ラインチャートは、通常、線分で表されます。
	 * これに、点線で表すことで、グラフの違いを、色だけでなく形状でも識別
	 * しやすくすることが可能です。
	 * 初期値は、false(線分)です。
	 *
	 * @param	useDottedLine	ドットラインにするかどうか [true:点線/false:線分]
	 */
	public void setUseDottedLine( final String useDottedLine ) {
		boolean flag = StringUtil.nval( getRequestParameter( useDottedLine ),false );
		chDataset.setUseDottedLine( flag );
	}

	/**
	 * 【TAG】複数チャート描画時のチャート色の繰返しパターンをCVS形式で複数指定します。
	 *
	 * @og.tag
	 * 通常、複数のチャートを同時に表示させる場合は、縦軸が共通であれば、
	 * １回のSelect分で複数データを取得します。
	 * この、データをシリーズと呼んでおり、これを区別する為に、色を分けます。
	 * 初期値は、JFreeChart が自動で割り振ります。
	 * これを、外部からCVS形式で、カンマ区切りで色コードを指定します。
	 * 指定データが多い場合は、多い分の色は使用されません。少ない場合は、
	 * 順番に繰り返して使用されます。
	 * 例えば、１色だけ指定した場合は、すべてのシリーズが同じ色で表されます。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 *
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @param	colors	チャート色の繰返しパターン(CVS形式)
	 * @see		java.awt.Color#BLACK
	 */
	public void setSeriesColors( final String colors ) {
		String clrs = StringUtil.nval( getRequestParameter( colors ),null );
		if( clrs != null ) {
			chDataset.setSeriesColors( StringUtil.csv2Array( clrs ) );
		}
	}

	/**
	 * 【TAG】Value(縦軸)の線を表示するかどうかを指定します(初期値:true[表示する])。
	 *
	 * @og.tag
	 * ValueAxis にて設定される、縦軸情報の、罫線を表示するかどうか指定します。
	 * 初期値は、true(表示する)です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	lineVisible	Value(縦軸)の線を表示するかどうか
	 */
//	public void setValueLineVisible( final String lineVisible ) {
//		boolean flag = StringUtil.nval( getRequestParameter( lineVisible ),true );
//		chDataset.setValueLineVisible( flag );
//	}

	/**
	 * 【TAG】Value(縦軸)のラベルを表示するかどうかを指定します(初期値:true[表示する])。
	 *
	 * @og.tag
	 * ValueAxis にて設定される、縦軸情報の、ラベルを表示するかどうか指定します。
	 * 初期値は、true(表示する)です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	labelsVisible	Value(縦軸)のラベルを表示するかどうか
	 */
	public void setValueLabelsVisible( final String labelsVisible ) {
		boolean flag = StringUtil.nval( getRequestParameter( labelsVisible ),true );
		chDataset.setValueLabelsVisible( flag );
	}

	/**
	 * 【TAG】Value(縦軸)のマーカーを表示するかどうかを指定します(初期値:true[表示する])。
	 *
	 * @og.tag
	 * ValueAxis にて設定される、縦軸情報の、マーカーを表示するかどうか指定します。
	 * 初期値は、true(表示する)です。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	marksVisible	Value(縦軸)のマーカーを表示するかどうか
	 */
	public void setValueMarksVisible( final String marksVisible ) {
		boolean flag = StringUtil.nval( getRequestParameter( marksVisible ),true );
		chDataset.setValueMarksVisible( flag );
	}

	/**
	 * 【TAG】データの値(itemText)を表示するかどうか[true/false/last]を指定します(初期値:false[表示しない])。
	 *
	 * @og.tag
	 * CategoryItemRenderer 関連のグラフの設定値をグラフ上に表示するかどうかを
	 * 指定します。
	 * true に設定した場合、通常の場合は、すべてのシリーズにラベル表示されます。
	 * false に設定すると、表示されません。
	 * last を設定すると、各シリーズの最後の値のみ表示されます。
	 * ChartCreate クラスに、seriesPikup が設定されている場合は、指定のシリーズ
	 * のみの設定値を表示し、他の値は、表示しません。
	 * 同様に、dynamicOCNo が指定されている場合(動的なマーカーライン)
	 * 最下位のシリーズは、閾値として使用されるため、設定値は表示されません。
	 * ラベルの表示位置は、表示する線グラフの傾きに応じてラベルの表示場所を
	 * 変えます。山形、右坂、谷形、左坂 に応じて、上中、下右、下中、上右 に
	 * 位置を設定します。右にずらすのは、10 ピクセル固定です。
	 * 初期値は、false(表示しない)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 * @og.rev 4.1.2.0 (2008/03/12) 文字列のまま、渡すように変更
	 *
	 * @param	itemLabelVisible	データの値の表示のさせ方(true,false,last)
	 */
	public void setItemLabelVisible( final String itemLabelVisible ) {
//		boolean flag = StringUtil.nval( getRequestParameter( itemLabelVisible ),false );
		String flag = StringUtil.nval( getRequestParameter( itemLabelVisible ),"false" );
		chDataset.setItemLabelVisible( flag );
	}

	/**
	 * 【TAG】データの値(itemText)の表示に、桁区切り文字を使用するかどうか[true/false]を指定します(初期値:true[使用する])。
	 *
	 * @og.tag
	 * itemLabelVisible=true 時に、表示されるデータ値ラベルで、
	 * NumberFormat していますが、３桁区切り文字(123,456,789.0) の
	 * 区切り記号を表示するかどうかを指定します。
	 * true を指定すると、表示します。false では、表示しません。
	 * 初期値は、true(使用する)です。
	 *
	 * @og.rev 4.1.2.0 (2008/03/12) 新規追加
	 *
	 * @param	used	データの桁区切り文字の使用有無
	 */
	public void setUseItemLabelSep( final String used ) {
		boolean useItemLabelSep = StringUtil.nval( getRequestParameter( used ),true );
		chDataset.setUseItemLabelSep( useItemLabelSep );
	}

	/**
	 * 【TAG】縦軸の表示領域INSET値(double)を指定します。
	 *
	 * @og.tag
	 * 縦軸文字表示領域(NumberAxis)の幅の追加値を設定します。
	 * これは、通常の表示領域の左側にスペースを挿入します。
	 * あくまで、追加する値なので、文字の長さは含まれません。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	inset	Value(縦軸)の表示領域INSET値
	 */
	public void setValueInset( final String inset ) {
		String valueInset = StringUtil.nval( getRequestParameter( inset ),null );
		chDataset.setValueInset( valueInset );
	}

	/**
	 * 【TAG】BOXチャートのバー幅(double)を指定します。
	 *
	 * @og.tag
	 * BOXチャートのバー幅(double)を指定します。
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 棒グラフのバー幅指定は、直接、CategoryItemRendererState に設定しています。
	 * 通常は、barMaxWidth(0.0～1.0)とbarItemMargin(0.0～1.0)を用いて比率で指定します。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	width	BOXチャートのバー幅(double)
	 */
	public void setBarWidth( final String width ) {
		String barWidth = StringUtil.nval( getRequestParameter( width ),null );
		chDataset.setBarWidth( barWidth );
	}

	/**
	 * 【TAG】グラフの書き出し位置のオフセットを指定します(初期値:0)。
	 *
	 * @og.tag
	 * グラフを重ねると、同じ位置から書き出しが始まります。
	 * このパラメータを指定する事で、グラフを重ねた場合に、少しずつ
	 * ずらして書き始める事が可能になります。
	 * これは、シリーズを、描画するときに、棒グラフの幅を広げる事で
	 * 同様の効果を得る事が出来ますが、検索条件等が異なる場合には、
	 * シリーズ化できませんので、こちらのオフセットを使用します。
	 * 初期値は、0(書き出し位置は標準のまま)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param	chartOffset	書き出し位置のオフセット
	 */
//	public void setChartOffset( final String chartOffset ) {
//		String offset = StringUtil.nval( getRequestParameter( chartOffset ),null );
//		if( offset != null ) {
//			chDataset.setChartOffset( Double.parseDouble(offset) );
//		}
//	}

	/**
	 * 【TAG】BOXチャートのバー幅の最大値(0.0～1.0)を指定します。
	 *
	 * @og.tag
	 * BOXチャートのバー幅の比率の最大値を指定します。
	 * 表示領域を１として小数点以下の数値で棒の幅を設定します。
	 * 設定した幅に無理がある時は適当なサイズに調整されます。
	 * (小さくしたときには棒が線のようになる)
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	barWidth	BOXチャートのバー幅の最大値(0.0～1.0)
	 */
	public void setBarMaxWidth( final String barWidth ) {
		String maxBarWidth = StringUtil.nval( getRequestParameter( barWidth ),null );
		chDataset.setBarMaxWidth( maxBarWidth );
	}

	/**
	 * 【TAG】BOXチャートのバーアイテムのマージン(0.0～1.0)を指定します。
	 *
	 * @og.tag
	 * BOXチャートのバーアイテムのマージンの比率を指定します。
	 * 棒の間を表示領域を１として小数点以下の数値で幅を設定します。
	 * 無理がある時は適当なサイズに調整されます。
	 * barMaxWidth より優先されます。
	 * 設定されるのは、org.jfree.chart.renderer.category.BarRenderer を
	 * 使用した描画のみです。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param	margin	BOXチャートのバーアイテムのマージン(0.0～1.0)
	 */
	public void setBarItemMargin( final String margin ) {
		String itemMargin = StringUtil.nval( getRequestParameter( margin ),null );
		chDataset.setBarItemMargin( itemMargin );
	}

	/**
	 * 【TAG】表示下限値(これ以下のデータは未表示)の値(double)を指定します。
	 *
	 * @og.tag
	 * HybsLine でのみ使用可能です。
	 * この設定値以下のデータは、存在しない扱いとします。
	 * Lineを引くとき、このデータと、存在しているデータ間にラインは引かれません。
	 * 何も指定しない場合は、設定しません。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param	limit	表示下限値(これ以下のデータは未表示)
	 */
	public void setVisibleLimit( final String limit ) {
		String visibleLimit = StringUtil.nval( getRequestParameter( limit ),null );
		chDataset.setVisibleLimit( visibleLimit );
	}

	/**
	 * 【TAG】グラフの書き出し位置の調整比率を指定します。
	 *
	 * @og.tag
	 * HybsStackedBar でのみ使用可能です。
	 * グラフを描画する場合の、書き出し位置を少しずらします。
	 * これは、グラフの幅に対して、比率で指定します。
	 * 0.0(初期値)の場合は、初期描画位置である、CategoryAnchor.Middle と
	 * 同じ箇所から、書き出されます。
	 * 1.0 の場合、中心から、グラフ幅の半分が加算され、END位置に寄ります。
	 * 同様に、-1.0 の場合は、グラフ幅の半分が減算され、START 位置になります。
	 * つまり、中心から、グラフ幅の半分単位で、前方/後方にずらす事が出来ます。
	 *   書き出し位置 ＝ 中心(Middle) + (domainMargin)＊幅/２
	 * 初期値は、0.0(真ん中:MIDDLE)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/14) 新規追加
	 *
	 * @param   margin double グラフの書き出し位置の調整比率
	 */
	public void setDomainMargin( final String margin ) {
		String domMgn = StringUtil.nval( getRequestParameter( margin ),null );
		if( domMgn != null ) {
			double domainMargin = Double.parseDouble( domMgn );
			chDataset.setDomainMargin( domainMargin ) ;	// double margin
		}
	}

	/**
	 * 【TAG】時刻を表す場合の表現の仕方[E1/E2/E3/H1/H2/H3]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * HybsNumberAxis にオーバーライドする 時間を表示する DecimalFormat の内部クラスを利用するに当たり、
	 * 時刻の表示方法を指定します。
	 * 外部から与える数字は、連続している必要があるため、１０進数です。
	 * たとえば、1700 → 17:00 , 2150 → 21:30 という感じです。
	 * 2400 を超えると日付違いになります。
	 *
	 * 英語表記(:)と日本語表記(時)の区別と、24時間を超える場合の表示方法によって、６種類のパターンが存在します。
	 * 　E1:そのまま、24:00 となり、加算されていく。
	 * 　E2:そのまま、0:00 に戻る。(日付は無視)
	 * 　E3:そのまま、1 00:00 と日付が付与される。
	 * 　H1:そのまま、24時00分 となり、加算されていく。
	 * 　H2:そのまま、00時00分 に戻る。(日付は無視)
	 * 　H3:そのまま、1日 00時00分 と日付が付与される。
	 * 初期値は、使用しない(-1)です。
	 *
	 * @og.rev 5.5.2.1 (2012/05/07) 新規追加
	 *
	 * @param   type 時刻の表示方法(E1,E2,E3,H1,H2,H3)
	 */
	public void setTimeFormatType( final String type ) {
		String timeFormatType = StringUtil.nval( getRequestParameter( type ),null );
		chDataset.setTimeFormatType( timeFormatType );
	}

	/**
	 * 【TAG】ラベルの表示向きを縦にするかどうか[false/true]を指定します(初期値:false[横書き])。
	 *
	 * @og.tag
	 * ChartCreate の rotationLabel は、角度を指定できましたが、NumberAxis では、
	 * 縦にするかどうかの指定しかできません。
	 * ここでは、true を指定するとラベルは、縦書きになります。
	 * 初期値は、false(横書き)です。
	 *
	 * @og.rev 5.5.2.1 (2012/05/07) 新規追加
	 *
	 * @param	useVLavels	ラベルの表示向き [false:横書き/true:縦書き]
	 */
	public void setUseVerticalLabels( final String useVLavels ) {
		boolean useVerticalLabels = StringUtil.nval( getRequestParameter( useVLavels ),false );
		chDataset.setUseVerticalLabels( useVerticalLabels );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクト(ChartDataset)は、シリアライズされません。
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクト(ChartDataset)は、読み込まれません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		chDataset = new ChartDataset();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		rtn.append( "[" ).append( this.getClass().getName() ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "dbid        [" ).append( dbid       ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "sql         [" ).append( sql        ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "dataset     [" ).append( chDataset  ).append( "]" ).append( HybsSystem.CR );

		return rtn.toString();
	}
}
