/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Types;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.Closer;

/**
 * 伝送要求に対してのPL/SQLを実行します。
 *
 * 実行するPL/SQLのPGIDは、実行対象で指定します。
 * 指定形式は、PG名( ?, ?, ... )  例) GEP00002.GEP00002( ?,?,0 ) です。
 * 引数は、従来のPL/SQLの実行が可能なように、第一引数はエラーコード、第二引数は、
 * エラーメッセージを返してきます。第三引数以降は、自由に指定できます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_PLSQL implements TransferExec {

	private static final int DB_MAX_QUERY_TIMEOUT = 6000;

	/**
	 * PL/SQLを実行します。
	 *
	 * @og.rev 5.5.2.4 (2012/05/16) クローズされない経路の処理
	 *
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		CallableStatement callStmt = null;
		try {
			String stmt = "{ call " + config.getExecObj() + "}";

			Connection conn = tran.getConnection( config.getExecDbid() );
//			CallableStatement callStmt = conn.prepareCall( stmt );
			callStmt = conn.prepareCall( stmt );						// 5.5.2.4 (2012/05/16)
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, Types.VARCHAR);
			callStmt.execute();

			int errCode = callStmt.getInt(1);
			String errMsg = callStmt.getString(2);

			if( errCode == ErrorMessage.OK ) { // 正常終了の場合
				tran.commit();
			}
			else {
				tran.rollback();
				errMsg = "PL/SQL実行時に例外が発生しました。" + errMsg;
				throw new RuntimeException( errMsg );
			}
		}
		catch( SQLException ex ) {
			tran.rollback();
			String errMsg = "PL/SQL呼出時に例外が発生しました。";
			throw new RuntimeException( errMsg, ex );
		}
		// 5.5.2.4 (2012/05/16)
		finally {
			Closer.stmtClose( callStmt );
		}
	}
}
