/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.util.HashMap;
import java.util.Map;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * <p>TableFilter_ROTATE は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。<br />
 *
 * ここではテーブルの回転、及びその逆回転を行います。
 *
 *  KEY_CLM    : キーカラム(複数指定可)
 *  ROTATE_CLM : 回転するカラム
 *  VALUE_CLM  : 回転カラムの値
 *  REVERSE    : 回転(false)・逆回転(true) (任意指定 初期値:false)
 *  MUST_CLM   : 必須属性を定義するカラム  (任意指定 初期値:false)
 *  DEF_CLM    : 初期値を定義するカラム    (任意指定)
 *  
 *  ※ それぞれに指定されたカラム名が存在しない場合は、処理されませんのでご注意下さい。
 *
 * ①回転
 *  キーカラムに指定された値が同じN行を1行として回転します。
 *  (キーカラムの値がブレイクしたタイミングで、行を変更します)
 *  このN行に含まれる回転カラムの値がカラム名に、回転カラム値が各カラムの値になります。
 *  キーカラムは、カンマ区切りで複数指定可能です。
 *
 *  生成されたテーブルモデルのカラムは、始めのMカラムがキーカラムに、その後ろのNカラムが
 *  回転されたカラムになります。
 *  
 *  また、元テーブルにMUST_CLMにより、各カラムの必須属性を定義することが
 *  できます。(MUST属性は、'1'又は'true'の場合に必須になります。)
 *
 * ②逆回転
 *  回転時の逆の挙動になります。
 *  "キーカラムに指定されたカラム以外"を回転カラムで指定されたカラムの値として分解します。
 *  各回転カラムの値は、回転カラム値に指定されたカラムに格納されます。
 *  
 *  分解後のカラム数は、キーカラム数 + 2 (回転カラム、回転カラム値)になります。
 *  また、行数は、(分解前の行数) x (回転カラム数)になります。
 *
 * @version  0.9.0  2000/10/17
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_ROTATE extends AbstractTableFilter {
	// * このプログラムのVERSION文字列を設定します。 {@value} */
	private static final String VERSION = "4.3.7.0 (2009/06/01)";

	private DBTableModel	table	 = null;			// 5.5.2.6 (2012/05/25) 共通に使うため、変数定義
	private ResourceManager	resource = null;			// 5.5.2.6 (2012/05/25) 共通に使うため、変数定義

	/**
	 * DBTableModel処理を実行します。<br />
	 * 
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * 
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		boolean reverse = StringUtil.nval( getValue( "REVERSE" ), false );

		table    = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		resource = getResource();			// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		// 逆回転
		if ( reverse ) {
			return getRecoverdTable();
		}
		// 回転
		else {
			return getRotateTable();
		}
	}

	/**
	 * 回転後のDBTableModelを返します。
	 * 
	 * @og.rev 5.1.8.0 (2010/07/01) メソッド名変更(setDefValue ⇒ setDefault)
	 * 
	 * @return 回転後のDBTableModel
	 */
	private DBTableModel getRotateTable() {
		String[] keyClm = StringUtil.csv2Array( getValue( "KEY_CLM" ) );
		int rotateNo = table.getColumnNo( getValue( "ROTATE_CLM" ), false );
		int valNo = table.getColumnNo( getValue( "VALUE_CLM" ), false );

		if ( keyClm == null || keyClm.length == 0 || rotateNo < 0 || valNo < 0 ) {
			return table;
		}

		int mustNo = table.getColumnNo( getValue( "MUST_CLM"), false );
		int defNo = table.getColumnNo( getValue( "DEF_CLM"), false );

		int clmCount = 0; // 回転後のカラム数
		// キーカラムのカラム番号を求め、カラム数としてカウントします。
		Map<String, Integer> clmMap = new HashMap<String, Integer>();
		int[] keyNos = new int[keyClm.length];
		for ( int i = 0; i < keyNos.length; i++ ) {
			keyNos[i] = table.getColumnNo( keyClm[i], false );
			if ( keyNos[i] < 0 ) {
				return table;
			}
			clmMap.put( keyClm[i], clmCount );
			clmCount++;
		}

		int rowCount = 0; // 回転後の行数
		// 回転カラムの値から回転後のカラム数を求めます。
		// また同時に、キーカラムの値のブレイク数により行数を求めます。
		Map<String, Integer> rowMap = new HashMap<String, Integer>();
		Map<String, Boolean> mustMap = new HashMap<String, Boolean>();
		Map<String, String> defaultMap = new HashMap<String, String>();
		for ( int i = 0; i < table.getRowCount(); i++ ) {
			String clmKey = table.getValue( i, rotateNo );
			if ( clmMap.get( clmKey ) == null ) {
				clmMap.put( clmKey, clmCount );
				clmCount++;
			}
			// 必須カラム抜き出し
			if( mustNo > -1 && StringUtil.nval( table.getValue( i, mustNo ), false ) ) {
				mustMap.put( clmKey, true );
			}
			// デフォルト値を書き換えるカラムの抜き出し
			if( defNo > -1 && table.getValue( i, defNo ) != null && table.getValue( i, defNo ).length() > 0 ) {
				defaultMap.put( clmKey, table.getValue( i, defNo ) );
			}

			String rowKey = getSeparatedValue( i, keyNos );
			if ( rowKey != null && rowKey.length() > 0 ) {
				if ( rowMap.get( rowKey ) == null ) {
					rowMap.put( rowKey, rowCount );
					rowCount++;
				}
			}
		}

		// 回転後のカラム一覧よりDBTableModelを初期化します。
		String names[] = new String[clmMap.size()];
		for ( Map.Entry<String, Integer> entry : clmMap.entrySet() ) {
			names[entry.getValue()] = entry.getKey();
		}
		DBTableModel nTable = DBTableModelUtil.newDBTable();
		nTable.init( names.length );
		for ( int i = 0; i < names.length; i++ ) {
			if( mustMap.get( names[i] ) != null ) {
				table.addMustType( i, "must" );
			}
			DBColumn column = resource.makeDBColumn( names[i] );
			if( defaultMap.get( names[i] ) != null ) {
				DBColumnConfig dbConfig = column.getConfig();
//				dbConfig.setDefValue( defaultMap.get( names[i] ) );
				dbConfig.setDefault( defaultMap.get( names[i] ) );		// 5.1.8.0 (2010/07/01)
				column = new DBColumn( dbConfig );
			}
//			nTable.setDBColumn( i, resource.makeDBColumn( names[i] ) );
			nTable.setDBColumn( i, column );							// 5.1.8.0 (2010/07/01)
		}

		// 値の一覧を作成し、DBTableModelに値をセットします。
		if( rowCount > 0 ) {
			String[][] vals = new String[rowCount][names.length];
			for ( int i = 0; i < table.getRowCount(); i++ ) {
				int row = rowMap.get( getSeparatedValue( i, keyNos ) );
				int clm = clmMap.get( table.getValue( i, rotateNo ) );

				for ( int j = 0; j < keyNos.length; j++ ) {
					vals[row][j] = table.getValue( i, keyNos[j] );
				}
				vals[row][clm] = table.getValue( i, valNo );
			}
			for ( int i = 0; i < vals.length; i++ ) {
				nTable.addColumnValues( vals[i] );
			}
		}

		return nTable;
	}

	/**
	 * 各行のキーとなるキーカラムの値を連結した値を返します。
	 * 
	 * @param row int
	 * @param clms int[]
	 * @return String 各行のキーとなるキーカラムの値を連結した値
	 */
	private String getSeparatedValue( final int row, final int[] clms ) {
		StringBuilder buf = new StringBuilder();
		for ( int i = 0; i < clms.length; i++ ) {
			String val = table.getValue( row, clms[i] );
			if( val != null && val.length() > 0 ) {
				if( i > 0 ) {
					buf.append( "__" );
				}
				buf.append( val );
			}
		}
		return buf.toString();
	}

	/**
	 * 逆回転後のDBTableModelを返します。
	 * 
	 * @return 逆回転後のDBTableModel
	 */
	private DBTableModel getRecoverdTable() {
		String[] keyClm = StringUtil.csv2Array( getValue( "KEY_CLM" ) );
		String rotateClm = getValue( "ROTATE_CLM" );
		String valClm = getValue( "VALUE_CLM" );

		if ( keyClm == null || keyClm.length == 0 || rotateClm == null || rotateClm.length() == 0
				|| valClm == null || valClm.length() == 0 ) {
			return table;
		}

		// キーカラムのカラム番号を求めます。
		int[] keyNos = new int[keyClm.length];
		for ( int i = 0; i < keyNos.length; i++ ) {
			keyNos[i] = table.getColumnNo( keyClm[i], false );
			if ( keyNos[i] < 0 ) {
				return table;
			}
		}

		// キーカラム以外(回転カラム以外)のカラム番号を求めます。
		int clmIdx = 0;
		int[] clmNos = new int[table.getColumnCount() - keyNos.length];
		for ( int i = 0; i < table.getColumnCount(); i++ ) {
			boolean isClm = true;
			for ( int j = 0; j < keyNos.length; j++ ) {
				if ( i == keyNos[j] ) {
					isClm = false;
				}
			}
			if ( isClm ) {
				clmNos[clmIdx] = i;
				clmIdx++;
			}
		}

		// テーブルモデルを初期化します。
		DBTableModel nTable = DBTableModelUtil.newDBTable();
		nTable.init( keyNos.length + 2 );
		for ( int i = 0; i < keyNos.length; i++ ) {
			nTable.setDBColumn( i, resource.makeDBColumn( keyClm[i] ) );
		}
		nTable.setDBColumn( keyNos.length, resource.makeDBColumn( rotateClm ) );
		nTable.setDBColumn( keyNos.length + 1, resource.makeDBColumn( valClm ) );

		// 各行を作成し、DBTableModelに登録します。
		for ( int i = 0; i < table.getRowCount(); i++ ) {
			for ( int j = 0; j < clmNos.length; j++ ) {
				String[] vals = new String[keyNos.length + 2];
				for ( int k = 0; k < keyNos.length; k++ ) {
					vals[k] = table.getValue( i, keyNos[k] );
				}
				vals[keyNos.length] = table.getColumnName( clmNos[j] );
				vals[keyNos.length + 1] = table.getValue( i, clmNos[j] );
				nTable.addColumnValues( vals );
			}
		}

		return nTable;
	}

}
