/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.ApplicationInfo;
// import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * systemId と lang に対応した画面データを作成するデータロードクラスです。
 *
 * 画面データは、画面ID(GUIKEY)に対して、各種画面情報を持っています。
 * 従来と異なるのは、同一画面IDに対して、アドレスやロールズを変えた情報を持てると言う
 * 事です。これは、カスタマイズ時に、画面IDは変えずに、実際のアクセスされるアドレスを
 * 変える事で、他のアプリケーションへの影響を最小限にして開発できます。
 * linkタグや、submit などの gamenID を指定するカスタムタグでは、実際のアクセス先は、
 * ログインユーザーのロールズでアクセス可能な画面のアドレスに転送されます。
 * 作番毎のカスタマイズや、ユーザーロールに応じた飛び先変更などにも使用できます。
 *
 * 画面データでは、複数階層持てるように、画面階層(GUILVL)を持っています。このレベルに
 * 応じて、分類(CLASSIFY)の表示方法が変わります。(擬似階層構造)
 *
 * 画面データでは、言語(LANG)は、条件から消えました。実際に名称を表示させる時は、
 * 画面カラムID(LABEL_CLM)に対応する ラベル定義より、言語に応じたラベルを取得します。
 * エンジン内部で使用している GUIInfo オブジェクト構築時に割り当てます。
 * 分類(CLASSIFY)は、コードリソースに登録します。
 *
 * 画面データを作成する場合は、同一画面IDで、作成区分(KBSAKU)違いの場合は、
 * 最も大きな作成区分を持つ画面情報を使用します。
 * 作成区分(KBSAKU)='0' のデータは、マスタリソースとして、エンジンとともに
 * 配布されるリソースになります。
 *
 * 画面データは、カラム定義のような、読込フラグ(FGLOAD)はありません。
 * 画面情報(GUIInfo)は、ユーザーログイン毎に作成されます。（キャッシュは
 * セッション情報に登録されます。）
 * これは、画面アクセス条件を、ログイン時に済ますことで、高速化を図っています。
 * 画面IDの件数が少ないことと、画面IDを自動作成した場合でも、
 * ほとんどのケースで、すべて使用される可能性が非常に高い為です。
 *
 * SYSTEM_ID='**' は、共通リソースです。
 * これは、システム間で共通に使用されるリソース情報を登録しておきます。
 *
 * @og.rev 4.0.0 (2004/12/31) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class GUIDataLoader {
	// リソースの接続先を、取得します。
	private final String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	// ＤＢリソースの初期一括読み込みのクエリー
	// ソート順は、画面IDオブジェクトの優先順(後優先)で、画面表示順ではありません。
	private static final String QUERY = "select GUIKEY,GUILVL,LABEL_CLM,ADDRESS,SEQNO"
//									+ ",GROUPS,CLASSIFY,ROLES,RWMODE,TARGET,PARAM,KBLINK"
//									+ ",GROUPS,'',ROLES,RWMODE,TARGET,PARAM,KBLINK"
									+ ",GROUPS,'',ROLES,RWMODE,TARGET,PARAM,KBLINK,DYUPD"
									+ " from GEA11 where SYSTEM_ID in ( ?,'**') and FGJ='1'"
									+ " order by SYSTEM_ID,KBSAKU,SEQNO,GUIKEY" ;

	private final List<GUIData>	pool = Collections.synchronizedList(new ArrayList<GUIData>());	// キャッシュ用プール
	private final String		SYSTEM_ID ;		// システムID

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	private final ApplicationInfo appInfo;

	/**
	 *  SystemId 毎に ファクトリオブジェクトを作成します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param systemId システムID
	 */
	GUIDataLoader( final String systemId ) {
		SYSTEM_ID = systemId;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "CodeDataLoader",null,null );
		}
		else {
			appInfo = null;
		}

		// ApplicationInfo の設定が終わってから実行します。
		loadDBResource();
	}

	/**
	 * ＤＢリソースより 画面データを取得、設定します。
	 * ＤＢリソースは、GUIKEY,GUILVL,LABEL_CLM,ADDRESS,SEQNO,GROUPS,
	 * CLASSIFY,ROLES,RWMODE,TARGET,PARAM,KBLINK の順番で、GUIKEY の重複を許します。
	 * 重複している場合（ロール違い等）は、一つのオブジェクトとして作成され、
	 * 個々のログインユーザー毎にユニークになるように、設定する必要があります。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/10/31) ロールの継承機能の追加・分類の取得を追加(暫定対応)
	 * @og.rev 5.3.1.0 (2011/01/01) 通常画面に対してアドレスを設定しない場合にロールが効かないバグを修正します。
	 * @og.rev 5.3.1.0 (2011/01/01) ロール継承機能廃止
	 */
	private void loadDBResource() {
		String[] args = new String[] { SYSTEM_ID };

		String[][] vals = DBUtil.dbExecute( QUERY,args,appInfo,DBID );
		int len = vals.length;

		int level ;
//		String rolesLv1 = null;
//		String rolesLv2 = null;
		String classify = "";
		for( int i=0; i<len; i++ ) {

			// ロールの継承対応
			level = Integer.parseInt( vals[i][GUIData.GUILVL] );
//			if( level == 1 ) {
//				rolesLv1 = vals[i][GUIData.ROLES];
//			}
//			else if( level == 2 ) {
			if( level == 2 ) {
//				rolesLv2 = vals[i][GUIData.ROLES];
				classify = vals[i][GUIData.GUIKEY]; // 暫定対応
			}
			else if( level >= 3 ) {
//				vals[i][GUIData.ROLES] = StringUtil.nval( vals[i][GUIData.ROLES], rolesLv2 );
//				vals[i][GUIData.ROLES] = StringUtil.nval( vals[i][GUIData.ROLES], rolesLv1 );
				vals[i][GUIData.CLASSIFY] = classify; // 暫定対応
			}

			// 5.3.1.0 (2011/01/01) 通常画面に対してアドレスを設定しない場合にロールが効かないバグを修正します。
//			if( vals[i][GUIData.ADDRESS] == null || vals[i][GUIData.ADDRESS].length() == 0 ) {
			if( ( level == 1 || level == 2 ) && ( vals[i][GUIData.ADDRESS] == null || vals[i][GUIData.ADDRESS].length() == 0 ) ) {
					vals[i][GUIData.ROLES] = null;
			}

			pool.add( new GUIData( vals[i] ) );
		}

		System.out.println( "  GUIDataLoader [" + len + "] loaded" );
	}

	/**
	 * すべてのGUIData オブジェクト配列を取得します。
	 * プールに持っているすべてのキャッシュを、GUIData オブジェクト配列
	 * にして返します。
	 * このリソースは、List で管理しており、読み込み時にすべてキャッシュされます。
	 *
	 * @return   GUIData[]  すべてのGUIData オブジェクト配列
	 */
	public GUIData[] getAllData() {
		if( pool.isEmpty() ) { loadDBResource(); }
		return pool.toArray( new GUIData[pool.size()] );
	}

	/**
	 * GUIData オブジェクトのキャッシュをクリアします。
	 *
	 */
	public void clear() {
		pool.clear();
	}
}
