/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import org.opengion.hayabusa.common.HybsSystem;

import java.util.Locale;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletContext;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Filter インターフェースを継承した HTMLデモ画面を作成するフィルタクラスです。
 * web.xml で filter 設定することにより、使用できます。
 * このフィルターでは、通常の画面アクセスを行うと、指定のフォルダに対して
 * JSPをHTMLに変換した形で、ファイルをセーブしていきます。このHTMLは、
 * デモサンプル画面として、使用できます。
 * 出来る限り、デモ画面として使えるように、画面間リンクや、ボタン制御を
 * JavaScript を挿入する事で実現しています。
 *
 * フィルターに対してweb.xml でパラメータを設定します。
 *   ・saveDir  :ファイルをセーブするディレクトリ
 *
 * パラメータがない場合は、G:/webapps/作番/filetemp/DIR/ 以下に自動設定されます。
 * また、ディレクトリが、相対パスの場合は、G:/webapps/作番/ 以下に、絶対パスの
 * 場合は、そのパスの下に作成されます。 *
 *
 * 【WEB-INF/web.xml】
 *     &lt;filter&gt;
 *         &lt;filter-name&gt;FileFilter&lt;/filter-name&gt;
 *         &lt;filter-class&gt;org.opengion.hayabusa.filter.FileFilter&lt;/filter-class&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;saveDir&lt;/param-name&gt;
 *             &lt;param-value&gt;filetemp/DIR/&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *     &lt;/filter&gt;
 *
 *     &lt;filter-mapping&gt;
 *         &lt;filter-name&gt;FileFilter&lt;/filter-name&gt;
 *         &lt;url-pattern&gt;/jsp/*&lt;/url-pattern&gt;
 *     &lt;/filter-mapping&gt;
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileFilter implements Filter {
	private String saveDir = null;	// "G:/webapps/gf/filetemp/DIR/" など

	/**
	 * Filter インターフェースの doFilter メソッド
	 *
	 * Filter クラスの doFilter メソッドはコンテナにより呼び出され、 最後のチェーンにおける
	 * リソースへのクライアントリクエストのために、 毎回リクエスト・レスポンスのペアが、
	 * チェーンを通して渡されます。 このメソッドに渡される FilterChain を利用して、Filter が
	 * リクエストやレスポンスをチェーン内の次のエンティティ(Filter)にリクエストとレスポンスを
	 * 渡す事ができます。
	 * このメソッドの典型的な実装は以下のようなパターンとなるでしょう。
	 * 1. リクエストの検査
	 * 2. オプションとして、入力フィルタリング用にコンテンツもしくはヘッダをフィルタリング
	 *    するためにカスタム実装によるリクエストオブジェクトのラップ
	 * 3. オプションとして、出力フィルタリング用にコンテンツもしくはヘッダをフィルタリング
	 *    するためにカスタム実装によるレスポンスオブジェクトラップ
	 * 4. 以下の a)、b) のどちらか
	 *    a) FileterChain オブジェクト(chain.doFilter()) を利用してチェーンの次のエンティティを呼び出す
	 *    b) リクエスト処理を止めるために、リクエスト・レスポンスのペアをフィルタチェーンの次の
	 *       エンティティに渡さない
	 * 5. フィルタチェーンの次のエンティティの呼び出した後、直接レスポンスのヘッダをセット
	 *
	 * @param req ServletRequest
	 * @param res ServletResponse
	 * @param chain FilterChain
	 * @throws IOException
	 * @throws ServletException
	 */
	public void doFilter( final ServletRequest req,
							final ServletResponse res,
							final FilterChain chain )
								throws IOException, ServletException {
		if(req instanceof HttpServletRequest && res instanceof HttpServletResponse) {
			HttpServletRequest  request  = (HttpServletRequest) req;
			HttpServletResponse response = (HttpServletResponse) res;

			try {
				request.setCharacterEncoding( "UTF-8" );
			}
			catch ( UnsupportedEncodingException ex ) {
				throw new RuntimeException( ex );
			}

			String filename = makeFileName( request );
			if( filename != null ) {
				FileResponseWrapper wrappedResponse = new FileResponseWrapper(response,filename);
				chain.doFilter(req, wrappedResponse);
				wrappedResponse.finishResponse();
			}
			else {
				chain.doFilter(req, res);
			}
		}
	}

	/**
	 * フィルターの初期処理メソッドです。
	 *
	 * フィルターに対してweb.xml で初期パラメータを設定します。
	 *   ・startTime:停止開始時刻
	 *   ・stopTime :停止終了時刻
	 *   ・filename :停止時メッセージ表示ファイル名
	 *
	 * @param filterConfig FilterConfig
	 */
	public void init(final FilterConfig filterConfig) {
		ServletContext context = filterConfig.getServletContext();
		String realPath = context.getRealPath( "/" );

		String dir = filterConfig.getInitParameter("saveDir");
		if( dir != null && dir.length() > 1 ) {
			dir = dir.replace( '\\','/' );
			if( dir.charAt(0) == '/' || dir.charAt(1) == ':' ) {
				saveDir = dir;
			}
			else {
				saveDir = realPath + dir ;
			}

			if( dir.charAt(dir.length()-1) != '/' ) {
				saveDir = saveDir + "/" ;
			}
		}
		else {
			saveDir = realPath + "filetemp/DIR/" ;
		}
	}

	/**
	 * Filter インターフェースの destroy メソッド (何もしません)。
	 *
	 * サービス状態を終えた事を Filter に伝えるために Web コンテナが呼び出します。
	 * Filter の doFilter メソッドが終了したか、タイムアウトに達した全てのスレッドにおいて、
	 * このメソッドを一度だけ呼び出されます。 Web コンテナがこのメソッドを呼び出した後は、
	 * Filter のこのインスタンスにおいて二度と doFilter メソッドを呼び出す事はありません。
	 *
	 * このメソッドは、フィルタに保持されている(例えば、メモリ、ファイルハンドル、スレッド)
	 * 様々なリソースを開放する機会を与え、 あらゆる永続性の状態が、メモリ上における Filter
	 * の現在の状態と同期しているように注意してください。
	 */
	public void destroy() {
		// noop
	}

	/**
	 * セーブするファイル名を、リクエスト情報より取得します。
	 *
	 * リクエストされたファイル(.jsp)を、HTMLファイル(.htm)にするだけでなく、
	 * 呼び出されたときの command を元に、ファイル名を作成します。
	 *   command="NEW"    + index.jsp   ⇒  "indexNW.htm"
	 *   command="NEW"    + forward.jsp ⇒  "forward.htm"
	 *   command="NEW"    + query.jsp   ⇒  "queryNW.htm"
	 *   command="RENEW"  + index.jsp   ⇒  "indexRNW.htm"
	 *   command="RENEW"  + forward.jsp ⇒  "renew.htm"
	 *   command="日本語名+ forward.jsp ⇒  "コマンド名.htm"
	 *   command="日本語名+ update.jsp  ⇒  "コマンド名.htm"
	 *   matrixMenu対応
	 *      GAMENID=XXXX&buttonRequest=true + index.jsp     ⇒ "jsp/indexXXXX.htm"
	 *      GAMENID=XXXX&buttonRequest=true + multiMenu.jsp ⇒ "jsp/menuXXXX.htm"
	 *   その他             xxxx.jsp    ⇒  "xxxx.htm"
	 *
	 * このメソッドは、フィルタに保持されている(例えば、メモリ、ファイルハンドル、スレッド)
	 * 様々なリソースを開放する機会を与え、 あらゆる永続性の状態が、メモリ上における Filter
	 * の現在の状態と同期しているように注意してください。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.3.3.0 (2008/10/01) Matrixメニュー対応
	 * @og.rev 5.5.2.5 (2012/05/21) update.jsp に出力されるファイルを、コマンド名.htm に出力するように機能追加
	 *
	 * @param request ServletRequest
	 * @return String セーブするファイル名
	 */
	private String makeFileName( final ServletRequest request ) {
		HttpServletRequest myReq = (HttpServletRequest) request;
		String requestURI = myReq.getRequestURI();

		int index2     = requestURI.lastIndexOf( '/' );
		String jspID   = requestURI.substring( index2+1 );

		int index1     = requestURI.lastIndexOf( '/',index2-1 );
		String gamenId = requestURI.substring( index1+1,index2 );

		String file = null;

		if( jspID != null && jspID.endsWith( ".jsp" ) ) {
			String cmd = request.getParameter( "command" );
			if( cmd != null && jspID.equals( "forward.jsp" ) ) {
				if( "NEW".equals( cmd ) ) { file = "forward.htm"; }
				else if( "RENEW".equals( cmd ) || "REVIEW".equals( cmd ) ) { file = "renew.htm"; }
				else {
					String xferVal = request.getParameter( HybsSystem.NO_XFER_KEY + cmd );
					// 5.5.2.5 (2012/05/21) update.jsp に出力されるファイルを、コマンド名.htm に出力するように機能追加
					if( "update.jsp".equals( xferVal ) ) {
						file = cmd + ".htm" ;
					}
					else if( xferVal != null && xferVal.endsWith( "jsp" ) ) {
						file = xferVal.toLowerCase(Locale.JAPAN).replace( "jsp","htm" );
					}
					else {
						String xferCmd = request.getParameter( HybsSystem.NO_XFER_KEY + cmd + "CMD" );
						if( xferCmd != null ) {
							file = xferCmd.toLowerCase(Locale.JAPAN) + ".htm";
						}
						else {
							file = cmd.toLowerCase(Locale.JAPAN) + ".htm";
						}
					}
				}
			}
			else if( "index.jsp".equals( jspID ) && ( "RENEW".equals( cmd ) || "REVIEW".equals( cmd ) ) ) {
				file = "indexRNW.htm";
			}
			else if( "index.jsp".equals( jspID ) && "NEW".equals( cmd ) ) {
				file = "indexNW.htm";
			}
			else if( "query.jsp".equals( jspID ) && "NEW".equals( cmd ) ) {
				file = "queryNW.htm";
			}
			else {
				file = jspID.substring( 0,jspID.length()-4 ) + ".htm" ;
			}

			// 4.3.3.0 (2008/10/01) Matrixメニュー対応
			//   GAMENID=XXXX&buttonRequest=true + index.jsp      ⇒ "jsp/indexXXXX.htm"
			//   GAMENID=XXXX&buttonRequest=true + multiMenu.jsp  ⇒ "jsp/menuXXXX.htm"
			//   group=YYYY&buttonRequest=true   + matrixMenu.jsp ⇒ "menu/matrixMenuYYYY.htm"
			String guiKey = request.getParameter( "GAMENID" );
			String btnReq = request.getParameter( "buttonRequest" );
			if( "true".equals( btnReq ) ) {
				if( guiKey != null ) {
					if( "index.jsp".equals( jspID ) ) {
						file = "jsp/index" + guiKey + ".htm";
					}
					else if( "multiMenu.jsp".equals( jspID ) ) {
						file = "jsp/menu" + guiKey + ".htm";
						gamenId = "jsp" ;	// トリッキー
					}
				}
				else {

					String group = request.getParameter( "group" );
					if( group  != null ) {
						if( "matrixMenu.jsp".equals( jspID ) ) {
							file = "menu/matrixMenu" + group + ".htm";
							gamenId = "jsp" ;	// トリッキー
						}
					}
				}
			}

//			int index1     = requestURI.lastIndexOf( '/',index2-1 );
//			String gamenId = requestURI.substring( index1+1,index2 );

			if( "jsp".equals( gamenId ) ) { file = saveDir + file; }
			else { file = saveDir + gamenId + "/" + file; }

			File fl = new File( saveDir + gamenId );
			if( !fl.exists() && !fl.mkdirs() ) {
				String errMsg = "所定のフォルダが作成できませんでした。[" + fl + "]" ;
				throw new RuntimeException( errMsg );
			}
		}
		return file;
	}
}
