/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ZipFileUtil;
import org.opengion.fukurou.xml.HybsXMLSave;

/**
 * システムの自動インストールと自動更新を行います。
 * 
 * (1)初期インストール・自動更新(#autoInsupd)
 *   ①初期自動インストールを行うには、起動時の環境変数にINSTALL_CONTEXTSが
 *     設定されている必要があります。
 *     この環境変数が設定されている場合、システムリソーステーブル(GE12)が存在しなければ、
 *     エンジンがインストールされていないと判断し、自動インストールを行います。
 *     INSTALL_CONTEXTSにge,gfが指定されている場合は、開発環境を含めたフルバージョンが
 *     インストールされます。
 *     geのみが指定されている場合は、コアモジュールであるge4のみがインストールされます。
 *
 *     インストールスクリプトは、
 *      webapps/[CONTEXT]/db/[DBNAME]/xml/install
 *     以下にあるXMLファイルが全て実行されます。
 *     また、同時に
 *      webapps/[CONTEXT]/db/xml
 *     以下にあるデータロードスクリプトも全て実行されます。
 *     
 *   ②自動更新については、システムリソーステーブル(GE12)の更新と、各システムの更新の2つがあります。
 *     GE12更新の判断基準は、システムID='**'に格納されているバージョン(同一のGE12を使用し
 *     ているシステムの最大バージョン番号)がアップした場合です。
 *     この場合に、エンジン内部で保持しているXMLファイルよりシステムリソースの再ロードを行います。
 *     各システムの更新の判断基準は、システムID=各システムのバージョン番号がアップされた場合です。
 * 
 *     更新スクリプトは、
 *      webapps/[CONTEXT]/db/[DBNAME]/xml/update
 *     以下にあるXMLファイルが全て実行されます。
 *     また、同時に
 *      webapps/[CONTEXT]/db/xml
 *     以下にあるデータロードスクリプトも全て実行されます。
 *  
 * (2)インストール(#install)
 *  自動インストールは、通常は画面からコンテキストのアーカイブを指定して行います。
 *  
 *  アーカイブの内容としては、アーカイブの直下がコンテキスト名のフォルダとなっている必要があります。
 *  このコンテキストフォルダをwebapps以下に展開します。
 *  
 *  また、Tomcatのコンテキストの設定ファイル、([CONTEXT].xml)が"WEB-INFの直下を配置している必要があります。
 *  
 *  このインストールでは、Tomcatに対するコンテキスト定義のXMLファイルの配備及び、
 *  各種DB、データのロードを行います。
 *  
 *  インストールスクリプトは、
 *   webapps/[CONTEXT]/db/[DBNAME]/xml/install
 *  以下にあるXMLファイルが全て実行されます。
 *  また、同時に
 *   webapps/[CONTEXT]/db/xml
 *  以下にあるデータロードスクリプトも全て実行されます。
 * 
 * @og.rev 4.3.6.6 (2009/05/15) 新規作成
 * @og.group 初期化
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public final class SystemInstaller {
	private final String VERSION;
	private final Connection connection;
	private final PrintWriter out;			// 5.1.9.0 (2010/08/01)

	/** エンジン共通パラメータ(SYSTEM_ID='**' KBSAKU='0')のXML ファイルの指定	{@value}	*/
	public static final String GE12_XML = "org/opengion/hayabusa/common/GE12.xml";

	/** エンジン共通パラメータ(SYSTEM_ID='**' KBSAKU='0')のENGINE_INFO 読み取りクエリー	{@value}	*/
	public static final String SEL_MAX_ENG = "select PARAM from GE12"
									+ " where SYSTEM_ID='**' and PARAM_ID='ENGINE_INFO'"
									+ " and FGJ='1' and KBSAKU='0'" ;

	/** エンジン個別(SYSTEM_ID='個別' KBSAKU='0'  CONTXT_PATH='自身')のバージョン情報を取得するクエリーー{@value} 4.3.6.6 (2009/05/15) */
	public static final String SEL_SYS_ENG = "select PARAM from GE12"
									+ " where SYSTEM_ID=? and PARAM_ID='ENGINE_INFO' and KBSAKU='0' and CONTXT_PATH=? and FGJ='1'";

	/**
	 * システムインストール・更新クラスのコンストラクタです
	 *
	 * @param conn Connection  登録用コネクション
	 * @param out PrintWriter 表示用のWriter
	 */
	public SystemInstaller( final Connection conn, final PrintWriter out ) {
		connection = conn;
		this.out = out;

		VERSION = BuildNumber.ENGINE_INFO;
	}

	/**
	 * システムの初期自動インストール・自動更新を行います。
	 * 
	 * 詳細は、クラスドキュメントを参照して下さい。
	 * 
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 * 
	 * @param systemId システムID
	 * @param context コンテキスト名
	 * @param hostUrl ホスト文字列
	 * @throws SQLException
	 * @throws UnsupportedEncodingException 
	 */
	public void autoInsUpd( final String systemId, final String context, final String hostUrl ) throws SQLException, UnsupportedEncodingException  {
		String oldMaxVersion = getOldMaxVersion();
		String oldSystemVersion = getOldSystemVersion( systemId, hostUrl );

		out.println( "    System Version Information ( " + systemId + " )" );
		out.println( "      Load Version [ " + VERSION + " ]" );
		out.println( "        -> Resource Version[ " + oldMaxVersion + " ]" );
		out.println( "        -> System   Version[ " + oldSystemVersion + " ]" );

		// 初期自動インストール
		if( "none".equalsIgnoreCase( oldMaxVersion ) ) {
			out.println( "      !!! openGion ENVIROMENT IS NOT INSTALLED !!!" );

			String INSTALL_CONTEXTS = System.getenv( "INSTALL_CONTEXTS" );
			if( INSTALL_CONTEXTS == null || INSTALL_CONTEXTS.length() == 0 ) {
				out.println( "        !!! \"INSTALL_CONTEXT\" IS NOT CONFIGURED\" !!!" );
				out.println( "        !!! \"SET ENRIVOMENT PARAMETER NAMED \"INSTALL_CONTEXT\" ON INIT_SCRIPT !!!" );
				return;
			}
			out.println( "      Start Initiall Enviroment Install : install type ( " + INSTALL_CONTEXTS + " )" );
			String[] insSys = StringUtil.csv2Array( INSTALL_CONTEXTS );
			for( int i=0; i<insSys.length; i++ ) {
				out.println( "        install    ( " + insSys[i] + " )" );
				loadXMLScript( "install", insSys[i] );
				connection.commit();
				out.println( "        completed  ( " + insSys[i] + " )" );
			}

			out.println( "      Start SystemParameter reload" );
			dbXMLResourceInsert();
			connection.commit();
			out.println( "      completed" );
		}
		// 自動更新
		else {
			if ( oldSystemVersion == null || oldSystemVersion.compareTo( VERSION ) < 0 ){
				out.println( "      Start Enviroment Update ( " + context + " )" );
				loadXMLScript( "update", context );
				connection.commit();
				out.println( "      completed               ( " + context + " )" );
			}

			if( oldMaxVersion == null || oldMaxVersion.compareTo( VERSION ) < 0 ){
				out.println( "      Start SystemParameter Reload" );
				dbXMLResourceInsert();
				connection.commit();
				out.println( "      completed" );
			}
		}
	}

	/**
	 * システムの自動インストールを行います。
	 * 
	 * 詳細は、クラスドキュメントを参照して下さい。
	 * 
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 * 
	 * @param buildArchive コンテキストのアーカイブファイル
	 * @throws SQLException
	 */
	public void install( final File buildArchive ) throws SQLException {
		final String FS       = File.separator ;
//		final String tempDir  = HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "FILE_URL" ) + String.valueOf( System.currentTimeMillis() + FS );
		final String tempDir  = HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "FILE_URL" ) + System.currentTimeMillis() + FS;
		final String ctxtXmlDir = System.getenv( "CATALINA_HOME" ) + FS + "conf" + FS + System.getenv( "ENGINE_NAME" ) + FS + "localhost" + FS;
		final String appBase  = System.getenv( "APP_BASE" ) + FS;

		out.println( "      Check Archive File and Enviroment" );

		// アーカイブの存在チェック
		if( !buildArchive.exists() ) {
			out.println( "        !!! Archive File does not exists File=[ " + buildArchive.getAbsolutePath() + "] !!!" );
			out.println( "        !!! Install Aborted !!! " );
			return;
		}

		// アーカイブを一時ファイルに展開します。
		ZipFileUtil.unCompress( tempDir, buildArchive.getAbsolutePath() );

		// アーカイブの内容チェック
		File[] ctxts = new File( tempDir ).listFiles();
		for( File ctxt : ctxts ) {
			// 5.1.9.0 (2010/08/01) if の条件を入れ替えます。(Avoid if (x != y) ..; else ..;)
			if( ctxt.isDirectory() ) {
				String context = ctxt.getName();

				// アーカイブ中に[CONTEXT].xmlが存在していない場合はエラー(何も処理しない)
				File srcCtxtXml = new File( tempDir + context + FS + "WEB-INF" + FS + context + ".xml" );
				if( !srcCtxtXml.exists() ) {
					out.println( "        !!! Context XML Does not exists =[ " + srcCtxtXml.getAbsolutePath() + "] !!!" );
					out.println( "        !!! Install Aborted !!! " );
					return;
				}

				// [CONTEXT].xmlが既に存在している場合はエラー(何も処理しない)
				File ctxtXml = new File(  ctxtXmlDir + context + ".xml" );
				if( ctxtXml.exists() ) {
					out.println( "        !!! Context XML File Already Installed File=[ " + ctxtXml.getAbsolutePath() + "] !!!" );
					out.println( "        !!! Install Aborted !!! " );
					return;
				}

				// webapps/[CONTEXT]が既に存在している場合はエラー(何も処理しない)
				File webAppsDir = new File( appBase + context );
				if( webAppsDir.exists() ) {
					out.println( "        !!! Context Path Already Exists Path=[ " + webAppsDir.getAbsolutePath() + "] !!!" );
					out.println( "        !!! Install Aborted !!! " );
					return;
				}

				out.println( "        This Archive includes SYSTEM ( " + ctxt.getName() + " ) for Install" );
			}
			// ファイルが含まれている場合はエラー(何も処理しない)
			else {
				out.println( "        !!! This Archive is not Installer. Because include FILE not DIRECTORY. File=[ " + ctxt.getName() + "] !!!" );
				out.println( "        !!! Install Aborted !!! " );
				return;
			}
		}

		// アーカイブをコンテキストファイル以下にコピー
		for( File ctxt : ctxts ) {
			String context = ctxt.getName();
			out.println( "      Start Enviroment Install ( " + context + " )" );

			// コンテキストのファイルをコピーします。
			FileUtil.copyDirectry( tempDir + context, appBase + context );

			// [CONTEXT].xmlをTomcatのconf以下に展開します。
			FileUtil.copy( tempDir + context + FS + "WEB-INF" + FS + context + ".xml", ctxtXmlDir + context + ".xml" );

			// DBスクリプトをロードします。
			loadXMLScript( "install", context );
			connection.commit();
			out.println( "      completed                ( " + context + " )" );
		}
		out.println( "      Install Process All Completed." );
	}

	/**
	 * インストール、更新用のXMLスクリプトをロードします。
	 * 
	 * @og.rev 5.0.0.2 (2009/09/15) .xmlファイル以外は読み込まないように修正
	 * @og.rev 5.1.1.0 (2009/12/01) コメントを出して、処理中ということが判る様にします。
	 * @og.rev 5.1.9.0 (2010/08/01) DB非依存の定義・データの読み込み対応
	 * 
	 * @param type String 更新タイプ( install or update )
	 * @param context String コンテキスト名
	 */
	private void loadXMLScript( final String type, final String context ) throws SQLException {
		final String FS       = File.separator ;
		final String APP_BASE = System.getenv( "APP_BASE" );
		final String DBNAME   = connection.getMetaData().getDatabaseProductName().toLowerCase( Locale.JAPAN );

		// DB名からスクリプトを格納しているフォルダを探します。
		String scriptBase = APP_BASE + FS + context.toLowerCase( Locale.JAPAN ) + FS + "db";
		File[] dbDir = new File( scriptBase ).listFiles();
		if( dbDir == null || dbDir.length == 0 ) {
			out.println( "             DB Folder not found. [" + scriptBase + "]"  );
			return;
		}

		String scriptPath = null;
		for ( int i = 0; i < dbDir.length; i++ ) {
			if ( DBNAME.indexOf( dbDir[i].getName() ) >= 0 ) {
				scriptPath = dbDir[i].getAbsolutePath();
				break;
			}
		}
		if( scriptPath == null ) {
			out.println( "             !!! Script Folder for [ " + DBNAME + " ] not found !!!" );
			return;
		}

		execScripts( scriptPath, type );

		// 5.1.9.0 (2010/08/01) DB非依存の定義・データの読み込み対応
		execScripts( scriptBase + FS + "common" + FS, type );

//		// webapps/[CONTEXT]/db/xml 内のスクリプトを実行します
//		File[] dataDir = new File( scriptBase + FS + "xml" ).listFiles();
//		if( dataDir != null && dataDir.length > 0 ) {
//			for ( int i=0; i<dataDir.length; i++ ) {
//				String dtNm = dataDir[i].getName() ;
//				if( dtNm.endsWith( ".xml" ) ) {		// 5.0.0.2 (2009/09/15)
//					Reader reader = new BufferedReader( FileUtil.getBufferedReader( dataDir[i], "UTF-8" ) );
//					HybsXMLSave save = new HybsXMLSave( connection, dtNm );
//					save.insertXML( reader );
//				}
//			}
//			out.println( "            DB Data Files Installed , [ " + dataDir.length + " ] files loaded " );
//		}
	}

	/**
	 * XMLファイルで定義されたDBスクリプトを実行します。
	 * 
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
	 * 
	 * @param scriptPath XMLファイルのあるパス
	 * @param type String 更新タイプ( install or update )
	 */
	private void execScripts( final String scriptPath, final String type ) {
		final String FS       = File.separator ;

		// webapps/[CONTEXT]/db/[DBNAME]/xml/(install|update) 内のスクリプトを実行します
		List<String> list = new ArrayList<String>();

		if( "install".equalsIgnoreCase( type ) ) {
			FileUtil.getFileList( new File( scriptPath  + FS + "xml" + FS + "install" ), true, list );
			FileUtil.getFileList( new File( scriptPath  + FS + "xml" + FS + "update" + FS + "const" ), true, list );
		}
		else {
			FileUtil.getFileList( new File( scriptPath  + FS + "xml" + FS + "update" + FS + "const" ), true, list );

			/*******************************************************************************
			 * updateの場合に、更新前のバージョンからの変更スクリプトを実行する機能が必要
			 *******************************************************************************/
		}

		if( ! list.isEmpty() ) {
			String dir1 = null;		// 5.1.1.0 (2009/12/01)
			for ( String name : list ) {
				if( name.endsWith( ".xml" ) ) {		// 5.0.0.2 (2009/09/15)
					File xml = new File( name );
					// 5.1.1.0 (2009/12/01) 処理中コメント：フォルダ単位に表示
					String dir2 = xml.getParent();
					if( dir1 == null || !dir1.equalsIgnoreCase( dir2 ) ) {
						out.println( "            processing ... " + dir2 );
						dir1 = dir2;
					}

					Reader reader = new BufferedReader( FileUtil.getBufferedReader( xml, "UTF-8" ) );
					HybsXMLSave save = new HybsXMLSave( connection, xml.getName() );
					save.insertXML( reader );
				}
			}
			out.println( "            DB Enviroment " + type + "ed , [ " + list.size() + " ] scripts loaded " );
		}
	}

	/**
	 * 最後に起動された際のバージョン番号を取得します。(システムID='**')
	 *
	 * エンジンがまだインストールされていない等の原因でエラーが発生した場合は、
	 * "none"という文字列を返します。
	 * 
	 * @og.rev 5.1.1.0 (2009/12/01) 実行エラー時に、rollback を追加(PostgreSQL対応)
	 * 
	 * @return String バージョン番号
	 */
	private String getOldMaxVersion() {
		// エンジンパラメータのエンジン情報(バージョン番号 + ビルドタイプ)を取得します。
		Statement 			stmt		= null;
		ResultSet			resultSet	= null;
		String				ver 		= null;
		try {
			stmt = connection.createStatement();
			resultSet = stmt.executeQuery( SEL_MAX_ENG );
			while( resultSet.next() ) {
				ver = resultSet.getString(1);
			}
		}
		catch( SQLException ex ) {
			ver = "none";
			Closer.rollback( connection );		// 5.1.1.0 (2009/12/01)
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( stmt );
		}
		return ver;
	}

	/**
	 * 最後に起動された際のバージョン番号を取得します。(システムID=各システム)
	 * 
	 * @og.rev 5.1.1.0 (2009/12/01) 実行エラー時に、rollback を追加(PostgreSQL対応)
	 * 
	 * @return String バージョン番号
	 */
	private String getOldSystemVersion( final String systemId, final String hostUrl ) {
		// エンジンパラメータのエンジン情報(バージョン番号 + ビルドタイプ)を取得します。
		PreparedStatement	pstmt		= null;
		ResultSet			resultSet	= null;
		String				ver 		= null;
		try {
			pstmt = connection.prepareStatement( SEL_SYS_ENG );
			pstmt.setString( 1, systemId );
			pstmt.setString( 2, hostUrl ); 
			resultSet = pstmt.executeQuery();
			while( resultSet.next() ) {
				ver = resultSet.getString(1);
			}
		}
		catch( SQLException ex ) {
			Closer.rollback( connection );		// 5.1.1.0 (2009/12/01)
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( pstmt );
		}
		return ver;
	}

	/**
	 * エンジン内部定義の初期リソース情報をDB(GE12)に登録します。
	 *
	 * 初期リソース情報は、KBSAKU='0' で登録されている情報で、一旦すべて削除
	 * してから、全てのリソース情報を追加するという形をとります。
	 * リソースは、すでに、Oracle XDK により XMLファイル化してあります。
	 * なお、この情報をDB登録する理由は、リソースの設定値を変えたい場合に、
	 * キーが判らない（JavaDOCからしか読み取れない）のでは不便な為に
	 * 用意しておくだけで、内部では SystemData オブジェクトとして定義
	 * されている値を使用するため、このデータベース値は、使用していません。
	 * 
	 * @og.rev 4.3.6.6 (2009/05/15) バージョン判定部分を分離
	 */
	private void dbXMLResourceInsert() throws UnsupportedEncodingException {
		// 新設定値を全件INSERTします。
		// common フォルダにセットして、ClassLoader で読み取る方法
		ClassLoader loader = Thread.currentThread().getContextClassLoader();
		InputStream stream = loader.getResourceAsStream( GE12_XML );

		Reader reader = new BufferedReader( new InputStreamReader( stream,"UTF-8" ) );
		HybsXMLSave save = new HybsXMLSave( connection,"GE12" );
		save.insertXML( reader );
		int insCnt = save.getInsertCount();
		int delCnt = save.getDeleteCount();

		out.print( "        XML Engine Resource Reconfiguration " );
		out.println( "DELETE=[" + delCnt + "],INSERT=[" + insCnt + "] finished." );
	}

}

