/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.URLConnect;

/**
 * 伝送要求に対してファイルを取得し、ローカルサーバーに保存します。
 *
 * この実行方法は、読取方法がHTTP受信(ファイル一覧)(HTTP_FILELIST)のみ使用することができます。
 * HTTP受信(ファイル一覧)により取得されたファイル一覧の各ファイルに対して、URL接続を行い、
 * 実行対象で指定された保存先にファイルを保存します。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_FILEGET implements TransferExec {

	// リモートコントロールサーブレット
	protected static final String REMOTE_SERVLET = "servlet/remoteControl";

	/**
	 * ファイルに書込みします。
	 * 
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		if( vals == null || vals.length == 0 ) { return; }

		String kbRead = config.getKbRead();
		if( !"HTTP_FILELIST".equals( kbRead ) ) {
			String errMsg = "実行方法(ファイル取得(FILEGET))を利用する場合、"
							+ "読取方法はHTTP受信(ファイル一覧)(HTTP_FILELIST)を指定して下さい。"
							+ "KBREAD=[" + kbRead + "]";
			throw new RuntimeException( errMsg );
		}

		String[] readObjArr = StringUtil.csv2Array( config.getReadObj(), ' ' );
		if( readObjArr[0] == null || readObjArr[0].length() == 0 ) {
			String errMsg = "受信元基準ディレクトリが取得できません。[READOBJ=" + config.getReadObj() + "]";
			throw new RuntimeException( errMsg );
		}
		File remoteFileDir = new File( readObjArr[0] );

		String hostPort = readObjArr[1];
		if( hostPort == null || hostPort.length() == 0 ) {
			String errMsg = "受信ホスト名が取得できません。[READOBJ=" + config.getReadObj() + "]";
			throw new RuntimeException( errMsg );
		}

		String saveBasePath = new File( config.getExecObj() ).getAbsolutePath();

		for( String val : vals ) {
			String saveFileName = null;
			if( remoteFileDir.isDirectory() ) {
				// 読取元がディレクトリの場合は、保存基準ディレクトリに相対パス名を付加して保存
				saveFileName = saveBasePath + val.replace( remoteFileDir.getAbsolutePath(), "" );
			}
			else {
				// 読取元がファイルの場合は、保存基準ディレクトリ＋ファイル名で保存
				String fileName = new File( val ).getName();
				saveFileName = saveBasePath + File.separatorChar + fileName;
			}

			File saveFile = new File( saveFileName );
			File parent = saveFile.getParentFile();
			if( !parent.exists() && !parent.mkdirs() ) {
				String errMsg = "保存ディレクトリの作成に失敗しました。file=[" + saveFileName + "]";
				throw new RuntimeException( errMsg );
			}

			URLConnect conn = null;
			InputStream is = null;
			OutputStream os = null;
			try {
				String url = hostPort + REMOTE_SERVLET + "?file=" + StringUtil.urlEncode( val );

				conn = new URLConnect( url, TransferConfig.HTTP_AUTH_USER_PASS );
				if( config.getProxyHost() != null && config.getProxyHost().length() > 0 ) {
					conn.setProxy( config.getProxyHost(),config.getProxyPort() );
				}

				conn.setCharset( "UTF-8" );
				conn.connect();
				is = conn.getInputStream();

				os = new BufferedOutputStream( new FileOutputStream( saveFileName ) ); 
				byte buf[] = new byte[4096]; 
				int len = 0; 
				while( ( len = is.read( buf ) ) != -1 ) { 
					os.write( buf, 0 ,len ); 
				}
				os.flush(); 
			}
			catch( IOException ex ) {
				String errMsg = "ファイル取得時にエラーが発生しました。file=[" + val + "]";
				throw new RuntimeException( errMsg, ex );
			}
			finally {
				Closer.ioClose( os );
				Closer.ioClose( is );

				if( conn != null ) { conn.disconnect(); }
			}
		}
	}
}
