/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.business;

import org.opengion.fukurou.util.ErrorMessage;

/**
 * 配列型テーブルモデルをメインカーソルとした業務ロジックの構造を定義します。
 * 
 * 配列型テーブルモデルについては、setTable( ArrayTableModel )によりセットします。
 * 配列型テーブルモデルが定義されていない場合、エラーとなります。
 * 
 * このクラスでは、以下に示すメソッドが呼び出されるタイミングのみを定義しています。
 * メソッドの中身については、サブクラスでオーバーライドし実装して下さい。
 * 
 * 処理が途中で中断される条件は、以下の3つです。
 * ①各メソッドの戻り値がfalseの場合
 * ②チェックメソッド(chk***())が全ての行で実行された後、エラーメッセージに"エラー"が含まれている場合
 * ③実行時エラーが発生した場合
 * 
 *  fstchk			  変更区分に関わらず   最初の行でのみ呼び出し
 *  befchk( int row ) 変更区分に関わらず   各行について呼び出し(insert,modify,deleteの前に呼び出し)
 *  inschk( int row ) 変更区分が"A"の場合 各行について呼び出し
 *  modchk( int row ) 変更区分が"C"の場合 各行について呼び出し
 *  delchk( int row ) 変更区分が"D"の場合 各行について呼び出し
 *	allchk( int row ) 変更区分に関わらず   各行について呼び出し(insert,modify,deleteの後に呼び出し)
 *  first()			  変更区分に関わらず   最初の行でのみ呼び出し
 *  befall( int row ) 変更区分に関わらず   各行について呼び出し(insert,modify,deleteの前に呼び出し)
 *  insert( int row ) 変更区分が"A"の場合 各行について呼び出し
 *  modify( int row ) 変更区分が"C"の場合 各行について呼び出し
 *  delete( int row ) 変更区分が"D"の場合 各行について呼び出し
 *  allrow( int row ) 変更区分に関わらず   各行について呼び出し(insert,modify,deleteの後に呼び出し)
 *  last()            変更区分に関わらず   最後の行でのみ呼び出し
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ロジック
 * 
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class BizLogic_TABLE extends AbstractBizLogic {

	/**
	 * 処理のメインロジックの前処理を記述します。
	 * (ここでは何もしません)
	 * 
	 * このメソッド自体は、protected属性であるため、サブクラスから直接参照することができます。
	 * 但し、これは、各業務ロジックで直接参照することを想定したものではなく、BizLogicの
	 * メイン構造を拡張するサブクラスを定義する際に使用することを想定しています。
	 * （この想定がなければ、本来は、package privateにすべきです)
	 * このため、業務ロジックを各実装クラスでは直接参照しないで下さい。
	 */
	@Override
	protected void init() {
		// Document empty method チェック対策
	}

	/**
	 * 処理のメインロジックを記述します。
	 * 
	 * このメソッド自体は、protected属性であるため、サブクラスから直接参照することができます。
	 * 但し、これは、各業務ロジックで直接参照することを想定したものではなく、BizLogicの
	 * メイン構造を拡張するサブクラスを定義する際に使用することを想定しています。
	 * （この想定がなければ、本来は、package privateにすべきです)
	 * このため、業務ロジックを各実装クラスでは直接参照しないで下さい。
	 *
	 * @og.rev 5.1.8.0 (2010/07/01) first,lastは行ループの中で呼び出し
	 * 
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean main() {
		if( table == null ) {
			throw new RuntimeException( "配列型テーブルモデルがセットされていません。" );
		}

		String modType = null;

		row = 0;
		if( !fstchk() ) { return false; }
		for( int i = 0; i < table.getRowCount(); i++ ) {
			if( !befchk( row ) ) { return false; }

			modType = table.getModifyType( row );
			if( "A".equals( modType ) ) {
				if( !inschk( row ) ) { return false; }
			}
			else if( "C".equals( modType ) ) {
				if( !modchk( row ) ) { return false; }
			}
			else if( "D".equals( modType ) ) {
				if( !delchk( row ) ) { return false; }
			}

			if( !allchk( row ) ) { return false; }

			row++;
		}
		if( getKekka() >= ErrorMessage.NG ) { return false; }

		row = 0;
//		if( !first() ) { return false; }
		for( int i = 0; i < table.getRowCount(); i++ ) {
			// 5.1.8.0 (2010/07/01) firstは行ループの中で呼び出し
			if( row == 0 ) {
				if( !first() ) { return false; }
			}

			if( !befall( row ) ) { return false; }

			modType = table.getModifyType( row );
			if( "A".equals( modType ) ) {
				if( !insert( row ) ) { return false; }
			}
			else if( "C".equals( modType ) ) {
				if( !modify( row ) ) { return false; }
			}
			else if( "D".equals( modType ) ) {
				if( !delete( row ) ) { return false; }
			}

			if( !allrow( row ) ) { return false; }

			// 5.1.8.0 (2010/07/01) lastは行ループの中で呼び出し
			if( row == table.getRowCount() - 1 ) {
				if( !last() ) { return false; }
			}
			row++;
		}
//		if( !last() ) { return false; }

		return true;
	}

	/**
	 * このクラスは、テーブルモデルが外部から指定されている必要があります。
	 * 
	 * このメソッド自体は、protected属性であるため、サブクラスから直接参照することができます。
	 * 但し、これは、各業務ロジックで直接参照することを想定したものではなく、BizLogicの
	 * メイン構造を拡張するサブクラスを定義する際に使用することを想定しています。
	 * （この想定がなければ、本来は、package privateにすべきです)
	 * このため、業務ロジックを各実装クラスでは直接参照しないで下さい。
	 * 
	 * @see AbstractBizLogic#isRequireTable()
	 * @return テーブルモデルが外部から指定されている必要があるかどうか（常にtrue）
	 */
	@Override
	protected boolean isRequireTable() {
		return true;
	}

	/**
	 * メインカーソルの一番初めで呼ばれるチェックロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @return 処理が正常終了したか
	 */
	protected boolean fstchk() {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の前)で呼ばれるチェックロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean befchk( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の後)で呼ばれるチェックロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean allchk( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分="A")で呼ばれるチェックロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean inschk( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分="C")で呼ばれるチェックロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean modchk( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分="D")で呼ばれるチェックロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean delchk( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの一番初めで呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @return 処理が正常終了したか
	 */
	protected boolean first() {
		return true;
	}

	/**
	 * メインカーソルの一番最後で呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @return 処理が正常終了したか
	 */
	protected boolean last() {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の前)で呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean befall( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の後)で呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean allrow( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分="A")で呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean insert( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分="C")で呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean modify( final int row ) {
		return true;
	}

	/**
	 * メインカーソルの各行(変更区分="D")で呼ばれるロジックを定義します。
	 * ここでは何も実装されていません。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	protected boolean delete( final int row ) {
		return true;
	}
}
