/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionImpl;

import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * コネクションを共有して、トランザクションを実現します。
 *
 * 通常のタグでは、コネクションプールより、その時々のコネクションを取り出して利用するため、
 * タグごとに異なるコネクションで処理されます。
 * また、commit や rollback などもそれぞれのタグで行われるため、連続処理時にエラーが
 * 発生しても、中途半端な状態になります。
 * ここでは、各 DBID 単位にコネクションを共有し、このタグの間は、同じオブジェクトを
 * commit や、rollback せずに使いまわすようにします。
 * これにより、複数タグ間のトランザクションや、異なる DBID 間のトランザクションを
 * 実現します。
 *
 * このタグは、doEndTag() メソッドが正常に呼び出されることで、トランザクションが成立します。
 * つまり、途中で、JSP出力が、SKIP_PAGE された場合は、commit もされません。
 * これは、データベースエラー以外のエラーでも、トランザクション処理されることを意味します。
 *
 * @og.formSample
 * ●形式：&lt;og:transaction &gt; ... &lt;/og:transaction &gt;
 * ●body：あり
 *
 * ●使用例
 *     &lt;og:transaction &gt;
 *         &lt;og:query command="NEW" dbid="SERVER1" &gt;
 *             insert into XX01 (aa,bb,cc) values ('AA','BB','CC') /&gt;
 *         &lt;/og:query &gt;
 *         &lt;og:query command="NEW" dbid="SERVER2" &gt;
 *             update YY02 set aa='AA',bb='BB',cc='CC' where uniq='00001' /&gt;
 *         &lt;/og:query &gt;
 *     &lt;/og:transaction &gt;
 *
 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
 * @og.group ＤＢ登録
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class TransactionTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.9.0 (2010/08/01)" ;
	private static final long serialVersionUID = 5190 ;	// 5.1.9.0 (2010/08/01)

	// TransactionTag では、Transaction インターフェースではなく、実装クラスで管理します。
	private TransactionImpl tran = null;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示( EVAL_BODY_INCLUDE )
	 */
	@Override
	public int doStartTag() {
		tran = new TransactionImpl( getApplicationInfo() );

		return( EVAL_BODY_INCLUDE );	// Body インクルード（ extends TagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// finish() は、TransactionImpl のメソッドです。
		if( tran != null ) { tran.finish(); }

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();

		// realClose() は、TransactionImpl のメソッドです。
		if( tran != null ) { tran.realClose(); }
		tran = null;
	}

	/**
	 * Transaction オブジェクトを返します。
	 *
	 * @return	Transaction オブジェクト
	 */
	protected Transaction getTransaction() {
		return tran ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
