/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsEntry;

import java.io.PrintWriter;
import java.util.List;

/**
 * TableWriter をXML形式で出力する為の実装クラスです。
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，XML形式ファイルの出力機能を実現しています。
 *
 * 出力のXML形式は、拡張オラクル ＸＤＫ形式のＸＭＬファイルです。
 * オラクルＸＤＫ形式のＸＭＬとは、下記のような ROWSET をトップとする ROW の
 * 集まりで１レコードを表し、各ROWには、カラム名をキーとするXMLになっています。
 *
 *   &lt;ROWSET&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;カラム1&gt;値1&lt;/カラム1&gt;
 *             ･･･
 *           &lt;カラムn&gt;値n&lt;/カラムn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *   &lt;ROWSET&gt;
 *
 * この形式であれば、XDK(Oracle XML Developer's Kit)を利用すれば、非常に簡単に
 * データベースとXMLファイルとの交換が可能です。
 * <a href="http://otn.oracle.co.jp/software/tech/xml/xdk/index.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * 拡張ＸＤＫ形式とは、ROW 以外に、SQL処理用タグ(EXEC_SQL)を持つ XML ファイルです。
 * これは、オラクルXDKで処理する場合、無視されますので、同様に扱うことが出来ます。
 * この、EXEC_SQL は、それそれの XMLデータをデータベースに登録する際に、
 * SQL処理を自動的に流す為の、SQL文を記載します。
 * この処理は、イベント毎に実行される為、その配置順は重要です。
 * このタグは、複数記述することも出来ますが、BODY部には、１つのSQL文のみ記述します。
 *
 *   &lt;ROWSET tablename="GEXX" &gt;
 *       &lt;EXEC_SQL&gt;                    最初に記載して、初期処理（データクリア等）を実行させる。
 *           delete from GEXX where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;カラム1&gt;値1&lt;/カラム1&gt;
 *             ･･･
 *           &lt;カラムn&gt;値n&lt;/カラムn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *       &lt;EXEC_SQL&gt;                    最後に記載して、項目の設定(整合性登録)を行う。
 *           update GEXX set AA='XX' , BB='XX' where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *   &lt;ROWSET&gt;
 *
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_XML extends TableWriter_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String	firstExecSql	= "" ;		// 4.0.0 (2005/01/31)
	private String	lastExecSql		= "" ;		// 4.0.0 (2005/01/31)
	private String	tableName		= "" ;		// 4.0.0 (2005/01/31)

//	/**
//	 * デフォルトコンストラクター
//	 *
//	 */
//	public TableWriter_XML() {
//		super();
//	}

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 *
	 * @param	writer PrintWriter
	 */
	public void writeDBTable( final PrintWriter writer )  {
		super.setHeaderSequence( "D" );
		super.writeDBTable( writer );
	}

	/**
	 * PrintWriter に DBTableModelのヘッダー情報を書き込みます。
	 * XML のヘッダーは、&lt;?xml version='1.0' encoding='encode値'?&gt; になります。
	 * encoding属性には、encode値をセットします。
	 * encoding属性に設定できる値は、"UTF-8","UTF-16","Shift_JIS" です。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 4.0.0 (2005/01/31) XML宣言部のencoding属性を、encode値をセットする。
	 *
	 * @param   writer PrintWriter
	 */
	protected void writeHeader( final PrintWriter writer ) {
		String encoding = getEncode();
		writer.println( "<?xml version='1.0' encoding='" + encoding + "'?>" ) ;
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * 出力のXML形式は、ORACLE XDK での出力ファイルと同じ形式ですので、直接データベースに
	 * 登録することができます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.8.0.1 (2005/06/17) DBTypeが NVAR の場合は、元のUnicodeに戻します。
	 * @og.rev 4.0.0 (2005/01/31) EXEC_SQL タグ機能の追加
	 * @og.rev 5.1.6.0 (2010/05/01) DbType の初期値(dbType)を利用する。
	 *
	 * @param   table DBTableModel
	 * @param   writer PrintWriter
	 */
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		int numberOfRows =  table.getRowCount();

		writer.print( "<ROWSET" );
		writer.print( tableName );
		writer.println( ">" );

		// 4.0.0 (2005/01/31)
		writer.println( firstExecSql );

		for( int row=0; row<numberOfRows; row++ ) {
			writer.print( "<ROW num=\"" );
			writer.print( row+1 );
			writer.println( "\">" );
			for( int i=0; i<numberOfColumns; i++ ) {
				int clm = clmNo[i];
				String val = table.getValue(row,clm);
//				if( "NVAR".equals( dbColumn[clm].getDbType()) ) {
				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}

				writer.print( "<" );
				writer.print( table.getColumnName(clm) );
				writer.print( ">" );
				writer.print( StringUtil.htmlFilter( val ) );
				writer.print( "</" );
				writer.print( table.getColumnName(clm) );
				writer.println( ">" );
			}
			writer.println( "</ROW>" );
		}

		// 4.0.0 (2005/01/31)
		writer.println( lastExecSql );
		writer.println( "</ROWSET>" );
	}

	/**
	 * パラメーターリストをセットします。
	 * 内部は、HybsEntry クラスを持っています。
	 * 引数が、null の場合は、何もしません。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @param   listParam List<HybsEntry> パラメーターリスト
	 */
	public void setParam( final List<HybsEntry> listParam ) {
		if( listParam != null && ! listParam.isEmpty() ) {
			StringBuilder first = new StringBuilder() ;
			StringBuilder last  = new StringBuilder() ;

			HybsEntry[] entries = listParam.toArray( new HybsEntry[listParam.size()] );
			for( int i=0; i<entries.length; i++ ) {
				String key = entries[i].getKey();
				String val = entries[i].getValue();
				if( "First".equalsIgnoreCase( key ) ) {
					first.append( "<EXEC_SQL>" );
					first.append( StringUtil.htmlFilter( val ) );
					first.append( "</EXEC_SQL>" );
					first.append( HybsSystem.CR );
				}
				else if( "Last".equalsIgnoreCase( key ) ) {
					last.append( "<EXEC_SQL>" );
					last.append( StringUtil.htmlFilter( val ) );
					last.append( "</EXEC_SQL>" );
					last.append( HybsSystem.CR );
				}
				// 複数の TableName があれば、最後のみが有効。(チェックなし)
				else if( "TableName".equalsIgnoreCase( key )
							&& val != null && val.length() > 0 ) {
					tableName = " tableName=\"" + val + "\"" ;
				}
			}
			firstExecSql = first.toString();
			lastExecSql  = last.toString();
		}
	}
}
