/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBLastSql;
// import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.html.ViewForm;
import org.opengion.hayabusa.html.ViewFormFactory;
import org.opengion.hayabusa.html.ViewMarker;
import org.opengion.hayabusa.html.TableFormatter;
import org.opengion.hayabusa.html.FormatterType;
import org.opengion.hayabusa.resource.UserInfo;
import org.opengion.hayabusa.resource.RoleMode;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.XHTMLTag;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.Locale ;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * データベースの検索結果を表示するタグです。
 *
 * 検索結果は、DBTableModel にセットされます。
 * このタグに、データ（DBTableModel)と、
 * コントローラ（ViewForm)を与えて、 外部からコントロールすることで、テキストフィールドやテーブルの形で表示したり、
 * 入力可/不可、表示可/不可の設定を行うことができます。
 *
 * @og.formSample
 * ●形式：&lt;og:view /&gt;
 *     &lt;og:view
 *         viewFormType = &quot;HTMLTable&quot;
 *         command      = &quot;{&#064;command}&quot;
 *         startNo      = &quot;0&quot;
 *         pageSize     = &quot;20&quot;
 *         language     = &quot;ja&quot; /&gt;
 *
 *     viewFormType : ViewFormオブジェクトを作成する時のキー（オプション)     ViewFormFactory参照
 *     command      : 処理コマンド（ PREV,NEXT,FIRST,LAST,VIEW ）（オプション）
 *     startNo      : データの書き始めの初期値を指定（オプション）
 *     pageSize     : データのページ数（画面表示件数）を指定（オプション）
 *     language     : ロケール（オプション）
 *
 * ●body：あり
 *
 * ●使用例
 *       &lt;og:view viewFormId="HTMLTable" command="NEW" startNo="0" pageSize="5" / &gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewFormTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  プレビュー {@value}  */
	public static final String CMD_PREV  = "PREV" ;
	/** command 引数に渡す事の出来る コマンド  ネクスト {@value}  */
	public static final String CMD_NEXT  = "NEXT" ;
	/** command 引数に渡す事の出来る コマンド  ファースト {@value}  */
	public static final String CMD_FIRST = "FIRST";
	/** command 引数に渡す事の出来る コマンド  ラスト {@value}  */
	public static final String CMD_LAST  = "LAST" ;
	/** command 引数に渡す事の出来る コマンド  新規 {@value}  */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value}  */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド  表示 {@value}  */
	public static final String CMD_VIEW  = "VIEW" ;
	/** command 引数に渡す事の出来る コマンド  再表示 {@value}  */
	public static final String CMD_REVIEW  = "REVIEW" ;
	/** command 引数に渡す事の出来る コマンド  リセット {@value}  */
	public static final String CMD_RESET  = "RESET" ;
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = new String[] {
			CMD_PREV , CMD_NEXT , CMD_FIRST , CMD_LAST , CMD_NEW , CMD_RENEW , CMD_VIEW , CMD_REVIEW , CMD_RESET };

	// 3.8.0.8 (2005/10/03)
	/** コマンドリストに存在しない場合の値 従来の CMD_VIEW と区別させるため。 */
	private static final String CMD_XXX  = "XXX" ;

	private   static final String START_NO_KEY		 = "startNo";
	private   static final String PAGE_SIZE_KEY 	 = "pageSize";
	private   static final String PAGE_PLUS_KEY 	 = "pagePlus";

	// 3.5.6.3 (2004/07/12) スクロールバー用のDIV 要素属性定義

	private String		tableId			= HybsSystem.TBL_MDL_KEY;
	private String		viewFormId		= HybsSystem.VIEWFORM_KEY;
	private String		viewFormType	= null;
	private String		viewLinkId		= HybsSystem.VIEWLINK_KEY;
	private String		viewMarkerId	= HybsSystem.VIEW_MARK_KEY;
	private String		editMarkerId	= HybsSystem.EDIT_MARK_KEY;		// 3.8.6.1 (2006/10/20)
	private String		command			= null;
	private int			startNo			= -1;		// HTML画面上の表示開始位置（初期値）
	private int			tempStartNo		= -1;		// 再検索時の表示開始位置
	private int			pageSize 		= -1;		// HTML画面上に表示する件数
	private int			pagePlus 		= -1;		// 3.5.6.4 (2004/07/16)
	private transient DBTableModel	table		= null;
	private transient ViewForm		form 		= null;
	private String		columnWritable	= null;
	private String		noWritable		= null;
	private String		columnDisplay	= null;
	private String		clmSeqDisplay	= null;		// 5.1.6.0 (2010/05/01) ViewForm_HTMLSeqClmTable 対応
	private String		columnGroup		= null;		// 3.8.5.0 (2006/03/20)
	private String		noDisplay		= null;
	private boolean		noMessage		= false;
	private String		writable 		= null;
	private String		checked			= null;
	private boolean		skip 			= ViewForm.DEFAULT_SKIP;
	private String		rowspan			= "2";
	private String		selectedType	= "checkbox";
	private String		numberType		= "sequence";		// 3.5.1.0 (2003/10/03) 追加
	private String		optTypeAttri	=  null;
//	private String[]	changeLabelKeys	= null;
//	private String[]	changeLabelVals	= null;
	private int			backLinkCount	= HybsSystem.sysInt( "VIEW_BACK_LINK_COUNT" );
	private int			headerSkipCount	= HybsSystem.sysInt( "VIEW_HEADER_SKIP_COUNT" );	// 0:通常ヘッダ、n:n回ごとに現れる
	private List<TableFormatter>		values			= null;
	private String		bodyString		= null;
	private int			useCheckControl = -1 ;
	private String		useTableSorter	= null ;	// 3.5.4.7 (2004/02/06)

	private long dyStart = 0;	// 3.5.4.7 (2004/02/06) 実行時間測定用のDIV要素を出力します。
	private boolean		useParam		= false;	// 3.5.4.8 (2004/02/23) パラメータTagの使用を宣言します。
	private Map<String,String> viewParam = null;	// 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを受け取ります。

	// 3.5.5.8 (2004/05/20) 整合性チェックを行う為の、Consistency キー を出力するかどうかを指定します。
	private boolean		useConsistency	= HybsSystem.sysBool( "USE_CONSISTENCY" ) ;

	// 3.5.6.2 (2004/07/05) ゼブラ模様の指定を、ViewForm としてサポート
	// 0(ゼブラなし)、-1（ワーニング)、-2以下（エラー）、1(ゼブラ）、2以上(行数まとめ）
	private int bgColorCycle = 1 ;

	private String		useScrollBar	= null ;

	// 3.6.0.0 (2004/09/17) カラムソートリンクを出すカラム名を指定。
	private String		sortKeys	= null;

	// 3.6.0.4 (2004/10/14) viewLink や viewMarker の情報取り込みを制御
	// 前後の , が重要です！
	private String		loadMarker	= ",NEW,RENEW,REVIEW,RESET,";

	// 3.7.1.1 (2005/05/23) SEL_ROW機能の有効/無効を指定します。
	private String		useSelectedRow	= null ;

	// 3.7.1.1 (2005/05/31) SEL_ROW機能使用時に、BG_COLOR_ROWSEL の使用 有/無を指定します。
	private String		useSelRowColor	= null;

	// 3.7.1.1 (2005/05/23) 処理時間やページサイズなどの情報出力の有効/無効を指定します。
	private boolean		useAfterHtmlTag	= true ;

	// 3.8.0.9 (2005/10/17) 設定値に "_" が含まれている場合に、レンデラーを使用します。
	private String		writableControl	= null;

	// 3.8.6.0 (2006/09/29) HILIGHT_ROW機能の有効/無効を指定します。
	private String		useHilightRow	= null;

	// 3.8.6.1 (2006/10/20) ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	private String		popupReturnKeys	= null ;

	// 4.0.0 (2007/04/16) tableタグを出力するときに付与します。
	private String		clazz			= "viewTable";
	
	// 4.2.0.0 (2008/03/18) ビューの幅と高さを指定できるようにします。
	private String		height			= null;
	private String		width			= null;

	// 4.3.0.0 (2008/07/04) fileUD ファイルダウンロード時の抜出カラム名(CSV形式)
	private String		fileDwnClms		= null ;
	
	// 4.3.1.0 (2008/09/08)
	private boolean		skipNoEdit		= false;

	// 4.3.3.0 (2008/10/01)
	private boolean		noTransition	= false;

	// 4.3.7.0 (2009/06/01) HILIGHT_ROW機能有効時に制御を行チェックボックスのみにします。
	private String		useHilightCheckOnly	= null;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) DBTableModelのソート機能 useTableSorter を追加
	 * @og.rev 3.5.4.7 (2004/02/06) 実行時間測定用に、開始時刻を取得します。
	 * @og.rev 3.5.4.8 (2004/02/23) useParam を、BODY部の処理判定に加えます。
	 * @og.rev 3.5.6.0 (2004/06/18) DBRowHeader のパッケージプライベート化に伴なう変更
	 * @og.rev 3.5.6.1 (2004/06/25) viewFormType変更時のviewForm 再セット
	 * @og.rev 3.5.6.2 (2004/07/05) command="VIEW" などで、ありえないとしていた処理を追加
	 * @og.rev 3.5.6.3 (2004/07/12) markerSet で table の整合性もチェックする。
	 * @og.rev 3.5.6.4 (2004/07/16) DBTableModelのソート時には、viewを初期設定する。
	 * @og.rev 3.6.0.0 (2004/09/17) scope が request の場合（検索）ヘッダーソートリンクは使えない。
	 * @og.rev 3.6.0.0 (2004/09/22) pageSize の form への設定見直し。
	 * @og.rev 3.6.0.0 (2004/09/25) viewLink , viewMarker のリクエストメモリ削除の中止
	 * @og.rev 3.6.0.4 (2004/10/14) viewLink や viewMarker の情報取り込みを制御する loadMarker を追加
	 * @og.rev 3.6.0.5 (2004/10/18) viewLink , viewMarker のリクエストメモリ削除を復活します。
	 * @og.rev 3.8.0.8 (2005/10/03) ヘッダーソート時に、startNo を 0 に戻します。
	 * @og.rev 3.8.8.5 (2007/03/09) markerSet する条件を強化する。
	 * @og.rev 3.8.8.9 (2007/05/29) ソート時のviewLink,viewMarker の取り込み
	 * @og.rev 3.8.9.3 (2007/08/28) soryKeysのみを指定した場合に、ソート用のテーブルモデルが作成されないバグを修正。
	 * @og.rev 4.1.0.0（2008/01/11）システムパラメータ（useCheckControl）の読取は doEndTag 関数から doStartTag 関数に移動します。
	 * @og.rev 5.1.6.0 (2010/05/01) ViewForm_HTMLSeqClmTable クラス によるカラム表示順処理を行います。
	 * @og.rev 5.1.6.0 (2010/05/01) 左右分割スクロール画面でヘッダースクロールが2回処理されるバグを修正
	 * @og.rev 5.1.6.0 (2010/05/01) ユーザー毎の画面項目並び替え対応
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();
		table = (DBTableModel)getObject( tableId );
		if( table == null ) { return(SKIP_BODY); }

		// 5.1.6.0 (2010/05/01) seqClmTableを使用するかどうかをチェックします
		checkUseSeqClmTable();

		// 3.8.8.9 (2007/05/29) ソート時の viewLink,viewMarker の取り込み
		boolean useSorter = false;

		// 3.5.4.7 (2004/02/06) DBTableModelのソート機能 追加
		// 登録後に同じソート順で再表示させるために、リクエストキャッシュに登録します。
		// 3.8.9.3 (2007/08/28) soryKeysのみを指定した場合に、ソート用のテーブルモデルが作成されないバグを修正。
		// 5.1.6.0 (2010/05/01) 左右分割スクロール画面でヘッダースクロールが2回処理されるバグを修正
		if( "session".equals( getScope() )
			&& ( nval( useTableSorter,sysBool( "VIEW_USE_TABLE_SORTER" ) ) || sortKeys != null )
			&& !( table.getConsistencyKey().equals( getRequestAttribute( HybsSystem.VIEW_SORT_KEY ) ) ) ) {

			String clm = getRequestValue( HybsSystem.SORT_COLUMNS );
			// 若干トリッキー。初めてクリックされるまで、DBTableModelSorter は作成しない。
			if( clm != null && clm.length() > 0 ) {
				useSorter = true;	// 3.8.8.9 (2007/05/29)
				startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
				int clmNo = table.getColumnNo( clm );
				boolean sortAscending = nval( getRequestValue( HybsSystem.SORT_ASCENDING ),true );
				if( table instanceof DBTableModelSorter ) {
					if( !CMD_RESET.equals( command ) ) {
						((DBTableModelSorter)table).sortByColumn( clmNo );
					}
						sortAscending = ((DBTableModelSorter)table).isAscending() ;	// ソート方向
				}
				else {
					DBTableModelSorter temp = new DBTableModelSorter();
					temp.setModel( table );
					temp.sortByColumn( clmNo,sortAscending );
					table = temp;
				}
				// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
				if( ! commitTableObject( tableId, table ) ) {
					jspPrint( "ViewFormTag Query処理が割り込まれました。DBTableModel は登録しません。" );
					return (SKIP_PAGE);
				}
				if( CMD_VIEW.equals( command ) ) {
					setRequestCacheData( HybsSystem.SORT_COLUMNS,clm );
					setRequestCacheData( HybsSystem.SORT_ASCENDING,String.valueOf( sortAscending ) );
				}

				// 5.1.6.0 (2010/05/01)
				setRequestAttribute( HybsSystem.VIEW_SORT_KEY, table.getConsistencyKey() );

	 			// 3.8.0.8 (2005/10/03) ヘッダーソート時に、startNo を 0 に戻します。
				startNo = 0;
				tempStartNo = startNo;
			}
		}

		if( CMD_NEW.equals( command ) || CMD_RENEW.equals( command ) || CMD_REVIEW.equals( command ) || CMD_RESET.equals( command ) ) {
			form = ViewFormFactory.newInstance( viewFormType );		// 3.5.6.1 (2004/06/25)
//			dbChangeLabel();
			form.init( table );
			table.setDefaultRowWritable( DBTableModel.DEFAULT_WRITABLE );
			table.setDefaultRowChecked(  DBTableModel.DEFAULT_CHECKED );
		}
		else {
			// 3.5.6.1 (2004/06/25) viewFormType変更時のviewForm 再セット
			ViewForm oldForm = (ViewForm)getObject( viewFormId );
			if( oldForm != null ) {
				if( oldForm.getViewFormType().equals( viewFormType ) ) {
					form = oldForm ;
				}
				else {
					form = ViewFormFactory.newInstance( viewFormType );
				}
				form.init( table );
				// 3.5.6.3 (2004/07/12) markerSet で table の整合性もチェックする。
				DBTableModel oldTable = oldForm.getDBTableModel();
				if( table.getConsistencyKey().equals( oldTable.getConsistencyKey() ) ) {
					// 3.8.8.5 (2007/03/09) markerSet する条件を強化する。
					// 3.8.8.9 (2007/05/29) ソート時の viewLink,viewMarker の取り込み
					if( "true".equalsIgnoreCase( loadMarker ) ||
						loadMarker.indexOf( "," + command + "," ) >= 0 ||
						useSorter ) {
							form.markerSet( oldForm );
					}
				}
			}
			else {
				// 3.5.6.2 (2004/07/05) command="VIEW" などで、ありえないとしていた処理を追加
				form = ViewFormFactory.newInstance( viewFormType );
				form.init( table );
			}
		}

		// 3.6.0.4 (2004/10/14) ViewMarker 等のセットは、loadMarker で制御します。
		if( "true".equalsIgnoreCase( loadMarker ) ||
			loadMarker.indexOf( "," + command + "," ) >= 0 ) {
			ViewMarker viewLink = (ViewMarker)getRequestAttribute( viewLinkId );
			if( viewLink != null ) {
				form.setViewLink( viewLink );
				removeRequestAttribute( viewLinkId );	// 3.6.0.5 (2004/10/18) 復活
			}

			ViewMarker viewMarker = (ViewMarker)getRequestAttribute( viewMarkerId );
			if( viewMarker != null ) {
				form.setViewMarker( viewMarker );
				removeRequestAttribute( viewMarkerId );	// 3.6.0.5 (2004/10/18) 復活
			}

			// 3.8.6.1 (2006/10/24) editMarker を追加する。
			ViewMarker editMarker = (ViewMarker)getRequestAttribute( editMarkerId );
			if( editMarker != null ) {
				form.setEditMarker( editMarker );
				removeRequestAttribute( editMarkerId );
			}
		}
		// 3.8.0.5 (2005/08/26) false の場合は、無条件に、form のviewLinkとViewMarkerをクリアします。
		if( "false".equalsIgnoreCase( loadMarker ) ) {
			form.setViewLink( null );
			form.setViewMarker( null );
			form.setEditMarker( null );		// 3.8.6.1 (2006/10/24)
		}
		// 4.1.0.0（2008/01/11）システムパラメータ（useCheckControl）の読取は doEndTag 関数から doStartTag 関数に移動します。
		if( useCheckControl < 0 ) {
			useCheckControl = intCase( sys( "VIEW_USE_CHECK_CONTROL" ) ) ;
		}
		setTableRowWritable( table );

		setDefaultParameter();

	//	if( pagePlus != 0 ) {	※ なぜ != だったのか謎。pagePlusがマイナス時は？
		if( pagePlus > 0 ) {
			form.setPageSize( pageSize + pagePlus );
		}
		else {
			form.setPageSize( pageSize );				// 3.6.0.0 (2004/09/22) 追加
		}

		// 3.5.4.8 (2004/02/23)
		if( form.canUseFormat() || useParam ) { return( EVAL_BODY_BUFFERED ); }	// JSP API 1.2

		return( SKIP_BODY );
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行ｖう。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 3.5.4.0 (2003/11/25) 外部からフォーマットが指定されない場合は、内部のフォーマットを登録する。
	 *
	 * @return	int
	 */
	@Override
	public int doAfterBody() {
		bodyString = getBodyString();
		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.0.0 (2003/03/28) noClass 属性の追加。および、ロジック整理。
	 * @og.rev 3.5.1.0 (2003/10/03) numberType 属性の追加
	 * @og.rev 3.5.2.0 (2003/10/20) headerSkipCount属性を追加
	 * @og.rev 3.5.2.0 (2003/10/20) form が null の場合の条件分岐を追加
	 * @og.rev 3.5.3.1 (2003/10/31) Table系のViewで、skip属性を使用する。
	 * @og.rev 3.5.4.0 (2003/11/25) command="RESET"時に、戻った画面のpageSize がクリアされてしまう対応。
	 * @og.rev 3.5.4.0 (2003/11/25) フォーマット文字列を、TableFormatter オブジェクトを使用するように変更。
	 * @og.rev 3.5.4.1 (2003/12/01) startNo を隠しフィールドに持つ処理を廃止。（キャッシュを使用）
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl属性を追加
	 * @og.rev 3.5.4.7 (2004/02/06) DBTableModelのソート機能 useTableSorter を追加
	 * @og.rev 3.5.4.7 (2004/02/06) makeStartNoKeyメソッドの廃止（setRequestCacheDataを利用）
	 * @og.rev 3.5.4.7 (2004/02/06) 実行時間測定用のDIV要素を出力しておきます。
	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 * @og.rev 3.5.5.5 (2004/04/23) DBTableModel の整合性パラメータを hidden で出力します。
	 * @og.rev 3.5.5.8 (2004/05/20) makeConsistencyKey メソッドに処理ロジックを移します。
	 * @og.rev 3.5.6.2 (2004/07/05) bgColorCycle 変数の追加。
	 * @og.rev 3.5.6.4 (2004/07/16) スクロールバー用のDIV要素の出力(開始)
	 * @og.rev 3.6.0.0 (2004/09/17) カラムソートリンクを出すカラム名を指定。
	 * @og.rev 3.6.0.0 (2004/09/22) pageSize の hidden 出力は、初期設定値とします。
	 * @og.rev 3.7.0.3 (2005/03/01) 指定の行番号まで画面をスクロールさせる機能を追加。
	 * @og.rev 3.7.1.1 (2005/05/23) 処理時間やページサイズなどの情報出力の有効/無効を指定します。
	 * @og.rev 3.8.0.3 (2005/07/15) ヘッダー固定のスクロールタイプをセットします。
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/09/29) useHilightRow 属性を追加します。
	 * @og.rev 3.8.6.1 (2006/10/20) ポップアップで値を返すカラム文字列(CSV形式)
	 * @og.rev 4.0.0.0 (2007/11/27) ヘッダーカラムのエディター、レンデラー適用対応
	 * @og.rev 4.1.2.0 (2008/03/12) columnWritable,noWritable に、"null" を指定可能とする。
	 * @og.rev 4.2.0.0 (2008/03/18) ビューの高さ、幅の指定を可能にします。
	 * @og.rev 4.2.1.0 (2008/04/11) null チェック方法変更
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードによる設定
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応。抜き出すカラムを指定します。
	 * @og.rev 4.3.1.0 (2008/09/08) canUseFormatのエラーハンドリングは各サブクラスで行う。editOnly属性対応。
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition 属性を追加します。
	 * @og.rev 4.3.6.0 (2009/04/01) eventColumn対応
	 * @og.rev 4.3.7.0 (2009/06/01) 行ハイライト機能でチェックボックスでのみON/OFFができるオプションを追加
	 * @og.rev 4.3.8.0 (2009/08/01) noTransition値取得メソッド名変更
	 * @og.rev 5.1.3.0 (2010/02/01) noTransitionのコントロールは、requestで行う。
	 * @og.rev 5.1.6.0 (2010/05/01) fileDwnClmsが指定されない場合は、Viewで表示されたカラムをセット
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// noTransition = isNoTransitionRequest() || noTransition; // 4.3.3.0 (2008/10/01) 追加
//		noTransition = isNoTransitionSession() || noTransition; // 4.3.8.0 (2009/08/01)
		noTransition = isNoTransitionRequest(); // 5.1.3.0 (2010/02/01)
		if( form == null ) { return(EVAL_PAGE); }	// 3.5.2.0 (2003/10/20)
		if( form.canUseFormat() ) {
//			if( values == null && bodyString != null ) { // 4.3.1.0 (2008/09/08)
			if( values == null && bodyString != null && !useParam ) {
				TableFormatter format = new TableFormatter();
//				format.setFormatType( TableFormatter.TYPE_HEAD );
				format.setFormatType( FormatterType.TYPE_HEAD );
				format.setFormat( bodyString );
				format.setRowspan( rowspan );
				values = new ArrayList<TableFormatter>();
				values.add( format );
				form.setFormatterList( values );
			}
			else if( values != null ) {
				form.setFormatterList( values );
			}
			// 4.3.1.0 (2008/09/08)
//			else {
//				String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
//				throw new HybsSystemException( errMsg );
//			}
		}

		if( table != null ) {
			// 4.1.2.0 (2008/03/12) columnWritable,noWritable に、"null" を指定可能とする。
			// 4.2.1.0 (2008/04/11) null チェック方法変更
			if( noWritable != null && columnWritable != null ) {
				String errMsg = "columnWritable と noWritable は同時に指定できません。["
								+ "columnWritable = " + columnWritable
								+ " , noWritable = " + noWritable
								+ "]";
				throw new HybsSystemException( errMsg );
			}

//			if( "null".equalsIgnoreCase( columnWritable ) ) {
//				columnWritable = "" ;	noWritable = null;
//			}
//			else if( "null".equalsIgnoreCase( noWritable ) ) {
//				columnWritable = null ;	noWritable = "";
//			}

			// 4.2.0.0 (2008/03/18) ビューの高さ、幅を指定する。null回避はAbstractViewFormで行う
			form.setHeight( height );
			form.setWidth( width );
			
			form.setColumnWritable( columnWritable );
			form.setNoWritable( noWritable );
			form.setColumnDisplay( columnDisplay );
			form.setColumnGroup( columnGroup );				// 3.8.5.0 (2006/03/20)
			form.setNoDisplay( noDisplay );
			form.setTableSorterKeys( sortKeys );			// 3.6.0.0 (2004/09/17) 追加
			form.setNoMessage( noMessage );
			form.setSelectedType( selectedType );
//			if( useCheckControl < 0 ) {
//				useCheckControl = intCase( sys( "VIEW_USE_CHECK_CONTROL" ) ) ;
//			}
			form.setUseCheckControl( useCheckControl );		// 3.5.4.3 (2004/01/05) 追加
			form.setNumberType( numberType );				// 3.5.1.0 (2003/10/03) 追加
			form.setOptionTypeAttributes( optTypeAttri );
			form.setBackLinkCount( backLinkCount );
			form.setHeaderSkipCount( headerSkipCount );		// 3.5.2.0 (2003/10/20) 追加
			if( useParam ) { form.setParam( viewParam ); }	// 3.5.4.8 (2004/02/23)
			form.setBgColorCycle( bgColorCycle );			// 3.5.6.2 (2004/07/05) 追加
			form.setWritableControl( writableControl );		// 3.8.0.9 (2005/10/17) 追加

			// 3.6.0.0 (2004/09/22) pageSize の hidden 出力は、初期設定値とします。
			int oldPageSize = pageSize;
			pageSize = form.getPageSize();
			commandExec( command );

			form.setStartNo( startNo );
			form.setSkip( skip );							// 3.5.3.1 (2003/10/31)
			form.setSkipNoEdit( skipNoEdit );				// 4.3.1.0 (2008/09/08)
			boolean useTblSorter = "session".equals( getScope() ) &&
						nval( useTableSorter,sysBool( "VIEW_USE_TABLE_SORTER" ) );
			form.setUseTableSorter( useTblSorter );		// 3.5.4.7 (2004/02/06)

			// 3.6.0.0 (2004/09/17)
			// tableSorter 時のリンクに追加するリクエスト変数のキャッシュMapを追加する。
			if( useTblSorter || sortKeys != null ) {
				// 4.0.0 (2005/01/31)
				String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );
				Map<?,?> mem = (Map<?,?>)getSessionAttribute( memKey );		// 4.3.3.6 (2008/11/15) Generics警告対応
				form.makeTableSorterQuery( mem );
			}

			// 3.5.6.4 (2004/07/16) スクロールバー用のDIV要素の出力(開始)
			// 3.8.0.3 (2005/07/15) ヘッダー固定のスクロールタイプをセットします。
			// 4.0.0 (2007/04/10) ヘッダー固定のスクロールタイプは、一つにする。
	//		String hctype = (String)getRequest().getAttribute( HybsSystem.HEADER_FIX_TYPE );
			boolean useScBar = nval( useScrollBar,sysBool( "VIEW_USE_SCROLLBAR" ) );
	//		form.setUseScrollBar( useScBar,hctype );
			form.setUseScrollBar( useScBar );

			// 3.8.6.1 (2006/10/20) ポップアップで値を返すカラム文字列(CSV形式)
			form.setPopupReturnKeys( popupReturnKeys );
			form.setTableClass( clazz );		// 4.0.0 (2007/04/16)

			// 3.7.0.3 (2005/03/01) 指定の行番号まで画面をスクロールさせる機能を追加。{I} は無視する。
			String rowval = nval( getRequestValue( "SEL_ROW" ),null ) ;

			boolean useSelRow = nval( useSelectedRow,sysBool( "VIEW_USE_SELECTED_ROW" ) );
			if( useSelRow && rowval != null && !rowval.equalsIgnoreCase( "{I}" ) ) {
				boolean useSelRowClr = nval( useSelRowColor,sysBool( "VIEW_USE_SELROW_COLOR" ) ) ;
				form.setScrollRowNo( Integer.parseInt( rowval ),useSelRowClr );
			}

			// 4.0.0 (2005/11/30) カラムロールを判定して、アクセス不可の場合は表示させないようにする。
			int clmSize = table.getColumnCount();
			UserInfo userInfo = getUser();
			for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
				DBColumn dbColumn = table.getDBColumn( clmNo );
				
				// 4.3.6.0 (2009/04/01) セッションに生SQLを登録
				String evCol = dbColumn.getEventColumn();
				if( evCol != null && evCol.length() > 0 ){
					addEventColumnSQL( dbColumn.getName(), dbColumn.getRawParam() );
				}
				
				// 4.3.0.0 (2008/07/04) ロールモードによる設定
				byte bitMode = userInfo.getAccessBitMode( dbColumn.getRoleMode() );
				// アクセス不許可の場合のみ、セットする。
				// それ以外は、columnDisplay ,noDisplay 属性が適用されている。
				if( !RoleMode.isRead( bitMode ) ) {
					form.setColumnDisplay( clmNo,false );
				}
				// 書込み不許可の場合のみ、セットする。
				// それ以外は、columnWritable ,noWritable 属性が適用されている。
				if( !RoleMode.isWrite( bitMode ) ) {
					form.setColumnWritable( clmNo,false );
				}
//				boolean ok = userInfo.isAccess( dbColumn.getRoles() );
//				// アクセス不許可の場合のみ、セットする。
//				// それ以外は、columnDisplay ,noDisplay 属性が適用されている。
//				if( !ok ) { form.setColumnDisplay( clmNo,false ); }
			}

			// 3.8.6.0 (2006/09/29) useHilightRow 属性追加
			boolean useHiRow = nval( useHilightRow,sysBool( "VIEW_USE_HILIGHT_ROW" ) );
			if( useHiRow ) {
//				jspPrint( "<div onclick=\"hilightRow();\">" );
				// 4.3.7.0 (2009/06/01)
				boolean useHiCkOnly = nval( useHilightCheckOnly,sysBool( "VIEW_USE_HILIGHT_CHECK_ONLY" ) );
				jspPrint( "<div onclick=\"hilightRow( event, " + useHiCkOnly + ");\">" );
			}

			// 4.0.0.0 (2007/11/27) 超特殊処理 クロス集計時にヘッダを動的に作成するために使用
			if( viewParam != null && "true".equalsIgnoreCase( viewParam.get( "useHeaderColumn" ) ) ) {
				form.setResourceManager( getResource() );
			}

			form.setNoTransition( noTransition ); // 4.3.3.0 (2008/10/01) 追加

			jspPrint( form.create( startNo,pageSize ) );

			if( useHiRow ) {
				jspPrint( "</div>" );
			}

			setRequestCacheData( START_NO_KEY,String.valueOf( tempStartNo ) );	// 3.5.4.7 (2004/02/06)
			// 3.6.0.0 (2004/09/22) pageSize の hidden 出力は、初期設定値とします。
			if( useAfterHtmlTag ) {	// 3.7.1.1 (2005/05/23)
				jspPrint( makePageSize( oldPageSize ) );		// 3.6.0.0 (2004/09/22)
			}

			makeConsistencyKey();		// 3.5.5.8 (2004/05/20)

			setObject( viewFormId, form );

			// 3.5.4.8 (2004/02/23) 超特殊処理 DBTableModel を View で置き換えます。
			if( viewParam != null ) {
				String saveTableId = viewParam.get( "saveTableId" );
				if( saveTableId != null ) {
					setObject( saveTableId, form.getDBTableModel() );
				}
			}
		}

		if( useAfterHtmlTag ) {	// 3.7.1.1 (2005/05/23)
			// 3.5.4.7 (2004/02/06)
			long dyTime = System.currentTimeMillis()-dyStart;
			jspPrint( "<div id=\"viewTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		}

		// 5.1.6.0 (2010/05/01) Viewで表示した項目をDBLastSqlにセットします。
		setViewClms();

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.2.0 (2003/10/20) headerSkipCount属性を追加
	 * @og.rev 3.5.3.0 (2003/10/27) backLinkCount,headerSkipCount属性の初期値をPropertyファイルから設定
	 * @og.rev 3.5.4.0 (2003/11/25) format 関係の文字列を廃止。TableFormatter クラスを使用する。
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl属性を追加
	 * @og.rev 3.5.4.7 (2004/02/06) DBTableModelのソート機能 useTableSorter を追加
	 * @og.rev 3.5.4.7 (2004/02/06) 実行時間測定用に、dyStart を追加します。
	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 * @og.rev 3.5.5.8 (2004/05/20) useConsistency 属性を追加します。
	 * @og.rev 3.5.6.2 (2004/07/05) bgColorCycle 変数の追加。
	 * @og.rev 3.5.6.3 (2004/07/12) useScrollBar 変数の追加。
	 * @og.rev 3.6.0.0 (2004/09/17) sortKeys 変数の追加。カラムソートリンクを出すカラム名を指定。
	 * @og.rev 3.6.0.4 (2004/10/14) loadMarker 変数の追加。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール変更(useCheckControl)
	 * @og.rev 3.7.1.1 (2005/05/23) SEL_ROW機能の有効/無効を指定します。
	 * @og.rev 3.7.1.1 (2005/05/23) 処理時間やページサイズなどの情報出力の有効/無効を指定します。
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 属性を追加します。
	 * @og.rev 3.8.6.0 (2006/09/29) useHilightRow 属性を追加します。
	 * @og.rev 3.8.6.1 (2006/10/20) editMarkerId , popupReturnKeys属性を追加します。
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD ファイルダウンロード時の抜出カラム名(CSV形式)
	 * @og.rev 4.3.1.0 (2008/09/08) isSkipNoEdit属性対応
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition 属性を追加します。
	 * @og.rev 4.3.7.0 (2009/06/01) useHilightCheckOnly 属性を追加します。
	 * @og.rev 5.1.6.0 (2010/05/01) clmSeqDisplay 属性を追加します。
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId				= HybsSystem.TBL_MDL_KEY;
		viewFormId			= HybsSystem.VIEWFORM_KEY;
		viewFormType		= null;
		viewLinkId			= HybsSystem.VIEWLINK_KEY;
		viewMarkerId		= HybsSystem.VIEW_MARK_KEY;
		editMarkerId		= HybsSystem.EDIT_MARK_KEY;		// 3.8.6.1 (2006/10/20)
		command				= null;
		startNo				= -1;		// HTML画面上の表示開始位置（初期値）
		tempStartNo			= -1;		// 再検索時の表示開始位置
		pageSize 			= -1;		// HTML画面上に表示する件数
		pagePlus 			= -1;		// 3.5.6.4 (2004/07/16)
		table				= null;
		form 				= null;
		columnWritable		= null;
		noWritable			= null;
		columnDisplay		= null;
		clmSeqDisplay		= null;		// 5.1.6.0 (2010/05/01) ViewForm_HTMLSeqClmTable 対応
		columnGroup			= null;		// 3.8.5.0 (2006/03/20)
		noDisplay			= null;
		noMessage			= false;
		writable 			= null;
		checked				= null;
		skip 				= ViewForm.DEFAULT_SKIP;
		rowspan				= "2";
		selectedType		= "checkbox";
		numberType			= "sequence";
		optTypeAttri		= null;
//		changeLabelKeys		= null;
//		changeLabelVals		= null;
		values				= null;
		backLinkCount		= HybsSystem.sysInt( "VIEW_BACK_LINK_COUNT" );
		headerSkipCount		= HybsSystem.sysInt( "VIEW_HEADER_SKIP_COUNT" );	// 0:通常ヘッダ、n:n回ごとに現れる
		bodyString			= null;
		useCheckControl		= -1 ;
		useTableSorter		= null ;	// 3.5.4.7 (2004/02/06)
		dyStart				= 0;		// 3.5.4.7 (2004/02/06)
		viewParam			= null;		// 3.5.4.8 (2004/02/23)
		useParam			= false;	// 3.5.4.8 (2004/02/23)
		useConsistency		= HybsSystem.sysBool( "USE_CONSISTENCY" );		// 3.5.5.8 (2004/05/20)
		bgColorCycle		= 1 ;		// 3.5.6.2 (2004/07/05)
		useScrollBar		= null;
		sortKeys			= null;		// 3.6.0.0 (2004/09/17)
//		loadMarker			= ",NEW,RENEW,REVIEW,RESET";					// 3.6.0.4 (2004/10/14)
		loadMarker			= ",NEW,RENEW,REVIEW,RESET,";					// 4.0.0.0 (2007/11/26)
		useSelectedRow		= null;
		useSelRowColor		= null;
		useAfterHtmlTag		= true;		// 3.7.1.1 (2005/05/23)
		writableControl		= null;		// 3.8.0.9 (2005/10/17)
		useHilightRow		= null;		// 3.8.6.0 (2006/09/29)
		popupReturnKeys		= null ;	// 3.8.6.1 (2006/10/20)
		clazz				= "viewTable";	// 4.0.0 (2007/04/16)
		height				= null;		// 4.2.0.0 (2008/03/18)
		width				= null;		// 4.2.0.0 (2008/03/18)
		fileDwnClms			= null ;	// 4.3.0.0 (2008/07/04)
		skipNoEdit			= false;	// 4.3.1.0 (2008/09/08)
		noTransition		= false;	// 4.3.3.0 (2008/10/01)
		useHilightCheckOnly	= null;		// 4.3.7.0 (2009/06/01)
	}

	/**
	 * DBTableModel に対して、各種初期設定を行います。
	 *
	 */
//	private void dbChangeLabel() {
//		if( changeLabelKeys != null &&
//			changeLabelVals != null &&
//			changeLabelKeys.length != 0 &&
//			changeLabelKeys.length == changeLabelVals.length ) {
//
//			for( int i=0; i<changeLabelKeys.length; i++ ) {
//				if( changeLabelKeys[i] != null &&
//					changeLabelVals[i] != null &&
//					changeLabelKeys[i].length() > 0 &&
//					changeLabelVals[i].length() > 0 ) {
//
//					int clmNo = table.getColumnNo( changeLabelKeys[i] );
//					DBColumn dbColumn = table.getDBColumn( clmNo );
//					DBColumnConfig config = dbColumn.getConfig();
//					config.setLabelData( getResource().getLabelData( changeLabelVals[i] ) );
//					table.setDBColumn( clmNo,new DBColumn( config ) );	// 4.0.0 (2005/01/31)
//				}
//			}
//		}
//	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，（内部ポインタを書き換えた）DBTableModel オブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId   = nval( getRequestParameter( id ),tableId );	// 3.8.0.9 (2005/10/17)
	}

	/**
	 * 【TAG】(通常使いません)sessionから取得する ViewForm オブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，（内部ポインタを書き換えた）ViewForm オブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.VIEWFORM_KEY です。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 3.5.6.4 (2004/07/16) VIEW_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id ViewForm オブジェクトID
	 */
	public void setViewFormId( final String id ) {
		String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewFormId = temp + TaglibUtil.VIEW_ID;
		}
	}

	/**
	 * 【TAG】(通常つかいません)requestから取得する ViewLink に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewForm オブジェクトに、ViewLink オブジェクトをこのキーより取得することにより、
	 * カラムに対して、リンク情報を付加して表示させる。
	 * 初期値は、HybsSystem.VIEWLINK_KEY です。
	 *
	 * @og.rev 3.1.4.0 (2003/04/18) 初期値を考慮して、属性登録を行うように変更。
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 3.5.6.4 (2004/07/16) LINK_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id ViewLink オブジェクトID
	 */
	public void setViewLinkId( final String id ) {
		String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewLinkId = temp + TaglibUtil.LINK_ID;
		}
	}

	/**
	 * 【TAG】(通常使いません)requestから取得する ViewMarker に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * ViewForm オブジェクトに、ViewMarker オブジェクトをこのキーより取得することにより、
	 * カラムに対して、マーカー情報を付加して表示させる。
	 * 初期値は、HybsSystem.VIEW_MARK_KEY です。
	 *
	 * @og.rev 3.1.4.0 (2003/04/18) 新規追加
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 3.5.6.4 (2004/07/16) MARK_ID を付加して、他のid と混同しないようにします。
	 *
	 * @param	id ViewMarker オブジェクトID
	 */
	public void setViewMarkerId( final String id ) {
		String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			viewMarkerId = temp + TaglibUtil.MARK_ID;
		}
	}

	/**
	 * 【TAG】(通常使いません)requestから取得する EditMarker に対応する Attributes オブジェクトの ID。
	 *
	 * @og.tag
	 * <p>ViewForm オブジェクトに、EditMarker オブジェクト(実際は、ViewMarker オブジェクト)を
	 * このキーより取得することにより、カラムに対して、マーカー情報を付加して表示させる。<br />
	 * デフォルトは、HybsSystem.EDIT_MARK_KEY です。</p>
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param	id EditMarker オブジェクトID(実際は、ViewMarker オブジェクト)
	 */
	public void setEditMarkerId( final String id ) {
		String temp = getRequestParameter( id ) ;
		if( temp != null && temp.length() > 0 ) {
			editMarkerId = temp + TaglibUtil.MARK_ID;
		}
	}

	/**
	 * 【TAG】ViewForm オブジェクトを作成するクラス(ViewForm_**** クラスの ****)を指定します。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.html 以下の ViewForm_**** クラスの **** を
	 * 与えます。これらは、ViewForm インターフェースを継承したサブクラスです。
	 * 属性クラス定義の
	 * <a href="/ge/jsp/DOC03/index.jsp?command=NEW&SORT=VERSION+DESC%2CSEQ%2CATTKEY&VALUENAME=viewFormType" target="CONTENTS">viewFormType</a>
	 * を参照願います。
	 *
	 * @param	id ViewForm タイプ
	 * @see		org.opengion.hayabusa.html.ViewForm  ViewFormのサブクラス
	 */
	public void setViewFormType( final String id ) {
		viewFormType = getRequestParameter( id );

		if( viewFormType == null || viewFormType.length() == 0 ) {
			String errMsg = "viewForm の id が設定されていません。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】コマンド(PREV,NEXT,FIRST,LAST,NEW,RENEW,VIEW,REVIEW,RESET)をセットします。
	 *
	 * @og.tag
	 * コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.ViewFormTag.CMD_PREV">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }

		if( command == null ||
			 (command.trim()).length() == 0 ||
			! check( command, COMMAND_LIST ) ) {
				// 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
				command = CMD_XXX ;
		}
	}

	/**
	 * 【TAG】表示データを作成する場合の表示の開始行番号をセットします。
	 *
	 * @og.tag
	 * セットされていない場合は，変更しません。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 * @param	no 表示開始行番号（先頭は ０）
	 */
	public void setStartNo( final String no ) {
		startNo = nval( getRequestParameter( no ),startNo );
		tempStartNo = startNo;
	}

	/**
	 * 【TAG】表示データを作成する場合の１ページの行数をセットします。
	 *
	 * @og.tag
	 * セットされていない場合は，変更しません。
	 *
	 * @param	sz １ページの行数
	 */
	public void   setPageSize( final String sz ) {
		pageSize = nval( getRequestParameter( sz ),pageSize );
	}

	/**
	 * 【TAG】１ページの行数の増加分をセットします(初期値:0)。
	 *
	 * @og.tag
	 * セットされていない場合は，０（増加なし）です。
	 *
	 * @param	pp １ページの行数
	 */
	public void   setPagePlus( final String pp ) {
		pagePlus = nval( getRequestParameter( pp ),pagePlus );
	}

	/**
	 * 【TAG】ページの先頭へのリンクの間隔をセットします(初期値:システムパラメータ の VIEW_BACK_LINK_COUNT)。
	 *
	 * @og.tag
	 * セットされていない場合は，システムパラメータ の VIEW_BACK_LINK_COUNTです。
	 * 0 にセットすると、リンクは作成されません。
	 *
	 * @param	blc リンクの間隔
	 */
	public void   setBackLinkCount( final String blc ) {
		backLinkCount = nval( getRequestParameter( blc ),backLinkCount );
	}

	/**
	 * 【TAG】ヘッダーを出力する間隔をセットします(初期値:システムパラメータ の VIEW_HEADER_SKIP_COUNT)。
	 *
	 * @og.tag
	 * 0を指定すると、繰り返しません（つまり、ヘッダーを最初に表示するだけです。）
	 * 数字を指定すると、その回数毎に、ヘッダーをデータの行に挿入します。
	 * 初期値は、システムパラメータ の VIEW_HEADER_SKIP_COUNT です。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 *
	 * @param	hsc ヘッダーの表示間隔 （ 0:通常ヘッダ、n:n回ごとに現れる ）
	 */
	public void   setHeaderSkipCount( final String hsc ) {
		headerSkipCount = nval( getRequestParameter( hsc ),headerSkipCount );
	}

	/**
	 * 【TAG】書き込み可能カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、書き込み不可カラム名の指定(noWritable)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * なお、カラム名の代わりに、"null" を指定すると、なにも指定しないこと
	 * になります。つまり、noWritable にすべてのカラムを指定することと
	 * 同じになります。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 4.1.2.0 (2008/03/12) 引数に、"null" を指定可能とする。
	 * @og.rev 4.2.1.0 (2008/04/11) null チェック方法変更
	 * @og.rev 4.2.1.1 (2008/04/30) 引数にA,B,{&#064;xxxx}等が指定された場合に{&#064;xxxx}の変換が行われないバグを修正
	 *
	 * @param	columnName String
	 */
	public void setColumnWritable( final String columnName ) {
//		if( columnName != null && columnName.startsWith( "{@" ) ) {
		if( columnName != null && columnName.indexOf( "{@" ) >= 0 ) {
			columnWritable = nval( getRequestParameter(columnName),null );
		}
		else {
			columnWritable = columnName ;
		}

		if( "null".equalsIgnoreCase( columnWritable ) ) { columnWritable = ""; }

//		if( noWritable == null || noWritable.length() == 0 ) {
//			columnWritable = nval( getRequestParameter(columnName),"" );
//		}
//		else {
//			if( !"null".equalsIgnoreCase( noWritable ) ) {
//				String errMsg = "すでに noWritable が設定されています。[" + noWritable + "]";
//				throw new HybsSystemException( errMsg );
//			}
//		}
	}

	/**
	 * 【TAG】書き込み不可カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、書き込み可能カラム名の指定(columnWritable)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * なお、カラム名の代わりに、"null" を指定すると、なにも指定しないこと
	 * になります。つまり、columnWritable にすべてのカラムを指定することと
	 * 同じになります。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 * @og.rev 4.1.2.0 (2008/03/12) 引数に、"null" を指定可能とする。
	 * @og.rev 4.2.1.0 (2008/04/11) null チェック方法変更
	 * @og.rev 4.2.1.1 (2008/04/30) 引数にA,B,{@xxxx}等が指定された場合に{@xxxx}の変換が行われないバグを修正
	 *
	 * @param	columnName String
	 */
	public void setNoWritable( final String columnName ) {
//		if( columnName != null && columnName.startsWith( "{@" ) ) {
		if( columnName != null && columnName.indexOf( "{@" ) >= 0 ) {
			noWritable = nval( getRequestParameter(columnName),null );
		}
		else {
			noWritable = columnName ;
		}

		if( "null".equalsIgnoreCase( noWritable ) ) { noWritable = ""; }

//		if( columnWritable == null || columnWritable.length() == 0 ) {
//			noWritable = nval( getRequestParameter(columnName),"" );
//		}
//		else {
//			if( !"null".equalsIgnoreCase( columnWritable ) ) {
//				String errMsg = "すでに columnWritable が設定されています。[" + columnWritable + "]";
//				throw new HybsSystemException( errMsg );
//			}
//		}
	}

	/**
	 * 【TAG】表示可能カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、表示不可カラム名の指定(noDisplay)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 *
	 * @param	columnName String
	 */
	public void setColumnDisplay( final String columnName ) {
		if( noDisplay == null ) {
			this.columnDisplay = nval( getRequestParameter(columnName),"" );
		}
		else {
			String errMsg = "すでに noDisplay が設定されています。[" + noDisplay + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】表示可能カラム名を、カラム順に、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、ViewForm_HTMLSeqClmTable 用に特別に用意された属性です。
	 * 機能的には、columnDisplay 属性で、カラムの表示順まで指定できる機能を
	 * 用意すべきですが、過去のタグとの互換性の関係で、別に用意しています。
	 * （この属性を使用すると、通常の表示より、処理速度が低下します。）
	 *  例："OYA,KO,HJO,SU,DYSET,DYUPD" この順番で表示され、指定以外のカラムは表示されません。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) clmSeqDisplay 属性を追加します。
	 *
	 * @param	columnName String
	 */
	public void setClmSeqDisplay( final String columnName ) {
		clmSeqDisplay = nval( getRequestParameter(columnName),null );
	}

	/**
	 * 【TAG】表示不可カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、表示可能カラム名の指定(columnDisplay)と同時にセットする
	 * ことは出来ません。 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 *
	 * @param	columnName String
	 */
	public void setNoDisplay( final String columnName ) {
		if( columnDisplay == null ) {
			this.noDisplay = nval( getRequestParameter(columnName),"" );
		}
		else {
			String errMsg = "すでに columnDisplay が設定されています。[" + columnDisplay + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】同一表示データをまとめる（表示しない）カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、カラムのグループ化指定を行います。
	 * 同じカラム番号で一つ上の行番号と同じ内容の場合に、表示しないというものです。
	 * 対応は、表示(Renderer)時のみとします。
	 * カラム単位なので、新しいゼブラによる色分けは行いません。(任意のカラムに適用できる為)
	 * また、ファイル出力などのデータ自身は変更されていませんので、そのままデータが落ちます。
	 *
	 * @og.rev 3.8.5.0 (2006/03/20) 新規追加
	 *
	 * @param	group String
	 */
	public void setColumnGroup( final String group ) {
		columnGroup = nval( getRequestParameter( group ),null );
	}

	/**
	 * 【TAG】カラム名リンクソートを表示するカラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * ヘッダーにソート用リンクを作成する useTableSorter 属性 に対して、
	 * カラム個別に作成する場合のカラム名をカンマ区切り文字で指定します。
	 * この sortKeys 属性は、useTableSorter 属性 と無関係に、指定した
	 * カラムのみ、リンクを表示します。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
	 *
	 * @param	columnName String
	 */
	public void setTableSorterKeys( final String columnName ) {
		sortKeys = nval( getRequestParameter(columnName),null );
	}

	/**
	 * 【TAG】検索結果メッセージを表示しないかどうか(true/false)を設定します(初期値:false[表示する])。
	 *
	 * @og.tag
	 * 初期値は、表示する（true以外)です。
	 *
	 * @param	flag 検索結果メッセージの制御 (true:表示しない / その他:表示する。）
	 */
	public void setNoMessage( final String flag ) {
		noMessage = nval( getRequestParameter( flag ),noMessage );
	}

	/**
	 * 【TAG】行が書き込み可能かどうか(true/false)を設定します(初期値:ViewForm.DEFAULT_ROW_WRITABLE)。
	 *
	 * @og.tag
	 * 書込み許可する場合は, true、許可しない場合は, true 以外を設定します。
	 * 初期値は、ViewForm.DEFAULT_ROW_WRITABLEです。
	 * ここで、false を設定すると全行書込み禁止になります。true を設定した
	 * 場合、デフォルトで書込み許可が与えられ、先頭カラムが writable であれば、
	 * その値を取り込んで、行ごとの書込み許可/不許可が決定されます。
	 * writable,checked,useCheckControl については、writable が最優先で
	 * 次に、useCheckControl="2"(または、checked)、次に、checked を判定します。
	 * checked="true"の場合は、useCheckControl が、"1"(true)以上の場合、
	 * "2"(checked)に設定されます。
	 *
	 * @param	flag 書込み許可(true)／書込み禁止（true 以外)
	 */
	public void setWritable( final String flag ) {
		writable = getRequestParameter( flag );
	}

	/**
	 * テーブルデータに書込み許可／チェック状態を設定します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) writable が false の場合全件チェック用チェックボックスを出さない。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロールのロジック変更(useCheckControl)
	 * @og.rev 3.8.8.5 (2007/03/09) 全件チェック(useCheckControl)と、個別チェック(checked)の整合性
	 *
	 * @param	table DBTableModel
	 */
	private void setTableRowWritable( final DBTableModel table ) {

		if( writable != null && writable.length() > 0 ) {
			boolean flag = "TRUE".equalsIgnoreCase( writable );
			table.setDefaultRowWritable( flag );
			if( ! flag ) { useCheckControl = 0; }			// 3.7.0.1 (2005/01/31)
		}

		// 全チェックONの場合は、個別チェックもONにする。
		if( useCheckControl == 2 ) { checked = "true"; }	// 3.8.8.5 (2007/03/09)

		if( checked != null && checked.length() > 0 ) {
			boolean flag = "TRUE".equalsIgnoreCase( checked );
			table.setDefaultRowChecked( flag );
			// 個別チェックONの場合は、全チェックもONにする。
			if( flag && useCheckControl >= 1 ) { useCheckControl = 2; }			// 3.8.8.5 (2007/03/09)
		}
	}

	/**
	 * 【TAG】行のチェックボックスの初期値を設定するかどうか(true/falseを)指定します。
	 *
	 * @og.tag
	 * 書き込み可能な行（rowWritable == true）のチェックボックスに対して
	 * 初期値を 選択済みにするか、非選択済みにするかを指定します。
	 * writable,checked,useCheckControl については、writable が最優先で
	 * 次に、useCheckControl="2"(または、checked)、次に、checked を判定します。
	 * checked="true"の場合は、useCheckControl が、"1"(true)以上の場合、
	 * "2"(checked)に設定されます。
	 *
	 * @param	flag 書込み許可(true)／書込み禁止（true 以外)
	 */
	public void setChecked( final String flag ) {
		checked = getRequestParameter( flag );
	}

	/**
	 * 【TAG】NEXT等で、選択行以外をスキップするのかどうか(true/false)を指定します(初期値:ViewForm.DEFAULT_SKIP)。
	 *
	 * @og.tag
	 * 明細情報の登録/編集時に、必要な個所のみを順番に処理する場合に使用します。
	 *
	 * @param	flag スキップする(true)／スキップしない（true 以外)
	 */
	public void setSkip( final String flag ) {
		skip = nval( getRequestParameter( flag ),skip );
	}

	/**
	 * 【TAG】表示データを作成する場合のフォーマットの行数をセットします(初期値:2)。
	 *
	 * @og.tag 表示データを作成する場合のフォーマットの行数をセットします。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 *
	 * @param	rowspan フォーマットの行数
	 */
	public void setRowspan( final String rowspan ) {
		this.rowspan = getRequestParameter(rowspan);
	}

	/**
	 * 【TAG】表示時の選択用オブジェクトのタイプ(checkbox,radio,hidden)を指定します(初期値:checkbox)。
	 *
	 * @og.tag
	 * 複数選択可能時は "checkbox" を指定します。
	 * 一つだけ選ばせる場合は, "radio" を指定します。
	 * 隠しフィールドで全件を選ぶ場合は、"hidden" を指定します。
	 * 初期値は、"checkbox" です。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) {&#064;XXXX} 変数が使用できるように修正。
	 *
	 * @param	type 選択用オブジェクトのタイプ（ / checkbox / radio / hidden / ）
	 */
	public void setSelectedType( final String type ) {
		selectedType = nval( getRequestParameter(type),selectedType );
	}

	/**
	 * 【TAG】表示時の行番号の表示方法(sequence,none,delete,skip=カラム名,view=カラム名)を指定します(初期値:sequence)。
	 *
	 * @og.tag
	 * ・sequenceは、１から始まる連番です。
	 * ・none を指定すると、番号は表示されません。
	 * ・delete を指定すると、行番号表示そのもののタグを取り除きます。(3.5.5.0 追加)
	 * ・skip=カラム名を指定すると、そのカラムの値が NULL の場合は、番号をスキップします。
	 * ・view=カラム名を指定すると、そのカラムの値が、使用されます。（ラベルは付きません）
	 * skip=XX と、view=XX は、=の前後にスペースを入れないで下さい。
	 * 初期値は、"sequence" です。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) 新規作成
	 *
	 * @param	type 行番号の表示方法のタイプ（ /sequence/none/delete/skip=カラム名/view=カラム名/ ）
	 */
	public void setNumberType( final String type ) {
		numberType = nval( getRequestParameter(type),numberType );
	}

	/**
	 * 【TAG】テーブル等のチェックボックスに属性を付加します。
	 *
	 * @og.tag
	 * JavaScript などの HTML基本タグ以外の属性を、そのまま
	 * チェックボックス/ラジオボタン等に使用します。
	 * そのときに、（'）は、（"）に置き換えます。
	 *
	 * @param	attributes HTML基本タグ以外の属性
	 */
	public void setOptionTypeAttributes( final String attributes ) {
		String optAttri = getRequestParameter( attributes );
		if( optAttri != null && optAttri.length() > 0 ) {
			optTypeAttri = optAttri.replace( '\'', '\"' );
		}
	}

//	/**
//	 * 【廃止】表示時のラベルを変更する時のカラムキー(参照：columnEditor タグ)。
//	 *
//	 * @og.tag
//	 * changeLabelVal とセットで使用されます。
//	 * 通常のラベル定義されている項目と置き換えたい場合は、
//	 * ＳＱＬ文の仮項目で指定します。
//	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
//	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
//	 *
//	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
//	 *
//	 * @param	key カラムキー
//	 * @deprecated  columnEditor タグで、ラベルを入れ替える方法で、代替してください。
//	 */
//	@Deprecated public void setChangeLabelKeys( final String key ) {
//		changeLabelKeys = getCSVParameter( key );
//	}

//	/**
//	 * 【廃止】表示時のラベルを変更する時のラベル(参照：columnEditor タグ)。
//	 *
//	 * @og.tag
//	 * changeLabelKey とセットで使用されます。
//	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
//	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
//	 *
//	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
//	 *
//	 * @param	val ラベル
//	 * @deprecated columnEditor タグで、ラベルを入れ替える方法で、代替してください。
//	 */
//	@Deprecated public void setChangeLabelVals( final String val ) {
//		changeLabelVals = getCSVParameter( val );
//	}

	/**
	 * 【TAG】チェックボックスの全チェックを選択するJavaScript の機能を使用するかどうかを指定します。
	 *
	 * @og.tag
	 * "true" で、コントロール用のチェックボックスが現れて、この機能を使用することができるようになります。
	 * false は、従来どおりです。
	 * 使用するにあたり、jsp/commpn/default.js にJavaScriptを設定しておきます。(設定済み）
	 * 初期値は、システムパラメータ の VIEW_USE_CHECK_CONTROL です。(false)
	 * false は、0でも代用可能です。true は、1でも代用可能です。
	 * checked は、2でも代用可能です。checked は、初期値を、チェック済みにします。
	 * 注意：0,1,2,true,false,checked 以外の文字をセットすると、エラーが発生します。
	 * writable,checked,useCheckControl については、writable が最優先で
	 * 次に、useCheckControl="2"(または、checked)、次に、checked を判定します。
	 * checked="true"の場合は、useCheckControl が、"1"(true)以上の場合、
	 * "2"(checked)に設定されます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロールのロジック変更(useCheckControl)
	 *
	 * @param	flag チェックボックスの全チェックを選択する機能を使用するかどうか
	 *			（true,1:使用する/false,0:使用しない/checked,2:初期値チェック済み）
	 */
	public void setUseCheckControl( final String flag ) {
		useCheckControl = intCase( getRequestParameter(flag) );

		if( useCheckControl < 0 ) {
			String errMsg = "useCheckControl は、false(0)/true(1)/2 のいずれかしか登録できません。["
						 + flag + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】ヘッダーにソート用リンクを作成するかどうかを指定します。
	 *
	 * @og.tag
	 * "true" で、ヘッダーにソート用リンクを作成します。
	 * false は、従来どおりです。
	 * 検索済みのDBTableModelを、メモリ上でソートし、再表示させます。
	 * 初期値は、システムパラメータ の VIEW_USE_TABLE_SORTER です。(false)
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規追加
	 *
	 * @param	flag ヘッダーにソート用リンクを作成するかどうか（true:作成する/false:作成しない）
	 */
	public void setUseTableSorter( final String flag ) {
		useTableSorter = nval( getRequestParameter(flag),useTableSorter );
	}

	/**
	 * 表示データをプレビュー可能かどうかを返します。
	 *
	 * 不可能な状態とは,すでに一番先頭から表示されている場合です。
	 *
	 * @return	プレビュー可能(true)／不可能（false)
	 */
	private boolean isPrev() {
		if( startNo <= 0 ) {
			return false;
		}
		return true;
	}

	/**
	 * 表示データをネクスト可能かどうかを返します。
	 *
	 * 不可能な状態とは,すでに一番後ろが表示されている場合です。
	 *
	 * @return	ネクスト可能(true)／不可能（false)
	 */
	private boolean	 isNext() {
		if( (startNo + pageSize) >= table.getRowCount() ) {
			return false;
		}
		return true;
	}

	/**
	 * コマンドを実行します。
	 *
	 * コマンドは,HTMLから（get/post)指定されますので,setCommand()メソッドで
	 * 登録します。
	 * コマンドを登録すると同時に,実行も行ないます。
	 *
	 * @og.rev 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
	 *
	 * @param	command コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.ViewFormTag.CMD_PREV">コマンド定数</a>
	 */
	private void commandExec( final String command ) {
		if( 	 CMD_PREV.equals(	command ) ) { prev() ;	}
		else if( CMD_NEXT.equals(	command ) ) { next() ;	}
		else if( CMD_FIRST.equals(	command ) ) { first() ; }
		else if( CMD_LAST.equals(	command ) ) { last() ;	}
	//	else if( CMD_NEW.equals(	command ) ) {} 		// 何もしない
	//	else if( CMD_REVIEW.equals( command ) ) {} 		// 何もしない
	//	else if( CMD_RESET.equals(	command ) ) {} 		// 何もしない
		else if( CMD_VIEW.equals(	command ) ) { view() ;	}
		// 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
		else if( CMD_XXX.equals(	command ) ) { view() ;	}
		this.command = null;
	}

	/**
	 * HTMLデータの表示をプレビュー（戻し）します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void prev() {
		if( ! isPrev() ) { return ; }			// prev 出来ない。
		tempStartNo = tempStartNo - pageSize;
		if( tempStartNo < 0 ) { tempStartNo = 0; }
		if( skip ) {
			for( int i=startNo-1; i>=0; i-- ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = startNo - pageSize;
		if( startNo < 0 ) {
			startNo = 0;
		}
	}

	/**
	 * HTMLデータの表示をネクスト（先送り）します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void next() {
		if( ! isNext() ) { return ; }			// next 出来ない。
		tempStartNo = tempStartNo + pageSize;
		if( skip ) {
			int end  = table.getRowCount();
			for( int i=startNo+1; i<end; i++ ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = startNo + pageSize;
	}

	/**
	 * HTMLデータの表示をファースト（最初）します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void first() {
		tempStartNo = 0;
		if( tempStartNo < 0 ) { tempStartNo = 0; }
		if( skip ) {
			int end  = table.getRowCount();
			for( int i=startNo; i<end; i++ ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = 0;
	}

	/**
	 * HTMLデータの表示をラスト（最後）します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 *
	 */
	private void last() {
		tempStartNo = table.getRowCount() - pageSize;
		if( tempStartNo < 0 ) { tempStartNo = 0; }
		if( skip ) {
			for( int i=startNo; i>=0; i-- ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}

		startNo = table.getRowCount() - pageSize;
		if( startNo < 0 ) { startNo = 0; }
	}

	/**
	 * HTMLデータの表示をします。
	 *
	 */
	private void view() {
		if( skip ) {
			int end  = table.getRowCount();
			for( int i=startNo; i<end; i++ ) {
				if( form.isMarked( i ) ) {
					startNo = i;
					break;
				}
			}
			return ;
		}
	}

	/**
	 * startNo,pageSize,pagePlus の初期値を設定する。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) startNo を隠しフィールドに持つ処理を、追加
	 * @og.rev 3.5.4.1 (2003/12/01) startNo をキャッシュから取り出すように変更。
	 * @og.rev 3.5.4.7 (2004/02/06) getStartNoKeyメソッドの廃止（getRequestCacheDataを利用）
	 * @og.rev 3.7.0.2 (2005/02/14) NEXT 時に何も選択しないで追加した場合は、１件目から表示する。
	 * @og.rev 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
	 *
	 */
	private void setDefaultParameter() {
		// 3.7.0.2 (2005/02/14)
		// 3.8.0.8 (2005/10/03) その他のコマンドを CMD_XXX にします。
		if( CMD_XXX.equals( command ) ) {
			String[] selected = getRequestValues( HybsSystem.ROW_SEL_KEY ) ;
			if( selected == null || selected.length == 0 ) {
				startNo = 0;
				tempStartNo = startNo;
			}
		}

		if( startNo<0 ) {
			startNo = nval( getRequestCacheData( START_NO_KEY ),form.getStartNo() ) ;
			tempStartNo = startNo;
		}

		if( pageSize<0 ) {
			pageSize = nval( getRequestValue( PAGE_SIZE_KEY ),form.getPageSize() ) ;
		}

		if( pagePlus<0 ) {
			pagePlus = nval( getRequestValue( PAGE_PLUS_KEY ),0 ) ;
		}

	}

	/**
	 * 内部タグの TheadTag / TbodyTag / TfootTag より、個々のフォーマットを定義する
	 * 為のTableFormatter オブジェクトを受け取る。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規登録（setAttributes メソッドより移行）
	 *
	 * @param	form TableFormatter
	 */
	protected void addFormatter( final TableFormatter form ) {
		if( values == null ) { values = new ArrayList<TableFormatter>(); }
		values.add( form );
	}

	/**
	 * pageSize をhidden属性で出力するタグを作成します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) command="RESET"時に、戻った画面のpageSize がクリアされてしまう対応。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 *
	 * @param	pageSize int
	 */
	private String makePageSize( final int pageSize ) {
		String name   = PAGE_SIZE_KEY;
		String value  = String.valueOf( pageSize );

		return XHTMLTag.hidden( name,value );	// 3.5.5.5 (2004/04/23)

	}

	/**
	 * consistencyKey をhidden属性で出力するタグを作成します。
	 * 出力する条件は、useConsistency="true" （初期値）、で
	 * scope が、 request 以外の場合です。
	 * tableId が設定された場合は、consistencyKey に、tableId を加えた
	 * キーにします。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) DBTableModel の整合性パラメータを hidden で出力します。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.5.5.8 (2004/05/20) makeConsistencyKey メソッドに処理ロジックを移します。
	 *
	 */
	private void makeConsistencyKey() {
		if( useConsistency && ! "request".equals( getScope() ) ) {
			String name   = HybsSystem.CONSISTENCY_KEY;
			String value  = table.getConsistencyKey() ;

			jspPrint( XHTMLTag.hidden( name,value ) );
		}
	}

	/**
	 * "false","true","checked" の文字列を、それぞれ、0,1,2 に対応した数字に変換します。
	 * それ以外の文字列は、-1 にします。
	 *
	 * @og.rev 3.7.0.1 (2005/01/31) 新規作成
	 *
	 * @param    flag 文字列 (true=1 , false=0 , checked=2) の文字列
	 * @return   設定値(0,1,2の数字、不正な場合は、-1 を返す。
	 */
	private static int intCase( final String flag ) {
		int rtn = -1;
		if( "FALSE".equalsIgnoreCase( flag ) || "0".equalsIgnoreCase( flag ) ) {
			rtn = 0;
		}
		else if( "TRUE".equalsIgnoreCase( flag ) || "1".equalsIgnoreCase( flag ) ) {
			rtn = 1;
		}
		else if( "CHECKED".equalsIgnoreCase( flag ) || "2".equalsIgnoreCase( flag ) ) {
			rtn = 2;
		}
		return rtn ;
	}

	/**
	 * ViewForm のサブクラスに渡すパラメータマップ
	 *
 	 * @og.rev 3.5.4.8 (2004/02/23) ViewParamTag のパラメータを追加します。
	 *
	 * @param	map Map
	 */
	protected void setParam( final Map<String,String> map ) {
		viewParam = map ;
	}

	/**
	 * 【TAG】パラメーターファイルの使用を指定します。
	 *
	 * @og.tag
	 * "true" で、パラメータファイルを読み込むため、BODY部を処理します。
	 * false は、パラメータファイうを使用しません。
	 * パラメータファイルは、各ViewFormのサブクラス用に用意します。
	 * 初期値は、false です。(false)
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) 新規追加
	 *
	 * @param	flag パラメーターファイルを使用するかどうか（true:使用する/false:使用しない）
	 */
	public void setUseParam( final String flag ) {
		useParam = nval( getRequestParameter(flag),useParam );
	}

	/**
	 * 【TAG】Consistency キー による整合性チェックを行うかどうかを指定します(初期値:true)
	 *
	 * @og.tag
	 * 検索結果を DBTableModel にセットする時に、整合性キーの Consistency キーを
	 * 作成します。これを、Viewタグでhidden出力しておき、Entryタグでデータ書き換え時に
	 * 整合性チェックを行います。これは、IEの戻るボタンで戻った場合に、画面の
	 * キーと検索結果の DBTableModel の内容が一致しない場合のエラーチェックに
	 * なります。
	 * この属性は、何らかのケースで、このエラーチェックを行いたくない場合に、
	 * false に設定することで、整合性チェックを行いません。
	 * 初期値は、true(整合性チェックを行う)です。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規登録
	 *
	 * @param	ck String  Consistency Key の使用するかどうか（初期値：true）
	 */
	public void setUseConsistency( final String ck ) {
		useConsistency = nval( getRequestParameter( ck ),useConsistency );
	}

	/**
	 * 【TAG】テーブルのバックグラウンドカラーの入れ替えのサイクルをセットします(初期値は、1:ゼブラ)。
	 *
	 * @og.tag
	 * 0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 * 初期値は、1(ゼブラ)です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 新規追加
	 *
	 * @param  sycle  0(ゼブラなし)、-1(ワーニング)、-2以下(エラー)、1(ゼブラ)、2以上(行数まとめ)
	 */
	public void setBgColorCycle( final String sycle ) {
		bgColorCycle = nval( getRequestParameter( sycle ),bgColorCycle );
	}

	/**
	 * 【TAG】スクロールバー用のDIV要素を出力するかどうか(初期値は、false)。
	 *
	 * @og.tag
	 * スクロールバー対応する為、テーブルの先頭に、DIV要素を出力します。
	 * 初期値は、システムパラメータ の VIEW_USE_SCROLLBAR です。
	 * ※ 互換性の関係より、false になっています。
	 *
	 * @og.rev 3.5.6.3 (2004/07/12) 新規追加
	 *
	 * @param  useBar  スクロールバー用のDIV要素を出力するかどうか(true/false)
	 */
	public void setUseScrollBar( final String useBar ) {
		useScrollBar = nval( getRequestParameter( useBar ),useScrollBar );
	}

	/**
	 * 【TAG】viewLink や viewMarker の情報取り込みを制御する(初期値は、NEW,RENEW,REVIEW,RESET)。
	 *
	 * @og.tag
	 * viewLink や viewMarker の情報取り込みは、command が、NEW,RENEW,REVIEW,RESET
	 * の場合のみ行われています。
	 * この属性で、"true" を指定すると、コマンドに無関係に取り込みます。
	 * "false" の場合は、form にキャッシュしているリンク、マーカーを削除します。
	 * 取り込み時に、viewLink や viewMarker オブジェクトが存在しない場合は、
	 * 何もしません。これは、viewMarker オブジェクト自身がすでに設定済みの場合（VIEWなど）
	 * NEXTやPREVでは、viewMarkerオブジェクトは、リクエスト変数に設定されない為です。
	 * この場合は、登録しませんが、既存の viewMarkerオブジェクトを持っているはずなので、
	 * なにも設定しなくても viewMarker は適用されます。
	 * 初期値は、NEW,RENEW,REVIEW,RESET です。
	 *
	 * @og.rev 3.6.0.4 (2004/10/14) 新規追加
	 *
	 * @param  load  viewLink や viewMarker の情報取り込みを制御する
	 */
	public void setLoadMarker( final String load ) {
		loadMarker = nval( getRequestParameter( load ),loadMarker );
	}

	/**
	 * 【TAG】SEL_ROW機能の有効/無効を指定します(初期値:VIEW_USE_SELECTED_ROW)
	 *
	 * @og.tag
	 * SEL_ROW機能は、戻るリンクから戻ったときや、登録したときに、選択していた
	 * 行番号や、先にクリックされた行番号を返す機能で、SEL_ROW リクエストに設定された
	 * 行がマーカーされます。この機能の有効(true)/無効(false)を指定します。
	 * この属性で、"true" を指定すると、有効になります。
	 * "false" の場合は、無効です。
	 * 初期値は、VIEW_USE_SELECTED_ROW で設定されます。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) 新規追加
	 *
	 * @param  selRow  SEL_ROW機能の有効/無効を指定(true/false)
	 */
	public void setUseSelectedRow( final String selRow ) {
		useSelectedRow = nval( getRequestParameter( selRow ),useSelectedRow );
	}

	/**
	 * 【TAG】SEL_ROW使用時の選択行色変更の有効/無効を指定します(初期値:VIEW_USE_SELROW_COLOR)
	 *
	 * @og.tag
	 * SEL_ROW機能は、現在選択されている行まで、スクロールする機能ですが、そのとき、
	 * 選択行にマーカーするかどうかを、指定します。
	 * この属性で、"true" を指定すると、有効になります。
	 * "false" の場合は、無効です。
	 * 初期値は、VIEW_USE_SELROW_COLOR で設定されます。
	 *
	 * @og.rev 3.7.1.1 (2005/05/31) 新規追加
	 *
	 * @param  selRowColor  SEL_ROW使用時の選択行色変更の有効/無効を指定(true/false)
	 */
	public void setUseSelRowColor( final String selRowColor ) {
		useSelRowColor = nval( getRequestParameter( selRowColor ),useSelRowColor );
	}

	/**
	 * 【TAG】 処理時間やページサイズなどの情報出力の有効/無効を指定します(初期値:true)
	 *
	 * @og.tag
	 * ViewForm で、CustomData などの 非HTML表示ビューを使用する場合、
	 * 最後に、処理時間とページサイズ（NEXT時に使用）などの情報を、hidden で
	 * 出力していますが、これを制御するフラグをセットします。
	 * true(有効)にすると、これらのHTMLが出力されます。false にすると、出力されません。
	 * 初期値は、true(有効) です。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) 新規追加
	 *
	 * @param  useTag  情報出力の有効/無効を指定(true/false)
	 */
	public void setUseAfterHtmlTag( final String useTag ) {
		useAfterHtmlTag = nval( getRequestParameter( useTag ),useAfterHtmlTag );
	}

	/**
	 * 【TAG】設定値に "_" が含まれている場合にレンデラーを使用するカラムをCSV形式で指定します。
	 *
	 * @og.tag
	 * これは、従来の カラム定義の WRITABLE エディターと同等の働きを行うように
	 * カラム属性を指定します。
	 * WRITABLE エディターは、設定値にアンダーバー "_" が含まれている場合に、
	 * その値を書込み禁止にする機能です。これは、エディター自身が値を判断して
	 * 書き込み許可か禁止かを判断しています。
	 * この動きを汎用的にするため、指定のカラムをカンマ区切り文字(CSV)で指定
	 * することにより、レンデラーとエディターを設定値によって動的に切り替える
	 * 機能を実現します。
	 * その場合、表示/編集ともに、先頭のアンダーバーは削除されます。
	 * また、全カラムについて、有効にする場合は、writableControl="*" を設定します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規追加
	 *
	 * @param  wrtCtrl 書き込み制御を行いたいカラムをCSV形式で指定
	 */
	public void setWritableControl( final String wrtCtrl ) {
		writableControl = nval( getRequestParameter( wrtCtrl ),writableControl );
	}

	/**
	 * 【TAG】HILIGHT_ROW機能の有効/無効を指定します(初期値:VIEW_USE_HILIGHT_ROW)
	 *
	 * @og.tag
	 * HILIGHT_ROW機能は、チェックボックスやラジオボタンで行選択した際に、どの行が選択されたか
	 * 判るように行のハイライト表示を行う機能です。
	 * さらに、行の任意の箇所をクリックしても、チェックが入り、ハイライト表示されます。
	 * この属性で、"true" を指定すると、有効になります。
	 * "false" の場合は、無効です。<br />
	 * 初期値は、VIEW_USE_HILIGHT_ROW で設定されます。
	 *
	 * @og.rev 3.8.6.0 (2006/09/29) HILIGHT_ROW機能の有効/無効を指定します。
	 *
	 * @param  hilightRow  HILIGHT_ROW機能の有効/無効を指定(true/false)
	 */
	public void setUseHilightRow( final String hilightRow ) {
		useHilightRow = nval( getRequestParameter( hilightRow ),useHilightRow );
	}

	/**
	 * 【TAG】ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	 *
	 * @og.tag
	 * popup の検索結果を返す画面で、結果のラジオボタンにイベントセットします。
	 * この場合、オープンもとのwindow に値を返しますが、そのキーをCSV形式で
	 * 指定します。
	 * なお、このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
	 * いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
	 * するようにします。(エラーにしません)
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  rtnKeys ogPopupで値を返すカラム文字列(CSV形式)
	 */
	public void setPopupReturnKeys( final String rtnKeys ) {
		popupReturnKeys = nval( getRequestParameter( rtnKeys ),popupReturnKeys );
	}

	/**
	 * 【TAG】table要素に対して class 属性を設定します(初期値:viewTable)。
	 *
	 * @og.tag
	 * Javaの言語使用上、class で作成できないため、代用として
	 * clazz を使用している。
	 * html で作成される属性は、 class で作成される。
	 * 初期値は、"viewTable" です。
	 *
	 * @og.rev 4.0.0 (2007/04/16) 新規追加
	 *
	 * @param   cls String class属性を表す文字列
	 */
	public void setClazz( final String cls ) {
		clazz = nval( getRequestParameter( cls ),clazz );
	}
	
	/**
	 * 【TAG】ビューの高さを指定します。
	 *
	 * @og.tag
	 * id=posDivのDIV要素に高さを指定します。
	 * javascriptからこの高さを見てビューの高さを決定します。
	 * 単位も指定して下さい。
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param   high String class属性を表す文字列
	 */
	public void setHeight( final String high ) {
		height = nval( getRequestParameter( high ), height );
	}

	/**
	 * 【TAG】ビューの幅を指定します。
	 *
	 * @og.tag
	 * id=posDivのDIV要素に幅を指定します。
	 * javascriptからこの値を見てビューの幅を決定します。
	 * 単位も指定して下さい。（%指定推奨）
	 *
	 * @og.rev 4.2.0.0 (2008/03/18) 新規追加
	 *
	 * @param   wide String 指定するビューの幅
	 */
	public void setWidth( final String wide ) {
		width = nval( getRequestParameter( wide ), width );
	}

	/**
	 * 【TAG】ファイルダウンロード時の抜出カラム名(CSV形式)を指定します。
	 *
	 * @og.tag
	 * fileUDのファイル抜出時のカラムをカンマ形式で指定します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) 新規追加
	 *
	 * @param   dwnClms String 抜出カラム名(CSV形式)
	 */
	public void setFileDwnClms( final String dwnClms ) {
		fileDwnClms = nval( getRequestParameter( dwnClms ), fileDwnClms );
	}
	
	/**
	 * 【TAG】編集行のみを表示するかを指定します。
	 *
	 * @og.tag
	 * 編集行のみを表示するかを指定します。trueがセットされた場合、編集行(改廃Cがついている行)
	 * のみが出力されます。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08)
	 *
	 * @param   sne String 編集行のみを出力するか
	 */
	public void setSkipNoEdit( final String sne ) {
		skipNoEdit = nval( getRequestParameter( sne ), skipNoEdit );
	}

	/**
	 * 【TAG】(通常は使いません)画面遷移を行わない形式の登録方法を使用するかを指定します。
	 *
	 * @og.tag
	 * 画面遷移なしの登録を行うかどうかを指定します。
	 * trueが指定された場合、HTML出力時に、各行に対して、行番号と改廃Cが属性情報として付加されます。
	 * ここで出力された情報を元に、JavaScriptの入れ替え処理を行います。
	 * 
	 * この値は、og:headタグで設定値、または前画面からの値を継承するため、通常、この属性ではセットしません。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 *
	 * @param   noTrnstn 画面遷移を行わない形式の登録方法を使用するか
	 */
	public void setNoTransition( final String noTrnstn ) {
//		noTransition = nval( getRequestParameter( noTrnstn ), noTransition );
		setNoTransitionRequest( nval( getRequestParameter( noTrnstn ), isNoTransitionRequest() ) );
	}

	/**
	 * 【TAG】選択行ハイライトの制御をチェックボックスのみで有効にするかを指定します。
	 *
	 * @og.tag
	 * 選択行ハイライトの制御をチェックボックスのみで有効にするかを指定します。
	 * trueの場合、チェックボックスのみで有効になります。
	 * falseの場合、行全体の各要素で有効になります。
	 * trueにした場合、チェックボックスが存在しない行に対しては、ハイライトすることはできません。
	 * 初期値はfalse(行全体の各要素で有効)です。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規追加
	 *
	 * @param   hiCkOnly 選択行ハイライトの制御をチェックボックスのみで有効にするか
	 */
	public void setUseHilightCheckOnly( final String hiCkOnly ) {
		useHilightCheckOnly = nval( getRequestParameter( hiCkOnly ), useHilightCheckOnly );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "view" ;
	}

	/**
	 * seqClmTableを使用するかどうかをチェックします
	 * 
	 * ここではtableId,scope,項目名などで厳密にチェックを行い、これらが元の情報と合致しない場合は、
	 * ViewFormTypeの置き換えを行わないようにします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 */
	private void checkUseSeqClmTable() {
		// 5.1.6.0 (2010/05/01)ViewForm_HTMLSeqClmTable クラス によるカラム表示順処理を行います。
		// clmSeqDisplay 属性が設定されていると、クラス指定を変更し、columnDisplay に、clmSeqDisplay を
		// 設定し、noDisplay を、削除(null をセット)します。
		if( clmSeqDisplay != null ) {
			viewFormType	= "HTMLSeqClmTable" ;
			columnDisplay	= clmSeqDisplay;
			noDisplay		= null;
		}
		// 5.1.6.0 (2010/05/01) ユーザー毎の画面項目並び替え対応
		else {
			String userViewClms = getUser().getAttribute( HybsSystem.USER_VIEW_CLMS + getGUIInfoAttri( "KEY" ) );
			if( userViewClms != null && userViewClms.length() > 0 ) {
				DBLastSql lastSql = (DBLastSql)getSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );
				if( lastSql != null && lastSql.isGuiMatch( getGUIInfoAttri( "KEY" ) ) && lastSql.isTableMatch( tableId, getScope() ) ) {
					// 1Queryに対して複数Viewがある場合(左右分割など)、Viewの各項目定義が'|'で分割されてセットされています。
					int vwCnt = StringUtil.nval( (String)getRequestAttribute( "h_vwCnt"), 0 );
					String[] clmGroup = StringUtil.csv2Array( userViewClms, '|' );
					if( clmGroup.length > vwCnt ) {
						// AAA,!BBB,CCC,DDD,!EEE から!がついたカラムのみを取り除く(!がある項目は非表示)
						String thisViewClms = ( "," + clmGroup[vwCnt] ).replaceAll( ",![^,]*", "" );
						if( thisViewClms.length() > 0 ) {
							thisViewClms = thisViewClms.substring( 1 );
						}
						String[] clm = StringUtil.csv2Array( thisViewClms );
						boolean isUse = true;
						for( int i=0; i<clm.length; i++ ) {
							if( table.getColumnNo( clm[i], false ) < 0 ) {
								isUse = false;
								break;
							}
						}
						if( isUse ) {
							viewFormType	= "HTMLSeqClmTable" ;
							columnDisplay	= thisViewClms;
							clmSeqDisplay	= thisViewClms;
							noDisplay		= null;
							setRequestAttribute( "h_vwCnt", String.valueOf( vwCnt+1 ) );
						}
					}
				}
			}
		}
	}

	/**
	 * Viewで表示した項目をDBLastSqlにセットします。
	 * 
	 * ここではtableId,scopeなどで厳密にチェックを行い、これらが元の情報と合致しない場合は、
	 * DBLastSqlへのセットを行わないようにします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 */
	private void setViewClms() {
		// 4.3.0.0 (2008/07/04) fileUD 対応。抜き出すカラムを指定します。
//		if( fileDwnClms != null ) {
		DBLastSql lastSql = (DBLastSql)getSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );
		if( lastSql != null ) {
			String guikey = getGUIInfoAttri( "KEY" );
			if( lastSql.isGuiMatch( guikey ) && lastSql.isTableMatch( tableId, getScope() ) && lastSql.isViewEditable() ) {
				// 5.1.6.0 (2010/05/01) fileDwnClmsが指定されない場合は、Viewで表示されたカラムをセット
				if( !form.isEditable() ) {
					lastSql.setViewNotEditable();
				}
				else {
					String clmNames = lastSql.getViewClmNames();
					if( clmNames == null || clmNames.length() == 0 ) {
						clmNames = StringUtil.nval( fileDwnClms, form.getViewClms() );
					}
					else {
						// 1Queryに対して複数Viewがある場合(左右分割など)、Viewの各項目定義が'|'で分割されてセットします。
						clmNames += "|" + StringUtil.nval( fileDwnClms, form.getViewClms() );
					}
					lastSql.setViewClmNames( clmNames );
					lastSql.setClmNames( clmNames.replace( "|", "," ) );
					setSessionAttribute( HybsSystem.DB_LAST_SQL_KEY,lastSql );
				}
			}
		}
//		}
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION			)
				.println( "tableId"			,tableId			)
				.println( "viewFormId"		,viewFormId			)
				.println( "viewFormType"	,viewFormType		)
				.println( "viewLinkId"		,viewLinkId			)
				.println( "viewMarkerId"	,viewMarkerId		)
				.println( "command"			,command			)
				.println( "startNo"			,startNo			)
				.println( "tempStartNo"		,tempStartNo		)
				.println( "pageSize"		,pageSize 			)
				.println( "pagePlus"		,pagePlus 			)
				.println( "columnWritable"	,columnWritable		)
				.println( "noWritable"		,noWritable			)
				.println( "columnDisplay"	,columnDisplay		)
				.println( "noDisplay"		,noDisplay			)
				.println( "columnGroup"		,columnGroup		)
				.println( "noMessage"		,noMessage			)
				.println( "writable"		,writable 			)
				.println( "checked"			,checked			)
				.println( "skip"			,skip 				)
				.println( "rowspan"			,rowspan			)
				.println( "selectedType"	,selectedType		)
				.println( "numberType"		,numberType			)
				.println( "optTypeAttri"	,optTypeAttri		)
//				.println( "changeLabelKeys"	,changeLabelKeys	)
//				.println( "changeLabelVals"	,changeLabelVals	)
				.println( "backLinkCount"	,backLinkCount		)
				.println( "headerSkipCount"	,headerSkipCount	)
				.println( "bodyString"		,bodyString			)
				.println( "useCheckControl"	,useCheckControl 	)
				.println( "useTableSorter"	,useTableSorter		)
				.println( "useParam"		,useParam			)
				.println( "useConsistency"	,useConsistency		)
				.println( "bgColorCycle"	,bgColorCycle		)
				.println( "useScrollBar"	,useScrollBar		)
				.println( "sortKeys"		,sortKeys			)
				.println( "loadMarker"		,loadMarker			)
				.println( "useSelectedRow"	,useSelectedRow		)
				.println( "useSelRowColor"	,useSelRowColor		)
				.println( "useAfterHtmlTag"	,useAfterHtmlTag	)
				.println( "writableControl"	,writableControl	)
				.println( "hight"			,height				)
				.println( "width"			,width				)
				.println( "skipNoEdit"		,skipNoEdit			)
				.println( "noTransition"	,noTransition		)
				.println( "useHilightCheckOnly",useHilightCheckOnly)
				.println( "COMMAND_LIST"	,COMMAND_LIST		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
