/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import org.opengion.fukurou.util.StringUtil;
import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * Where句を作成するための条件を指定します。
 *
 * このタグのvalue 値に、{&#064;XXXX} 変数が含まれている場合、そのリクエスト値が
 * ない場合は、このタグそのものがなにも出力しません。（つまり条件から消えます。）
 * startKeyは、value を連結する場合の頭に置かれる文字列で、where句の最初には表示されず、
 * それ以降について、表示されます。（つまり、where VALUE1 and VALUE2 and VALUE3 … です。）
 * startKey の初期値は、"and" です。
 * multi は、{&#064;XXXX} 変数に、値が複数含まれている場合の処理を規定します。
 * 複数の値とは、同一nameでチェックボックス指定や、メニューでの複数指定した場合、
 * リクエストが配列で送られます。multi="true" とすると、'xx1','xx2','xx3', ･･･ という
 * 形式に変換されます。
 * 具体的には、"where PN in ( {&#064;PN} )" という文字列に対して、
 * "where PN in ( 'xx1','xx2','xx3' )" を作成することができます。
 * multi の初期値は、"false" です。
 * SystemData の USE_SQL_INJECTION_CHECK が true か、quotCheck 属性が true の場合は、
 * ＳＱＬインジェクション対策用のクォーティションチェックを行います。リクエスト引数に
 * クォーティション(')が含まれると、エラーになります。
 * 同様にUSE_XSS_CHECKがtrueか、xssCheck属性がtrueの場合は、
 * クロスサイトススクリプティング(XSS)対策のためless/greater than signのチェックを行います。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。
 *
 * @og.formSample
 * ●形式：&lt;og:and startKey="[and|or|…]" value="…" multi="[false|true]" /&gt;
 * ●body：あり
 *
 * ●使用例
 *     &lt;og:query command="NEW"&gt;
 *         &lt;jsp:text&gt;
 *             select PN,YOBI,NMEN,HINM from XX01
 *         &lt;/jsp:text&gt;
 *         &lt;og:where&gt;
 *             &lt;og:and value="PN   =    '{&#064;PN}'"    /&gt;
 *             &lt;og:and value="YOBI like '{&#064;YOBI}%'" /&gt;
 *         &lt;/og:where&gt;
 *         &lt;jsp:text&gt;
 *             order by PN
 *         &lt;/jsp:text&gt;
 *     &lt;/og:query&gt;
 *
 *          ・検索条件が入力された時(PN=AAA , YOBI=BBB)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = 'AAA' and YOBI like 'BBB%' order by PN
 *
 *          ・検索条件が片方入力されなかった時(PNがNULLのとき, YOBI=BBB)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where YOBI like 'BBB%' order by PN
 *
 *          ・検索条件が入力されなかった時(PNがNULL, YOBIがNULL) WHERE句がなくなる。
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 order by PN
 *
 *        注意:WhereTagを使わない場合に、検索条件が入力されなかった場合は、下記のようになります。
 *            select PN,YOBI,NMEN,HINM from XX01 where PN = '' and YOBI like '%' order by PN
 *
 *    --------------------------------------------------------------------------------------------------------------
 *
 *     &lt;og:query command="NEW"&gt;
 *         &lt;jsp:text&gt;
 *             select PN,YOBI,NMEN,HINM from XX01 where PN="11111"
 *         &lt;/jsp:text&gt;
 *         &lt;og:where startKey="and"&gt;
 *             &lt;og:and value="YOBI in   ({&#064;YOBI})" multi="true" /&gt;
 *             &lt;og:and value="HINM like '{&#064;HINM}%'"             /&gt;
 *         &lt;/og:where&gt;
 *         &lt;jsp:text&gt;
 *             order by PN
 *         &lt;/jsp:text&gt;
 *     &lt;/og:query&gt;
 *
 *          ・YOBI を複数選択し、in で検索する時(YOBI=AA,BB,CC を選択)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = '11111'
 *                             and YOBI in ( 'AA','BB','CC' ) and HINM like 'BBB%' order by PN
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SqlAndTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 3562 ;	// 3.5.6.2 (2004/07/05)

	private String	startKey	= "and";
	private String	value		= "";
	private String	instrVals	= null;		// 3.8.8.1 (2007/01/06)
	private boolean	multi		= false;
	private boolean	quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
	private boolean	xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.0.0.2 (2009/09/15)
	
	private boolean	allNull	 	= false;	// 5.0.0.2 (2009/09/15)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0 (2006/12/05) BODY 部の値を value に使用する機能追加
	 * @og.rev 4.0.0 (2005/08/31) useQuotCheck() によるＳＱＬインジェクション対策
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		useQuotCheck( quotCheck );
		// 5.0.0.2 (2009/09/15) XSS対策
		useXssCheck( xssCheck );
		
		value = getRequestParameter( value );

		if( value != null && value.length() > 0 ) {
			return( SKIP_BODY );			// Body を評価しない
		}
		else {
			return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
		}
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 4.0.0 (2006/12/05) BODY 部の値を value に使用する機能追加
	 *
	 * @return  int
	 */
	@Override
	public int doAfterBody() {
		value = getBodyString();
		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.8.1 (2007/01/06) makeInstrVals を加味する。
	 * @og.rev 5.0.0.2 (2009/09/15) multi時のallNull対応
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		SqlWhereTag where = (SqlWhereTag)findAncestorWithClass( this,SqlWhereTag.class );
		if( where == null ) {
			String errMsg = "<b>このタグは、where タグの内部におく必要があります。</b>";
			throw new HybsSystemException( errMsg );
		}

		// if( ! isNull() ) {
		if( ! isNull() && ! allNull ) { // 5.0.0.2 (2009/09/15)
			value = makeInstrVals( instrVals,value );	// 3.8.8.1 (2007/01/06)
			if( value != null ) {
				set( "keyWord", startKey );
				set( "value"  , value );
				where.setAttributes( getAttributes() );
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.8.1 (2007/01/06) instrVals 属性追加
	 * @og.rev 4.0.0 (2005/08/31) quotCheck 属性の追加
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.0.0.2 (2009/09/15) multi時のallNull対応
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		startKey	= "and";
		value		= "";
		instrVals	= null;		// 3.8.8.1 (2007/01/06)
		multi		= false;
		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.0.0.2 (2009/09/15)
		allNull	 	= false;	// 5.0.0.2 (2009/09/15)
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * これは、通常のgetRequestParameter 処理の中で呼ばれる getRequestValue を
	 * オーバーライトしています。
	 * 
	 * @og.rev 5.0.0.2 (2009/09/15) valuesの全NULL/空文字をisNull扱いにする
	 *
	 * @param    key キー
	 * @return   リクエスト情報の文字列
	 */
	protected String getRequestValue( final String key ) {
		String rtn = "";

		if( multi ) {
			String[] array = getRequestValues( key );
			allNull = true; // 5.0.0.2 (2009/09/15) arrayの内容が全てnull/空文字か
			if( ! isNull() ) {
				// 5.0.0.2 (2009/09/15) 全てnull/空文字の場合はnullと扱い
				for( int i = 0; i < array.length; i++ ) {
					if( array[i] != null && array[i].length() > 0 ) {
						allNull = false;
						break;
					}
				}
				if( ! allNull ){
					rtn = makeCSVvalue( array );
				}
			}
		}
		else {
			rtn = super.getRequestValue( key );
		}
		return rtn ;
	}

	/**
	 * 複数の値を 'xx1','xx2','xx3', ･･･ という形式に変換します。
	 *
	 * この処理は、in などで使用するためのリクエストを配列で受け取って処理
	 * する場合の文字列を加工します。
	 *
	 * @param	array String[] 元の配列文字列
	 * @return  連結後の文字列
	 */
	private String makeCSVvalue( final String[] array ) {
		if( array == null || array.length == 0 ) {
			String errMsg = "array 引数に、null や、サイズゼロの配列は使用できません。";
			throw new HybsSystemException( errMsg );
		}

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append( "'" );
		buf.append( array[0] );
		buf.append( "'" );
		for(int i=1; i < array.length; i++) {
			buf.append( ",'" );
			buf.append( array[i] );
			buf.append( "'" );
		}
		return buf.toString();
	}

	/**
	 * スペースで区切られた複数の値を and 接続で連結します。
	 *
	 * value="CLM" instrVals="ABC DEF GHI" と指定すると、
	 * value="CLM LIKE '%ABC%' AND CLM LIKE '%DEF%'  AND CLM LIKE '%GHI%' "
	 * という文字列を作成します。
	 * 個別にLIKE検索項目を AND 連結する為、現れる場所に依存しません。
	 * 逆に、現れる順序を指定する場合は、ABC%DEF の様に指定可能です。
	 * ただし、columnMarker の instrVals で、複数文字のマーカーを行う場合、
	 * ABC%DEF という文字列は、オリジナルでないので、マークアップされません。
	 *
	 * @param	instrVals  String 繰返し処理を行う 値
	 * @param	value    String 繰返し処理を行う value
	 * @return  連結後の文字列
	 * @see		#setInstrVals( String )
	 * @see		ColumnMarkerTag#setInstrVals( String )
	 */
	private String makeInstrVals( final String instrVals , final String value ) {
		if( instrVals == null || instrVals.length() == 0 ) { return value; }

		String reqVals = nval( getRequestParameter( instrVals ),null );
		if( reqVals == null || reqVals.length() == 0 ) { return null; }

		final String[] vals ;
		if( multi ) {
			// multi のときは、makeCSVvalue で加工された値になっている。
			vals = StringUtil.csv2Array( reqVals,',' );
			// 前後の ' はずし
			for( int i=0; i<vals.length; i++ ) {
				vals[i] = vals[i].substring( 1,vals[i].length()-1 );
			}
		}
		else {
			vals = StringUtil.csv2Array( reqVals,' ' );
		}

		if( vals == null || vals.length == 0 ) { return null; }

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

		buf.append( value );
		buf.append( " LIKE '%" );
		buf.append( vals[0] );
		buf.append( "%'" );
		for(int i=1; i < vals.length; i++) {
			buf.append( " and " );
			buf.append( value );
			buf.append( " LIKE '%" );
			buf.append( vals[i] );
			buf.append( "%'" );
		}

		return buf.toString();
	}

	/**
	 * 【TAG】SQL条件句の最初の演算子を指定します(初期値:and)。
	 *
	 * @og.tag
	 * value を連結する場合の頭に置かれる文字列で、where句の最初には表示されず、
	 * それ以降について、表示されます。
	 * （つまり、where VALUE1 and VALUE2 and VALUE3 … です。）
	 * startKey の初期値は、"and" です。
	 *
	 * @param	skey String
	 */
	public void setStartKey( final String skey ) {
		if( skey != null && skey.length() > 0 ) { startKey = skey; }
	}

	/**
	 * 【TAG】valueを セットします。
	 *
	 * @og.tag
	 * value 値に、{&#064;XXXX} 変数が含まれている場合、そのリクエスト値がない場合は、
	 * このタグそのものがなにも出力しません。（つまり条件から消えます。）
	 * BODY 部に記述することが可能です。その場合は、value 属性になにも設定できません。
	 *
	 * @param	val value
	 */
	public void setValue( final String val ) {
		value = val;
	}

	/**
	 * 【TAG】スペースで区切られた複数の値すべてを含む条件を作成します。
	 *
	 * @og.tag
	 * 通常、value="CLM LIKE 'ABC%'" という文字列を指定しますが、
	 * value="CLM" instrVals="ABC DEF GHI" と指定すると、
	 * value="CLM LIKE '%ABC%' AND CLM LIKE '%DEF%'  AND CLM LIKE '%GHI%' "
	 * という文字列を作成します。
	 * これは、instrVals に指定した引数に対して、スペース区切りで分割し、
	 * 前方の value に複数のAND検索を同時に実現できるように指定します
	 * 個別にLIKE検索項目を AND 連結する為、現れる場所に依存しません。
	 * 逆に、現れる順序を指定する場合は、ABC%DEF の様に指定可能です。
	 * ただし、columnMarker の instrVals で、複数文字のマーカーを行う場合、
	 * ABC%DEF という文字列は、オリジナルでないので、マークアップされません。
	 *
	 * @param	val value
	 * @see		ColumnMarkerTag#setInstrVals( String )
	 */
	public void setInstrVals( final String val ) {
		instrVals = val;
	}

	/**
	 * 【TAG】複数の引数に対して処理するかどうか(true/false)を設定します(初期値:false)。
	 *
	 * @og.tag
	 * {&#064;XXXX} 変数に、値が複数含まれている場合の処理を規定します。
	 * multi="true" に設定すると、複数の引数は、'xx1','xx2','xx3', ･･･ という
	 * 形式に変換します。
	 * where 条件で言うと、 "where PN in ( {&#064;PN} )" という文字列に対して、
	 * "where PN in ( 'xx1','xx2','xx3' )" を作成することになります。
	 * 初期値は、 false (マルチ変換しない) です。
	 *
	 * @param   flag マルチ変換する ("true")／しない (それ以外)
	 */
	public void setMulti( final String flag ) {
		multi = nval( flag,multi );
	}

	/**
	 * 【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか(true/false)を設定します(初期値:USE_SQL_INJECTION_CHECK)。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * 初期値は、SystemData#USE_SQL_INJECTION_CHECK です。
	 *
	 * @og.rev 4.0.0 (2005/08/31) 新規追加
	 *
	 * @param   flag クォーティションチェックする ("true")／しない (それ以外)
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}
	
	/**
	 * 【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか(true/false)を設定します(初期値:USE_XSS_CHECK)。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * 初期値は、SystemData#USE_XMLLTAG_CHECK です。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 新規追加
	 *
	 * @param flag boolean XSSチェックする (true)／しない (false)
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "and" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "startKey"		,startKey	)
				.println( "value"			,value		)
				.println( "instrVals"		,instrVals	)
				.println( "multi"			,multi		)
				.println( "quotCheck"		,quotCheck	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
