/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report2;

import java.io.File;

import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ZipFileUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.report.ExcelInsert;
import org.opengion.hayabusa.report.ProgramRun;
import org.opengion.hayabusa.report.RFIDPrintRequest;

/**
 * 帳票要求に設定された実行方法により、各種出力、Excel取り込み、RFID出力処理を行います。
 * 1.出力
 *  雛形ファイルを一時ディレクトリに展開した後、帳票データを埋め込み、最後にOpenOffice.orgの
 *  プロセスを利用して出力を行います。
 *  対応している出力方法は、印刷、PDF出力、Excel出力です。
 *  一時ディレクトリは、システムリソースのREPORT_FILE_URLで定義されたディレクトリです。
 *  これが定義されていない場合は、システムリソースのFILE_URLで定義されたディレクト以下の/REPORTに
 *  展開されます。
 *  一時ファイルは、処理が正常に終了した場合、削除されます。(ODS出力のみにした場合は、出力直前の
 *  ODSファイルは残ります)
 *  処理でエラーが発生した場合は、一時ファイルはデバッグのため、削除されません。
 * 2.取り込み
 *  旧帳票システムの取り込み処理及びその後のPG起動を行います。
 * 3.RFID出力
 *  旧帳票システムのRFID出力処理を行います。
 * 
 * 実行方法により、出力、入力、RFID出力を行います。
 *
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class ExecProcess {

	/** 帳票処理キュー */
	private final ExecQueue queue;

	/** 出力タイプ */
	private final String type;

//	/** 実行方法 */
//	private final static String OUT_ODS_ONLY		= "1";
//	private final static String OUT_PRINT_ONLY		= "2";
//	private final static String OUT_ODS_PRINT		= "3";
//	private final static String OUT_ODS_PDF			= "P";
//	private final static String OUT_ODS_PRINT_PDF	= "Q";
//	private final static String OUT_ODS_EXCEL		= "E";
//	private final static String IN_INPUT_ONLY		= "5";
//	private final static String IN_EXEC_ONLY		= "6";
//	private final static String IN_INPUT_EXEC		= "7";
//	private final static String RFID_PRINT			= "A";
//	private final static String RFID_ALLPRINT		= "B";
//	private final static String RFID_ALLERASE		= "C";
//	private final static String RFID_SEQERASE		= "D";

	final long start = System.currentTimeMillis();
	private final boolean debug; // 4.3.0.0 (2008/07/15) デバッグの追加

	/**
	 * コンストラクタ
	 * 
	 * @og.rev 4.3.0.0 (2008/07/15)引数にdebugを追加
	 * @param qu
	 * @param debugFlag
	 */
	public ExecProcess( final ExecQueue qu , final boolean debugFlag ) {
		queue = qu;
		type = qu.getOutputType();
		debug = debugFlag;
	}

	/**
	 * 帳票処理プロセスを実行します。
	 * 
	 * @og.rev 4.3.0.0 (2008/07/15) debugの追加
	 * @og.rev 4.3.3.4 (2008/11/01) ODS出力追加
	 * @og.rev 5.1.2.0 (2010/01/01) 256シートを超えた場合の対応
	 */
	public void process() {
		// 処理開始
		addDebugMsg( "[INFO]EXEC-TIME:START=" + start );

		// 5.1.2.0 (2010/01/01) 基本的には1回で終了。256シートを超える場合のみ内部でfalseを立てる(2回目を処理させる)
		queue.setEnd( true );

		/*
		 * ======================================================================
		 * = 出力処理
		 * ======================================================================
		 */
		// パース
		if( ExecQueue.OUT_ODS_ONLY.equals( type )
				|| ExecQueue.OUT_ODS_PRINT.equals( type ) || ExecQueue.OUT_ODS_PDF.equals( type ) || ExecQueue.OUT_ODS_EXCEL.equals( type )
				|| ExecQueue.OUT_ODS_PRINT_PDF.equals( type ) || ExecQueue.OUT_ODS_ODS.equals( type ) ) {
			parse();
		}

		// 印刷
		if( ExecQueue.OUT_PRINT_ONLY.equals( type ) || ExecQueue.OUT_ODS_PRINT.equals( type ) ) {
			output( "PRINT" );
		}
		// PDF出力
		else if( ExecQueue.OUT_ODS_PDF.equals( type ) ) {
			output( "PDF" );
		}
		// EXCEL出力
		else if( ExecQueue.OUT_ODS_EXCEL.equals( type ) ) {
			output( "EXCEL" );
		}
		// 印刷 + PDF出力
		else if( ExecQueue.OUT_ODS_PRINT_PDF.equals( type ) ) {
			output( "PRINT", "PDF" );
		}
		// 4.3.3.4 (2008/11/01) 追加 ODS出力 
		else if( ExecQueue.OUT_ODS_ODS.equals( type ) ) {
			output( "ODS" );
		}

		/*
		 * ======================================================================
		 * = 取込処理
		 * ======================================================================
		 */
		// 取込
		if( ExecQueue.IN_INPUT_ONLY.equals( type ) || ExecQueue.IN_INPUT_EXEC.equals( type ) ) {
			input();
		}

		// PG起動
		if( ExecQueue.IN_EXEC_ONLY.equals( type ) || ExecQueue.IN_INPUT_EXEC.equals( type ) ) {
			pgexec();
		}

		/*
		 * ======================================================================
		 * = RFID出力処理
		 * ======================================================================
		 */
		// RFID出力
		if( ExecQueue.RFID_PRINT.equals( type ) || ExecQueue.RFID_ALLPRINT.equals( type )
				|| ExecQueue.RFID_ALLERASE.equals( type ) || ExecQueue.RFID_SEQERASE.equals( type ) ) {
			rfid();
		}

		addDebugMsg( "[INFO]EXEC-TIME:END=" + System.currentTimeMillis() );
	}

	/**
	 * 雛形ファイルを解析し、帳票データを挿入します。
	 * 
	 */
	private void parse() {
		String template = queue.getTemplateName() + ".ods";

		String tmp = 
			HybsSystem.url2dir( StringUtil.nval( HybsSystem.sys( "REPORT_FILE_URL" ) ,HybsSystem.sys( "FILE_URL" ) + "REPORT/" ) )
			+ queue.getSystemId() + File.separator + queue.getListId() + File.separator + queue.getYkno();
		String tmpdir = tmp + File.separator;
		String tmpods = tmp + ".ods";

		// 一時ファイルを削除(エラー発生時の前のファイルを削除)
		FileUtil.deleteFiles( new File( tmpdir ) );

		// 雛形ODSをテンポラリフォルダに解凍
		ZipFileUtil.unCompress( tmpdir, template );
		addDebugMsg( "[INFO]EXEC-TIME:UNCOMP=" + ( System.currentTimeMillis() - start ) );

		// DBTableModelのセット
		queue.setData();
		addDebugMsg( "[INFO]EXEC-TIME:DATA=" + ( System.currentTimeMillis() - start ) );

		// content.xml,meta.xmlのパース
		OdsContentParser contentParser = new OdsContentParser( queue, tmpdir );
		contentParser.exec();
		addDebugMsg( "[INFO]EXEC-TIME:PARSE=" + ( System.currentTimeMillis() - start ) );

		// 雛形ODSを再圧縮
		ZipFileUtil.compress( tmpdir, tmpods );
		addDebugMsg( "[INFO]EXEC-TIME:COMP=" + ( System.currentTimeMillis() - start ) );

		// 一時ファイルを削除
		FileUtil.deleteFiles( new File( tmpdir ) );
		addDebugMsg( "[INFO]EXEC-TIME:DELETE=" + ( System.currentTimeMillis() - start ) );
	}

	/**
	 * 出力処理を行います。
	 * 
	 * @og.rev 4.2.3.1 (2008/06/04) 中間ファイルの存在チェックを追加
	 * @og.rev 4.3.0.0 (2008/07/18) プリント時のプリンタ名チェック追加
	 * @og.rev 4.3.0.0 (2008/07/18) 出力ファイル名を指定していない場合に要求番号にする
	 * @og.rev 4.3.3.4 (2008/11/01) ODS出力追加
	 * @og.rev 5.1.2.0 (2010/01/01) 例外発生時にCalcオブジェクトをCloseしていないバグを修正
	 * @og.rev 5.1.6.0 (2010/05/01) 変換クラスの大幅見直しによる修正(元のコードも削除します)
	 * 
	 * @param types
	 */
	private void output( final String... types ) {
		String tmpods =
			HybsSystem.url2dir( StringUtil.nval( HybsSystem.sys( "REPORT_FILE_URL" ) ,HybsSystem.sys( "FILE_URL" ) + "REPORT/" ) )
			+ queue.getSystemId() + File.separator + queue.getListId() + File.separator + queue.getYkno() + ".ods";
		
		// 4.2.3.1 (2008/06/04) 中間ファイルの存在チェック
		if( ! new File( tmpods ).exists() ){
			queue.addMsg( "中間ファイルが存在しません。" + tmpods + HybsSystem.CR );
			throw new HybsSystemException();
		}

		// 変換クラスの生成
		DocConverter dc = new DocConverter( tmpods );
		try {
			// 起動
			dc.open();
			addDebugMsg( "[INFO]EXEC-TIME:OPEN=" + ( System.currentTimeMillis() - start ) );

			for( int i=0; i<types.length; i++ ) {
				if( "PRINT".equals( types[i] ) ) {
					// 4.3.0.0 (2008/07/18) プリント時のプリンタ名チェック
					if( queue.getPrinterName() == null || queue.getPrinterName().length() ==0 ){
						queue.addMsg( "出力先マスタが正しく設定されていません。" + HybsSystem.CR );
						throw new Exception();
					}
					dc.print( queue.getPrinterName() );
				}
				else if( "PDF".equals( types[i] ) ) {
					dc.pdf( queue.getOutputName(), queue.getPdfPasswd() );
				}
				else if( "EXCEL".equals( types[i] ) ) {					
					dc.xls( queue.getOutputName() );
				}
				// 4.3.3.4 (2008/11/01) 追加
				else if( "ODS".equals( types[i] ) ) {					
					dc.ods( queue.getOutputName() );
				}
				addDebugMsg( "[INFO]EXEC-TIME:EXEC["+types[i]+"]=" + ( System.currentTimeMillis() - start ) );
			}

			// Calcを閉じる
			dc.close();
			addDebugMsg( "[INFO]EXEC-TIME:RELEASE=" + ( System.currentTimeMillis() - start ) );
		}
		catch( Throwable th ) {
			// Calcを閉じる(エラー発生時)
			dc.close( true );
			queue.addMsg( "[INFO]EXEC-TIME:ERROR=" + ( System.currentTimeMillis() - start ) + HybsSystem.CR );
			throw new HybsSystemException( th );
		}
		// 一時ファイルの削除
		FileUtil.deleteFiles( new File( tmpods ) );
		addDebugMsg( "[INFO]EXEC-TIME:DELETE=" + ( System.currentTimeMillis() - start ) );
	}

	/**
	 * 取込処理を行います。
	 * 
	 * @og.rev 4.3.0.0 (2008/07/15) debugの追加
	 */
	private void input() {
		boolean flag = false;

		// エクセル入力の基底となるパス
		String excelinUrl = HybsSystem.url2dir( StringUtil.nval( HybsSystem.sys( "EXCEL_IN_FILE_URL" ), HybsSystem.sys( "FILE_URL" ) + "EXCELIN/" ) );
		String excelinDir = excelinUrl + queue.getSystemId() + HybsSystem.FS + queue.getListId();

		ExcelInsert ei = new ExcelInsert( queue.getSystemId(), queue.getYkno(), queue.getListId(), excelinDir, false );
		flag = ei.execute();
		if( !flag ) {
			queue.addMsg( ei.getErrMsg() );
			queue.addMsg( "取り込み処理に失敗しました" );
			throw new HybsSystemException();
		}
		addDebugMsg( "[INFO]EXEC-TIME:INPUT=" + ( System.currentTimeMillis() - start ) );
	}

	/**
	 * Excel取込後のPG起動処理を行います。
	 * 
	 * @og.rev 4.3.0.0 (2008/07/15) debugの追加
	 */
	private void pgexec() {
		boolean flag = false;

		ProgramRun pr = new ProgramRun( queue.getSystemId(), queue.getYkno(), queue.getListId(), false );
		flag = pr.execute();
		if( !flag ) {
			queue.addMsg( "取り込み後のPG起動に失敗しました" );
			queue.addMsg( pr.getErrMsg() );
			throw new HybsSystemException();
		}
		addDebugMsg( "[INFO]EXEC-TIME:PGEXEC=" + ( System.currentTimeMillis() - start ) );
	}

	/**
	 * RFID出力処理を行います。
	 * 
	 * @og.rev 4.3.0.0 (2008/07/15) debugの追加
	 * @og.rev 4.3.3.0 (2008/10/01) 板金RFID対応
	 */
	private void rfid() {
		boolean flag = false;

		// RFIDPrintRequest rpr = new RFIDPrintRequest( queue.getSystemId(), queue.getYkno(), queue.getListId(), queue.getLang(), type, queue.getPrinterName(), false );
		// 4.3.3.0 (2008/10/01) 板金RFID対応。
		RFIDPrintRequest rpr = new RFIDPrintRequest( queue.getSystemId(), queue.getYkno(), queue.getListId(), queue.getLang(), type, queue.getPrtId()
													 ,queue.getPrgDir(), queue.getPrgFile(), queue.getOutputName(), false );
		flag = rpr.initialDataSet();
		if( flag ) {
			flag = rpr.execute();
		}
		if( !flag ) {
			queue.addMsg( "RFID出力処理に失敗しました" );
			queue.addMsg( rpr.getErrMsg() );
			throw new HybsSystemException();
		}
		addDebugMsg( "[INFO]EXEC-TIME:RFID=" + ( System.currentTimeMillis() - start ) );
	}
	
	/**
	 * デバッグ用のメッセージを出力します。
	 * 
	 * @param msg
	 */
	private void addDebugMsg( final String msg ) {
		if( debug ){
			queue.addMsg( msg + HybsSystem.CR );
		}
	}
}
