//
// ptreetest.cpp --- test for ptree.h
//

#include <cstdio>
#include <cstdarg>
#include <omt/common.h>
#include <omt/ptree.h>
#include <omt/utest.h>

using namespace std;
using namespace omt;

// Test Data Class
// -----------------------------------------------------------------------------
class test {
        int     m_int;
        char*   m_str;

        static int cntr;

    public:
        test( int n, const char* b ) : m_int( n ), m_str( strdup( b ))
        {
                ++cntr;
#ifdef _DEBUG
                printf( "%03d: ==> alloc test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
        }
        ~test()
        {
#ifdef _DEBUG
                printf( "%03d: <== free test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
                --cntr;
                if ( m_str ) delete [] m_str;
        }

        void print() const
        {
                printf( "(%d, \"%s\")", m_int, m_str ? m_str : "<null>" );
        }

        const char*     cstr() const { return m_str; }
        int             num()  const { return m_int; }

        static int getCntr() { return cntr; }

    private:
        test( const test& t );
        test& operator=( const test& t );
};

int test::cntr = 0;

// -----------------------------------------------------------------------------
bool operator== ( const test& a, const test& b )
{
        return a.num() == b.num() && eql_fn<const char*>()( a.cstr(), b.cstr());
}

// -----------------------------------------------------------------------------
test* dup( const test* p )
{
        return p ? new test( p->num(), p->cstr()) : 0;
}

// -----------------------------------------------------------------------------
namespace omt {
        template<> struct cmp_fn<const test*>
        {
                int operator()( const test* p, const test* q ) const
                {
                        return ( !p && !q ) ? 0 :
				!p ? -1 :
				!q ? 1 :
				( p->num() < q->num()) ? -1 : 
				( p->num() > q->num()) ? 1 :
				cmp_fn<const char*>()( p->cstr(), q->cstr());
                }
        };
}

#ifdef _DEBUG
// -----------------------------------------------------------------------------
// use as 'tr.printree( print )' for test ptree
void print( test* t )
{
	if ( t ) t->print(); else printf( "()" );
}
#endif

// Utilities
// -----------------------------------------------------------------------------
template<typename P, typename Q>
bool check_testtree( const ptree<test*,P,const char*,Q>& tr, size_t n, ... )
{
        va_list                                               ap;
        typename ptree<test*,P,const char*,Q>::const_itor     i( tr );
        size_t                                                j;

        va_start( ap, n );
        for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		const char*	k = va_arg( ap, const char* );
                test*           p = va_arg( ap, test* );
                test*           q = i.get();

                if ( p && q == 0 || p == 0 && q ||
		     p && q && !( *p == *q  && strcmp( k, i.key()) == 0 )) break;
        }
        va_end( ap );

        return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
template<typename P, typename Q>
bool check_wtesttree( const ptree<test*,P,const test*,Q>& tr, size_t n, ... )
{
        va_list                                               ap;
        typename ptree<test*,P,const test*,Q>::const_itor     i( tr );
        size_t                                                j;

        va_start( ap, n );
        for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		const test*	k = va_arg( ap, const test* );
                test*           p = va_arg( ap, test* );
                test*           q = i.get();

                if ( p && q == 0 || p == 0 && q ||
		     p && q && !( *p == *q && *k == *i.key())) break;
        }
        va_end( ap );

        return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
template<typename P>
bool check_intkytree( const ptree<test*,P,int,refer_policy<int> >& tr, size_t n, ... )
{
        va_list                                                    ap;
        typename ptree<test*,P,int,refer_policy<int> >::const_itor i( tr );
        size_t                                                     j;

        va_start( ap, n );
        for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		int		k = va_arg( ap, int );
                test*           p = va_arg( ap, test* );
                test*           q = i.get();

                if ( p && q == 0 || p == 0 && q ||
		     p && q && !( *p == *q && k == i.key())) break;
        }
        va_end( ap );

        return !i.cont() && j == n;
}

// Unit Test Class
// -----------------------------------------------------------------------------
class ptreetest : public unittest
{
        // test data
        int i0;
        int i1;
        int i2;
        int i3;
        int i4;
        int i5;
        int i6;
        int i7;
        int i8;

        const char* s0;
        const char* s1;
        const char* s2;
        const char* s3;
        const char* s4;
        const char* s5;
        const char* s6;
        const char* s7;
        const char* s8;

        test t0;
        test t1;
        test t2;
        test t3;
        test t4;
        test t5;
        test t6;
        test t7;
        test t8;

        int     m_cnt;

    public:
        void test11();
        void test12();
        void test13();
        void test14();
        void test15();
        void test16();
        void test17();

        void test21();
        void test22();
        void test23();
        void test24();
        void test25();
        void test26();
        void test27();

        void test31();
        void test32();
        void test33();
        void test34();
        void test35();
        void test36();
        void test37();
        void test38();

        void test41();
        void test42();
        void test43();
        void test44();
        void test45();
        void test46();
        void test47();

	void test50();

        void test61();
        void test62();
        void test63();
        void test64();
        void test65();
        void test66();
        void test67();

        ptreetest() : unittest( "test for ptree.h" ),

                i0( 0 ),
                i1( 1 ),
                i2( 2 ),
                i3( 3 ),
                i4( 4 ),
                i5( 5 ),
                i6( 3 ),		// same value with i3
                i7( 7 ),
                i8( 8 ),

                s0( "" ),
                s1( "Hello" ),
                s2( "World" ),
                s3( "Open" ),
                s4( "Client" ),
                s5( "Server" ),
                s6( "Open" ),           // same value with s3
                s7( "Middleware" ),
                s8( "Web site" ),

                t0( 0, "" ),
                t1( i1, s1 ),
                t2( i2, s2 ),
                t3( i3, s3 ),
                t4( i4, s4 ),
                t5( i5, s5 ),
                t6( i3, s6 ),           // same value with s3
                t7( i7, s7 ),
                t8( i8, s8 )
        {
                m_cnt = test::getCntr();
        }
        ~ptreetest()
        {
                this->set_subtitle( "test class allocation" );
                AssertEqual( m_cnt, test::getCntr());
        }
        void run();
};


// ptree operations
// -----------------------------------------------------------------------------
void ptreetest::test11() 
{
	this->set_subtitle( "operations for ptree - refer/copy policy" );
	
	ptree<test*>	tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_testtree( tr, 0 ));
	AssertZero( tr.set( s1, &t1 ));
	AssertNonZero( check_testtree( tr, 1, s1, &t1 ));
	AssertZero( tr.set( s2, &t2 ));
	AssertZero( tr.set( s3, &t3 ));
	AssertZero( tr.set( s4, &t4 ));
	AssertZero( tr.set( s5, &t5 ));
	AssertEqual( tr.set( s6, &t6 ), &t3 );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s3, &t6, s5, &t5, s2, &t2 ));

	// find
	this->set_subtitle( "ptree::find()" );
	AssertEqual( tr.find( s1 ), &t1 );
	AssertEqual( tr.find( s2 ), &t2 );
	AssertEqual( tr.find( s3 ), &t6 );
	AssertEqual( tr.find( s4 ), &t4 );
	AssertEqual( tr.find( s5 ), &t5 );
	AssertZero( tr.find( s0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( s2, &t7 ));
	AssertNonZero( tr.replace( s3, 0 ));
	AssertZero( tr.replace( s8, &t8 ));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s3, 0, s5, &t5, s2, &t7 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( s6, &t6 ));
	AssertNonZero( tr.insert( s7, 0 ));
	AssertNonZero( tr.insert( s8, &t8 ));
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s7, 0, s3, 0, s5, &t5, s8, &t8, s2, &t7 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( s6 ));
	AssertZero( tr.remove( s7 ));
	AssertEqual( tr.remove( s8 ), &t8 );
	AssertZero( tr.remove( s0 ));
	AssertNonZero( check_testtree( tr, 4, s4, &t4, s1, &t1, s5, &t5, s2, &t7 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test12() 
{
	this->set_subtitle( "operations for ptree - store/copy policy" );
	
	ptree<test*,store_policy<test*> >	tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_testtree( tr, 0 ));
	AssertZero( tr.set( s1, dup( &t1 )));
	AssertNonZero( check_testtree( tr, 1, s1, &t1 ));
	AssertZero( tr.set( s2, dup( &t2 )));
	AssertZero( tr.set( s3, dup( &t3 )));
	AssertZero( tr.set( s4, dup( &t4 )));
	AssertZero( tr.set( s5, dup( &t5 )));
	AssertZero( tr.set( s6, dup( &t6 )));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s3, &t6, s5, &t5, s2, &t2 ));

	// find 
	this->set_subtitle( "ptree::find()" );
	AssertEqual( *tr.find( s1 ), t1 );
	AssertEqual( *tr.find( s2 ), t2 );
	AssertEqual( *tr.find( s3 ), t6 );
	AssertEqual( *tr.find( s4 ), t4 );
	AssertEqual( *tr.find( s5 ), t5 );
	AssertZero( tr.find( s0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( s2, dup( &t7 )));
	AssertNonZero( tr.replace( s3, 0 ));
	AssertZero( tr.replace( s8, dup( &t8 )));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s3, 0, s5, &t5, s2, &t7 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( s6, dup( &t6 )));
	AssertNonZero( tr.insert( s7, 0 ));
	AssertNonZero( tr.insert( s8, dup( &t8 )));
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s7, 0, s3, 0, s5, &t5, s8, &t8, s2, &t7 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( s6 ));
	AssertZero( tr.remove( s7 ));
	AssertZero( tr.remove( s8 ));
	AssertZero( tr.remove( s0 ));
	AssertNonZero( check_testtree( tr, 4, s4, &t4, s1, &t1, s5, &t5, s2, &t7 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test13()
{
	this->set_subtitle( "operations for ptree - copy/copy policy" );
	
	ptree<test*,copy_policy<test*> >	tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_testtree( tr, 0 ));
	AssertZero( tr.set( s1, &t1 ));
	AssertNonZero( check_testtree( tr, 1, s1, &t1 ));
	AssertZero( tr.set( s2, &t2 ));
	AssertZero( tr.set( s3, &t3 ));
	AssertZero( tr.set( s4, &t4 ));
	AssertZero( tr.set( s5, &t5 ));
	AssertZero( tr.set( s6, &t6 ));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s3, &t6, s5, &t5, s2, &t2 ));

	// find
	this->set_subtitle( "ptree::find()" );
	AssertEqual( *tr.find( s1 ), t1 );
	AssertEqual( *tr.find( s2 ), t2 );
	AssertEqual( *tr.find( s3 ), t6 );
	AssertEqual( *tr.find( s4 ), t4 );
	AssertEqual( *tr.find( s5 ), t5 );
	AssertZero( tr.find( s0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( s2, &t7 ));
	AssertNonZero( tr.replace( s3, 0 ));
	AssertZero( tr.replace( s8, &t8 ));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s3, 0, s5, &t5, s2, &t7 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( s6, &t6 ));
	AssertNonZero( tr.insert( s7, 0 ));
	AssertNonZero( tr.insert( s8, &t8 ));
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s7, 0, s3, 0, s5, &t5, s8, &t8, s2, &t7 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( s6 ));
	AssertZero( tr.remove( s7 ));
	AssertZero( tr.remove( s8 ));
	AssertZero( tr.remove( s0 ));
	AssertNonZero( check_testtree( tr, 4, s4, &t4, s1, &t1, s5, &t5, s2, &t7 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test14() 
{
	this->set_subtitle( "operations for ptree - store/refer<int> policy" );
	
	ptree<test*,store_policy<test*>,int,refer_policy<int> >	tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_intkytree( tr, 0 ));
	AssertZero( tr.set( i1, dup( &t1 )));
	AssertNonZero( check_intkytree( tr, 1, i1, &t1 ));
	AssertZero( tr.set( i2, dup( &t2 )));
	AssertZero( tr.set( i3, dup( &t3 )));
	AssertZero( tr.set( i4, dup( &t4 )));
	AssertZero( tr.set( i5, dup( &t5 )));
	AssertZero( tr.set( i6, dup( &t6 )));
	AssertNonZero( check_intkytree( tr, 5, i1, &t1, i2, &t2, i3, &t6, i4, &t4, i5, &t5 ));

	// find 
	this->set_subtitle( "ptree::find()" );
	AssertEqual( *tr.find( i1 ), t1 );
	AssertEqual( *tr.find( i2 ), t2 );
	AssertEqual( *tr.find( i3 ), t6 );
	AssertEqual( *tr.find( i4 ), t4 );
	AssertEqual( *tr.find( i5 ), t5 );
	AssertZero( tr.find( i0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( i2, dup( &t7 )));
	AssertNonZero( tr.replace( i3, 0 ));
	AssertZero( tr.replace( i8, dup( &t8 )));
	AssertNonZero( check_intkytree( tr, 5, i1, &t1, i2, &t7, i3, 0, i4, &t4, i5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( i6, dup( &t6 )));
	AssertNonZero( tr.insert( i7, 0 ));
	AssertNonZero( tr.insert( i8, dup( &t8 )));
	AssertNonZero( check_intkytree( tr, 7, i1, &t1, i2, &t7, i3, 0, i4, &t4, i5, &t5, i7, 0, i8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( i6 ));
	AssertZero( tr.remove( i7 ));
	AssertZero( tr.remove( i8 ));
	AssertZero( tr.remove( i0 ));
	AssertNonZero( check_intkytree( tr, 4, i1, &t1, i2, &t7, i4, &t4, i5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test15()
{
	this->set_subtitle( "operations for ptree - store/refer<obj> policy" );
	
	ptree<test*,store_policy<test*>,const test*,refer_policy<const test*> > tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_wtesttree( tr, 0 ));
	AssertZero( tr.set( &t1, dup( &t1 )));
	AssertNonZero( check_wtesttree( tr, 1, &t1, &t1 ));
	AssertZero( tr.set( &t2, dup( &t2 )));
	AssertZero( tr.set( &t3, dup( &t3 )));
	AssertZero( tr.set( &t4, dup( &t4 )));
	AssertZero( tr.set( &t5, dup( &t5 )));
	AssertZero( tr.set( &t6, dup( &t6 )));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t3, &t6, &t4, &t4, &t5, &t5 ));

	// find 
	this->set_subtitle( "ptree::find()" );
	AssertEqual( *tr.find( &t1 ), t1 );
	AssertEqual( *tr.find( &t2 ), t2 );
	AssertEqual( *tr.find( &t3 ), t6 );
	AssertEqual( *tr.find( &t4 ), t4 );
	AssertEqual( *tr.find( &t5 ), t5 );
	AssertZero( tr.find( &t0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( &t2, dup( &t7 )));
	AssertNonZero( tr.replace( &t3, 0 ));
	AssertZero( tr.replace( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t7, &t3, 0, &t4, &t4, &t5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( &t6, dup( &t6 )));
	AssertNonZero( tr.insert( &t7, 0 ));
	AssertNonZero( tr.insert( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t7, &t3, 0, &t4, &t4, &t5, &t5, &t7, 0, &t8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( &t6 ));
	AssertZero( tr.remove( &t7 ));
	AssertZero( tr.remove( &t8 ));
	AssertZero( tr.remove( &t0 ));
	AssertNonZero( check_wtesttree( tr, 4, &t1, &t1, &t2, &t7, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test16()
{
	this->set_subtitle( "operations for ptree - store/store<obj> policy" );
	
	ptree<test*,store_policy<test*>,const test*,store_policy<const test*> > tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_wtesttree( tr, 0 ));
	AssertZero( tr.set( dup( &t1 ), dup( &t1 )));
	AssertNonZero( check_wtesttree( tr, 1, &t1, &t1 ));
	AssertZero( tr.set( dup( &t2 ), dup( &t2 )));
	AssertZero( tr.set( dup( &t3 ), dup( &t3 )));
	AssertZero( tr.set( dup( &t4 ), dup( &t4 )));
	AssertZero( tr.set( dup( &t5 ), dup( &t5 )));
	AssertZero( tr.set( dup( &t6 ), dup( &t6 )));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t3, &t6, &t4, &t4, &t5, &t5 ));

	// find 
	this->set_subtitle( "ptree::find()" );
	AssertEqual( *tr.find( &t1 ), t1 );
	AssertEqual( *tr.find( &t2 ), t2 );
	AssertEqual( *tr.find( &t3 ), t6 );
	AssertEqual( *tr.find( &t4 ), t4 );
	AssertEqual( *tr.find( &t5 ), t5 );
	AssertZero( tr.find( &t0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( &t2, dup( &t7 )));
	AssertNonZero( tr.replace( &t3, 0 ));
	AssertZero( tr.replace( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t7, &t3, 0, &t4, &t4, &t5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( dup( &t6 ), dup( &t6 )));
	AssertNonZero( tr.insert( dup( &t7 ), 0 ));
	AssertNonZero( tr.insert( dup( &t8 ), dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t7, &t3, 0, &t4, &t4, &t5, &t5, &t7, 0, &t8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( &t6 ));
	AssertZero( tr.remove( &t7 ));
	AssertZero( tr.remove( &t8 ));
	AssertZero( tr.remove( &t0 ));
	AssertNonZero( check_wtesttree( tr, 4, &t1, &t1, &t2, &t7, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test17()
{
	this->set_subtitle( "operations for ptree - store/copy<obj> policy" );
	
	ptree<test*,store_policy<test*>,const test*,copy_policy<const test*> > tr;

	// set
	this->set_subtitle( "ptree::set()" );
	AssertNonZero( check_wtesttree( tr, 0 ));
	AssertZero( tr.set( &t1, dup( &t1 )));
	AssertNonZero( check_wtesttree( tr, 1, &t1, &t1 ));
	AssertZero( tr.set( &t2, dup( &t2 )));
	AssertZero( tr.set( &t3, dup( &t3 )));
	AssertZero( tr.set( &t4, dup( &t4 )));
	AssertZero( tr.set( &t5, dup( &t5 )));
	AssertZero( tr.set( &t6, dup( &t6 )));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t3, &t6, &t4, &t4, &t5, &t5 ));

	// find 
	this->set_subtitle( "ptree::find()" );
	AssertEqual( *tr.find( &t1 ), t1 );
	AssertEqual( *tr.find( &t2 ), t2 );
	AssertEqual( *tr.find( &t3 ), t6 );
	AssertEqual( *tr.find( &t4 ), t4 );
	AssertEqual( *tr.find( &t5 ), t5 );
	AssertZero( tr.find( &t0 ));

	// replace
	this->set_subtitle( "ptree::replace()" );
	AssertNonZero( tr.replace( &t2, dup( &t7 )));
	AssertNonZero( tr.replace( &t3, 0 ));
	AssertZero( tr.replace( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t7, &t3, 0, &t4, &t4, &t5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( &t6, dup( &t6 )));
	AssertNonZero( tr.insert( &t7, 0 ));
	AssertNonZero( tr.insert( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t7, &t3, 0, &t4, &t4, &t5, &t5, &t7, 0, &t8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( &t6 ));
	AssertZero( tr.remove( &t7 ));
	AssertZero( tr.remove( &t8 ));
	AssertZero( tr.remove( &t0 ));
	AssertNonZero( check_wtesttree( tr, 4, &t1, &t1, &t2, &t7, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test21() 
{
	this->set_subtitle( "operations for multi-value ptree - refer/copy policy" );
	
	ptree<test*>	tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_testtree( tr, 0 ));
	tr.put( s1, &t1 );
	AssertNonZero( check_testtree( tr, 1, s1, &t1 ));
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	AssertNonZero( check_testtree( tr, 6, s4, &t4, s1, &t1, s3, &t3, s6, &t6, s5, &t5, s2, &t2 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( s6, &t6 ));
	AssertNonZero( tr.insert( s7, 0 ));
	AssertNonZero( tr.insert( s8, &t8 ));
	AssertNonZero( check_testtree( tr, 8, s4, &t4, s1, &t1, s7, 0, s3, &t3, s6, &t6, s5, &t5, s8, &t8, s2, &t2 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertEqual( tr.remove( s6 ), &t3 );
	AssertZero( tr.remove( s7 ));
	AssertEqual( tr.remove( s8 ), &t8 );
	AssertZero( tr.remove( s0 ));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s6, &t6, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test22() 
{
	this->set_subtitle( "operations for multi-value ptree - store/copy policy" );
	
	ptree<test*,store_policy<test*> >	tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_testtree( tr, 0 ));
	tr.put( s1, dup( &t1 ));
	AssertNonZero( check_testtree( tr, 1, s1, &t1 ));
	tr.put( s2, dup( &t2 ));
	tr.put( s3, dup( &t3 ));
	tr.put( s4, dup( &t4 ));
	tr.put( s5, dup( &t5 ));
	tr.put( s6, dup( &t6 ));
	AssertNonZero( check_testtree( tr, 6, s4, &t4, s1, &t1, s3, &t3, s6, &t6, s5, &t5, s2, &t2 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( s6, dup( &t6 )));
	AssertNonZero( tr.insert( s7, 0 ));
	AssertNonZero( tr.insert( s8, dup( &t8 )));
	AssertNonZero( check_testtree( tr, 8, s4, &t4, s1, &t1, s7, 0, s3, &t3, s6, &t6, s5, &t5, s8, &t8, s2, &t2 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( s6 ));
	AssertZero( tr.remove( s7 ));
	AssertZero( tr.remove( s8 ));
	AssertZero( tr.remove( s0 ));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s6, &t6, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test23()
{
	this->set_subtitle( "operations for multi-value ptree - copy/copy policy" );
	
	ptree<test*,copy_policy<test*> >	tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_testtree( tr, 0 ));
	tr.put( s1, &t1 );
	AssertNonZero( check_testtree( tr, 1, s1, &t1 ));
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	AssertNonZero( check_testtree( tr, 6, s4, &t4, s1, &t1, s3, &t3, s6, &t6, s5, &t5, s2, &t2 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( s6, &t6 ));
	AssertNonZero( tr.insert( s7, 0 ));
	AssertNonZero( tr.insert( s8, &t8 ));
	AssertNonZero( check_testtree( tr, 8, s4, &t4, s1, &t1, s7, 0, s3, &t3, s6, &t6, s5, &t5, s8, &t8, s2, &t2 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( s6 ));
	AssertZero( tr.remove( s7 ));
	AssertZero( tr.remove( s8 ));
	AssertZero( tr.remove( s0 ));
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t1, s6, &t6, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test24() 
{
	this->set_subtitle( "operations for multi-value ptree - store/refer<int> policy" );
	
	ptree<test*,store_policy<test*>,int,refer_policy<int> >	tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_intkytree( tr, 0 ));
	tr.put( i1, dup( &t1 ));
	AssertNonZero( check_intkytree( tr, 1, i1, &t1 ));
	tr.put( i2, dup( &t2 ));
	tr.put( i3, dup( &t3 ));
	tr.put( i4, dup( &t4 ));
	tr.put( i5, dup( &t5 ));
	tr.put( i6, dup( &t6 ));
	AssertNonZero( check_intkytree( tr, 6, i1, &t1, i2, &t2, i3, &t3, i6, &t6, i4, &t4, i5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( i6, dup( &t6 )));
	AssertNonZero( tr.insert( i7, 0 ));
	AssertNonZero( tr.insert( i8, dup( &t8 )));
	AssertNonZero( check_intkytree( tr, 8, i1, &t1, i2, &t2, i3, &t3, i6, &t6, i4, &t4, i5, &t5, i7, 0, i8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( i6 ));
	AssertZero( tr.remove( i7 ));
	AssertZero( tr.remove( i8 ));
	AssertZero( tr.remove( i0 ));
	AssertNonZero( check_intkytree( tr, 5, i1, &t1, i2, &t2, i6, &t6, i4, &t4, i5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test25()
{
	this->set_subtitle( "operations for multi-value ptree - store/refer<obj> policy" );
	
	ptree<test*,store_policy<test*>,const test*,refer_policy<const test*> > tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_wtesttree( tr, 0 ));
	tr.put( &t1, dup( &t1 ));
	AssertNonZero( check_wtesttree( tr, 1, &t1, &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	AssertNonZero( check_wtesttree( tr, 6, &t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4, &t5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( &t6, dup( &t6 )));
	AssertNonZero( tr.insert( &t7, 0 ));
	AssertNonZero( tr.insert( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 8, 
		&t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4, &t5, &t5, &t7, 0, &t8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( &t6 ));
	AssertZero( tr.remove( &t7 ));
	AssertZero( tr.remove( &t8 ));
	AssertZero( tr.remove( &t0 ));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t6, &t6, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test26()
{
	this->set_subtitle( "operations for multi-value ptree - store/store<obj> policy" );
	
	ptree<test*,store_policy<test*>,const test*,store_policy<const test*> > tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_wtesttree( tr, 0 ));
	tr.put( dup( &t1 ), dup( &t1 ));
	AssertNonZero( check_wtesttree( tr, 1, &t1, &t1 ));
	tr.put( dup( &t2 ), dup( &t2 ));
	tr.put( dup( &t3 ), dup( &t3 ));
	tr.put( dup( &t4 ), dup( &t4 ));
	tr.put( dup( &t5 ), dup( &t5 ));
	tr.put( dup( &t6 ), dup( &t6 ));
	AssertNonZero( check_wtesttree( tr, 6, &t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4, &t5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( dup( &t6 ), dup( &t6 )));
	AssertNonZero( tr.insert( dup( &t7 ), 0 ));
	AssertNonZero( tr.insert( dup( &t8 ), dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 8, 
		&t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4, &t5, &t5, &t7, 0, &t8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( &t6 ));
	AssertZero( tr.remove( &t7 ));
	AssertZero( tr.remove( &t8 ));
	AssertZero( tr.remove( &t0 ));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t6, &t6, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test27()
{
	this->set_subtitle( "operations for multi-value ptree - store/copy<obj> policy" );
	
	ptree<test*,store_policy<test*>,const test*,copy_policy<const test*> > tr;

	// put
	this->set_subtitle( "ptree::put()" );
	AssertNonZero( check_wtesttree( tr, 0 ));
	tr.put( &t1, dup( &t1 ));
	AssertNonZero( check_wtesttree( tr, 1, &t1, &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	AssertNonZero( check_wtesttree( tr, 6, &t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4, &t5, &t5 ));

	// insert
	this->set_subtitle( "ptree::insert()" );
	AssertZero( tr.insert( &t6, dup( &t6 )));
	AssertNonZero( tr.insert( &t7, 0 ));
	AssertNonZero( tr.insert( &t8, dup( &t8 )));
	AssertNonZero( check_wtesttree( tr, 8, 
		&t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4, &t5, &t5, &t7, 0, &t8, &t8 ));

	// remove
	this->set_subtitle( "ptree::remove()" );
	AssertZero( tr.remove( &t6 ));
	AssertZero( tr.remove( &t7 ));
	AssertZero( tr.remove( &t8 ));
	AssertZero( tr.remove( &t0 ));
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t6, &t6, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test31() 
{
	this->set_subtitle( "itor/riotr for ptree - refer/copy policy" );

	typedef ptree<test*> tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( s1, &t1 );
	tr.set( s2, &t2 );
	tr.set( s3, &t3 );
	tr.set( s4, &t4 );
	tr.set( s5, &t5 );
	tr.set( s6, &t6 );
	
	// itor
	for ( n = 0, it.find( s1 ); it.ne( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t1 ); AssertZero( strcmp( it.key(), s1 )); break;
			case 1: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( s7 ); it.le( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 1: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
			case 2: AssertEqual( n, *it.get(), t2 ); AssertZero( strcmp( it.key(), s2 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( s4 ); it.lt( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t1 ); AssertZero( strcmp( it.key(), s1 )); break;
			case 1: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( s5 ); ir.ne( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertZero( strcmp( ir.key(), s1 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( s2 ); ir.ge( s7 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t2 ); AssertZero( strcmp( ir.key(), s2 )); break;
			case 1: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 2: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( s2 ); ir.gt( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertZero( strcmp( ir.key(), s1 )); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( s1 ); it.ne( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, it.set( 0 ),   &t1 ); break;
			case 1: AssertEqual( n, it.set( &t7 ), &t6 ); break;
			case 2: AssertEqual( n, it.set( &t8 ), &t5 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, 0, s3, &t7, s5, &t8, s2, &t2 ));

	for ( n = 0, ir.find( s5 ); ir.ne( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, ir.set( 0 ),   &t8 ); break;
			case 1: AssertEqual( n, ir.set( &t1 ), &t7 ); break;
			case 2: AssertZero( n,  ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t2, s3, &t1, s5, 0, s2, &t2 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( s1, &t1 );
	tr.set( s2, &t2 );
	tr.set( s3, &t3 );
	tr.set( s4, &t4 );
	tr.set( s5, &t5 );
	for ( n = 0, it.find( s3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( it.del(), &t3 ); break;
			case 1: AssertEqual( it.del(), &t5 ); break;
			case 2: AssertEqual( it.del(), &t2 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 2, s4, &t4, s1, &t1 ));

	tr.clear();
	tr.set( s1, &t1 );
	tr.set( s2, &t2 );
	tr.set( s3, &t3 );
	tr.set( s4, &t4 );
	tr.set( s5, &t5 );
	for ( n = 0, ir.find( s3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( ir.del(), &t3 ); break;
			case 1: AssertEqual( ir.del(), &t1 ); break;
			case 2: AssertEqual( ir.del(), &t4 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 2, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test32() 
{
	this->set_subtitle( "itor/riotr for ptree - store/copy policy" );

	typedef ptree<test*, store_policy<test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( s1, dup( &t1 ));
	tr.set( s2, dup( &t2 ));
	tr.set( s3, dup( &t3 ));
	tr.set( s4, dup( &t4 ));
	tr.set( s5, dup( &t5 ));
	tr.set( s6, dup( &t6 ));
	
	// itor
	for ( n = 0, it.find( s1 ); it.ne( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t1 ); AssertZero( strcmp( it.key(), s1 )); break;
			case 1: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( s7 ); it.le( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 1: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
			case 2: AssertEqual( n, *it.get(), t2 ); AssertZero( strcmp( it.key(), s2 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( s4 ); it.lt( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t1 ); AssertZero( strcmp( it.key(), s1 )); break;
			case 1: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( s5 ); ir.ne( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertZero( strcmp( ir.key(), s1 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( s2 ); ir.ge( s7 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t2 ); AssertZero( strcmp( ir.key(), s2 )); break;
			case 1: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 2: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( s2 ); ir.gt( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertZero( strcmp( ir.key(), s1 )); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( s1 ); it.ne( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( dup( &t7 ))); break;
			case 2: AssertZero( n, it.set( dup( &t8 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, 0, s3, &t7, s5, &t8, s2, &t2 ));

	for ( n = 0, ir.find( s5 ); ir.ne( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( dup( &t1 ))); break;
			case 2: AssertZero( n, ir.set( dup( &t2 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t2, s3, &t1, s5, 0, s2, &t2 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( s1, dup( &t1 ));
	tr.set( s2, dup( &t2 ));
	tr.set( s3, dup( &t3 ));
	tr.set( s4, dup( &t4 ));
	tr.set( s5, dup( &t5 ));
	for ( n = 0, it.find( s3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t5 ); break;
			case 2: AssertEqual( *it.del(), t2 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 2, s4, &t4, s1, &t1 ));

	tr.clear();
	tr.set( s1, dup( &t1 ));
	tr.set( s2, dup( &t2 ));
	tr.set( s3, dup( &t3 ));
	tr.set( s4, dup( &t4 ));
	tr.set( s5, dup( &t5 ));
	for ( n = 0, ir.find( s3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t3 ); break;
			case 1: AssertEqual( *ir.del(), t1 ); break;
			case 2: AssertEqual( *ir.del(), t4 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 2, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test33() 
{
	this->set_subtitle( "itor/riotr for ptree - copy/copy policy" );

	typedef ptree<test*, copy_policy<test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( s1, &t1 );
	tr.set( s2, &t2 );
	tr.set( s3, &t3 );
	tr.set( s4, &t4 );
	tr.set( s5, &t5 );
	tr.set( s6, &t6 );
	
	// itor
	for ( n = 0, it.find( s1 ); it.ne( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t1 ); AssertZero( strcmp( it.key(), s1 )); break;
			case 1: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( s7 ); it.le( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 1: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
			case 2: AssertEqual( n, *it.get(), t2 ); AssertZero( strcmp( it.key(), s2 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( s4 ); it.lt( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t1 ); AssertZero( strcmp( it.key(), s1 )); break;
			case 1: AssertEqual( n, *it.get(), t6 ); AssertZero( strcmp( it.key(), s6 )); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertZero( strcmp( it.key(), s5 )); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( s5 ); ir.ne( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertZero( strcmp( ir.key(), s1 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( s2 ); ir.ge( s7 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t2 ); AssertZero( strcmp( ir.key(), s2 )); break;
			case 1: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 2: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( s2 ); ir.gt( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t5 ); AssertZero( strcmp( ir.key(), s5 )); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); AssertZero( strcmp( ir.key(), s6 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertZero( strcmp( ir.key(), s1 )); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( s1 ); it.ne( s2 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 ));   break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, 0, s3, &t7, s5, &t8, s2, &t2 ));

	for ( n = 0, ir.find( s5 ); ir.ne( s4 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 ));   break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 5, s4, &t4, s1, &t2, s3, &t1, s5, 0, s2, &t2 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( s1, &t1 );
	tr.set( s2, &t2 );
	tr.set( s3, &t3 );
	tr.set( s4, &t4 );
	tr.set( s5, &t5 );
	for ( n = 0, it.find( s3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t5 ); break;
			case 2: AssertEqual( *it.del(), t2 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 2, s4, &t4, s1, &t1 ));

	tr.clear();
	tr.set( s1, &t1 );
	tr.set( s2, &t2 );
	tr.set( s3, &t3 );
	tr.set( s4, &t4 );
	tr.set( s5, &t5 );
	for ( n = 0, ir.find( s3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t3 ); break;
			case 1: AssertEqual( *ir.del(), t1 ); break;
			case 2: AssertEqual( *ir.del(), t4 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 2, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test34() 
{
	this->set_subtitle( "itor/riotr for ptree - copy/refer<int> policy" );

	typedef ptree<test*, copy_policy<test*>, int, refer_policy<int> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( i1, &t1 );
	tr.set( i2, &t2 );
	tr.set( i3, &t3 );
	tr.set( i4, &t4 );
	tr.set( i5, &t5 );
	tr.set( i6, &t6 );
	
	// itor
	for ( n = 0, it.find( i2 ); it.ne( i5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 ); AssertEqual( it.key(), i2 ); break;
			case 1: AssertEqual( n, *it.get(), t3 ); AssertEqual( it.key(), i3 ); break;
			case 2: AssertEqual( n, *it.get(), t4 ); AssertEqual( it.key(), i4 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( i3 ); it.le( i5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); AssertEqual( it.key(), i3 ); break;
			case 1: AssertEqual( n, *it.get(), t4 ); AssertEqual( it.key(), i4 ); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertEqual( it.key(), i5 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( i1 ); it.lt( i5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 ); AssertEqual( it.key(), i2 ); break;
			case 1: AssertEqual( n, *it.get(), t3 ); AssertEqual( it.key(), i3 ); break;
			case 2: AssertEqual( n, *it.get(), t4 ); AssertEqual( it.key(), i4 ); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( i4 ); ir.ne( i1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 ); AssertEqual( ir.key(), i4 ); break;
			case 1: AssertEqual( n, *ir.get(), t3 ); AssertEqual( ir.key(), i3 ); break;
			case 2: AssertEqual( n, *ir.get(), t2 ); AssertEqual( ir.key(), i2 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( i3 ); ir.ge( i1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t3 ); AssertEqual( ir.key(), i3 ); break;
			case 1: AssertEqual( n, *ir.get(), t2 ); AssertEqual( ir.key(), i2 ); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertEqual( ir.key(), i1 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( i5 ); ir.gt( i1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 ); AssertEqual( ir.key(), i4 ); break;
			case 1: AssertEqual( n, *ir.get(), t3 ); AssertEqual( ir.key(), i3 ); break;
			case 2: AssertEqual( n, *ir.get(), t2 ); AssertEqual( ir.key(), i2 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( i2 ); it.ne( i5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 ));   break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_intkytree( tr, 5, i1, &t1, i2, 0, i3, &t7, i4, &t8, i5, &t5 ));

	for ( n = 0, ir.find( i4 ); ir.ne( i1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 ));   break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_intkytree( tr, 5, i1, &t1, i2, &t2, i3, &t1, i4, 0, i5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( i1, &t1 );
	tr.set( i2, &t2 );
	tr.set( i3, &t3 );
	tr.set( i4, &t4 );
	tr.set( i5, &t5 );
	for ( n = 0, it.find( i3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t4 ); break;
			case 2: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_intkytree( tr, 2, i1, &t1, i2, &t2 ));

	tr.clear();
	tr.set( i1, &t1 );
	tr.set( i2, &t2 );
	tr.set( i3, &t3 );
	tr.set( i4, &t4 );
	tr.set( i5, &t5 );
	for ( n = 0, ir.find( i3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t3 ); break;
			case 1: AssertEqual( *ir.del(), t2 ); break;
			case 2: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_intkytree( tr, 2, i4, &t4, i5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test35() 
{
	this->set_subtitle( "itor/riotr for ptree - copy/refer<obj> policy" );

	typedef ptree<test*, copy_policy<test*>, const test*, refer_policy<const test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( &t1, &t1 );
	tr.set( &t2, &t2 );
	tr.set( &t3, &t3 );
	tr.set( &t4, &t4 );
	tr.set( &t5, &t5 );
	tr.set( &t6, &t6 );
	
	// itor
	for ( n = 0, it.find( &t2 ); it.ne( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 );
				AssertEqual( it.key(), const_cast<const test* const>( &t2 )); break;
			case 1: AssertEqual( n, *it.get(), t3 );
				AssertEqual( it.key(), const_cast<const test* const>( &t3 )); break;
			case 2: AssertEqual( n, *it.get(), t4 );
				AssertEqual( it.key(), const_cast<const test* const>( &t4 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( &t3 ); it.le( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 );
				AssertEqual( it.key(), const_cast<const test* const>( &t3 )); break;
			case 1: AssertEqual( n, *it.get(), t4 );
				AssertEqual( it.key(), const_cast<const test* const>( &t4 )); break;
			case 2: AssertEqual( n, *it.get(), t5 );
				AssertEqual( it.key(), const_cast<const test* const>( &t5 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( &t1 ); it.lt( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 );
				AssertEqual( it.key(), const_cast<const test* const>( &t2 )); break;
			case 1: AssertEqual( n, *it.get(), t3 );
				AssertEqual( it.key(), const_cast<const test* const>( &t3 )); break;
			case 2: AssertEqual( n, *it.get(), t4 );
				AssertEqual( it.key(), const_cast<const test* const>( &t4 )); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( &t4 ); ir.ne( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t4 )); break;
			case 1: AssertEqual( n, *ir.get(), t3 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t3 )); break;
			case 2: AssertEqual( n, *ir.get(), t2 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( &t3 ); ir.ge( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t3 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t3 )); break;
			case 1: AssertEqual( n, *ir.get(), t2 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t2 )); break;
			case 2: AssertEqual( n, *ir.get(), t1 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t1 )); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( &t5 ); ir.gt( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t4 )); break;
			case 1: AssertEqual( n, *ir.get(), t3 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t3 )); break;
			case 2: AssertEqual( n, *ir.get(), t2 );
				AssertEqual( ir.key(), const_cast<const test* const>( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( &t2 ); it.ne( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 ));   break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, 0, &t3, &t7, &t4, &t8, &t5, &t5 ));

	for ( n = 0, ir.find( &t4 ); ir.ne( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 ));   break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t3, &t1, &t4, 0, &t5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( &t1, &t1 );
	tr.set( &t2, &t2 );
	tr.set( &t3, &t3 );
	tr.set( &t4, &t4 );
	tr.set( &t5, &t5 );
	for ( n = 0, it.find( &t3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t4 ); break;
			case 2: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 2, &t1, &t1, &t2, &t2 ));

	tr.clear();
	tr.set( &t1, &t1 );
	tr.set( &t2, &t2 );
	tr.set( &t3, &t3 );
	tr.set( &t4, &t4 );
	tr.set( &t5, &t5 );
	for ( n = 0, ir.find( &t3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t3 ); break;
			case 1: AssertEqual( *ir.del(), t2 ); break;
			case 2: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 2, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test36() 
{
	this->set_subtitle( "itor/riotr for ptree - copy/store_policy<obj> policy" );

	typedef ptree<test*, copy_policy<test*>, const test*, store_policy<const test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( dup( &t1 ), &t1 );
	tr.set( dup( &t2 ), &t2 );
	tr.set( dup( &t3 ), &t3 );
	tr.set( dup( &t4 ), &t4 );
	tr.set( dup( &t5 ), &t5 );
	tr.set( dup( &t6 ), &t6 );
	
	// itor
	for ( n = 0, it.find( &t2 ); it.ne( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 ); AssertEqual( *it.key(), t2 ); break;
			case 1: AssertEqual( n, *it.get(), t3 ); AssertEqual( *it.key(), t3 ); break;
			case 2: AssertEqual( n, *it.get(), t4 ); AssertEqual( *it.key(), t4 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( &t3 ); it.le( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); AssertEqual( *it.key(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t4 ); AssertEqual( *it.key(), t4 ); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertEqual( *it.key(), t5 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( &t1 ); it.lt( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 ); AssertEqual( *it.key(), t2 ); break;
			case 1: AssertEqual( n, *it.get(), t3 ); AssertEqual( *it.key(), t3 ); break;
			case 2: AssertEqual( n, *it.get(), t4 ); AssertEqual( *it.key(), t4 ); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( &t4 ); ir.ne( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 ); AssertEqual( *ir.key(), t4 ); break;
			case 1: AssertEqual( n, *ir.get(), t3 ); AssertEqual( *ir.key(), t3 ); break;
			case 2: AssertEqual( n, *ir.get(), t2 ); AssertEqual( *ir.key(), t2 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( &t3 ); ir.ge( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t3 ); AssertEqual( *ir.key(), t3 ); break;
			case 1: AssertEqual( n, *ir.get(), t2 ); AssertEqual( *ir.key(), t2 ); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertEqual( *ir.key(), t1 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( &t5 ); ir.gt( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 ); AssertEqual( *ir.key(), t4 ); break;
			case 1: AssertEqual( n, *ir.get(), t3 ); AssertEqual( *ir.key(), t3 ); break;
			case 2: AssertEqual( n, *ir.get(), t2 ); AssertEqual( *ir.key(), t2 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( &t2 ); it.ne( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 ));   break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, 0, &t3, &t7, &t4, &t8, &t5, &t5 ));

	for ( n = 0, ir.find( &t4 ); ir.ne( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 ));   break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t3, &t1, &t4, 0, &t5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( dup( &t1 ), &t1 );
	tr.set( dup( &t2 ), &t2 );
	tr.set( dup( &t3 ), &t3 );
	tr.set( dup( &t4 ), &t4 );
	tr.set( dup( &t5 ), &t5 );
	for ( n = 0, it.find( &t3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t4 ); break;
			case 2: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 2, &t1, &t1, &t2, &t2 ));

	tr.clear();
	tr.set( dup( &t1 ), &t1 );
	tr.set( dup( &t2 ), &t2 );
	tr.set( dup( &t3 ), &t3 );
	tr.set( dup( &t4 ), &t4 );
	tr.set( dup( &t5 ), &t5 );
	for ( n = 0, ir.find( &t3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t3 ); break;
			case 1: AssertEqual( *ir.del(), t2 ); break;
			case 2: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 2, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test37() 
{
	this->set_subtitle( "itor/riotr for ptree - copy/copy<obj> policy" );

	typedef ptree<test*, copy_policy<test*>, const test*, copy_policy<const test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.set( &t1, &t1 );
	tr.set( &t2, &t2 );
	tr.set( &t3, &t3 );
	tr.set( &t4, &t4 );
	tr.set( &t5, &t5 );
	tr.set( &t6, &t6 );
	
	// itor
	for ( n = 0, it.find( &t2 ); it.ne( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 ); AssertEqual( *it.key(), t2 ); break;
			case 1: AssertEqual( n, *it.get(), t3 ); AssertEqual( *it.key(), t3 ); break;
			case 2: AssertEqual( n, *it.get(), t4 ); AssertEqual( *it.key(), t4 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_ge( &t3 ); it.le( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); AssertEqual( *it.key(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t4 ); AssertEqual( *it.key(), t4 ); break;
			case 2: AssertEqual( n, *it.get(), t5 ); AssertEqual( *it.key(), t5 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, it.find_gt( &t1 ); it.lt( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t2 ); AssertEqual( *it.key(), t2 ); break;
			case 1: AssertEqual( n, *it.get(), t3 ); AssertEqual( *it.key(), t3 ); break;
			case 2: AssertEqual( n, *it.get(), t4 ); AssertEqual( *it.key(), t4 ); break;
		}
	}
	AssertEqual( n, 3u );

	// ritor
	for ( n = 0, ir.find( &t4 ); ir.ne( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 ); AssertEqual( *ir.key(), t4 ); break;
			case 1: AssertEqual( n, *ir.get(), t3 ); AssertEqual( *ir.key(), t3 ); break;
			case 2: AssertEqual( n, *ir.get(), t2 ); AssertEqual( *ir.key(), t2 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_le( &t3 ); ir.ge( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t3 ); AssertEqual( *ir.key(), t3 ); break;
			case 1: AssertEqual( n, *ir.get(), t2 ); AssertEqual( *ir.key(), t2 ); break;
			case 2: AssertEqual( n, *ir.get(), t1 ); AssertEqual( *ir.key(), t1 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find_lt( &t5 ); ir.gt( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t4 ); AssertEqual( *ir.key(), t4 ); break;
			case 1: AssertEqual( n, *ir.get(), t3 ); AssertEqual( *ir.key(), t3 ); break;
			case 2: AssertEqual( n, *ir.get(), t2 ); AssertEqual( *ir.key(), t2 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( &t2 ); it.ne( &t5 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 ));   break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, 0, &t3, &t7, &t4, &t8, &t5, &t5 ));

	for ( n = 0, ir.find( &t4 ); ir.ne( &t1 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 ));   break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 5, &t1, &t1, &t2, &t2, &t3, &t1, &t4, 0, &t5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.set( &t1, &t1 );
	tr.set( &t2, &t2 );
	tr.set( &t3, &t3 );
	tr.set( &t4, &t4 );
	tr.set( &t5, &t5 );
	for ( n = 0, it.find( &t3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t4 ); break;
			case 2: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 2, &t1, &t1, &t2, &t2 ));

	tr.clear();
	tr.set( &t1, &t1 );
	tr.set( &t2, &t2 );
	tr.set( &t3, &t3 );
	tr.set( &t4, &t4 );
	tr.set( &t5, &t5 );
	for ( n = 0, ir.find( &t3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t3 ); break;
			case 1: AssertEqual( *ir.del(), t2 ); break;
			case 2: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 2, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test41() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - refer/copy policy" );

	typedef ptree<test*> tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( s1, &t1 );
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	tr.put( s3, &t7 );

	for ( n = 0, it.find( s3 ); it.eq( s3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( s3 ); ir.eq( s3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( s3 ); it.eq( s3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, it.set( 0 ),   &t3 ); break;
			case 1: AssertEqual( n, it.set( &t7 ), &t6 ); break;
			case 2: AssertEqual( n, it.set( &t8 ), &t7 ); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s3, 0, s6, &t7, s3, &t8, s5, &t5, s2, &t2 ));

	for ( n = 0, ir.find( s6 ); ir.eq( s6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, ir.set( 0 ),   &t8 ); break;
			case 1: AssertEqual( n, ir.set( &t1 ), &t7 ); break;
			case 2: AssertZero( n,  ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s3, &t2, s6, &t1, s3, 0, s5, &t5, s2, &t2 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( s1, &t1 );
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	for ( n = 0, it.find( s3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( it.del(), &t3 ); break;
			case 1: AssertEqual( it.del(), &t6 ); break;
			case 2: AssertEqual( it.del(), &t5 ); break;
			case 3: AssertEqual( it.del(), &t2 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_testtree( tr, 2, s4, &t4, s1, &t1 ));

	tr.clear();
	tr.put( s1, &t1 );
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	for ( n = 0, ir.find( s3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( ir.del(), &t6 ); break;
			case 1: AssertEqual( ir.del(), &t3 ); break;
			case 2: AssertEqual( ir.del(), &t1 ); break;
			case 3: AssertEqual( ir.del(), &t4 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_testtree( tr, 2, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test42() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - store/copy policy" );

	typedef ptree<test*,store_policy<test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( s1, dup( &t1 ));
	tr.put( s2, dup( &t2 ));
	tr.put( s3, dup( &t3 ));
	tr.put( s4, dup( &t4 ));
	tr.put( s5, dup( &t5 ));
	tr.put( s6, dup( &t6 ));
	tr.put( s3, dup( &t7 ));

	for ( n = 0, it.find( s3 ); it.eq( s3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( s3 ); ir.eq( s3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( s3 ); it.eq( s3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( dup( &t7 ))); break;
			case 2: AssertZero( n, it.set( dup( &t8 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s3, 0, s6, &t7, s3, &t8, s5, &t5, s2, &t2 ));

	for ( n = 0, ir.find( s6 ); ir.eq( s6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( dup( &t1 ))); break;
			case 2: AssertZero( n, ir.set( dup( &t2 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s3, &t2, s6, &t1, s3, 0, s5, &t5, s2, &t2 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( s1, dup( &t1 ));
	tr.put( s2, dup( &t2 ));
	tr.put( s3, dup( &t3 ));
	tr.put( s4, dup( &t4 ));
	tr.put( s5, dup( &t5 ));
	tr.put( s6, dup( &t6 ));
	for ( n = 0, it.find( s3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t6 ); break;
			case 1: AssertEqual( *it.del(), t6 ); break;
			case 2: AssertEqual( *it.del(), t5 ); break;
			case 3: AssertEqual( *it.del(), t2 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_testtree( tr, 2, s4, &t4, s1, &t1 ));

	tr.clear();
	tr.put( s1, dup( &t1 ));
	tr.put( s2, dup( &t2 ));
	tr.put( s3, dup( &t3 ));
	tr.put( s4, dup( &t4 ));
	tr.put( s5, dup( &t5 ));
	tr.put( s6, dup( &t6 ));
	for ( n = 0, ir.find( s3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t6 ); break;
			case 1: AssertEqual( *ir.del(), t3 ); break;
			case 2: AssertEqual( *ir.del(), t1 ); break;
			case 3: AssertEqual( *ir.del(), t4 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_testtree( tr, 2, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test43() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - copy/copy policy" );

	typedef ptree<test*,copy_policy<test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( s1, &t1 );
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	tr.put( s3, &t7 );

	for ( n = 0, it.find( s3 ); it.eq( s3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( s3 ); ir.eq( s3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( s3 ); it.eq( s3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s3, 0, s6, &t7, s3, &t8, s5, &t5, s2, &t2 ));

	for ( n = 0, ir.find( s6 ); ir.eq( s6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_testtree( tr, 7, s4, &t4, s1, &t1, s3, &t2, s6, &t1, s3, 0, s5, &t5, s2, &t2 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( s1, &t1 );
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	for ( n = 0, it.find( s3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t6 ); break;
			case 2: AssertEqual( *it.del(), t5 ); break;
			case 3: AssertEqual( *it.del(), t2 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_testtree( tr, 2, s4, &t4, s1, &t1 ));

	tr.clear();
	tr.put( s1, &t1 );
	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	for ( n = 0, ir.find( s3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t6 ); break;
			case 1: AssertEqual( *ir.del(), t3 ); break;
			case 2: AssertEqual( *ir.del(), t1 ); break;
			case 3: AssertEqual( *ir.del(), t4 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_testtree( tr, 2, s5, &t5, s2, &t2 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test44() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - copy/refer<int> policy" );

	typedef ptree<test*,copy_policy<test*>,int,refer_policy<int> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( i1, &t1 );
	tr.put( i2, &t2 );
	tr.put( i3, &t3 );
	tr.put( i4, &t4 );
	tr.put( i5, &t5 );
	tr.put( i6, &t6 );
	tr.put( i3, &t7 );

	for ( n = 0, it.find( i3 ); it.eq( i3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( i3 ); ir.eq( i3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( i3 ); it.eq( i3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( &t7 )); break;
			case 2: AssertZero( n, it.set( &t8 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_intkytree( tr, 7, i1, &t1, i2, &t2, i3, 0, i6, &t7, i3, &t8, i4, &t4, i5, &t5 ));

	for ( n = 0, ir.find( i6 ); ir.eq( i6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( &t1 )); break;
			case 2: AssertZero( n, ir.set( &t2 )); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_intkytree( tr, 7, i1, &t1, i2, &t2, i3, &t2, i6, &t1, i3, 0, i4, &t4, i5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( i1, &t1 );
	tr.put( i2, &t2 );
	tr.put( i3, &t3 );
	tr.put( i4, &t4 );
	tr.put( i5, &t5 );
	tr.put( i6, &t6 );
	for ( n = 0, it.find( i3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t6 ); break;
			case 2: AssertEqual( *it.del(), t4 ); break;
			case 3: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_intkytree( tr, 2, i1, &t1, i2, &t2 ));

	tr.clear();
	tr.put( i1, &t1 );
	tr.put( i2, &t2 );
	tr.put( i3, &t3 );
	tr.put( i4, &t4 );
	tr.put( i5, &t5 );
	tr.put( i6, &t6 );
	for ( n = 0, ir.find( i3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t6 ); break;
			case 1: AssertEqual( *ir.del(), t3 ); break;
			case 2: AssertEqual( *ir.del(), t2 ); break;
			case 3: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_intkytree( tr, 2, i4, &t4, i5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test45() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - store/refer<obj> policy" );

	typedef ptree<test*,store_policy<test*>,const test*,refer_policy<const test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( &t1, dup( &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	tr.put( &t3, dup( &t7 ));

	for ( n = 0, it.find( &t3 ); it.eq( &t3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( &t3 ); ir.eq( &t3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( &t3 ); it.eq( &t3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( dup( &t7 ))); break;
			case 2: AssertZero( n, it.set( dup( &t8 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t2, &t3, 0, &t6, &t7, &t3, &t8, &t4, &t4, &t5, &t5 ));

	for ( n = 0, ir.find( &t6 ); ir.eq( &t6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( dup( &t1 ))); break;
			case 2: AssertZero( n, ir.set( dup( &t2 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t2, &t3, &t2, &t6, &t1, &t3, 0, &t4, &t4, &t5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( &t1, dup( &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	for ( n = 0, it.find( &t3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t6 ); break;
			case 2: AssertEqual( *it.del(), t4 ); break;
			case 3: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_wtesttree( tr, 2, &t1, &t1, &t2, &t2 ));

	tr.clear();
	tr.put( &t1, dup( &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	for ( n = 0, ir.find( &t3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t6 ); break;
			case 1: AssertEqual( *ir.del(), t3 ); break;
			case 2: AssertEqual( *ir.del(), t2 ); break;
			case 3: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_wtesttree( tr, 2, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test46() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - store/store<obj> policy" );

	typedef ptree<test*,store_policy<test*>,const test*,store_policy<const test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( dup( &t1 ), dup( &t1 ));
	tr.put( dup( &t2 ), dup( &t2 ));
	tr.put( dup( &t3 ), dup( &t3 ));
	tr.put( dup( &t4 ), dup( &t4 ));
	tr.put( dup( &t5 ), dup( &t5 ));
	tr.put( dup( &t6 ), dup( &t6 ));
	tr.put( dup( &t3 ), dup( &t7 ));

	for ( n = 0, it.find( &t3 ); it.eq( &t3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( &t3 ); ir.eq( &t3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( &t3 ); it.eq( &t3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( dup( &t7 ))); break;
			case 2: AssertZero( n, it.set( dup( &t8 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t2, &t3, 0, &t6, &t7, &t3, &t8, &t4, &t4, &t5, &t5 ));

	for ( n = 0, ir.find( &t6 ); ir.eq( &t6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( dup( &t1 ))); break;
			case 2: AssertZero( n, ir.set( dup( &t2 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t2, &t3, &t2, &t6, &t1, &t3, 0, &t4, &t4, &t5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( dup( &t1 ), dup( &t1 ));
	tr.put( dup( &t2 ), dup( &t2 ));
	tr.put( dup( &t3 ), dup( &t3 ));
	tr.put( dup( &t4 ), dup( &t4 ));
	tr.put( dup( &t5 ), dup( &t5 ));
	tr.put( dup( &t6 ), dup( &t6 ));
	for ( n = 0, it.find( &t3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t6 ); break;
			case 2: AssertEqual( *it.del(), t4 ); break;
			case 3: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_wtesttree( tr, 2, &t1, &t1, &t2, &t2 ));

	tr.clear();
	tr.put( dup( &t1 ), dup( &t1 ));
	tr.put( dup( &t2 ), dup( &t2 ));
	tr.put( dup( &t3 ), dup( &t3 ));
	tr.put( dup( &t4 ), dup( &t4 ));
	tr.put( dup( &t5 ), dup( &t5 ));
	tr.put( dup( &t6 ), dup( &t6 ));
	for ( n = 0, ir.find( &t3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t6 ); break;
			case 1: AssertEqual( *ir.del(), t3 ); break;
			case 2: AssertEqual( *ir.del(), t2 ); break;
			case 3: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_wtesttree( tr, 2, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test47() 
{
	this->set_subtitle( "itor/riotr for multi-value ptree - store/copy<obj> policy" );

	typedef ptree<test*,store_policy<test*>,const test*,copy_policy<const test*> > tree;

	tree		tr;
	tree::itor	it( tr );
	tree::ritor	ir( tr );
	unsigned	n;

	tr.put( &t1, dup( &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	tr.put( &t3, dup( &t7 ));

	for ( n = 0, it.find( &t3 ); it.eq( &t3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *it.get(), t3 ); break;
			case 1: AssertEqual( n, *it.get(), t6 ); break;
			case 2: AssertEqual( n, *it.get(), t7 ); break;
		}
	}
	AssertEqual( n, 3u );

	for ( n = 0, ir.find( &t3 ); ir.eq( &t3 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertEqual( n, *ir.get(), t7 ); break;
			case 1: AssertEqual( n, *ir.get(), t6 ); break;
			case 2: AssertEqual( n, *ir.get(), t3 ); break;
		}
	}
	AssertEqual( n, 3u );

	// itor::set(), ritor::set()
	for ( n = 0, it.find( &t3 ); it.eq( &t3 ); ++n, it.next()) {
		switch ( n ) {
			case 0: AssertZero( n, it.set( 0 )); break;
			case 1: AssertZero( n, it.set( dup( &t7 ))); break;
			case 2: AssertZero( n, it.set( dup( &t8 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t2, &t3, 0, &t6, &t7, &t3, &t8, &t4, &t4, &t5, &t5 ));

	for ( n = 0, ir.find( &t6 ); ir.eq( &t6 ); ++n, ir.next()) {
		switch ( n ) {
			case 0: AssertZero( n, ir.set( 0 )); break;
			case 1: AssertZero( n, ir.set( dup( &t1 ))); break;
			case 2: AssertZero( n, ir.set( dup( &t2 ))); break;
		}
	}
	AssertEqual( n, 3u );
	AssertNonZero( check_wtesttree( tr, 7, 
		&t1, &t1, &t2, &t2, &t3, &t2, &t6, &t1, &t3, 0, &t4, &t4, &t5, &t5 ));

	// itor::del(), ritor::del()
	tr.clear();
	tr.put( &t1, dup( &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	for ( n = 0, it.find( &t3 ); it.cont(); ++n, it.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *it.del(), t3 ); break;
			case 1: AssertEqual( *it.del(), t6 ); break;
			case 2: AssertEqual( *it.del(), t4 ); break;
			case 3: AssertEqual( *it.del(), t5 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_wtesttree( tr, 2, &t1, &t1, &t2, &t2 ));

	tr.clear();
	tr.put( &t1, dup( &t1 ));
	tr.put( &t2, dup( &t2 ));
	tr.put( &t3, dup( &t3 ));
	tr.put( &t4, dup( &t4 ));
	tr.put( &t5, dup( &t5 ));
	tr.put( &t6, dup( &t6 ));
	for ( n = 0, ir.find( &t3 ); ir.cont(); ++n, ir.next())
	{
		switch ( n ) {
			case 0: AssertEqual( *ir.del(), t6 ); break;
			case 1: AssertEqual( *ir.del(), t3 ); break;
			case 2: AssertEqual( *ir.del(), t2 ); break;
			case 3: AssertEqual( *ir.del(), t1 ); break;
		}
	}
	AssertEqual( n, 4u );
	AssertNonZero( check_wtesttree( tr, 2, &t4, &t4, &t5, &t5 ));
}

// -----------------------------------------------------------------------------
void ptreetest::test50() 
{
	this->set_subtitle( "balance" );

	typedef ptree<test*, store_policy<test*> > tree;

	tree		tr;
	char		buf[ 16 ];

	tr.balance();
	AssertEqual( tr.length(), 0u );

	snprintf( buf, 16, "%u", 0 );
	test* p = new test( 0, buf );
	tr.put( buf, p );	
	tr.balance();
	AssertEqual( tr.length(), 1u );

	for ( unsigned i = 1; i < 3000; ++i ) {
		snprintf( buf, 16, "%u", i );
		test* p = new test( i, buf );
		tr.put( buf, p );
	}
	tr.balance();

#ifdef _DEBUG
	tr.printtree( print );
#endif
	AssertEqual( tr.length(), 3000u );
}

// -----------------------------------------------------------------------------
void ptreetest::test61() 
{
	this->set_subtitle( "dup - refer/copy policy" );

	typedef ptree<test*> tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 0 ));
	delete p;

	tr.put( s1, &t1 );
	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 1, s1, &t1 ));
	delete p;

	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	tr.put( s7, &t7 );
	tr.put( s8, &t8 );
	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 8, s4, &t4, s1, &t1, s7, &t7, s3, &t3, s6, &t6,
						s5, &t5, s8, &t8, s2, &t2 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::test62() 
{
	this->set_subtitle( "dup - store/copy policy" );

	typedef ptree<test*, store_policy<test*> > tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 0 ));
	delete p;

	tr.put( s1, dup( &t1 ));
	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 1, s1, &t1 ));
	delete p;

	tr.put( s2, dup( &t2 ));
	tr.put( s3, dup( &t3 ));
	tr.put( s4, dup( &t4 ));
	tr.put( s5, dup( &t5 ));
	tr.put( s6, dup( &t6 ));
	tr.put( s7, dup( &t7 ));
	tr.put( s8, dup( &t8 ));
	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 8, s4, &t4, s1, &t1, s7, &t7, s3, &t3, s6, &t6,
						s5, &t5, s8, &t8, s2, &t2 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::test63() 
{
	this->set_subtitle( "dup - copy/copy policy" );

	typedef ptree<test*, copy_policy<test*> > tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 0 ));
	delete p;

	tr.put( s1, &t1 );
	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 1, s1, &t1 ));
	delete p;

	tr.put( s2, &t2 );
	tr.put( s3, &t3 );
	tr.put( s4, &t4 );
	tr.put( s5, &t5 );
	tr.put( s6, &t6 );
	tr.put( s7, &t7 );
	tr.put( s8, &t8 );
	p = dup( &tr );
	AssertNonZero( check_testtree( *p, 8, s4, &t4, s1, &t1, s7, &t7, s3, &t3, s6, &t6,
						s5, &t5, s8, &t8, s2, &t2 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::test64() 
{
	this->set_subtitle( "dup - copy/refer<int> policy" );

	typedef ptree<test*, copy_policy<test*>, int, refer_policy<int> > tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_intkytree( *p, 0 ));
	delete p;

	tr.put( i1, &t1 );
	p = dup( &tr );
	AssertNonZero( check_intkytree( *p, 1, i1, &t1 ));
	delete p;

	tr.put( i2, &t2 );
	tr.put( i3, &t3 );
	tr.put( i4, &t4 );
	tr.put( i5, &t5 );
	tr.put( i6, &t6 );
	tr.put( i7, &t7 );
	tr.put( i8, &t8 );
	p = dup( &tr );
	AssertNonZero( check_intkytree( *p, 8, i1, &t1, i2, &t2, i3, &t3, i6, &t6, i4, &t4,
						i5, &t5, i7, &t7, i8, &t8 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::test65() 
{
	this->set_subtitle( "dup - copy/refer<obj> policy" );

	typedef ptree<test*, copy_policy<test*>, const test*, refer_policy<const test*> > tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 0 ));
	delete p;

	tr.put( &t1, &t1 );
	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 1, &t1, &t1 ));
	delete p;

	tr.put( &t2, &t2 );
	tr.put( &t3, &t3 );
	tr.put( &t4, &t4 );
	tr.put( &t5, &t5 );
	tr.put( &t6, &t6 );
	tr.put( &t7, &t7 );
	tr.put( &t8, &t8 );
	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 8, &t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4,
						&t5, &t5, &t7, &t7, &t8, &t8 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::test66() 
{
	this->set_subtitle( "dup - copy/store<obj> policy" );

	typedef ptree<test*, copy_policy<test*>, const test*, store_policy<const test*> > tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 0 ));
	delete p;

	tr.put( dup( &t1 ), &t1 );
	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 1, &t1, &t1 ));
	delete p;

	tr.put( dup( &t2 ), &t2 );
	tr.put( dup( &t3 ), &t3 );
	tr.put( dup( &t4 ), &t4 );
	tr.put( dup( &t5 ), &t5 );
	tr.put( dup( &t6 ), &t6 );
	tr.put( dup( &t7 ), &t7 );
	tr.put( dup( &t8 ), &t8 );
	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 8, &t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4,
						&t5, &t5, &t7, &t7, &t8, &t8 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::test67() 
{
	this->set_subtitle( "dup - copy/copy<obj> policy" );

	typedef ptree<test*, copy_policy<test*>, const test*, copy_policy<const test*> > tree;

	tree		tr;
	tree*		p;
	unsigned	n;

	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 0 ));
	delete p;

	tr.put( &t1, &t1 );
	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 1, &t1, &t1 ));
	delete p;

	tr.put( &t2, &t2 );
	tr.put( &t3, &t3 );
	tr.put( &t4, &t4 );
	tr.put( &t5, &t5 );
	tr.put( &t6, &t6 );
	tr.put( &t7, &t7 );
	tr.put( &t8, &t8 );
	p = dup( &tr );
	AssertNonZero( check_wtesttree( *p, 8, &t1, &t1, &t2, &t2, &t3, &t3, &t6, &t6, &t4, &t4,
						&t5, &t5, &t7, &t7, &t8, &t8 ));
	delete p;
}

// -----------------------------------------------------------------------------
void ptreetest::run()
{
        test11();
        test12();
        test13();
        test14();
        test15();
        test16();
        test17();

        test21();
        test22();
        test23();
        test24();
        test25();
        test26();
        test27();

	test31();
	test32();
	test33();
	test34();
	test35();
	test36();
	test37();

	test41();
	test42();
	test43();
	test44();
	test45();
	test46();
	test47();

	test50();

	test61();
	test62();
	test63();
	test64();
	test65();
	test66();
	test67();
}

// -----------------------------------------------------------------------------
int main()
{
        ptreetest().run();

        return 0;
}

