//
// plisttest.cpp
//
//		Copyright (C) Kazunari Saitoh, 2003, 2006 all right reserved.
//

#include <cstdio>
#include <cstdarg>
#include <omt/common.h>
#include <omt/plist.h>
#include <omt/utest.h>

using namespace std;
using namespace omt;

// Test Data Class
// -----------------------------------------------------------------------------
class test {
	int	m_int;
	char*	m_str;

	static int cntr;

    public:
	test( int n, const char* b ) : m_int( n ), m_str( strdup( b ))
	{
		++cntr;
#ifdef _DEBUG
                printf( "%03d: ==> alloc test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
	}
	~test()
	{
#ifdef _DEBUG
                printf( "%03d: <== free test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
		--cntr;
		if ( m_str ) delete [] m_str;
	}

	void print() const
	{
		printf( "(%d, \"%s\"), ", m_int, m_str ? m_str : "<null>" );
	}

	const char*	cstr() const { return m_str; }
	int		num()  const { return m_int; }

	static int getCntr() { return cntr; }

    private:
	test( const test& t );
	test& operator=( const test& t );
};

int test::cntr = 0;

// -----------------------------------------------------------------------------
bool operator== ( const test& a, const test& b )
{
        return a.num() == b.num() && eql_fn<const char*>()( a.cstr(), b.cstr());
}

// -----------------------------------------------------------------------------
test* dup( test* p )
{
	return p ? new test( p->num(), p->cstr()) : 0;
}

// Utilities
// -----------------------------------------------------------------------------
template<typename P>
void print( const plist<test*,P>& ls )
{
	typename plist<test*,P>::const_itor	i( ls );

	printf( "---> " );
	for ( i.init(); i.cont(); i.next())
		if ( test* p = i.get()) p->print(); else printf( "null, " );
	printf( " <---" );
}

// -----------------------------------------------------------------------------
template<typename P>
void print( const plist<const char*,P>& ls )
{
	typename plist<const char*,P>::const_itor	i( ls );

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) 
		if ( const char* p = i.get()) printf( "\"%s\", ", p ); else printf( "null, " );
	printf( " <---" );
}

// -----------------------------------------------------------------------------
void print( const plist<int>& ls )
{
	plist<int>::const_itor	i( ls );

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) printf( "%d, ", i.get());
	printf( " <---" );
}

// -----------------------------------------------------------------------------
template<typename P>
bool check_testlist( const plist<test*,P>& ls, size_t n, ... )
{
	va_list					ap;
	typename plist<test*,P>::const_itor	i( ls );
	size_t					j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		test*	p = va_arg( ap, test* );
		test*	q = i.get();

		if ( p && q == 0 || p == 0 && q || p && q && !( *p == *q )) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
template<typename P>
bool check_cstrlist( const plist<const char*,P>& ls, size_t n, ... )
{
	va_list						ap;
	typename plist<const char*,P>::const_itor	i( ls );
	size_t						j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		const char*	p = va_arg( ap, const char* );
		const char*	q = i.get();

		if ( p && q == 0 || p == 0 && q || p && q && strcmp( p, q )) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
bool check_intlist( const plist<int>& ls, size_t n, ... )
{
	va_list			ap;
	plist<int>::const_itor	i( ls );
	size_t			j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		int	p = va_arg( ap, int );
		int	q = i.get();

		if ( p && q == 0 || p == 0 && q || p && q && p != q ) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// Unit Test Class
// -----------------------------------------------------------------------------
class plisttest : public unittest
{
	// test data
	int i1;
	int i2;
	int i3;
	int i4;
	int i5;
	int i6;
	int i7;
	int i8;

	const char* s1;
	const char* s2;
	const char* s3;
	const char* s4;
	const char* s5;
	const char* s6;
	const char* s7;
	const char* s8;

	test t1;
	test t2;
	test t3;
	test t4;
	test t5;
	test t6;
	test t7;
	test t8;

	size_t	m_cnt;

        // test functions
        void test11();
        void test12();
        void test13();

        void test21();
        void test22();
        void test23();

        void test31();
        void test32();
        void test33();
        void test34();
        void test35();
        void test36();
        void test37();

        void test41();
        void test42();
        void test43();

        void test51();
        void test52();
        void test53();
        void test54();
        void test55();
        void test56();
        void test57();

    public:
        plisttest() : unittest( "test for plist.h" ), 

		i1( 1 ),
		i2( 2 ),
		i3( 3 ),
		i4( 4 ),
		i5( 5 ),
		i6( 6 ),
		i7( 7 ),
		i8( 8 ),
	
		s1( "Hello" ),
		s2( "World" ),
		s3( "Open" ),
		s4( "Client" ),
		s5( "Server" ),
		s6( "Open" ),
		s7( "Middleware" ),
		s8( "Web site" ),

		t1( i1, s1 ),
		t2( i2, s2 ),
		t3( i3, s3 ),
		t4( i4, s4 ),
		t5( i5, s5 ),
		t6( i6, s6 ),
		t7( i7, s7 ),
		t8( i8, s8 )
	{
		m_cnt = test::getCntr();
	}
	~plisttest()
	{
		this->set_subtitle( "test class allocation" );
		AssertNonZero( test::getCntr() == m_cnt ); 
	}
        void run();
};

// lsack operation
// -----------------------------------------------------------------------------
void plisttest::test11() 
{
	this->set_subtitle( "object stack operation" );

	plist<test*>	ls;

	ls.push( &t1 );
	ls.push( &t2 );
	ls.push( &t3 );
	ls.push( &t4 );
	ls.push( &t5 );

	AssertEqual( t5, *ls.pop());
	AssertEqual( t4, *ls.pop());
	AssertEqual( t3, *ls.pop());
	AssertEqual( t2, *ls.pop());
	AssertEqual( t1, *ls.pop());
	AssertZero( ls.pop());
}

// -----------------------------------------------------------------------------
void plisttest::test12() 
{
	this->set_subtitle( "c_str stack operation" ); 

	plist<const char*>	ls;

	ls.push( s1 ); 
	ls.push( s2 );
	ls.push( s3 );
	ls.push( s4 );
	ls.push( s5 );

	AssertEqual( s5, ls.pop());
	AssertEqual( s4, ls.pop());
	AssertEqual( s3, ls.pop());
	AssertEqual( s2, ls.pop());
	AssertEqual( s1, ls.pop());
	AssertZero( ls.pop());
}

// -----------------------------------------------------------------------------
void plisttest::test13() 
{
	this->set_subtitle( "int stack operation" );

	plist<int>	ls;

	ls.push( i1 );
	ls.push( i2 );
	ls.push( i3 );
	ls.push( i4 );
	ls.push( i5 );

	AssertEqual( i5, ls.pop());
	AssertEqual( i4, ls.pop());
	AssertEqual( i3, ls.pop());
	AssertEqual( i2, ls.pop());
	AssertEqual( i1, ls.pop());
	AssertZero( ls.pop());
}

// -----------------------------------------------------------------------------
void plisttest::test21()
{
	this->set_subtitle( "object queue operation" );

	plist<test*>	ls;

	ls.enq( &t1 );
	ls.enq( &t2 );
	ls.enq( &t3 );
	ls.enq( &t4 );
	ls.enq( &t5 );

	AssertEqual( t1, *ls.deq());
	AssertEqual( t2, *ls.deq());
	AssertEqual( t3, *ls.deq());
	AssertEqual( t4, *ls.deq());
	AssertEqual( t5, *ls.deq());
	AssertZero( ls.pop());
}

// -----------------------------------------------------------------------------
void plisttest::test22()
{
	this->set_subtitle( "c_str queue operation" );

	plist<const char*>	ls;

	ls.enq( s1 );
	ls.enq( s2 );
	ls.enq( s3 );
	ls.enq( s4 );
	ls.enq( s5 );

	AssertEqual( s1, ls.deq());
	AssertEqual( s2, ls.deq());
	AssertEqual( s3, ls.deq());
	AssertEqual( s4, ls.deq());
	AssertEqual( s5, ls.deq());
	AssertZero( ls.deq());
}
// -----------------------------------------------------------------------------
void plisttest::test23()
{
	this->set_subtitle( "int queue operation" );

	plist<int>	ls;

	ls.enq( i1 );
	ls.enq( i2 );
	ls.enq( i3 );
	ls.enq( i4 );
	ls.enq( i5 );

	AssertEqual( i1, ls.deq());
	AssertEqual( i2, ls.deq());
	AssertEqual( i3, ls.deq());
	AssertEqual( i4, ls.deq());
	AssertEqual( i5, ls.deq());
	AssertZero( ls.deq());
}

// -----------------------------------------------------------------------------
void plisttest::test31()
{
	this->set_subtitle( "object list miscellaneous operations - refer policy" );

	plist<test*>	ls;
	plist<test*>	la;
	plist<test*>	l0;
	plist<test*>	r0;
	plist<test*>	r1;
	plist<test*>	r5;

	r1.enq( &t1 );

	r5.enq( &t5 );
	r5.enq( &t4 );
	r5.enq( &t3 );
	r5.enq( &t2 );
	r5.enq( &t1 );

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( &t1 );
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( &t2 );
	ls.enq( &t3 );
	ls.enq( &t4 );
	ls.enq( &t5 );
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertEqual( t5, *ls.getfirst());
	AssertEqual( t1, *ls.getlast());

	this->set_subtitle( "setfirst()/setlast()" );
	AssertEqual( t5, *ls.setfirst( &t6 ));
	AssertEqual( t6, *ls.getfirst());
	AssertZero( l0.setfirst( &t6 ));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertEqual( t1, *ls.setlast( &t7 ));
	AssertEqual( t7, *ls.getlast());
	AssertZero( l0.setlast( &t7 ));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertEqual( t6, *ls.nth( 0 ));	
	AssertEqual( t3, *ls.nth( 2 ));	
	AssertEqual( t7, *ls.nth( 4 ));	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test32()
{
	this->set_subtitle( "object list miscellaneous operations - store policy" );

	plist<test*, store_policy<test*> >	ls;
	plist<test*, store_policy<test*> >	la;
	plist<test*, store_policy<test*> >	l0;
	plist<test*, store_policy<test*> >	r0;
	plist<test*, store_policy<test*> >	r1;
	plist<test*, store_policy<test*> >	r5;
	test*					p;

	r1.enq( dup( &t1 ));

	r5.enq( dup( &t5 ));
	r5.enq( dup( &t4 ));
	r5.enq( dup( &t3 ));
	r5.enq( dup( &t2 ));
	r5.enq( dup( &t1 ));

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( dup( &t1 ));
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( dup( &t2 ));
	ls.enq( dup( &t3 ));
	ls.enq( dup( &t4 ));
	ls.enq( dup( &t5 ));
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertEqual( t5, *ls.getfirst());
	AssertEqual( t1, *ls.getlast());

	this->set_subtitle( "setfirst()/setlast()" );
	AssertZero( ls.setfirst( dup( &t6 )));
	AssertEqual( t6, *ls.getfirst());
	AssertZero( l0.setfirst( dup( &t6 )));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertZero( ls.setlast( dup( &t7 )));
	AssertEqual( t7, *ls.getlast());
	AssertZero( l0.setlast( dup( &t7 )));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertEqual( t6, *ls.nth( 0 ));	
	AssertEqual( t3, *ls.nth( 2 ));	
	AssertEqual( t7, *ls.nth( 4 ));	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test33()
{
	this->set_subtitle( "object list miscellaneous operations - copy policy" );

	plist<test*, copy_policy<test*> >	ls;
	plist<test*, copy_policy<test*> >	la;
	plist<test*, copy_policy<test*> >	l0;
	plist<test*, copy_policy<test*> >	r0;
	plist<test*, copy_policy<test*> >	r1;
	plist<test*, copy_policy<test*> >	r5;

	r1.enq( &t1 );

	r5.enq( &t5 );
	r5.enq( &t4 );
	r5.enq( &t3 );
	r5.enq( &t2 );
	r5.enq( &t1 );

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( &t1 );
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( &t2 );
	ls.enq( &t3 );
	ls.enq( &t4 );
	ls.enq( &t5 );
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertEqual( t5, *ls.getfirst());
	AssertEqual( t1, *ls.getlast());

	this->set_subtitle( "setfirst()/setlast()" );
	AssertZero( ls.setfirst( &t6 ));
	AssertEqual( t6, *ls.getfirst());
	AssertZero( l0.setfirst( &t6 ));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertZero( ls.setlast( &t7 ));
	AssertEqual( t7, *ls.getlast());
	AssertZero( l0.setlast( &t7 ));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertEqual( t6, *ls.nth( 0 ));	
	AssertEqual( t3, *ls.nth( 2 ));	
	AssertEqual( t7, *ls.nth( 4 ));	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test34()
{
	this->set_subtitle( "c_str list miscellaneous operations - refer policy" );

	plist<const char*>	ls;
	plist<const char*>	la;
	plist<const char*>	l0;
	plist<const char*>	r0;
	plist<const char*>	r1;
	plist<const char*>	r5;

	r1.enq( s1 );

	r5.enq( s5 );
	r5.enq( s4 );
	r5.enq( s3 );
	r5.enq( s2 );
	r5.enq( s1 );

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( s1 );
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( s2 );
	ls.enq( s3 );
	ls.enq( s4 );
	ls.enq( s5 );
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertEqual( s5, ls.getfirst());
	AssertEqual( s1, ls.getlast());

	this->set_subtitle( "setfirst()/setlast()" );
	AssertEqual( s5, ls.setfirst( s6 ));
	AssertEqual( s6, ls.getfirst());
	AssertZero( l0.setfirst( s6 ));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertEqual( s1, ls.setlast( s7 ));
	AssertEqual( s7, ls.getlast());
	AssertZero( l0.setlast( s7 ));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertEqual( s6, ls.nth( 0 ));	
	AssertEqual( s3, ls.nth( 2 ));	
	AssertEqual( s7, ls.nth( 4 ));	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test35()
{
	this->set_subtitle( "c_str list miscellaneous operations - store policy" );

	plist<const char*, store_policy<const char*> > ls;
	plist<const char*, store_policy<const char*> > la;
	plist<const char*, store_policy<const char*> > l0;
	plist<const char*, store_policy<const char*> > r0;
	plist<const char*, store_policy<const char*> > r1;
	plist<const char*, store_policy<const char*> > r5;

	r1.enq( dup( s1 ));

	r5.enq( dup( s5 ));
	r5.enq( dup( s4 ));
	r5.enq( dup( s3 ));
	r5.enq( dup( s2 ));
	r5.enq( dup( s1 ));

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( dup( s1 ));
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( dup( s2 ));
	ls.enq( dup( s3 ));
	ls.enq( dup( s4 ));
	ls.enq( dup( s5 ));
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertZero( strcmp( s5, ls.getfirst()) );
	AssertZero( strcmp( s1, ls.getlast()) );

	this->set_subtitle( "setfirst()/setlast()" );
	AssertZero( ls.setfirst( dup( s6 )) );
	AssertZero( strcmp( s6, ls.getfirst()) );
	AssertZero( l0.setfirst( dup( s6 )));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertZero( ls.setlast( dup( s7 )) );
	AssertZero( strcmp( s7, ls.getlast()) );
	AssertZero( l0.setlast( dup( s7 )));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertZero( strcmp( s6, ls.nth( 0 )) );	
	AssertZero( strcmp( s3, ls.nth( 2 )) );	
	AssertZero( strcmp( s7, ls.nth( 4 )) );	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test36()
{
	this->set_subtitle( "c_str list miscellaneous operations - copy policy" );

	plist<const char*, copy_policy<const char*> >	ls;
	plist<const char*, copy_policy<const char*> >	la;
	plist<const char*, copy_policy<const char*> >	l0;
	plist<const char*, copy_policy<const char*> >	r0;
	plist<const char*, copy_policy<const char*> >	r1;
	plist<const char*, copy_policy<const char*> >	r5;

	r1.enq( s1 );

	r5.enq( s5 );
	r5.enq( s4 );
	r5.enq( s3 );
	r5.enq( s2 );
	r5.enq( s1 );

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( s1 );
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( s2 );
	ls.enq( s3 );
	ls.enq( s4 );
	ls.enq( s5 );
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertZero( strcmp( s5, ls.getfirst()) );
	AssertZero( strcmp( s1, ls.getlast()) );

	this->set_subtitle( "setfirst()/setlast()" );
	AssertZero( ls.setfirst( s6 ));
	AssertZero( strcmp( s6, ls.getfirst()) );
	AssertZero( l0.setfirst( s6 ));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertZero( ls.setlast( s7 ));
	AssertZero( strcmp( s7, ls.getlast()) );
	AssertZero( l0.setlast( s7 ));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertZero( strcmp( s6, ls.nth( 0 )) );	
	AssertZero( strcmp( s3, ls.nth( 2 )) );	
	AssertZero( strcmp( s7, ls.nth( 4 )) );	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test37()
{
	this->set_subtitle( "int list miscellaneous operations" );

	plist<int>	ls;
	plist<int>	la;
	plist<int>	l0;
	plist<int>	r0;
	plist<int>	r1;
	plist<int>	r5;

	r1.enq( i1 );

	r5.enq( i5 );
	r5.enq( i4 );
	r5.enq( i3 );
	r5.enq( i2 );
	r5.enq( i1 );

	this->set_subtitle( "length()" );
	AssertEqual( static_cast<size_t>( 0 ), r0.length());
	AssertEqual( static_cast<size_t>( 1 ), r1.length());
	AssertEqual( static_cast<size_t>( 5 ), r5.length());

	this->set_subtitle( "reverse()" );
	ls.reverse();
	AssertEqual( ls, r0 );

	ls.enq( i1 );
	ls.reverse();
	AssertEqual( ls, r1 );

	ls.enq( i2 );
	ls.enq( i3 );
	ls.enq( i4 );
	ls.enq( i5 );
	ls.reverse();
	AssertEqual( ls, r5 );

	ls.reverse();
	ls.reverse();
	AssertEqual( ls, r5 );

	this->set_subtitle( "getfirst()/getlast()" );
	AssertEqual( i5, ls.getfirst());
	AssertEqual( i1, ls.getlast());

	this->set_subtitle( "setfirst()/setlast()" );
	AssertEqual( i5, ls.setfirst( i6 ));
	AssertEqual( i6, ls.getfirst());
	AssertZero( l0.setfirst( i6 ));
	AssertZero( l0.getfirst());
	l0.clear();
	AssertEqual( i1, ls.setlast( i7 ));
	AssertEqual( i7, ls.getlast());
	AssertZero( l0.setlast( i7 ));
	AssertZero( l0.getlast());
	l0.clear();

	this->set_subtitle( "nth()" );
	AssertEqual( i6, ls.nth( 0 ));	
	AssertEqual( i3, ls.nth( 2 ));	
	AssertEqual( i7, ls.nth( 4 ));	
	AssertZero( ls.nth( 5 ));

	this->set_subtitle( "append()" );
	la.append( r0 );
	AssertEqual( la, r0 );
	la.append( ls );
	AssertEqual( ls, la );
	ls.append( r0 );
	AssertEqual( ls, la );
	ls.append( la );
	AssertEqual( static_cast<size_t>( 10 ), ls.length());
	la.append( la );
	AssertEqual( static_cast<size_t>( 10 ), la.length());
	AssertEqual( ls, la );

	this->set_subtitle( "clear()" );
	ls.clear();
	AssertEqual( ls, r0 );
	ls.clear();
	AssertEqual( ls, r0 );
}

// -----------------------------------------------------------------------------
void plisttest::test41() 
{
	this->set_subtitle( "iterator for object list" );

	typedef plist<test*, store_policy<test*> > pslist;

	pslist		l;
	pslist		r;
	pslist::itor	i( l );
	test*		p;
	size_t		n;

	l.enq( dup( &t1 ));
	l.enq( dup( &t2 ));
	l.enq( dup( &t3 ));
	l.enq( 0 );
	l.enq( dup( &t4 ));
	l.enq( dup( &t5 ));
	l.enq( 0 );
	l.enq( dup( &t6 ));
	l.enq( dup( &t7 ));

	// ----------------------------------------
	this->set_subtitle( "itor::get()" );

	r.append( l );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		test*	a = i.get();
		test*	b = r.nth( n );

		if ( a ) AssertEqual( n, *a, *b );
		else	 AssertZero( n, b );
	}
	AssertZero( i.get());

	// ----------------------------------------
	this->set_subtitle( "itor::set()" );

	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) i.set( dup( &t1 ));
	AssertNonZero( check_testlist( l, 9, &t1, &t2, &t3, &t1, &t4, &t5, &t1, &t6, &t7 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && p->num() > 4 ) i.set( dup( &t3 ));
	AssertNonZero( check_testlist( l, 9, &t1, &t2, &t3, &t1, &t4, &t3, &t1, &t3, &t3 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && p->num() % 4 == 0 ) i.set( 0 );
	AssertNonZero( check_testlist( l, 9, &t1, &t2, &t3, &t1, 0, &t3, &t1, &t3, &t3 ));

	// ----------------------------------------
	this->set_subtitle( "itor::ins()" );

	i.init();
	i.ins( dup( &t5 ));
	AssertNonZero( check_testlist( l, 10, &t5, &t1, &t2, &t3, &t1, 0, &t3, &t1, &t3, &t3 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( !i.get()) i.ins( dup( &t6 ));
	AssertNonZero( check_testlist( l, 11, &t5, &t1, &t2, &t3, &t1, &t6, 0, &t3, &t1, &t3, &t3 ));

	i.ins( dup( &t7 ));
	i.ins( dup( &t8 ));
	AssertNonZero( check_testlist( l, 13, &t5, &t1, &t2, &t3, &t1, &t6, 0, &t3, &t1, &t3, &t3, &t7, &t8 ));

	// ----------------------------------------
	this->set_subtitle( "itor::del()" );

	l.enq( dup( &t4 ));
	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p->cstr(), "Open" ) == 0 ) i.del();
	AssertNonZero( check_testlist( l, 9, &t5, &t1, &t2, &t1, 0, &t1, &t7, &t8, &t4 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p->cstr(), "Open" ) == 0 ) i.del();
	AssertNonZero( check_testlist( l, 9, &t5, &t1, &t2, &t1, 0, &t1, &t7, &t8, &t4 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p->cstr(), "Hello" ) == 0 ) i.del();
	AssertNonZero( check_testlist( l, 6, &t5, &t2, 0, &t7, &t8, &t4 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p->cstr(), "Middleware" ) == 0 ) i.del();
	AssertNonZero( check_testlist( l, 5, &t5, &t2, 0, &t8, &t4 ));

	for ( i.init(); i.cont(); i.next())	// delete all, better to use deq() or pop()
		i.del();
	AssertNonZero( l.is_empty());
	
	l.enq( dup( &t8 ));
	l.push( dup( &t4 ));
	AssertNonZero( check_testlist( l, 2, &t4, &t8 ));
}

// -----------------------------------------------------------------------------
void plisttest::test42() 
{
	this->set_subtitle( "iterator for c_str list" );

	typedef plist<const char*, store_policy<const char*> > pslist;

	pslist		l;
	pslist		r;
	pslist::itor	i( l );
	const char*	p;
	size_t		n;

	l.enq( dup( s1 ));
	l.enq( dup( s2 ));
	l.enq( dup( s3 ));
	l.enq( 0 );
	l.enq( dup( s4 ));
	l.enq( dup( s5 ));
	l.enq( 0 );
	l.enq( dup( s6 ));
	l.enq( dup( s7 ));

	// ----------------------------------------
	this->set_subtitle( "itor::get()" );

	r.append( l );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		const char*	a = i.get();
		const char*	b = r.nth( n );

		if ( a ) AssertEqual( n, *a, *b );
		else	 AssertZero( n, b );
	}
	AssertZero( i.get());

	// ----------------------------------------
	this->set_subtitle( "itor::set()" );

	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) i.set( dup( s1 ));
	AssertNonZero( check_cstrlist( l, 9, s1, s2, s3, s1, s4, s5, s1, s6, s7 ));

	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) 
		if ( n > 6 ) i.set( dup( s3 ));
	AssertNonZero( check_cstrlist( l, 9, s1, s2, s3, s1, s4, s5, s1, s3, s3 ));

	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) 
		if ( n % 4 == 3 ) i.set( 0 );
	AssertNonZero( check_cstrlist( l, 9, s1, s2, s3, 0, s4, s5, s1, 0, s3 ));

	// ----------------------------------------
	this->set_subtitle( "itor::ins()" );

	i.init();
	i.ins( dup( s5 ));
	AssertNonZero( check_cstrlist( l, 10, s5, s1, s2, s3, 0, s4, s5, s1, 0, s3 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( !i.get()) i.ins( dup( s6 ));
	AssertNonZero( check_cstrlist( l, 12, s5, s1, s2, s3, s6, 0, s4, s5, s1, s6, 0, s3 ));

	i.ins( dup( s7 ));
	i.ins( dup( s8 ));
	AssertNonZero( check_cstrlist( l, 14, s5, s1, s2, s3, s6, 0, s4, s5, s1, s6, 0, s3, s7, s8 ));

	// ----------------------------------------
	this->set_subtitle( "itor::del()" );

	l.enq( dup( s4 ));
	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p, "Open" ) == 0 ) i.del();
	AssertNonZero( check_cstrlist( l, 11, s5, s1, s2, 0, s4, s5, s1, 0, s7, s8, s4 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p, "Open" ) == 0 ) i.del();
	AssertNonZero( check_cstrlist( l, 11, s5, s1, s2, 0, s4, s5, s1, 0, s7, s8, s4 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) != 0 && strcmp( p, "Hello" ) == 0 ) i.del();
	AssertNonZero( check_cstrlist( l, 9, s5, s2, 0, s4, s5, 0, s7, s8, s4 ));

	for ( i.init(); i.cont(); i.next()) 
		if (( p = i.get()) == 0 || strcmp( p, "Middleware" ) == 0 ) i.del();
	AssertNonZero( check_cstrlist( l, 6, s5, s2, s4, s5, s8, s4 ));

	for ( i.init(); i.cont(); i.next())	// delete all, better to use deq() or pop()
		i.del();
	AssertNonZero( l.is_empty());
	
	l.enq( dup( s8 ));
	l.push( dup( s4 ));
	AssertNonZero( check_cstrlist( l, 2, s4, s8 ));
}

// -----------------------------------------------------------------------------
void plisttest::test43() 
{
	this->set_subtitle( "iterator for int list" );

	plist<int>		l;
	plist<int>		r;
	plist<int>::itor	i( l );
	int			p;
	size_t			n;

	l.enq( i1 );
	l.enq( i2 );
	l.enq( i3 );
	l.enq( 0 );
	l.enq( i4 );
	l.enq( i5 );
	l.enq( 0 );
	l.enq( i6 );
	l.enq( i7 );

	// ----------------------------------------
	this->set_subtitle( "itor::get()" );

	r.append( l );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		int	a = i.get();
		int	b = r.nth( n );

		if ( a ) AssertEqual( n, a, b );
		else	 AssertZero( n, b );
	}
	AssertZero( i.get());

	// ----------------------------------------
	this->set_subtitle( "itor::set()" );

	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) i.set( i1 );
	AssertNonZero( check_intlist( l, 9, i1, i2, i3, i1, i4, i5, i1, i6, i7 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( i.get() > 4 ) i.set( i3 );
	AssertNonZero( check_intlist( l, 9, i1, i2, i3, i1, i4, i3, i1, i3, i3 ));

	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) 
		if ( n % 4 == 3 ) i.set( 0 );
	AssertNonZero( check_intlist( l, 9, i1, i2, i3, 0, i4, i3, i1, 0, i3 ));

	// ----------------------------------------
	this->set_subtitle( "itor::ins()" );

	i.init();
	i.ins( i5 );
	AssertNonZero( check_intlist( l, 10, i5, i1, i2, i3, 0, i4, i3, i1, 0, i3 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( !i.get()) i.ins( i6 );
	AssertNonZero( check_intlist( l, 12, i5, i1, i2, i3, i6, 0, i4, i3, i1, i6, 0, i3 ));

	i.ins( i7 );
	i.ins( i8 );
	AssertNonZero( check_intlist( l, 14, i5, i1, i2, i3, i6, 0, i4, i3, i1, i6, 0, i3, i7, i8 ));

	// ----------------------------------------
	this->set_subtitle( "itor::del()" );

	l.enq( i4 );
	for ( i.init(); i.cont(); i.next()) 
		if ( i.get() % 3 == 0 ) i.del();
	AssertNonZero( check_intlist( l, 8, i5, i1, i2, i4, i1, i7, i8, i4 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( i.get() % 3 == 0 ) i.del();
	AssertNonZero( check_intlist( l, 8, i5, i1, i2, i4, i1, i7, i8, i4 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( i.get() == 1 ) i.del();
	AssertNonZero( check_intlist( l, 6, i5, i2, i4, i7, i8, i4 ));

	for ( i.init(); i.cont(); i.next()) 
		if ( i.get() == 7 ) i.del();
	AssertNonZero( check_intlist( l, 5, i5, i2, i4, i8, i4 ));

	for ( i.init(); i.cont(); i.next())	// delete all, better to use deq() or pop()
		i.del();
	AssertNonZero( l.is_empty());
	
	l.enq( i8 );
	l.push( i4 );
	AssertNonZero( check_intlist( l, 2, i4, i8 ));
}

// -----------------------------------------------------------------------------
void plisttest::test51() 
{
	this->set_subtitle( "dup() for object list with refer policy" );

	plist<test*>		l;
	plist<test*>*		p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( &t1 );
	p = dup( &l );
	AssertNonZero( check_testlist( *p, 1, &t1 ));
	delete p;

	l.enq( &t2 );
	l.enq( &t3 );
	l.enq( 0 );
	l.enq( &t4 );
	l.enq( &t5 );
	p = dup( &l );
	AssertNonZero( check_testlist( *p, 6, &t1, &t2, &t3, 0, &t4, &t5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::test52() 
{
	this->set_subtitle( "dup() for object list with store policy" );

	typedef plist<test*, store_policy<test*> > pslist;

	pslist	l;
	pslist*	p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( dup( &t1 ));
	p = dup( &l );
	AssertNonZero( check_testlist( *p, 1, &t1 ));
	delete p;

	l.enq( dup( &t2 ));
	l.enq( dup( &t3 ));
	l.enq( 0 );
	l.enq( dup( &t4 ));
	l.enq( dup( &t5 ));
	p = dup( &l );
	AssertNonZero( check_testlist( *p, 6, &t1, &t2, &t3, 0, &t4, &t5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::test53() 
{
	this->set_subtitle( "dup() for object list with copy policy" );

	typedef plist<test*, copy_policy<test*> > pclist;

	pclist	l;
	pclist*	p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( &t1 );
	p = dup( &l );
	AssertNonZero( check_testlist( *p, 1, &t1 ));
	delete p;

	l.enq( &t2 );
	l.enq( &t3 );
	l.enq( 0 );
	l.enq( &t4 );
	l.enq( &t5 );
	p = dup( &l );
	AssertNonZero( check_testlist( *p, 6, &t1, &t2, &t3, 0, &t4, &t5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::test54() 
{
	this->set_subtitle( "dup() for c_str list with refer policy" );

	plist<const char*>	l;
	plist<const char*>*	p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( s1 );
	p = dup( &l );
	AssertNonZero( check_cstrlist( *p, 1, s1 ));
	delete p;

	l.enq( s2 );
	l.enq( s3 );
	l.enq( 0 );
	l.enq( s4 );
	l.enq( s5 );
	p = dup( &l );
	AssertNonZero( check_cstrlist( *p, 6, s1, s2, s3, 0, s4, s5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::test55() 
{
	this->set_subtitle( "dup() for c_str list with store policy" );

	typedef plist<const char*, store_policy<const char*> > pslist;

	pslist	l;
	pslist*	p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( dup( s1 ));
	p = dup( &l );
	AssertNonZero( check_cstrlist( *p, 1, s1 ));
	delete p;

	l.enq( dup( s2 ));
	l.enq( dup( s3 ));
	l.enq( 0 );
	l.enq( dup( s4 ));
	l.enq( dup( s5 ));
	p = dup( &l );
	AssertNonZero( check_cstrlist( *p, 6, s1, s2, s3, 0, s4, s5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::test56() 
{
	this->set_subtitle( "dup() for c_str list with copy policy" );

	typedef plist<const char*, copy_policy<const char*> > pclist;

	pclist	l;
	pclist*	p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( s1 );
	p = dup( &l );
	AssertNonZero( check_cstrlist( *p, 1, s1 ));
	delete p;

	l.enq( s2 );
	l.enq( s3 );
	l.enq( 0 );
	l.enq( s4 );
	l.enq( s5 );
	p = dup( &l );
	AssertNonZero( check_cstrlist( *p, 6, s1, s2, s3, 0, s4, s5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::test57() 
{
	this->set_subtitle( "dup() for int list" );

	plist<int>	l;
	plist<int>*	p;

	p = dup( &l );
	AssertNonZero( p->is_empty());
	delete p;

	l.enq( i1 );
	p = dup( &l );
	AssertNonZero( check_intlist( *p, 1, i1 ));
	delete p;

	l.enq( i2 );
	l.enq( i3 );
	l.enq( 0 );
	l.enq( i4 );
	l.enq( i5 );
	p = dup( &l );
	AssertNonZero( check_intlist( *p, 6, i1, i2, i3, 0, i4, i5 ));
	delete p;
}

// -----------------------------------------------------------------------------
void plisttest::run()
{
	test11();
	test12();
	test13();

	test21();
	test22();
	test23();

	test31();
	test32();
	test33();
	test34();
	test35();
	test36();
	test37();

	test41();
	test42();
	test43();

	test51();
	test52();
	test53();
	test54();
	test55();
	test56();
	test57();
}

// -----------------------------------------------------------------------------
int main()
{
	plisttest().run();

	return 0;
}
