//
// filter.cpp -- common stream filter for Open Middleware
//
//      Copyright (C) 2006 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#include <omt/gstream.h>
#include <omt/filter.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

// Simple Copy
// =============================================================================
void copy_cstr( gistream& in, gostream& out, int trm )
{
        int     a;

        while (( a = in.get()) != trm && a >= 0 ) out.put( a );
};

// C-String style escape/unescape
// -----------------------------------------------------------------------------
void escape_cstr( gistream& in, gostream& out, int trm )
{
        int     a;

        while (( a = in.get()) >= 0 ) {
                if ( 0x20 <= a && a < 0x7F ) {
                        if ( a == '\\' || a == trm ) out.put( '\\' );
                        out.put( a );
                } else {
                        out.put( '\\' );
                        switch ( a ) {
                                case '\a': out.put( 'a' ); break;
                                case '\b': out.put( 'b' ); break;
                                case '\f': out.put( 'f' ); break;
                                case '\n': out.put( 'n' ); break;
                                case '\r': out.put( 'r' ); break;
                                case '\t': out.put( 't' ); break;
                                case '\v': out.put( 'v' ); break;
                                default:
                                        out.put( 'x' );
                                        out.put( hexchar(( a & 0xF0 ) >> 4 ));
                                        out.put( hexchar( a & 0x0F ));
                                        break;
                        }
                }
        }
};

// -----------------------------------------------------------------------------
void unescape_cstr( gistream& in, gostream& out, int trm )
{
        int     a;

        while (( a = in.get()) >= 0 && a != trm ) {
                if ( a == '\\' ) {
                        int     x = 0;

			if (( a = in.get()) == trm || a == '\\' ) {
				out.put( a );
			} else switch ( a ) {
                                case 'a': out.put( '\a' ); break;
                                case 'b': out.put( '\b' ); break;
                                case 'f': out.put( '\f' ); break;
                                case 'n': out.put( '\n' ); break;
                                case 'r': out.put( '\r' ); break;
                                case 't': out.put( '\t' ); break;
                                case 'v': out.put( '\v' ); break;
                                case 'x':
                                        if (( x = gethexa( a = in.get() )) >= 0 ) {
                                                int n = x << 4;
                                                if (( x = gethexa( a = in.get() )) >= 0 )
                                                        out.put( n + x );
                                        }
                                        break;
                                default:
                                        if (( x = getocta( a )) >= 0 ) {
                                                int n = x << 6;
                                                if (( x = getocta( a = in.get() )) >= 0 ) {
                                                        n += x << 3;
                                                        if (( x = getocta( a = in.get())) >= 0 )
                                                                out.put( n + x );
                                                }
                                        }
                                        break;
                        }
#ifdef _DEBUG
                        if ( x < 0 ) {
                                // error handling
                                omt_error( ERR_PARSER, "Illegal escape seq '%c'", a );
                                break;
                        }
#endif
                } else {
                        out.put( a );
                }
        }
}

// Base64 Encode/Decode
// ============================================================================
static const unsigned char B64Pad = '=';
static const unsigned char Bin_2_B64[ 64 ] = {
        'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
        'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
        'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
        'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/'
};

static const signed char B64_2_Bin[ 128 ] = {
        // -1: error, -2: pad, -3: blank
        -1, -1, -1, -1, -1, -1, -1, -1,  -3, -1, -3, -1, -1, -3, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1,  -1, -1, -1, -1, -1, -1, -1, -1,
        -3, -1, -1, -1, -1, -1, -1, -1,  -1, -1, -1, 62, -1, -1, -1, 63,
        52, 53, 54, 55, 56, 57, 58, 59,  60, 61, -1, -1, -1, -2, -1, -1,
        -1,  0,  1,  2,  3,  4,  5,  6,   7,  8,  9, 10, 11, 12, 13, 14,
        15, 16, 17, 18, 19, 20, 21, 22,  23, 24, 25, -1, -1, -1, -1, -1,
        -1, 26, 27, 28, 29, 30, 31, 32,  33, 34, 35, 36, 37, 38, 39, 40,
        41, 42, 43, 44, 45, 46, 47, 48,  49, 50, 51, -1, -1, -1, -1, -1
};

// ----------------------------------------------------------------------------
static int decode64aux( gistream& in, int term )
{
        int     a, r = -1;

        while (( a = in.get()) >= 0 && a != term ) {
                r = B64_2_Bin[ a & 0x7F ];
                if ( r == -1 ) in.unget( a );
                if ( r != -3 ) break;   // not blank
        }
        return r;
}

// ----------------------------------------------------------------------------
void decode64( gistream& in, gostream& out, int term )
{
        while ( !in.is_eof() ) {
                long    l = 1;
                int     n;

                if (( n = decode64aux( in, term )) < 0 ) break;
                l <<= 6; l |= n;
                if (( n = decode64aux( in, term )) < 0 ) break;
                l <<= 6; l |= n;
                if (( n = decode64aux( in, term )) == -1 ) break;
                if ( n >= 0 ) { l <<= 6; l |= n; }
                if (( n = decode64aux( in, term )) == -1 ) break;
                if ( n >= 0 ) { l <<= 6; l |= n; }

                if ( l & 0x1000000 ) {
                        out.put( static_cast<char>(( l >> 16 ) & 0xFF ));
                        out.put( static_cast<char>(( l >>  8 ) & 0xFF ));
                        out.put( static_cast<char>( l & 0xFF ));
                } else if ( l & 0x40000 ) {
                        out.put( static_cast<char>(( l >> 10 ) & 0xFF ));
                        out.put( static_cast<char>(( l >>  2 ) & 0xFF ));
                } else if ( l & 0x1000 ) {
                        out.put( static_cast<char>(( l >>  4 ) & 0xFF ));
                }
        }
}

// ----------------------------------------------------------------------------
void encode64( gistream& in, gostream& out, int term )
{
        while ( 1 ) {
                unsigned long   l = 0;
                int             a;

                if (( a = in.get()) >= 0 && a != term ) {
                        l = ( unsigned char )a;
                        if (( a = in.get()) >= 0 && a != term ) {
                                l <<= 8; l |= ( unsigned char )a;
                                if (( a = in.get()) >= 0 && a != term ) {
                                        l <<= 8; l |= ( unsigned char )a;
                                        out.put( Bin_2_B64[ ( l >> 18 ) & 0x3F ] );
                                        out.put( Bin_2_B64[ ( l >> 12 ) & 0x3F ] );
                                        out.put( Bin_2_B64[ ( l >>  6 ) & 0x3F ] );
                                        out.put( Bin_2_B64[ l & 0x3F ] );
                                } else {
                                        l <<= 2;
                                        out.put( Bin_2_B64[ ( l >> 12 ) & 0x3F ] );
                                        out.put( Bin_2_B64[ ( l >>  6 ) & 0x3F ] );
                                        out.put( Bin_2_B64[ l & 0x3F ] );
                                        out.put( B64Pad );
                                        break;
                                }
                        } else {
                                l <<= 4;
                                out.put( Bin_2_B64[ ( l >>  6 ) & 0x3F ] );
                                out.put( Bin_2_B64[ l & 0x3F ] );
                                out.put( B64Pad );
                                out.put( B64Pad );
                                break;
                        }
                } else {
                        break;
                }
        }
}

#ifndef NO_NAMESPACE
}
#endif

