//
// ptree.h -- pointer based binary search tree for Open Middleware
//
//	version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_PTREE_H
#define __OMT_PTREE_H

#include <omt/common.h>
#include <omt/pcontainer.h>
#include <omt/plist.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

//
// --- Specification ---
//

// -----------------------------------------------------------------------------
template <typename T,
	  typename P = refer_policy<T>,
	  typename K = const char*,
	  typename Q = copy_policy<K>,
          typename C = cmp_fn<K> >
class ptree
{
	class item;
	class itor_base;

	item*		m_top;

    public:
	typedef	T	value_type;
	typedef K	key_type;
	typedef	P	policy_type;
	typedef Q	key_policy_type;
	typedef C	comp_function_type;

	// --- constructor/destructor/assignment operator -----
	ptree() : m_top( 0 ) { }
	~ptree () { clear(); }

	// --- unique value tree operations -------------------
	T set( const K& ky, const T& dt );
	bool replace( const K& ky, const T& dt );
	bool insert( const K& ky, const T& dt );
	T find( const K& ky ) const;
	T remove( const K& ky );

	// --- multi value tree operations -----------------------
	void put( const K& ky, const T& dt );
	
	// --- miscellaneous operations --------------------------
	void balance();

	bool is_empty() const { return m_top ? false : true; }
	size_t length() const { return treesize( m_top ); }

	void clear();
	void swap( ptree<T,P,K,Q,C>& rt );

	// --- omt style iterator --------------------------------
    public:	
	class itor : private itor_base
	{
	    public:
		explicit itor( ptree<T,P,K,Q,C>& rt ) : itor_base( rt ) { }
		itor( ptree<T,P,K,Q,C>& rt, K ky ) : itor_base( rt ) { find( ky ); }
		~itor() { itor_base::next(); }

		void init() { itor_base::init(); }
		void next() { itor_base::next(); }
		bool cont() const { return itor_base::cont(); } 

		bool eq( const K& ky ) const { return itor_base::eq( ky ); }
		bool lt( const K& ky ) const { return itor_base::lt( ky ); }
		bool le( const K& ky ) const { return itor_base::le( ky ); }
		bool ge( const K& ky ) const { return itor_base::ge( ky ); }
		bool gt( const K& ky ) const { return itor_base::gt( ky ); }
		bool ne( const K& ky ) const { return itor_base::ne( ky ); }

		T find( const K& ky ) { return itor_base::find( ky ); }
		T find_ge( const K& ky ) { return itor_base::find_ge( ky ); }
		T find_gt( const K& ky ) { return itor_base::find_gt( ky ); }

		const K key() const { return itor_base::key(); }
		const T get() const { return itor_base::get(); }
		T set( const T& dt ) { return itor_base::set( dt ); }
		T del() { return itor_base::del(); }

		// for ptree only
		typename ptree<T,P,K,Q,C>::item* pitem() { return itor_base::pitem(); }

	    private:
		itor( const itor& );
		itor& operator=( const itor& );

		item*& left( item* p ) { return p->m_left; }
		item*& right( item* p ) { return p->m_right; }
		int cmp( const K& k1, const K& k2 ) { return C()( k1, k2 ); }
	};

	class const_itor : private itor_base
	{
	    public:
		explicit const_itor( const ptree<T,P,K,Q,C>& rt )
			: itor_base( const_cast<ptree<T,P,K,Q,C>&>( rt )) { }
		const_itor( const ptree<T,P,K,Q,C>& rt, K ky )
			: itor_base( const_cast<ptree<T,P,K,Q,C>&>( rt )) { find( ky ); }
		~const_itor() { }

		void init() { itor_base::init(); }
		void next() { itor_base::next(); }
		bool cont() const { return itor_base::cont(); } 

		bool eq( const K& ky ) const { return itor_base::eq( ky ); }
		bool lt( const K& ky ) const { return itor_base::lt( ky ); }
		bool le( const K& ky ) const { return itor_base::le( ky ); }
		bool ge( const K& ky ) const { return itor_base::ge( ky ); }
		bool gt( const K& ky ) const { return itor_base::gt( ky ); }
		bool ne( const K& ky ) const { return itor_base::ne( ky ); }

		const T find( const K& ky ) { return itor_base::find( ky ); }
		const T find_ge( const K& ky ) { return itor_base::find_ge( ky ); }
		const T find_gt( const K& ky ) { return itor_base::find_gt( ky ); }

		const K key() const { return itor_base::key(); }
		const T get() const { return itor_base::get(); }

	    private:
		const_itor( const const_itor& );
		const_itor& operator=( const const_itor& );

		item*& left( item* p ) { return p->m_left; }
		item*& right( item* p ) { return p->m_right; }
		int cmp( const K& k1, const K& k2 ) { return C()( k1, k2 ); }
	};

	class ritor : private itor_base
	{
	    public:
		explicit ritor( ptree<T,P,K,Q,C>& rt ) : itor_base( rt ) { }
		ritor( ptree<T,P,K,Q,C>& rt, K ky ) : itor_base( rt ) { find( ky ); }
		~ritor() { itor_base::next(); }

		void init() { itor_base::init(); }
		void next() { itor_base::next(); }
		bool cont() const { return itor_base::cont(); } 

		bool eq( const K& ky ) const { return itor_base::eq( ky ); }
		bool lt( const K& ky ) const { return itor_base::lt( ky ); }
		bool le( const K& ky ) const { return itor_base::le( ky ); }
		bool ge( const K& ky ) const { return itor_base::ge( ky ); }
		bool gt( const K& ky ) const { return itor_base::gt( ky ); }
		bool ne( const K& ky ) const { return itor_base::ne( ky ); }

		T find( const K& ky ) { return itor_base::find( ky ); }
		T find_le( const K& ky ) { return itor_base::find_le( ky ); }	
		T find_lt( const K& ky ) { return itor_base::find_lt( ky ); }	

		const K key() const { return itor_base::key(); }
		const T get() const { return itor_base::get(); }
		T set( const T& dt ) { return itor_base::set( dt ); }
		T del() { return itor_base::del(); }

	    private:
		ritor( const ritor& );
		ritor& operator=( const ritor& );

		item*& left( item* p ) { return p->m_right; }
		item*& right( item* p ) { return p->m_left; }
		int cmp( const K& k1, const K& k2 ) { return C()( k2, k1 ); }
	};

	class const_ritor : private itor_base
	{
	    public:
		explicit const_ritor( const ptree<T,P,K,Q,C>& rt )
			: itor_base( const_cast<ptree<T,P,K,Q,C>&>( rt )) { }
		const_ritor( const ptree<T,P,K,Q,C>& rt, K ky )
			: itor_base( const_cast<ptree<T,P,K,Q,C>&>( rt )) { find( ky ); }
		~const_ritor() { }

		void init() { itor_base::init(); }
		void next() { itor_base::next(); }
		bool cont() const { return itor_base::cont(); } 

		bool eq( const K& ky ) const { return itor_base::eq( ky ); }
		bool lt( const K& ky ) const { return itor_base::lt( ky ); }
		bool le( const K& ky ) const { return itor_base::le( ky ); }
		bool ge( const K& ky ) const { return itor_base::ge( ky ); }
		bool gt( const K& ky ) const { return itor_base::gt( ky ); }
		bool ne( const K& ky ) const { return itor_base::ne( ky ); }

		const T find( const K& ky ) { return itor_base::find( ky ); }
		const T find_le( const K& ky ) { return itor_base::find_le( ky ); }	
		const T find_lt( const K& ky ) { return itor_base::find_lt( ky ); }	

		const K key() const { return itor_base::key(); }
		const T get() const { return itor_base::get(); }

	    private:
		const_ritor( const const_ritor& );
		const_ritor& operator=( const const_ritor& );

		item*& left( item* p ) { return p->m_right; }
		item*& right( item* p ) { return p->m_left; }
		int cmp( const K& k1, const K& k2 ) { return C()( k2, k1 ); }
	};

#ifdef _DEBUG
	int	m_ckdp;
	int	m_cklr;
	int	m_cklx;
	int	m_ckxr;
	int	m_ckxx;

	void check();
	int check_aux( item* p, int d );

	void printtree( void ( *pf )( T ), item* p = 0, int n = 0 );
#endif

    private:
	ptree( const ptree& );
	ptree& operator=( const ptree& );

	item** find_item( const K& ky, item** h = 0 ) const;
	size_t treesize( item* ) const;
	static item* balance_aux( item* a[], size_t n );

	// ---------------------------------------------------------------------
	struct item
	{
		K	m_key;
		T	m_data;
		item*	m_left;
		item*	m_right;

		item( const K ky, T dt )
			: m_key( Q::set( ky )), m_data( P::set( dt )), m_left( 0 ), m_right( 0 ) { }
		~item()
		{
			P::del( m_data );
			Q::del( m_key );
			if ( m_left  ) delete m_left;
			if ( m_right ) delete m_right;
		}

		item* dup() const
		{
			return new item( Q::copy( m_key ), P::copy( m_data ));
		}
		T setdata( const T& p )
		{
			T	r = P::del( m_data );

			m_data = P::set( p );
			return	r;
		}
	    private:
		item( const item& );
		item& operator= ( const item& );
	};

	// ---------------------------------------------------------------------
	class itor_base
	{
	    protected:
		// if m_hitem is not null, *m_hitem will never be null,
		// except after delete_item().
		item**			m_hitem;
		plist<item**>		m_stack;
		ptree<T,P,K,Q,C>&	m_ptree;
		bool			m_dflag;

		itor_base( ptree<T,P,K,Q,C>& rt ) : m_hitem( 0 ), m_ptree( rt ), m_dflag( false ) { }
		virtual ~itor_base() { }	

		void init();
		void next();
		bool cont() const { return m_hitem ? true : false; }

		bool eq( const K& ky ) const { return m_hitem && C()(( *m_hitem )->m_key, ky ) == 0; } 
		bool lt( const K& ky ) const { return m_hitem && C()(( *m_hitem )->m_key, ky ) <  0; } 
		bool le( const K& ky ) const { return m_hitem && C()(( *m_hitem )->m_key, ky ) <= 0; } 
		bool ge( const K& ky ) const { return m_hitem && C()(( *m_hitem )->m_key, ky ) >= 0; } 
		bool gt( const K& ky ) const { return m_hitem && C()(( *m_hitem )->m_key, ky ) >  0; } 
		bool ne( const K& ky ) const { return m_hitem && C()(( *m_hitem )->m_key, ky ) != 0; } 
		T find( const K& ky );
		T find_ge( const K& ky );	// find first item of the same or greater than key
		T find_gt( const K& ky );	// find first item greater than key
		T find_le( const K& ky );	// find first item of the same or less than key
		T find_lt( const K& ky );	// find first item less than key

		const K key() const { return m_hitem ? ( *m_hitem )->m_key  : K(); }
		const T get() const { return m_hitem ? ( *m_hitem )->m_data : T(); }
		T set( const T& dt ) { return ( m_hitem ) ? ( *m_hitem )->setdata( dt ) : T(); }
		T del() { m_dflag = true; return get(); }

		// for ptree only
		typename ptree<T,P,K,Q,C>::item* pitem() { return m_hitem ? *m_hitem : 0; }
#ifdef _DEBUG
		void check( void ( *pf )( T ));
#endif
	    private:
		itor_base( const itor_base& );
		itor_base& operator=( const itor_base& );

		void find_prep( const K& ky );
		bool delete_item( item** h );

		virtual item*& left( item* p ) = 0;
		virtual item*& right( item* p ) = 0;
		virtual int cmp( const K& k1, const K& k2 ) = 0;
	};
};

// utility templates
// -----------------------------------------------------------------------------
template<>
template <typename T, typename P, typename K, typename Q, typename C>
struct dup_fn<ptree<T,P,K,Q,C>*>
{
        ptree<T,P,K,Q,C>* operator()( const ptree<T,P,K,Q,C>* p )
        {
                ptree<T,P,K,Q,C>*   r = 0;

                if ( p ) {
                        r = new ptree<T,P,K,Q,C>;

                        typename ptree<T,P,K,Q,C>::const_itor    i( *p );
                        for ( i.init(); i.cont(); i.next())
                                r->put( Q::copy( i.key()), P::copy( i.get()) );
                }
		r->balance();
                return r;
        }
};

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
bool operator== ( const ptree<T,P,K,Q,C>& a, const ptree<T,P,K,Q,C>& b )
{
        typename ptree<T,P,K,Q,C>::const_itor       i( a );
        typename ptree<T,P,K,Q,C>::const_itor       j( b );
        eql_fn<T>                                   f_eql;

        for ( i.init(), j.init(); i.cont() && j.cont(); i.next(), j.next())
                if ( !f_eql( i.get(), j.get()) ) break;

        return !( i.cont() || j.cont());
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void swap( ptree<T,P,K,Q,C>& a, ptree<T,P,K,Q,C>& b )
{
        a.swap( b );
}

//
// --- Implementation ---
//

// ptree member functions
// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::find( const K& ky ) const
{
	item**	h = find_item( ky );
	return *h ? ( *h )->m_data : 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::set( const K& ky, const T& dt )
{
	item**	h = find_item( ky );
	T	r = T();

	if ( *h ) {
		r = ( *h )->setdata( dt );
		Q::release( ky );
	} else {
		*h = new item( ky, dt );
	}
	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline bool ptree<T,P,K,Q,C>::replace( const K& ky, const T& dt )
{
	item**	h = find_item( ky );
	bool	r = ( *h ) ? true : false;

	if ( *h )
		( *h )->setdata( dt );
	else
		P::release( dt );

	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline bool ptree<T,P,K,Q,C>::insert( const K& ky, const T& dt )
{
	item**	h = find_item( ky );
	bool	r = *h ? false : true;

	if ( *h ) {
		P::release( dt );
		Q::release( ky );
	} else {
		*h = new item( ky, dt );
	}
	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::remove( const K& ky )
{
	itor	i( *this, ky );
	T	r = T();

	if ( i.cont()) {
		r = i.set( r );
		i.del();
		i.next();
	}
	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::put( const K& ky, const T& dt )
{
	item**	h = &m_top;

	while ( *( h = find_item( ky, h )) != 0 ) h = &( *h )->m_right; 

	*h = new item( ky, dt );
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::clear()
{
	if ( m_top ) delete m_top;
	m_top = 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::swap( ptree<T,P,K,Q,C>& rt )
{
	item**	bk = m_top;
	m_top = rt.m_top;
	rt.m_top = bk;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline typename ptree<T,P,K,Q,C>::item**
	ptree<T,P,K,Q,C>::find_item( const K& ky, typename ptree<T,P,K,Q,C>::item** h ) const
{
	if ( h == 0 ) h = const_cast<item**>( &m_top );

	while ( *h ) {
		int	c = C()( ky, ( *h )->m_key );

		if      ( c > 0 ) h = &( *h )->m_right;
		else if ( c < 0 ) h = &( *h )->m_left;
		else    break;
	}

	return h;	
}
	
// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline size_t ptree<T,P,K,Q,C>::treesize( typename ptree<T,P,K,Q,C>::item* p ) const
{
	return p ? treesize( p->m_left ) + treesize( p->m_right ) + 1 : 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline typename ptree<T,P,K,Q,C>::item*
	ptree<T,P,K,Q,C>::balance_aux( typename ptree<T,P,K,Q,C>::item* a[], size_t n )
{
	item*	r;

	if ( n == 1 ) {
		r = a[ 0 ]; r->m_left = r->m_right = 0;
	} else if ( n == 2 ) {
		r = a[ 1 ]; 
		r->m_left = a[ 0 ]; r->m_left->m_left = r->m_left->m_right = 0;
		r->m_right = 0;
	} else if ( n == 3 ) {		// for performance
		r = a[ 1 ];
		r->m_left  = a[ 0 ]; r->m_left->m_left  = r->m_left->m_right  = 0;
		r->m_right = a[ 2 ]; r->m_right->m_left = r->m_right->m_right = 0;
	} else {
		size_t	m = n / 2;
		r = a[ m ];
		r->m_left  = balance_aux( a, m );
		r->m_right = balance_aux( &a[ m + 1 ], n - m - 1 );
	}
	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::balance()
{
	if ( size_t n = treesize( m_top )) {
		item**	a = new item*[ n ];
		size_t	j;

		typename ptree<T,P,K,Q,C>::itor	i( *this );

		for ( i.init(), j = 0; i.cont(); i.next(), ++j ) a[ j ] = i.pitem();

		m_top = balance_aux( a, n );
		delete [] a;
	}
}

#ifdef _DEBUG
// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline int ptree<T,P,K,Q,C>::check_aux( typename ptree<T,P,K,Q,C>::item* p, int d )
{
	int	r = 1;

	if ( p->m_left && p->m_right ) ++m_cklr;
	else if ( p->m_left )          ++m_cklx;
	else if ( p->m_right )         ++m_ckxr;
	else			       ++m_ckxx;

	if ( m_ckdp < d ) m_ckdp = d;

	if ( p->m_left  ) r += check_aux( p->m_left,  d + 1 );
	if ( p->m_right ) r += check_aux( p->m_right, d + 1 );

	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::check()
{
	m_ckdp = m_cklr = m_cklx = m_ckxr = m_ckxx = 0;	

	int n = check_aux( m_top, 0 );

	printf( "*** size:%d, depth:%d, lr-lx-xr-xx:%d-%d-%d-%d\n",
		n, m_ckdp, m_cklr, m_cklx, m_ckxr, m_ckxx );
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::printtree( void ( *pf )( T ), typename ptree<T,P,K,Q,C>::item* p, int n )
{
	if ( !p ) p = m_top;

	if ( p->m_left ) printtree( pf, p->m_left,  n + 1 );
	for ( int i = 0; i < n; i++ ) printf( "  " );
	( *pf )( p->m_data );
	printf( "\n" );
	if ( p->m_right ) printtree( pf, p->m_right, n + 1 );
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::itor_base::check( void ( *pf )( T )) {
	typename plist<item*>::itor i( m_stack );

	m_ptree.check();
	printf( "*** current: " );
	( *pf )( get());
	printf( "stack: " );
	for ( i.init(); i.cont(); i.next()) if ( *i.get()) ( *pf )(( *i.get())->m_data );
	printf( "\n" );
}
#endif

// ptree::itor member functions
// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::itor_base::init()
{
	if ( m_ptree.m_top ) {
		m_stack.clear();
		for ( m_hitem = &m_ptree.m_top; left( *m_hitem ); m_hitem = &left( *m_hitem )) {
			m_stack.push( m_hitem );
		}
	}
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::itor_base::next()
{
	if ( m_hitem ) {
		if ( !m_dflag || delete_item( m_hitem )) { 
			if ( *m_hitem && right( *m_hitem ) != 0 ) {
				for ( m_hitem = &right( *m_hitem );
				      left( *m_hitem );
				      m_hitem = &left( *m_hitem ))
					m_stack.push( m_hitem );
			} else {
				m_hitem = m_stack.pop();
			}
		}
		m_dflag = false;
	}
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline void ptree<T,P,K,Q,C>::itor_base::find_prep( const K& ky )
{
	m_stack.clear();
	m_hitem = 0;

	typename ptree<T,P,K,Q,C>::item** h = &m_ptree.m_top;

	while ( *h ) {
		int	c = cmp( ky, ( *h )->m_key );

		if ( c > 0 ) {
			h = &right( *h ); 
		} else if ( c < 0 ) {
			m_stack.push( h );
			h = &left( *h ); 
		} else {
			m_hitem = h;
			m_stack.push( h );
			if ( left( *h ) == 0 ) break; 
			h = &left( *h );
		}
	}
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline bool ptree<T,P,K,Q,C>::itor_base::delete_item( typename ptree<T,P,K,Q,C>::item** h )
{
	bool					r = true;
	typename ptree<T,P,K,Q,C>::item**	g;

	if ( left( *h )) {
		for ( g = &left( *h ); right( *g ); g = &right( *g )) ;
		omt::swap(( *h )->m_data, ( *g )->m_data );
		omt::swap(( *h )->m_key,  ( *g )->m_key );
		delete_item( g );
	} else if ( right( *h )) {	
		for ( g = &right( *h ); left( *g ); g = &left( *g )) ;
		omt::swap(( *h )->m_data, ( *g )->m_data );
		omt::swap(( *h )->m_key,  ( *g )->m_key );
		delete_item( g );
		r = false;
	} else {
		delete *h;
		*h = 0;
	}
	return r;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::itor_base::find( const K& ky ) 
{
	find_prep( ky );

	if ( m_hitem ) while ( m_stack.pop() != m_hitem ) ; else m_stack.clear();
	return m_hitem ? ( *m_hitem )->m_data : 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::itor_base::find_ge( const K& ky )
{
	find_prep( ky );

	if ( m_hitem ) while ( m_stack.pop() != m_hitem ) ; else m_hitem = m_stack.pop();
	return m_hitem ? ( *m_hitem )->m_data : 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::itor_base::find_gt( const K& ky )
{
	find_prep( ky );

	if ( m_hitem ) {
		while ( m_stack.pop() != m_hitem ) ;
		next();
	} else {
		m_hitem = m_stack.pop();
	}
	return m_hitem ? ( *m_hitem )->m_data : 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::itor_base::find_le( const K& ky )
{
	find_prep( ky );

	if ( m_hitem ) while ( m_stack.pop() != m_hitem ) ; else m_hitem = m_stack.pop();
	return m_hitem ? ( *m_hitem )->m_data : 0;
}

// -----------------------------------------------------------------------------
template <typename T, typename P, typename K, typename Q, typename C>
inline T ptree<T,P,K,Q,C>::itor_base::find_lt( const K& ky )
{
	find_prep( ky );

	if ( m_hitem ) {
		while ( m_stack.pop() != m_hitem ) ;
		next();
	} else {
		m_hitem = m_stack.pop();
	}
	return m_hitem ? ( *m_hitem )->m_data : 0;
}

#ifndef NO_NAMESPACE
}
#endif

#endif // __OMT_PTREE_H

