//
// plist.h -- pointer based single linked list for Open Middleware
//
//	version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_PLIST_H
#define __OMT_PLIST_H

#include <omt/common.h>
#include <omt/pcontainer.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

//
// Specification
//
// -----------------------------------------------------------------------------
template <typename P, typename Policy = refer_policy<P> > class plist
{
	struct item;

	item*	m_top;
	item**	m_end;

    public:
	typedef P	value_type;
	typedef Policy	policy_type;

	// constructor/destructor ----------------------------------------------
	plist() : m_top( 0 ), m_end( &m_top ) { }
	~plist() { clear(); }

	// stack operation -----------------------------------------------------
	void push( const P& dt );
	P pop();

	// queue operation -----------------------------------------------------
	void enq( const P& dt );
	P deq() { return pop(); }

	// miscellaneous operation ---------------------------------------------
	bool is_empty() const { return m_top ? false : true; }

	P getfirst() const { return m_top ? m_top->m_data : 0; }
	P getlast() const { return m_top ? pnext2item( m_end )->m_data : 0; }
	P setfirst( const P& p ) { return m_top ? m_top->setdata( p ) : Policy::release( p ); }
	P setlast( const P& p ) { return m_top ? pnext2item( m_end )->setdata( p ) : Policy::release( p ); }

	P nth( size_t n ) const;
	size_t length() const;

	void clear();
	plist& reverse();
	void swap( plist& l );
	template <typename Q> void append( const plist<P,Q>& r );
	template <typename Q> void append( const plist<P,Q>* p ) { if ( p ) append( *p ); }

	// iterator ------------------------------
	class itor
	{
		plist<P,Policy>&			m_plst;
		typename plist<P,Policy>::item**	m_hpos;
		bool					m_dflg;

	    public:
		itor( plist<P,Policy>& r ) : m_hpos( &r.m_top ), m_plst( r ), m_dflg( false ) { }
		~itor() { next(); } 			// for del() without calling next()

		void init() { m_hpos = &m_plst.m_top; }
		void next();
		bool cont() const { return *m_hpos ? true : false; }

		const P get() const  { return *m_hpos ? ( *m_hpos )->m_data : 0; }
		P set( const P& dt ) { return *m_hpos ? ( *m_hpos )->setdata( dt ) : 0; }
		void ins( const P& dt );		// insert before
		P del()		     { m_dflg = true; return get(); }

	    private:
		itor( const itor& );
		itor& operator=( const itor& );
	};
	class const_itor
	{
		const plist<P,Policy>&				m_plst;
		const typename plist<P,Policy>::item*const*	m_hpos;

	    public:
		const_itor( const plist<P,Policy>& r ) : m_hpos( &r.m_top ), m_plst( r ) { }
		~const_itor() { }

		void init() { m_hpos = &m_plst.m_top; }
		void next() { if ( *m_hpos ) m_hpos = &( *m_hpos )->m_next; }
		bool cont() const { return *m_hpos ? true : false; }

		const P get() const { return *m_hpos ? ( *m_hpos )->m_data : 0; }

	    private:
		const_itor( const itor& );
		const_itor& operator=( const itor& );
	};

    private:
	plist( const plist& );
	plist& operator=( const plist& );

	struct item
	{
		P	m_data;
		item*	m_next;

		item( P dt, item* nx ) : m_data( Policy::set( dt )), m_next( nx ) { } 
		~item() { Policy::del( m_data ); }

		P setdata( const P& p ) 
		{
			P	r = Policy::del( m_data );

			m_data = Policy::set( p );
			return r;
		}
	};

	item* pnext2item( item** nx ) const
	{
		// nx must not be null, a little tricky implementation
		return ( item* )(( char* )nx - (( char* )&m_top->m_next - ( char* )m_top ));
	}
};

// utility templates
// -----------------------------------------------------------------------------
template<>
template<typename P, typename Policy>
struct dup_fn<plist<P,Policy>*>
{
        // default: new with copy constructor
        plist<P,Policy>* operator()( const plist<P,Policy>* p )
	{
		plist<P,Policy>*	r = 0;

		if ( p ) {
			r = new plist<P,Policy>;

			typename plist<P,Policy>::const_itor	i( *p );
			for ( i.init(); i.cont(); i.next()) r->enq( Policy::copy( i.get()) ); 
		}
		return r;
	}
};

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
bool operator== ( const plist<P,Policy>& a, const plist<P,Policy>& b )
{
	typename plist<P,Policy>::const_itor	i( a );
	typename plist<P,Policy>::const_itor	j( b );
	eql_fn<P>				f_eql;

	for ( i.init(), j.init(); i.cont() && j.cont(); i.next(), j.next()) 
		if ( !f_eql( i.get(), j.get()) ) break;

	return !( i.cont() || j.cont());
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void swap( plist<P,Policy>& a, plist<P,Policy>& b )
{
	a.swap( b );
}

//
// Implementation
//
// -----------------------------------------------------------------------------

// plist members
// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void plist<P,Policy>::push( const P& dt )
{
	m_top = new item( dt, m_top );
	if ( !m_top->m_next ) m_end = &( m_top->m_next );
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline P plist<P,Policy>::pop()
{
	P	r = 0;

	if ( m_top ) {
		item* d = m_top;
		r = d->m_data;
		d->m_data = 0;
		m_top = d->m_next;
		if ( !m_top ) m_end = &m_top;
		delete d;
	}
	return r;
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void plist<P,Policy>::enq( const P& dt )
{
	*m_end = new item( dt, 0 );
	if ( !m_top ) m_top = *m_end;
	m_end = &(( *m_end )->m_next );
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline P plist<P,Policy>::nth( size_t n ) const
{
	size_t	i;
	item*	p;

	for ( i = 0, p = m_top; i < n && p; i++, p = p->m_next ) ;

	return p ? p->m_data : 0;
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline size_t plist<P,Policy>::length() const
{
	size_t	r;
	item*	p;

	for ( r = 0, p = m_top; p; r++, p = p->m_next ) ;

	return r;
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void plist<P,Policy>::clear()
{
	item*	p = m_top;

	while ( p ) {
		item* b = p;
		p = p->m_next;
		delete b;
	}
	m_top = 0;
	m_end = &m_top;
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline plist<P,Policy>& plist<P,Policy>::reverse()
{
	if ( item* p = m_top ) {
		m_end = &m_top->m_next;
		m_top = 0;

		while ( p ) { 
			item* n = p->m_next;
			p->m_next = m_top;
			m_top = p;
			p = n;
		}
	}
	return *this;
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void plist<P,Policy>::swap( plist<P,Policy>& pl )
{
	item*	top = m_top;
	m_top = pl.m_top;
	pl.m_top = top;

	item**	end = m_end;
	m_end = pl.m_end;
	pl.m_end = end;
}

// -----------------------------------------------------------------------------
template <typename P, typename Policy>
template <typename Q>
inline void plist<P,Policy>::append( const plist<P,Q>& r )
{
	if ( reinterpret_cast<const plist<P,Policy>*>( &r ) == this ) {
		plist<P> t;
		t.append( r );
		append( t );
	} else {
		typename plist<P,Q>::const_itor	i( r );
		for ( i.init(); i.cont(); i.next()) enq( Policy::copy( i.get()) );
	}
}

// plist::itor members
// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void plist<P,Policy>::itor::next()
{
	if ( *m_hpos ) {
		if ( m_dflg ) {
			typename plist<P,Policy>::item* p = *m_hpos;

			if (( *m_hpos = p->m_next ) == 0 ) m_plst.m_end = m_hpos;
			delete p;
			m_dflg = false;
		} else {
			m_hpos = &( *m_hpos )->m_next;
		}
	}
}

// -----------------------------------------------------------------------------
template<typename P, typename Policy>
inline void plist<P,Policy>::itor::ins( const P& dt )
{
	*m_hpos = new item( dt, *m_hpos );
	m_hpos = &( *m_hpos )->m_next;

	if ( ! *m_hpos ) m_plst.m_end = m_hpos;
}

#ifndef NO_NAMESPACE
}
#endif
#endif // __OMT_PLIST_H

// -----------------------------------------------------------------------------
// Note: 
// If you use multiple itors for the same plist, never use itor::del(), or cause seriouse problem.
//
// Note:
// Using store_ or copy_policy plist and calling itor::del(), deleting item will delay until itor::next().
// This feature make possible following code, even if plist policy is store or copy:
//		for ( i.init(); i.cont(); i.next()) {	
//			if ( some_condition( i.get())) {
//				P p = i.del();
//
//				// ... able to handle data pointed by p ...
//			}
//		}
// If you break after itor::del(), deleting item will deleted when the itor is destroyed.
// Or you can call itor::next() to force deleting item: 
//		P p = i.del();
//		...
//		i.next();	// delete data pointed by p here
//		break;
//
