//
// gstream.h --- generic stream with minimum I/O functions
//
//	Version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_GSTREAM_H
#define __OMT_GSTREAM_H

#include <cstdio>
#include <cstdlib>

#ifndef NO_CPPSTD
#include <string>
#include <iostream>
#endif

#include <omt/xbuf.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

// Generic Stream ( base class )
// -----------------------------------------------------------------------------
class gstream
{
    public:
	gstream() { }
	virtual ~gstream() { }

	virtual operator bool() const = 0;

    private:
	gstream( const gstream& );
	gstream& operator=( const gstream& );
};

// Generic Input Stream
// -----------------------------------------------------------------------------
class gistream : public gstream
{
    public:
	virtual int get() = 0;
	virtual int unget( int a ) = 0;
	virtual int peek() = 0;
	virtual bool is_eof() const = 0;

	operator bool() const { return !is_eof(); }
};

// -----------------------------------------------------------------------------
template <typename T> class gin;

// -----------------------------------------------------------------------------
template <> class gin<FILE*> : public gistream
{
	FILE*   	m_file;
	bool		m_close;

    public:
	explicit gin( FILE* f = stdin ) : m_file( f ), m_close( false ) { }
	~gin() { if ( m_close ) fclose( m_file ); }

	int get() { return getc( m_file ); }
	int unget( int a ) { return ungetc( a, m_file ); }
	int peek() { int r = get(); if ( r >= 0 ) unget( r ); return r; }
	bool is_eof() const { return feof( m_file ) != 0 || ferror( m_file ) != 0; }
};

// -----------------------------------------------------------------------------
template <> class gin<gistream> : public gistream
{
	gistream&	m_in;

    public:
	explicit gin( gistream& in ) : m_in( in ) { }
	~gin() { }

	int get() { return m_in.get(); }
	int unget( int a ) { return m_in.unget( a ); }
	int peek() { return m_in.peek(); }
	bool is_eof() const { return m_in.is_eof(); }
};

// -----------------------------------------------------------------------------
template <> class gin<char*> : public gistream
{
	const char*     m_buf;
	const char*     m_pos;

    public:
	explicit gin( const char* s ) : m_buf( s ? s : "" ), m_pos( m_buf ) { }
	~gin() { }

	int get() { return *m_pos ? *m_pos++ : -1; }
	int unget( int a ) { return ( m_pos != m_buf ) ? *--m_pos : -1; }
	int peek() { return *m_pos ? *m_pos : -1; }
	bool is_eof() const { return *m_pos == '\0'; }
};

// -----------------------------------------------------------------------------
template <> class gin<xstr> : public gistream
{
	const xstr&     m_xstr;

    public:
	explicit gin( const xstr& s ) : m_xstr( s ) { }
	~gin() { }

	int get() { return m_xstr.get(); }
	int unget( int a ) { m_xstr.unget(); return a; }
	int peek() { return m_xstr.peek(); }
	bool is_eof() const { return m_xstr.is_eos(); }
};

#ifndef NO_CPPSTD
// -----------------------------------------------------------------------------
template <> class gin<std::istream> : public gistream
{
	std::istream&   m_in;

    public:
	explicit gin( std::istream& in ) : m_in( in ) { }
	~gin() { }

	int get() { return m_in.get(); }
	int unget( int a ) { m_in.unget(); return m_in.fail() ? -1 : a; }
	int peek() { return m_in.peek(); }
	bool is_eof() const { return !m_in.good(); }
};

// -----------------------------------------------------------------------------
template <> class gin<std::string> : public gistream
{
	const char*     m_buf;
	const char*     m_pos;

    public:
	explicit gin( const std::string& s ) : m_buf( s.c_str() ), m_pos( m_buf ) { }
	~gin() { }

	int get() { return *m_pos ? *m_pos++ : -1; }
	int unget( int a ) { return ( m_pos != m_buf ) ? *--m_pos : -1; }
	int peek() { return *m_pos ? *m_pos : -1; }
	bool is_eof() const { return *m_pos == '\0'; }
};
#endif

// Generic Output Stream
// -----------------------------------------------------------------------------
class gostream : public gstream
{
	// use 'limits.h' for strict implementation
	static const int bufsize = 32;
	char m_buf[ bufsize ];

    public:
	virtual void put( int a ) = 0;
	virtual bool is_err() const = 0;

	virtual void write_str( const char* s ) = 0;
	virtual void write_int( long n )
	{
		snprintf( m_buf, bufsize, "%ld", n );
		write_str( m_buf );
	}
	virtual void write_flt( double n )
	{
		snprintf( m_buf, bufsize, "%g", n );
		write_str( m_buf );
	}
	virtual void write_hex( unsigned long n )
	{
		snprintf( m_buf, bufsize, "%lX", n );
		write_str( m_buf );
	}

	operator bool() const { return !is_err(); }
};

// -----------------------------------------------------------------------------
template <typename T> class gout;

// -----------------------------------------------------------------------------
template <> class gout<FILE*> : public gostream
{
	FILE*	m_out;

    public:
	explicit gout( FILE* f = stdout ) : m_out( f ) { }
	~gout() { }

	void put( int a ) { putc( a, m_out ); }
	bool is_err() const { return ferror( m_out ); }

	void write_str( const char* s ) { if ( s ) fprintf( m_out, "%s", s ); }
	void write_int( long n ) { fprintf( m_out, "%ld", n ); }
	void write_flt( double n ) { fprintf( m_out, "%g", n ); }
	void write_hex( unsigned long n ) { fprintf( m_out, "%lX", n ); }
};

// -----------------------------------------------------------------------------
template <> class gout<gostream> : public gostream
{
	gostream&	m_out;

    public:
	explicit gout( gostream& out ) : m_out( out ) { }
	~gout() { }

	void put( int a ) { m_out.put( a ); }
	bool is_err() const { return m_out.is_err(); }

	void write_str( const char* s ) { m_out.write_str( s ); }
	void write_int( long n ) { m_out.write_int( n ); }
	void write_flt( double n ) { m_out.write_flt( n ); }
	void write_hex( unsigned long n ) { m_out.write_hex( n ); }
};

// -----------------------------------------------------------------------------
template <> class gout<xstr> : public gostream
{
	xstr&		m_out;

    public:
	explicit gout( xstr& s ) : m_out( s ) { }
	~gout() { }

	void put( int a ) { m_out.put( a ); }
	bool is_err() const { return false; }

	void write_str( const char* s ) { m_out.append( s ); }
};

#ifndef NO_CPPSTD
// -----------------------------------------------------------------------------
template <> class gout<std::ostream> : public gostream
{
	std::ostream&	m_out;

    public:
	explicit gout( std::ostream& s ) : m_out( s ) { }
	~gout() { }

	void put( int a ) { m_out.put( a ); }
	bool is_err() const { return m_out.fail(); }

	void write_str( const char* s ) { if ( s ) m_out << s; }
};

// -----------------------------------------------------------------------------
template <> class gout<std::string> : public gostream
{
	std::string&	m_out;

    public:
	explicit gout( std::string& s ) : m_out( s ) { }
	~gout() { }

	void put( int a ) { m_out.push_back( a ); }
	bool is_err() const { return false; }

	void write_str( const char* s ) { m_out.append( s ); }
};
#endif

// Generic I/O Stream
// -----------------------------------------------------------------------------

// will be implemented if needed


#ifndef NO_NAMESPACE
}
#endif

#endif // __OMT_GSTREAM_H

