//
// common.h --- common definitions for Open Web Middleware
//
//	version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_COMMON_H
#define __OMT_COMMON_H

#ifndef NO_CPPSTD
#include <locale>
#endif

#include <cwchar>
#include <cstring>

#ifndef NO_NAMESPACE
namespace omt {
#endif

// Common Constants
// ============================================================================
void* const nullpointer = 0;
const char* const nullcstr = "";

const unsigned MaxCharsInLine = 1024;

// Common Function Objects
// ============================================================================
//
// hash and hash_fn
// ----------------------------------------------------------------------------
template<typename K> struct hash_fn_cstr 
{
	unsigned long operator()( const K* const& key ) const
	{
		unsigned long	r = 0;

		for ( const K* p = key; *p != K(); ++p ) r += ( r << 2 ) + *p;

		return r;
	}
};

struct hash_fn_num 
{
        unsigned long operator()( const unsigned long& key ) const
        {
		unsigned long	r = 0;

		for ( unsigned long x = key; x; x >>= 5 ) r += x + x + x;
		return r;
	}
};

struct hash_fn_pvoid 
{
        unsigned long operator()( const void* const& key ) const
	{
		return hash_fn_num()( reinterpret_cast<unsigned long>( key ));
	}
};

template<typename K> struct hash_fn { /* not implemented */ };
template<> struct hash_fn<char*> : public hash_fn_cstr<char> { };
template<> struct hash_fn<const char*> : public hash_fn_cstr<char> { };
template<> struct hash_fn<wchar_t*> : public hash_fn_cstr<wchar_t> { };
template<> struct hash_fn<const wchar_t*> : public hash_fn_cstr<wchar_t> { };

template<> struct hash_fn<unsigned long> : public hash_fn_num { }; 
template<> struct hash_fn<long> : public hash_fn_num { };
template<> struct hash_fn<unsigned int> : public hash_fn_num { };
template<> struct hash_fn<int> : public hash_fn_num { };
template<> struct hash_fn<unsigned short> : public hash_fn_num { };
template<> struct hash_fn<short> : public hash_fn_num { };

template<> struct hash_fn<void*> : public hash_fn_pvoid { };
template<> struct hash_fn<const void*> : public hash_fn_pvoid { };

// eql and eql_fn
// ----------------------------------------------------------------------------
template<typename K> struct eql_fn_cstr 
{
	bool operator()( const K* const& k1, const K* const& k2 ) const
	{
		if      ( k1 == 0 && k2 == 0 )	return  true;
		else if ( k1 == 0 || k2 == 0 )	return false;

		const K*	a = k1;
		const K*	b = k2;

		for ( ; *a != K() && *b != K() && *a == *b; ++a, ++b ) ;

		if ( *a == K() && *b == K())	return  true;
		else 				return false;
	}
};

struct eql_fn_pvoid 
{
	bool operator()( const void* const& k1, const void* const& k2 ) const { return k1 == k2; }
};

template<typename K> struct eql_fn 
{
	bool operator()( const K& k1, const K& k2 ) const { return k1 == k2; }
};
template<typename K> struct eql_fn<K*> 
{
	bool operator()( const K* const& k1, const K* const& k2 ) const
	{
		if      ( k1 == 0 && k2 == 0 )	return  true;
		else if ( k1 == 0 || k2 == 0 )	return  false;
		else if ( *k1 == *k2 )		return  true;
		else				return  false;
	}
};
template<> struct eql_fn<char*> : public eql_fn_cstr<char> { };
template<> struct eql_fn<const char*> : public eql_fn_cstr<char> { };
template<> struct eql_fn<wchar_t*> : public eql_fn_cstr<wchar_t> { };
template<> struct eql_fn<const wchar_t*> : public eql_fn_cstr<wchar_t> { };

template<> struct eql_fn<void*> : public eql_fn_pvoid { }; 
template<> struct eql_fn<const void*> : public eql_fn_pvoid { }; 

// cmp and cmp_fn
// ----------------------------------------------------------------------------
template<typename K> struct cmp_fn_cstr 
{
	int operator()( const K* const& k1, const K* const& k2 ) const
	{
		if      ( k1 == 0 && k2 == 0 )	return  0;
		else if ( k1 == 0 )		return -1;
		else if ( k2 == 0 )		return  1;

		const K*	a = k1;
		const K*	b = k2;

		for ( ; *a != K() && *b != K() && *a == *b; ++a, ++b ) ;

		if ( *a == K() && *b == K())	return  0;
		else if ( *a == K())		return -1;
		else if ( *b == K())		return  1;
		else if ( *a <  *b )		return -1;
		else				return  1;
	}
};

struct cmp_fn_pvoid 
{
	int operator()( const void* const& k1, const void* const& k2 ) const
	{
		if	( k1 <  k2 ) return -1;
		else if ( k1 == k2 ) return  0;	
		else		     return  1; 
	}
};

template<typename K> struct cmp_fn 
{
	int operator()( const K& k1, const K& k2 ) const
	{
		if	( k1 <  k2 ) return -1;
		else if ( k1 == k2 ) return  0;	
		else		     return  1; 
	}
};
template<typename K> struct cmp_fn<K*> 
{
	int operator()( const K* const& k1, const K* const& k2 ) const
	{
		if      ( k1 == 0 && k2 == 0 )	return  0;
		else if ( k1 == 0 )		return -1;
		else if ( k2 == 0 )		return  1;
		else if ( *k1 == *k2 )		return  0;
		else if ( *k1 <  *k2 )		return -1;
		else				return  1;
	}
};
template<> struct cmp_fn<char*> : public cmp_fn_cstr<char> { };
template<> struct cmp_fn<const char*> : public cmp_fn_cstr<char> { };
template<> struct cmp_fn<wchar_t*> : public cmp_fn_cstr<wchar_t> { };
template<> struct cmp_fn<const wchar_t*> : public cmp_fn_cstr<wchar_t> { };

template<> struct cmp_fn<void*> : public cmp_fn_pvoid { };
template<> struct cmp_fn<const void*> : public cmp_fn_pvoid { };

// dup_fn
// ----------------------------------------------------------------------------
#ifndef NO_CPPSTD
template<typename K> struct dup_fn_cstr 
{
	K* operator()( const K* const& p )
	{
		K* r = 0;

		if ( p ) {
			size_t n = std::char_traits<K>::length( p );
			r = new K[ n + 1 ];
			std::char_traits<K>::copy( r, p, n );
			r[ n ] = K();	// ends
		}
		return r;
	}
};
#else
template<typename K> struct dup_fn_cstr;
template<> struct dup_fn_cstr<char>
{
	char* operator()( const char* const& p )
	{
		char*	r = 0;
		if ( p ) {
			size_t n = strlen( p );
			r = new char[ n + 1 ];
			strncpy( r, p, n );
			r[ n ] = '\0';
		}
		return r;
	}
};
template<> struct dup_fn_cstr<wchar_t>
{
	wchar_t* operator()( const wchar_t* const& p )
	{
		wchar_t*	r = 0;
		if ( p ) {
			size_t n = wcslen( p );
			r = new wchar_t[ n + 1 ];
			wmemcpy( r, p, n );
			r[ n ] = wchar_t();
		}
		return r;
	}
};
#endif

template<typename K> struct dup_fn { /* not implemented */ };
template<typename K> struct dup_fn<K*> 
{
	// default: new with copy constructor
	K* operator()( const K* const& p ) { return p ? new K( *p ) : 0; }
};	

template<> struct dup_fn<char*>  : public dup_fn_cstr<char> { };
template<> struct dup_fn<const char*>  : public dup_fn_cstr<char> { };
template<> struct dup_fn<wchar_t*>  : public dup_fn_cstr<wchar_t> { };
template<> struct dup_fn<const wchar_t*>  : public dup_fn_cstr<wchar_t> { };

// utility templates
// ============================================================================

// typename deducer
template<typename K> unsigned long hash( const K& key ) { return hash_fn<K>()( key ); }
template<typename K> bool eql( const K& k1, const K& k2 ) { return eql_fn<K>()( k1, k2 ); }
template<typename K> int cmp( const K& k1, const K& k2 ) { return cmp_fn<K>()( k1, k2 ); }
template<typename T> inline T* dup( const T* p ) { return dup_fn<T*>()( p ); }

// swap
template<typename T> inline void swap( T& p, T& q ) { T x = p; p = q; q = x; }

#ifndef NO_NAMESPACE
}
#endif

#endif // __OMT_COMMON_H

