/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: boardlist_ui.c,v 1.19 2003/11/05 11:00:17 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_utils_2ch.h"
#include "worker.h"

#include "ochusha_ui.h"
#include "bbs_thread_ui.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "boardlist_view.h"
#include "bulletin_board_ui.h"
#include "paned_notebook.h"
#include "text_search_window.h"
#include "ugly_gtk2utils.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <errno.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>


#if DEBUG_WIDGET_MOST
static void select_category_cb(BoardlistView *view,
			       OchushaBoardCategory *category,
			       gpointer user_data);
#endif
static void category_expanded_cb(BoardlistView *view,
				 OchushaBoardCategory *category,
				 OchushaApplication *application);
static void category_collapsed_cb(BoardlistView *view,
				  OchushaBoardCategory *category,
				  OchushaApplication *application);
static void toggle_hide_category_cb(BoardlistView *view,
				    OchushaBoardCategory *category,
				    OchushaApplication *application);
static void toggle_hide_board_cb(BoardlistView *view,
				 OchushaBulletinBoard *board,
				 OchushaApplication *application);
static void kill_category_cb(BoardlistView *view,
			     OchushaBoardCategory *category,
			     OchushaApplication *application);
static void kill_board_cb(BoardlistView *view, OchushaBulletinBoard *board,
			  OchushaApplication *application);

static void boardlist_read_category_element_cb(OchushaBBSTable *table,
					       OchushaBoardCategory *category,
					       GHashTable *cat_attributes,
					       OchushaApplication *appl);
static void boardlist_read_board_element_cb(OchushaBBSTable *table,
					    OchushaBulletinBoard *board,
					    GHashTable *cat_attributes,
					    OchushaApplication *appl);

static void boardlist_write_category_element_cb(OchushaBBSTable *table,
						OchushaBoardCategory *category,
						FILE *boardlist_xml,
						OchushaApplication *appl);
static void boardlist_write_board_element_cb(OchushaBBSTable *table,
					     OchushaBulletinBoard *board,
					     FILE *boardlist_xml,
					     OchushaApplication *appl);

static void write_threadlist(gpointer data, gpointer user_data);

static void build_boardlist(WorkerThread *employee, gpointer args);

static void setup_boardlist(OchushaApplication *application);

static void filter_toggled_cb(GtkToggleButton *toggle_button,
			      OchushaApplication *application);
static void item_view_required_cb(PanedNotebook *paned_notebook,
				  gpointer item, GtkWidget **item_view,
				  const gchar **title, GtkWidget **tab_label,
				  OchushaApplication *application);
static void item_view_being_closed_cb(PanedNotebook *paned_notebook,
				      OchushaBulletinBoard *board,
				      GtkWidget *item_view,
				      OchushaApplication *application);
static void item_view_closed_cb(PanedNotebook *paned_notebook,
				OchushaBulletinBoard *board, GtkWidget *item_view,
				OchushaApplication *application);
static void page_switched_cb(PanedNotebook *paned_notebook,
			     GtkWidget *previous_page, GtkWidget *new_page,
			     OchushaApplication *application);

#if 0
static void make_new_thread_button_cb(GtkWidget *widget,
				      OchushaApplication *application);
#endif
static void open_board_properties_button_cb(GtkWidget *widget,
					    OchushaApplication *application);
static void refresh_board_button_cb(GtkWidget *widget,
				    OchushaApplication *application);
static void start_threadlist_search_button_cb(GtkWidget *widget,
					      OchushaApplication *application);



/* Υå˰٤ǤⳫ(OchushaBulletinBoard *)Υꥹȡ*/
static GSList *board_list;

static pthread_mutex_t board_list_lock;


#define BOARD_LIST_LOCK					\
  if (pthread_mutex_lock(&board_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define BOARD_LIST_UNLOCK				\
  if (pthread_mutex_unlock(&board_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


typedef struct _BoardCategoryGUIInfo
{
  gboolean expanded;
} BoardCategoryGUIInfo;


static GQuark category_gui_info_id;


void
prepare_boardlist_ui_initialization(OchushaApplication *application)
{
  if (pthread_mutex_init(&board_list_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  category_gui_info_id
    = g_quark_from_static_string("BoardlistUI::CategoryInfo");

#if HAVE_ONIGURUMA
  reg_set_encoding(REG_ENCODING_UTF8);
#endif
}


void
initialize_boardlist(OchushaApplication *application)
{
  GtkToolbar *toolbar;
  GtkWidget *filter_button;

  board_list = NULL;

  application->boardlist_view = boardlist_view_new();
  g_signal_connect(G_OBJECT(application->boardlist_view), "category_expanded",
		   G_CALLBACK(category_expanded_cb), application);
  g_signal_connect(G_OBJECT(application->boardlist_view), "category_collapsed",
		   G_CALLBACK(category_collapsed_cb), application);
  g_signal_connect(G_OBJECT(application->boardlist_view),
		   "toggle_hide_category",
		   G_CALLBACK(toggle_hide_category_cb), application);
  g_signal_connect(G_OBJECT(application->boardlist_view),
		   "toggle_hide_board",
		   G_CALLBACK(toggle_hide_board_cb), application);
  g_signal_connect(G_OBJECT(application->boardlist_view), "kill_category",
		   G_CALLBACK(kill_category_cb), application);
  g_signal_connect(G_OBJECT(application->boardlist_view), "kill_board",
		   G_CALLBACK(kill_board_cb), application);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(application->boardlist_view), "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &application->boardlist_view);
#endif
  gtk_widget_show(application->boardlist_view);

  application->contents_window
    = paned_notebook_new_with_selector(application->boardlist_pane_style,
				       application->boardlist_view);
  gtk_paned_set_position(GTK_PANED(application->contents_window),
			 application->boardlist_width);
  paned_notebook_set_always_show_contents_pane(PANED_NOTEBOOK(application->contents_window),
					       TRUE);
  paned_notebook_set_tab_shrinkable(PANED_NOTEBOOK(application->contents_window), application->board_tab_shrinkable);
  paned_notebook_set_minimum_tab_label_size(PANED_NOTEBOOK(application->contents_window), application->board_tab_minimum_size);
  paned_notebook_set_tooltips(PANED_NOTEBOOK(application->contents_window),
			      application->board_tab_enable_tooltips);

  toolbar = paned_notebook_get_toolbar(PANED_NOTEBOOK(application->contents_window));

#if 0
  gtk_toolbar_prepend_space(toolbar);
  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_NEW,
			   _("Make a new thread at the current board"),
			   "make_new_thread_at_current_board",
			   GTK_SIGNAL_FUNC(make_new_thread_button_cb),
			   application,
			   0);
#endif

  gtk_toolbar_prepend_space(toolbar);
  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_PROPERTIES,
			   _("Open the current board's properties"),
			   "open_the_current_board_properties",
			   GTK_SIGNAL_FUNC(open_board_properties_button_cb),
			   application,
			   0);
  filter_button = gtk_toggle_button_new_with_label(_("Filter"));
  gtk_widget_show(filter_button);
  application->filter_button = GTK_TOGGLE_BUTTON(filter_button);
  gtk_toolbar_prepend_widget(toolbar, filter_button,
			     _("Apply filter to the current threadlist"),
			     "apply_filter_to_the_current_threadlist");
  g_signal_connect(G_OBJECT(filter_button), "toggled",
		   G_CALLBACK(filter_toggled_cb), application);
#if TRACE_MEMORY_USAGE
  g_signal_connect(G_OBJECT(filter_button), "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &application->filter_button);
#endif

  gtk_toolbar_prepend_space(toolbar);
  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_REFRESH,
			   _("Refresh current threadlist"),
			   "refresh_current_threadlist",
			   GTK_SIGNAL_FUNC(refresh_board_button_cb),
			   application,
			   0);

  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_FIND,
			   _("Start text search on current threadlist"),
			   "start_text_search_on_threadlist",
			   GTK_SIGNAL_FUNC(start_threadlist_search_button_cb),
			   application,
			   0);

#if DEBUG_WIDGET_MOST
  g_signal_connect(G_OBJECT(application->boardlist_view), "select_category",
		   G_CALLBACK(select_category_cb), application);
#endif
  g_signal_connect(G_OBJECT(application->contents_window),
		   "item_view_required", G_CALLBACK(item_view_required_cb),
		   application);
  g_signal_connect(G_OBJECT(application->contents_window),
		   "item_view_being_closed",
		   G_CALLBACK(item_view_being_closed_cb), application);
  g_signal_connect(G_OBJECT(application->contents_window),
		   "item_view_closed", G_CALLBACK(item_view_closed_cb),
		   application);
  g_signal_connect(G_OBJECT(application->contents_window),
		   "page_switched", G_CALLBACK(page_switched_cb),
		   application);

  application->threadlist_search_window = NULL;

  application->table = ochusha_bbs_table_new();

  /* γĥǥ¸ */
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_read_category_element",
		   G_CALLBACK(boardlist_read_category_element_cb),
		   application);
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_read_board_element",
		   G_CALLBACK(boardlist_read_board_element_cb),
		   application);
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_write_category_element",
		   G_CALLBACK(boardlist_write_category_element_cb),
		   application);
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_write_board_element",
		   G_CALLBACK(boardlist_write_board_element_cb),
		   application);

  if (ochusha_bbs_table_read_boardlist_xml(application->table,
					   &application->config))
    {
      setup_boardlist(application);
      return;
    }

  refresh_boardlist(application);
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static BoardCategoryGUIInfo *
ensure_board_category_info(OchushaBoardCategory *category)
{
  BoardCategoryGUIInfo *info = g_object_get_qdata(G_OBJECT(category),
						  category_gui_info_id);
  if (info == NULL)
    {
      info = G_NEW0(BoardCategoryGUIInfo, 1);
      g_object_set_qdata_full(G_OBJECT(category), category_gui_info_id,
			      info, (GDestroyNotify)TRACE_FREE);
    }

  return info;
}


void
refresh_boardlist(OchushaApplication *application)
{
  WorkerJob *job = G_NEW0(WorkerJob, 1);

  job->canceled = FALSE;
  job->job = build_boardlist;
  job->args = application;

  commit_job(job);
}


void
finalize_boardlist(OchushaApplication *application)
{
  if (application->contents_window != NULL)
    {
      PanedNotebook *boardlist = PANED_NOTEBOOK(application->contents_window);
      GtkWidget *current_board;
      application->boardlist_width
	= gtk_paned_get_position(GTK_PANED(boardlist));
      current_board = paned_notebook_get_current_item_view(boardlist);

      if (current_board != NULL)
	application->threadlist_height
	  = gtk_paned_get_position(GTK_PANED(current_board));
    }

  if (application->table != NULL)
    {
      if (!ochusha_bbs_table_write_boardlist_xml(application->table,
						 &application->config))
	fprintf(stderr, "Couldn't write boardlist2.xml.\n");
    }

  g_slist_foreach(board_list, write_threadlist, application);
}


#if 0
void
make_new_thread(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  contents_window = PANED_NOTEBOOK(application->contents_window);

  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  /*
   * TODO: 빽ݤʤΤǺ٤ˤ褦
   */
}
#endif


void
open_current_board_properties(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  contents_window = PANED_NOTEBOOK(application->contents_window);

  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  open_board_properties(application, board);
}


void
refresh_current_threadlist(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  contents_window = PANED_NOTEBOOK(application->contents_window);

  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  refresh_threadlist_view(application, board);
}


static void
refresh_threadlist_helper(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  OchushaApplication *application = (OchushaApplication *)user_data;
  refresh_threadlist_view(application, board);
  sleep(1);
}


void
refresh_current_threadlist_all(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  paned_notebook_foreach_item(contents_window, refresh_threadlist_helper,
			      application);
}


static void
redraw_threadlist_helper(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  OchushaApplication *application = (OchushaApplication *)user_data;
  redraw_threadlist_view(application, board);
}


void
redraw_current_threadlist_all(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  paned_notebook_foreach_item(contents_window, redraw_threadlist_helper,
			      application);
  fix_filter_button_state(application);
}


static ThreadlistSearchQueryData search_query_data =
{
  NULL,		/* key */
  TEXT_SEARCH_DIRECTION_DOWN,
  NULL,		/* last_view */
#if ENABLE_REGEXP
  {},		/* regexp */
  FALSE,	/* regexp_available */
#endif
  FALSE,	/* enable_wrap */
  FALSE,	/* match_case */
  FALSE,	/* use_regexp */
};


static void
text_search_window_response_cb(TextSearchWindow *window, gint response_id,
			       OchushaApplication *application)
{
  switch (response_id)
    {
    case GTK_RESPONSE_DELETE_EVENT:
      application->threadlist_search_window = NULL;
      /* fall through */

    case GTK_RESPONSE_CANCEL:
      /* XXX: ɥξȥϿ٤ */
      if (search_query_data.last_view != NULL)
	threadlist_view_set_being_searched(search_query_data.last_view, FALSE);
      search_query_data.last_view = NULL;
      gtk_widget_hide(GTK_WIDGET(window));
      return;
    }
}


static gboolean
do_search_on_current_threadlist(OchushaApplication *application,
				gboolean include_current_pos)
{
  GtkWidget *widget;
  ThreadlistView *view;

  g_return_val_if_fail(application != NULL
		       && application->contents_window != NULL, FALSE);

  widget = paned_notebook_get_current_item_view(
				PANED_NOTEBOOK(application->contents_window));
  if (widget == NULL)
    return FALSE;
  g_return_val_if_fail(IS_PANED_NOTEBOOK(widget), FALSE);

  widget = paned_notebook_get_selector(PANED_NOTEBOOK(widget));
  if (widget == NULL)
    return FALSE;
  g_return_val_if_fail(IS_THREADLIST_VIEW(widget), FALSE);
  view = THREADLIST_VIEW(widget);
  search_query_data.last_view = view;
  threadlist_view_set_being_searched(view, TRUE);

  return threadlist_view_find_next(view,
				   search_query_data.direction,
				   search_query_data.enable_wrap,
				   include_current_pos,
				   &search_query_data);
}


static gboolean
text_search_window_query_changed_cb(TextSearchWindow *window, const gchar *key,
				    TextSearchDirection direction,
				    gboolean enable_wrap, gboolean match_case,
				    gboolean use_regexp,
				    OchushaApplication *application)
{
  gboolean result;
  gchar *normalized_key;

  if (search_query_data.key != NULL)
    G_FREE(search_query_data.key);

  normalized_key = g_utf8_normalize(key, -1, G_NORMALIZE_ALL);
#if TRACE_MEMORY_USAGE
  {
    gchar *tmp_key = normalized_key;
    normalized_key = G_STRDUP(tmp_key);
    g_free(tmp_key);
  }
#endif
  if (!match_case)
    {
      gchar *case_normalized_key = g_utf8_casefold(normalized_key, -1);
      G_FREE(normalized_key);
#if TRACE_MEMORY_USAGE
      normalized_key = G_STRDUP(case_normalized_key);
      g_free(case_normalized_key);
#else
      normalized_key = case_normalized_key;
#endif
    }
  search_query_data.key = normalized_key;
  search_query_data.direction = direction;
  search_query_data.enable_wrap = enable_wrap;
  search_query_data.match_case = match_case;
#if ENABLE_REGEXP
  search_query_data.use_regexp = use_regexp;
  if (use_regexp)
    {
      if (search_query_data.regexp_available)
	regfree(&search_query_data.regexp);

      if (regcomp(&search_query_data.regexp,
		  normalized_key, REG_EXTENDED | REG_NEWLINE | REG_NOSUB) == 0)
	{
	  search_query_data.regexp_available = TRUE;
	  result = do_search_on_current_threadlist(application, TRUE);
	}
      else
	{
	  regfree(&search_query_data.regexp);
	  search_query_data.regexp_available = FALSE;
	  result = FALSE;
	}
    }
  else
    result = do_search_on_current_threadlist(application, TRUE);
#else
  search_query_data.use_regexp = FALSE;
  result = do_search_on_current_threadlist(application, TRUE);
#endif

  return result;
}


static gboolean
text_search_window_find_next_cb(TextSearchWindow *window, const gchar *key,
				TextSearchDirection direction,
				gboolean enable_wrap, gboolean match_case,
				gboolean use_regexp,
				OchushaApplication *application)
{
  return do_search_on_current_threadlist(application, FALSE);
}


static void
setup_threadlist_search_window(OchushaApplication *application)
{
  if (search_query_data.key != NULL)
    {
      G_FREE(search_query_data.key);
      search_query_data.key = NULL;
    }

  application->threadlist_search_window = text_search_window_new();
  text_search_window_set_enable_incremental_search(
		TEXT_SEARCH_WINDOW(application->threadlist_search_window),
		TRUE);
  g_signal_connect(application->threadlist_search_window, "response",
		   G_CALLBACK(text_search_window_response_cb),
		   application);
  g_signal_connect(application->threadlist_search_window, "query_changed",
		   G_CALLBACK(text_search_window_query_changed_cb),
		   application);
  g_signal_connect(application->threadlist_search_window, "find_next",
		   G_CALLBACK(text_search_window_find_next_cb),
		   application);
#if TRACE_MEMORY_USAGE
  g_signal_connect(application->threadlist_search_window, "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &application->threadlist_search_window);
#endif
  gtk_window_set_title(GTK_WINDOW(application->threadlist_search_window),
		       _("Find in the current threadlist"));
  gtk_window_set_transient_for(
			GTK_WINDOW(application->threadlist_search_window),
			application->top_level);
}


void
advance_threadlist_search(OchushaApplication *application,
			  ThreadlistView *view,
			  ThreadlistSearchAction search_action)
{
  switch (search_action)
    {
    case THREADLIST_SEARCH_ACTION_FORWARD:
    case THREADLIST_SEARCH_ACTION_REGEXP_FORWARD:
      search_query_data.direction = THREADLIST_SEARCH_DIRECTION_FORWARD;
      break;

    case THREADLIST_SEARCH_ACTION_BACKWARD:
    case THREADLIST_SEARCH_ACTION_REGEXP_BACKWARD:
      search_query_data.direction = THREADLIST_SEARCH_DIRECTION_BACKWARD;
      break;

    default:
      /* SHOULD NOT reached here */
      /* THREADLIST_SEARCH_ACTION_TERMINATEcaller¦ǽ */
      break;
    }

  if (application->threadlist_search_window == NULL
      || !GTK_WIDGET_VISIBLE(view)
      || search_query_data.last_view != view)
    {
      if (search_query_data.last_view != NULL)
	threadlist_view_set_being_searched(search_query_data.last_view,
					   FALSE);
      start_current_threadlist_search(application);
      search_query_data.last_view = view;
    }
  else
    do_search_on_current_threadlist(application,
				    search_query_data.last_view != view);
}


void
terminate_threadlist_search(OchushaApplication *application,
			    ThreadlistView *view)
{
  TextSearchWindow *window;
  window = TEXT_SEARCH_WINDOW(application->threadlist_search_window);
  if (window != NULL)
    g_object_ref(G_OBJECT(window));

  if (window != NULL)
    {
      text_search_window_response(window, GTK_RESPONSE_CANCEL);
      g_object_unref(G_OBJECT(window));
    }
}


void
start_current_threadlist_search(OchushaApplication *application)
{
  if (application->threadlist_search_window == NULL)
    setup_threadlist_search_window(application);

  text_search_window_set_key_selected(
		TEXT_SEARCH_WINDOW(application->threadlist_search_window));
  gtk_widget_show(application->threadlist_search_window);
  gtk_widget_grab_focus(application->threadlist_search_window);
}


/*
 * ϤʤΤѰդषOchushaBulletinBoardѲưŪ
 * ˻ΤƼΧŪб٤ʤΤʡ
 */
void
fix_filter_button_state(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;
  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  gtk_toggle_button_set_active(application->filter_button,
		ensure_bulletin_board_info(board, application)->enable_filter);
}


void
update_threadlist_entry(OchushaApplication *application, OchushaBBSThread *thread)
{
  PanedNotebook *board_view;
  ThreadlistView *threadlist_view;
  PanedNotebook *boardlist = (PanedNotebook *)application->contents_window;

  g_return_if_fail(boardlist != NULL && thread != NULL);

  gdk_threads_enter();
  board_view = (PanedNotebook *)paned_notebook_get_item_view(boardlist,
							     thread->board);

  if (board_view == NULL)
    goto finish_update;	/* ĤɽƤʤ⤢ */

  threadlist_view = (ThreadlistView *)paned_notebook_get_selector(board_view);

  if (threadlist_view == NULL)
    {
      goto finish_update;	/* ե륿ߤǥɽƤʤ
				 * ⤢롣
				 */
    }

  update_threadlist_entry_style(application, threadlist_view, thread);

 finish_update:
  gdk_threads_leave();
}


static void
filter_toggled_cb(GtkToggleButton *toggle_button,
		  OchushaApplication *application)
{
  PanedNotebook *board_pane = PANED_NOTEBOOK(application->contents_window);
  OchushaBulletinBoard *board
    = (OchushaBulletinBoard *)paned_notebook_get_current_item(board_pane);
  BulletinBoardGUIInfo *info;

  if (board == NULL)
    return;

  info = ensure_bulletin_board_info(board, application);
  if (info->enable_filter == gtk_toggle_button_get_active(toggle_button))
    return;

  info->enable_filter = !info->enable_filter;

  gtk_toggle_button_set_active(toggle_button, info->enable_filter);
  gdk_threads_leave();
  redraw_threadlist_view(application, board);
  gdk_threads_enter();
}


#if DEBUG_WIDGET_MOST
static void
select_category_cb(BoardlistView *view, OchushaBoardCategory *category,
		   gpointer user_data)
{
  fprintf(stderr, "select_category_cb: %p, %p, %p\n",
	  view, category, user_data);
}
#endif


static void
item_view_required_cb(PanedNotebook *paned_notebook, gpointer item,
		      GtkWidget **item_view, const gchar **title,
		      GtkWidget **tab_label,
		      OchushaApplication *application)
{
  OchushaBulletinBoard *board;
  if (!OCHUSHA_IS_BULLETIN_BOARD(item))
    {
      *tab_label = NULL;
      *item_view = NULL;
      return;
    }

  board = OCHUSHA_BULLETIN_BOARD(item);
  *tab_label = get_tab_label(board->name);
  *item_view = open_bulletin_board(application, board, ICON_LABEL(*tab_label));

  if (*item_view == NULL)
    {
      gtk_object_sink(GTK_OBJECT(*tab_label));
      *tab_label = NULL;
      return;
    }

  BOARD_LIST_LOCK
    {
      if (g_slist_find(board_list, board) == NULL)
	board_list = g_slist_append(board_list, board);
    }
  BOARD_LIST_UNLOCK;

  gtk_widget_show(*item_view);
  *title = board->name;
}


static void
item_view_being_closed_cb(PanedNotebook *paned_notebook, OchushaBulletinBoard *board,
			  GtkWidget *item_view,
			  OchushaApplication *application)
{
  OchushaAsyncBuffer *buffer = snatch_threadlist_source_buffer_for_board(
							board, application);

#ifndef SUSPEND_ALL_FOR_DESTRUCTION	/* ¸ */
  /*
   * MEMO: ɽƤ륦åȤºݤ˼ΤƤ륿ߥ
   *       򤺤餹
   * 
   * GtkTreeViewŪidleؿϿƤꡢ줬Ƥ֤˥
   * åȤȲ줿åȤidleؿưƤޤ
   * ͡ˡʤΤǡǤϥåȤref_count1äƤ
   * Υץ饤ƥidleؿˤäơˡref_count򸺤餹
   * Υץ饤ƥidleؿưȤȤϡGTK+Ū
   * ȤäƤidleؿλŻλƤȤȤˤʤΤǡ
   * פ
   */
  g_object_ref(G_OBJECT(item_view));

  /*
   * MEMO: Ĥ褦ȤƤ륹Ρsubject.txtɤ߹ߡפ
   *       ֥󥰡פԤäƤthreadŪ˽λ롣
   */
  if (buffer != NULL)
    {
      gdk_threads_leave();

      ochusha_async_buffer_suspend(buffer);
      ochusha_async_buffer_terminate(buffer);
      ochusha_async_buffer_resume(buffer);

      gdk_threads_enter();
    }
#else
  /*
   * ΥɽĤ褦ȤƤΤǡϢthreadߤᡢ
   * ¾GUIϢthread򥵥ڥɤίäƤ륤٥Ȥ
   * Ƥ롣ǽʸ¤᤯٥Ȥò뤿ᡢƤthread
   * ڥɤˡ˴Ϣthread϶Ūߤ롣
   *
   * ġĤ褦ˤ顢ब򤱤褦ʵΤġġ
   */

  gdk_threads_leave();

  ochusha_async_buffer_suspend_all();

  if (buffer != NULL)
    ochusha_async_buffer_terminate(buffer);

  gdk_threads_enter();

  while (gtk_events_pending())
    gtk_main_iteration();

  ochusha_async_buffer_resume_all();
#endif

  if (buffer != NULL)
    g_object_unref(G_OBJECT(buffer));
  return;
}


static void
item_view_closed_cb(PanedNotebook *paned_notebook, OchushaBulletinBoard *board,
		    GtkWidget *item_view, OchushaApplication *application)
{
  if (paned_notebook_get_current_item_view(paned_notebook) == NULL)
    {
      /* ɽˤǤƤڡĤ줿 */
      if (item_view != NULL && IS_PANED_NOTEBOOK(item_view))
	{
	  application->threadlist_height
	    = gtk_paned_get_position(GTK_PANED(item_view));
	}
    }

  finalize_bulletin_board_view(board, item_view);
#ifndef SUSPEND_ALL_FOR_DESTRUCTION
  delayed_g_object_unref(G_OBJECT(item_view));
#endif
}


static void
page_switched_cb(PanedNotebook *paned_notebook,
		 GtkWidget *previous_page, GtkWidget *new_page,
		 OchushaApplication *application)
{
  g_return_if_fail((previous_page != NULL && IS_PANED_NOTEBOOK(previous_page))
		   || (new_page != NULL && IS_PANED_NOTEBOOK(new_page)));

  if (previous_page != NULL)
    {
      gpointer threadlist_view;
      application->threadlist_height
	= gtk_paned_get_position(GTK_PANED(previous_page));
      threadlist_view
	= paned_notebook_get_selector(PANED_NOTEBOOK(previous_page));

      if (IS_THREADLIST_VIEW(threadlist_view)
	  && search_query_data.last_view == threadlist_view)
	{
	  threadlist_view_set_being_searched(THREADLIST_VIEW(threadlist_view),
					     FALSE);
	  search_query_data.last_view = NULL;
	}
    }

  if (new_page != NULL)
    {
      OchushaBulletinBoard *board
	= (OchushaBulletinBoard *)paned_notebook_get_item(paned_notebook, new_page);
      BulletinBoardGUIInfo *info;

      g_return_if_fail(board != NULL);
      info = ensure_bulletin_board_info(board, application);
      gtk_paned_set_position(GTK_PANED(new_page),
			     application->threadlist_height);
      gtk_toggle_button_set_active(application->filter_button,
				   info->enable_filter);
    }
}


static void
boardlist_read_category_element_cb(OchushaBBSTable *table,
				   OchushaBoardCategory *category,
				   GHashTable *category_attributes,
				   OchushaApplication *application)
{
  BoardCategoryGUIInfo *info = ensure_board_category_info(category);
  info->expanded = ochusha_utils_get_attribute_boolean(category_attributes,
						       "expanded");
  return;
}


static void
boardlist_read_board_element_cb(OchushaBBSTable *table,
				OchushaBulletinBoard *board,
				GHashTable *board_attributes,
				OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);
  info->enable_filter = ochusha_utils_get_attribute_boolean(board_attributes,
							    "enable_filter");
  info->filter.rule = ochusha_utils_get_attribute_string(board_attributes,
							 "filter.rule");
  info->filter.ignore_threshold
    = ochusha_utils_get_attribute_int(board_attributes,
				      "filter.ignore_threshold");
  info->last_name = ochusha_utils_get_attribute_string(board_attributes,
						       "last_name");
  info->last_mail = ochusha_utils_get_attribute_string(board_attributes,
						       "last_mail");
  return;
}


#define OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(file, category, attribute)	\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <boolean val=\"%s\"/>\n"				\
	  "      </attribute>\n",					\
	  (category)->attribute ? "true" : "false")


static void
boardlist_write_category_element_cb(OchushaBBSTable *table,
				    OchushaBoardCategory *category,
				    FILE *boardlist_xml,
				    OchushaApplication *application)
{
  BoardCategoryGUIInfo *info = ensure_board_category_info(category);

  OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(boardlist_xml, info, expanded);
}


#define OUTPUT_BOARD_ATTRIBUTE_BOOLEAN(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <boolean val=\"%s\"/>\n"				\
	  "        </attribute>\n",					\
	  (board)->attribute ? "true" : "false")

#define OUTPUT_BOARD_ATTRIBUTE_INT(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <int val=\"%d\"/>\n"				\
	  "        </attribute>\n", (board)->attribute)

#define OUTPUT_BOARD_ATTRIBUTE_STRING(file, board, attribute)		\
  if ((board)->attribute != NULL)					\
    fprintf(file,							\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <string>%s</string>\n"				\
	  "        </attribute>\n", (board)->attribute)


static void
boardlist_write_board_element_cb(OchushaBBSTable *table,
				 OchushaBulletinBoard *board,
				 FILE *boardlist_xml,
				 OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board, application);
  OUTPUT_BOARD_ATTRIBUTE_BOOLEAN(boardlist_xml, info, enable_filter);
  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, info, last_name);
  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, info, last_mail);

  if (info->filter.rule != NULL)
    {
      OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, info, filter.rule);
      OUTPUT_BOARD_ATTRIBUTE_INT(boardlist_xml, info, filter.ignore_threshold);
    }

  return;
}


static void
write_threadlist(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  OchushaApplication *application = (OchushaApplication *)user_data;

  /* XXX: 顼 */
  if (!ochusha_bulletin_board_write_threadlist_xml(board, &application->config))
    fprintf(stderr, "Couldn't write threadlist2.xml.\n");
}


static void
category_expanded_cb(BoardlistView *view, OchushaBoardCategory *category,
		     OchushaApplication *application)
{
  BoardCategoryGUIInfo *info = ensure_board_category_info(category);
  info->expanded = TRUE;
}


static void
category_collapsed_cb(BoardlistView *view, OchushaBoardCategory *category,
		      OchushaApplication *application)
{
  BoardCategoryGUIInfo *info = ensure_board_category_info(category);
  info->expanded = FALSE;
}


static void
toggle_hide_category_cb(BoardlistView *view, OchushaBoardCategory *category,
			OchushaApplication *application)
{
  category->hidden = !(category->hidden);
}


static void
toggle_hide_board_cb(BoardlistView *view, OchushaBulletinBoard *board,
		     OchushaApplication *application)
{
  board->hidden = !(board->hidden);
}


static void
kill_category_cb(BoardlistView *view, OchushaBoardCategory *category,
		 OchushaApplication *application)
{
  category->killed = TRUE;
}


static void
kill_board_cb(BoardlistView *view, OchushaBulletinBoard *board,
	      OchushaApplication *application)
{
  board->killed = TRUE;
}


static gboolean
prepare_category(OchushaBoardCategory *category, gpointer user_data)
{
  ensure_board_category_info(category);
  return TRUE;
}


static gboolean
prepare_board(OchushaBulletinBoard *board, gpointer user_data)
{
  ensure_bulletin_board_info(board, (OchushaApplication *)user_data);
  return TRUE;
}


static OchushaBulletinBoard *
process_board_move(OchushaBulletinBoard *old_board,
		   OchushaBulletinBoard *new_board,
		   gpointer user_data)
{
#if 0
  OchushaApplication *application = (OchushaApplication *)user_data;
#endif  
  /*
   * TODO: DATեʤɥå夷ƤǡΰžԤ٤
   *       İžɤߤGUIɲäɬפ⡣
   */
#if DEBUG_GUI
  fprintf(stderr, "process_board_move: old_board=%p new_board=%p\n",
	  old_board, new_board);
#endif
  return new_board;	/* ǤϿΤͥ */
}


void
redraw_boardlist(OchushaApplication *application)
{
  int status;

  if ((status = pthread_mutex_trylock(&board_list_lock)) != 0)
    {
      if (status == EBUSY)
	return;		/* İι1åɤ */

      fprintf(stderr, "Internal error: invalid mutex.\n");
      abort();
    }

  setup_boardlist(application);

  if (pthread_mutex_unlock(&board_list_lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }
}


static void
build_boardlist(WorkerThread *employee, gpointer args)
{
  int status;
  OchushaAsyncBuffer *buffer;

  OchushaApplication *application = (OchushaApplication *)args;
  OchushaBBSTable *table = application->table;

  if ((status = pthread_mutex_trylock(&board_list_lock)) != 0)
    {
      if (status == EBUSY)
	return;		/* İι1åɤ */

      fprintf(stderr, "Internal error: invalid mutex.\n");
      abort();
    }

  buffer = ochusha_utils_2ch_get_bbsmenu(application->broker);

  if (buffer == NULL)
    {
      /* XXX: 顼 */
      fprintf(stderr, "Couldn't get %s\n", application->config.bbsmenu_url);
      goto cleanup_before_return;
      return;
    }

  if (ochusha_utils_2ch_analyze_bbsmenu(table, buffer,
					prepare_category,
					prepare_board,
					process_board_move,
					application))
    {
      gdk_threads_enter();
      setup_boardlist(application);
      gdk_threads_leave();

      ochusha_bbs_table_write_boardlist_xml(table, &application->config);
    }

 cleanup_before_return:
  if (buffer != NULL)
    g_object_unref(G_OBJECT(buffer));

  if (pthread_mutex_unlock(&board_list_lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }
}


static void
restore_expanded_state(OchushaBoardCategory *category, BoardlistView *view)
{
  BoardCategoryGUIInfo *info = ensure_board_category_info(category);
  if (info->expanded)
    boardlist_view_expand_category(view, category);
}


static void
setup_boardlist(OchushaApplication *application)
{
  boardlist_view_open(BOARDLIST_VIEW(application->boardlist_view),
		      application->table->category_list,
		      application->hide_hidden_boardlist_items);
  g_slist_foreach(application->table->category_list,
		  (GFunc)restore_expanded_state, application->boardlist_view);

  gtk_widget_show_all(application->boardlist_view);
  gtk_widget_show_all(application->contents_window);
}


#if 0
static void
make_new_thread_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  make_new_thread(application);
}
#endif


static void
open_board_properties_button_cb(GtkWidget *widget,
				OchushaApplication *application)
{
  open_current_board_properties(application);
}


static void
refresh_board_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  refresh_current_threadlist(application);
}


static void
start_threadlist_search_button_cb(GtkWidget *widget,
				  OchushaApplication *application)
{
  start_current_threadlist_search(application);
}

