/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: paned_notebook.c,v 1.23.2.6 2005/02/18 17:33:16 fuyu Exp $
 */

#define GTK_COMPILATION	1	/* 
				 * GtkPanedinternal functionȤΤ
				 * Τ
				 */

#include "config.h"

#include "ochusha_private.h"

#include "ochusha_ui.h"
#include "paned_notebook.h"

#include "marshal.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>


static void paned_notebook_class_init(PanedNotebookClass *klass);
static void paned_notebook_set_property(GObject *object, guint prop_id,
					const GValue *value,
					GParamSpec *pspec);
static void paned_notebook_get_property(GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec);
static void item_view_required_cb(PanedNotebook *paned_notebook, gpointer item,
				  GtkWidget **item_view,
				  const gchar **item_title,
				  GtkWidget **tab_label);
static void paned_notebook_init(PanedNotebook *paned_notebook);
static void paned_notebook_finalize(GObject *object);
static void paned_notebook_destroy(GtkObject *object);
static void paned_notebook_size_request(GtkWidget *widget,
					GtkRequisition *requisition);
static void paned_notebook_size_allocate (GtkWidget *widget,
					  GtkAllocation *allocation);
typedef struct _ItemViewRelation ItemViewRelation;
#if GTK_MINOR_VERSION <= 2
static ItemViewRelation *item_view_relation_new(gpointer item,
						GtkWidget *item_view,
						GtkWidget *menu_item,
						GtkWidget *tab_label);
static int compare_items(gconstpointer list_data, gconstpointer user_data);
static ItemViewRelation *find_item_view_relation(PanedNotebook *paned_notebook,
						 gconstpointer key);
static void foreach_item_func(gpointer data, gpointer user_data);
static void foreach_item_view_func(gpointer data, gpointer user_data);
#else
static ItemViewRelation *find_item_view_relation(PanedNotebook *paned_notebook,
						 gconstpointer key,
						 ItemViewRelation *relation);
typedef struct _ForeachItemArgs ForeachItemArgs;
static gboolean foreach_item_func(GtkTreeModel *model, GtkTreePath *path,
				  GtkTreeIter *iter, ForeachItemArgs *args);
static gboolean foreach_item_view_func(GtkTreeModel *model, GtkTreePath *path,
				       GtkTreeIter *iter,
				       ForeachItemArgs *args);
#endif
static void close_nth_page(PanedNotebook *paned_notebook, int page);

static void switch_page_cb(ModNotebook *notebook, ModNotebookPage *page,
			   guint page_num, PanedNotebook *paned_notebook);
static gboolean button_press_event_cb(ModNotebook *notebook,
				      GdkEventButton *event,
				      PanedNotebook *paned_notebook);
static void close_button_clicked_cb(ModNotebook *notebook,
				    PanedNotebook *paned_notebook);
#if GTK_MINOR_VERSION <= 2
static void menu_changed_cb(GtkOptionMenu *option_menu,
			    PanedNotebook *paned_notebook);
static void menu_item_activate_cb(GtkMenuItem *menu_item,
				  PanedNotebook *paned_notebook);
#else
static void menu_changed_cb(GtkComboBox *combo_box,
			    PanedNotebook *paned_notebook);
#endif
static void go_left_button_cb(gpointer unused, PanedNotebook *paned_notebook);
static void go_right_button_cb(gpointer unused, PanedNotebook *paned_notebook);


GType
paned_notebook_get_type(void)
{
  static GType pn_type = 0;

  if (pn_type == 0)
    {
      static const GTypeInfo pn_info =
	{
	  sizeof(PanedNotebookClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)paned_notebook_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(PanedNotebook),
	  0,    /* n_preallocs */
	  (GInstanceInitFunc)paned_notebook_init,
	};

      pn_type = g_type_register_static(GTK_TYPE_PANED,
				       "PanedNotebook", &pn_info, 0);
    }

  return pn_type;
}


GType
paned_style_get_type(void)
{
  static GType s_type = 0;

  if (s_type == 0)
    {
      static const GEnumValue values[] =
	{
	  { HPANED_STYLE, (char *)"HPANED_STYLE", (char *)"hpaned" },
	  { VPANED_STYLE, (char *)"VPANED_STYLE", (char *)"vpaned" },
	  { 0, NULL, NULL }
	};

      s_type = g_enum_register_static("PanedStyle", values);
    }

  return s_type;
}

#if GTK_MINOR_VERSION <= 2
struct _ItemViewRelation
{
  gpointer item;
  GtkWidget *item_view;
  GtkWidget *menu_item;
  GtkWidget *tab_label;
};
#else
struct _ItemViewRelation
{
  gpointer item;
  GtkWidget *item_view;
  GtkWidget *tab_label;
  GtkTreeIter *iter;
};


typedef enum
{
  NOTEBOOK_ITEM = 0,
  NOTEBOOK_ITEM_VIEW,
  NOTEBOOK_ITEM_TITLE,
  NOTEBOOK_ITEM_TAB_LABEL,
  NOTEBOOK_ITEM_N_COLUMNS
} NotebookItem;
#endif


enum {
  ITEM_VIEW_REQUIRED_SIGNAL,
  ITEM_VIEW_BEING_CLOSED_SIGNAL,
  ITEM_VIEW_CLOSED_SIGNAL,
  PAGE_SWITCHED_SIGNAL,
  PAGE_DOUBLE_CLICKED_SIGNAL,
  LAST_SIGNAL
};


static GtkPanedClass *parent_class = NULL;
static int paned_notebook_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0 };


enum {
  PROP_0,
  PROP_PANED_STYLE,
  PROP_TAB_POLICY,
  PROP_ALWAYS_SHOW_CONTENTS_PANE,
};


static void
paned_notebook_class_init(PanedNotebookClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS(klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = paned_notebook_finalize;
  o_class->set_property = paned_notebook_set_property;
  o_class->get_property = paned_notebook_get_property;

  g_object_class_install_property(o_class,
				  PROP_PANED_STYLE,
				  g_param_spec_enum("paned_style",
						       _("PanedStyle"),
						       _("What type of paned style to be used"),
						    PANED_STYLE_TYPE,
						    HPANED_STYLE,
						    G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_TAB_POLICY,
				  g_param_spec_enum("tab_policy",
						    _("TabPolicy"),
						    _("When bookmark(tab)s should be shown"),
						    GTK_TYPE_POLICY_TYPE,
						    GTK_POLICY_AUTOMATIC,
						    G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_ALWAYS_SHOW_CONTENTS_PANE,
				  g_param_spec_boolean("always_show_contents_pane",
						    _("AlwaysShowContentsPane"),
						    _("Whether contents pane should always be shown"),
						       FALSE,
						       G_PARAM_READWRITE));

  /* GtkObject signals */
  object_class->destroy = paned_notebook_destroy;

  /* GtkWidget signals */
  widget_class->size_request = paned_notebook_size_request;
  widget_class->size_allocate = paned_notebook_size_allocate;

  paned_notebook_signals[ITEM_VIEW_REQUIRED_SIGNAL] =
    g_signal_new("item_view_required",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, item_view_required),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_POINTER_POINTER_POINTER,
		 G_TYPE_NONE, 4,
		 G_TYPE_POINTER,
		 G_TYPE_POINTER,
		 G_TYPE_POINTER,
		 G_TYPE_POINTER);
  paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL] =
    g_signal_new("item_view_being_closed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, item_view_being_closed),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_OBJECT,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 GTK_TYPE_WIDGET);
  paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL] =
    g_signal_new("item_view_closed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, item_view_closed),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_OBJECT,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 GTK_TYPE_WIDGET);
  paned_notebook_signals[PAGE_SWITCHED_SIGNAL] =
    g_signal_new("page_switched",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, page_switched),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT_OBJECT,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_WIDGET,
		 GTK_TYPE_WIDGET);
  paned_notebook_signals[PAGE_DOUBLE_CLICKED_SIGNAL] =
    g_signal_new("page_double_clicked",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, page_double_clicked),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_OBJECT,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 GTK_TYPE_WIDGET);

  klass->item_view_required = item_view_required_cb;
  klass->item_view_accepted = NULL;
  klass->item_view_being_closed = NULL;
  klass->item_view_closed = NULL;
  klass->page_switched = NULL;
  klass->page_double_clicked = NULL;
}


static void
paned_notebook_set_property(GObject *object, guint prop_id,
			    const GValue *value, GParamSpec *pspec)
{
  PanedNotebook *paned_notebook;

  g_return_if_fail(IS_PANED_NOTEBOOK(object));

  paned_notebook = PANED_NOTEBOOK(object);

  switch (prop_id)
    {
    case PROP_PANED_STYLE:
      paned_notebook_set_paned_style(paned_notebook, g_value_get_enum(value));
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "setting paned_style: %d\n", g_value_get_enum(value));
#endif
      break;

    case PROP_TAB_POLICY:
      paned_notebook_set_tab_policy(paned_notebook, g_value_get_enum(value));
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "setting tab_policy: %d\n", g_value_get_enum(value));
#endif
      break;

    case PROP_ALWAYS_SHOW_CONTENTS_PANE:
      paned_notebook_set_always_show_contents_pane(paned_notebook,
						   g_value_get_boolean(value));
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "setting always_show: %d\n", g_value_get_boolean(value));
#endif
      break;

    default:
      break;
    }
}


static void
paned_notebook_get_property(GObject *object, guint prop_id,
			    GValue *value, GParamSpec *pspec)
{
  PanedNotebook *paned_notebook;

  g_return_if_fail(IS_PANED_NOTEBOOK(object));

  paned_notebook = PANED_NOTEBOOK(object);

  switch (prop_id)
    {
    case PROP_PANED_STYLE:
      g_value_set_enum(value, paned_notebook->paned_style);
      break;

    case PROP_TAB_POLICY:
      g_value_set_enum(value, paned_notebook->tab_policy);
      break;

    case PROP_ALWAYS_SHOW_CONTENTS_PANE:
      g_value_set_boolean(value, paned_notebook->always_show_contents_pane);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


static void
item_view_required_cb(PanedNotebook *paned_notebook, gpointer item,
		      GtkWidget **item_view, const gchar **title,
		      GtkWidget **tab_label)
{
  /* TODO: 顼Ф */
}


static void
paned_notebook_init(PanedNotebook *paned_notebook)
{
  GtkPaned *paned;
  GtkWidget *tmp_arrow;
#if GTK_MINOR_VERSION <= 2
  GtkWidget *tmp_button;
#else
  GtkCellRenderer *renderer;
  GtkToolItem *tmp_item;
  GtkToolItem *tool_button;

  static const char *toolbar_ui =
    "<ui>"
    "  <toolbar name=\"PanedNotebookLeftToolbar\"/>"
    "</ui>";

  GError *error;
#endif

  g_return_if_fail(GTK_IS_PANED(paned_notebook));

  paned_notebook->selector = NULL;
  paned_notebook->contents_pane = gtk_vbox_new(FALSE, 0);

  paned_notebook->toolbar_box = GTK_BOX(gtk_hbox_new(FALSE, 0));

#if GTK_MINOR_VERSION <= 2
  paned_notebook->toolbar = GTK_TOOLBAR(gtk_toolbar_new());
#else
  paned_notebook->ui_manager = gtk_ui_manager_new();
  error = NULL;
  if (!gtk_ui_manager_add_ui_from_string(paned_notebook->ui_manager,
					 toolbar_ui, -1, &error))
    {
      g_message("building toolbar failed: %s", error->message);
      g_error_free(error);
      exit(EXIT_FAILURE);
    }

  paned_notebook->toolbar
    = GTK_TOOLBAR(gtk_ui_manager_get_widget(paned_notebook->ui_manager,
					    "/PanedNotebookLeftToolbar"));
#endif
  gtk_toolbar_set_style(paned_notebook->toolbar, GTK_TOOLBAR_ICONS);
  gtk_box_pack_start(paned_notebook->toolbar_box,
		     GTK_WIDGET(paned_notebook->toolbar),
		     TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(paned_notebook->contents_pane),
		     GTK_WIDGET(paned_notebook->toolbar_box),
		     FALSE, TRUE, 0);

#if GTK_MINOR_VERSION <= 2
  paned_notebook->contents_menu = GTK_OPTION_MENU(gtk_option_menu_new());
  paned_notebook->contents_list = GTK_MENU(gtk_menu_new());
  gtk_option_menu_set_menu(paned_notebook->contents_menu,
			   GTK_WIDGET(paned_notebook->contents_list));
#else
  paned_notebook->contents_list
    = gtk_list_store_new(NOTEBOOK_ITEM_N_COLUMNS,
			 G_TYPE_POINTER,	/* item */
			 G_TYPE_POINTER,	/* item_view */
			 G_TYPE_STRING,		/* title */
			 G_TYPE_POINTER		/* tab_label */);
  paned_notebook->contents_menu
    = GTK_COMBO_BOX(gtk_combo_box_new_with_model(GTK_TREE_MODEL(paned_notebook->contents_list)));
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(paned_notebook->contents_menu),
			     renderer, FALSE);
  gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(paned_notebook->contents_menu),
				 renderer, "text", NOTEBOOK_ITEM_TITLE, NULL);
#endif

  g_signal_connect(G_OBJECT(paned_notebook->contents_menu),
		   "changed",
		   G_CALLBACK(menu_changed_cb), paned_notebook);

  /* MEMO: tooltipΥåpropertyǽˤ٤ */

#if GTK_MINOR_VERSION <= 2
  tmp_arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_ETCHED_OUT);
  tmp_button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(tmp_button), GTK_RELIEF_NONE);
  gtk_widget_show(tmp_arrow);
  gtk_container_add(GTK_CONTAINER(tmp_button), tmp_arrow);
  GTK_WIDGET_UNSET_FLAGS(tmp_button, GTK_CAN_FOCUS);
  gtk_widget_show(tmp_button);
  g_signal_connect(G_OBJECT(tmp_button), "clicked",
		   G_CALLBACK(go_left_button_cb), paned_notebook);
  gtk_toolbar_append_widget(paned_notebook->toolbar, tmp_button,
			    _("Go to the Left Page"),
			    "go_left_page");
  paned_notebook->left_arrow = tmp_button;
  gtk_widget_set_sensitive(tmp_button, FALSE);

  tmp_arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_ETCHED_OUT);
  tmp_button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(tmp_button), GTK_RELIEF_NONE);
  gtk_widget_show(tmp_arrow);
  gtk_container_add(GTK_CONTAINER(tmp_button), tmp_arrow);
  gtk_widget_show(tmp_button);
  g_signal_connect(G_OBJECT(tmp_button), "clicked",
		   G_CALLBACK(go_right_button_cb), paned_notebook);
  GTK_WIDGET_UNSET_FLAGS(tmp_button, GTK_CAN_FOCUS);
  gtk_toolbar_append_widget(paned_notebook->toolbar, tmp_button,
			    _("Go to the Right Page"),
			    "go_right_page");
  paned_notebook->right_arrow = tmp_button;
  gtk_widget_set_sensitive(tmp_button, FALSE);


  gtk_toolbar_append_widget(paned_notebook->toolbar,
			    GTK_WIDGET(paned_notebook->contents_menu),
			    _("Select the Page to be Shown"),
			    "item_selection_menu");
#else
  tmp_arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(tmp_arrow);

  tool_button = gtk_tool_button_new(tmp_arrow, NULL);
  GTK_WIDGET_UNSET_FLAGS(tool_button, GTK_CAN_FOCUS);
  gtk_tool_item_set_tooltip(tool_button,
			    paned_notebook->toolbar->tooltips,
			    _("Go to the Left Page"),
			    "go_left_page");
  gtk_widget_show(GTK_WIDGET(tool_button));
  g_signal_connect(G_OBJECT(tool_button), "clicked",
		   G_CALLBACK(go_left_button_cb), paned_notebook);
  gtk_toolbar_insert(paned_notebook->toolbar, tool_button, -1);
  paned_notebook->left_arrow = GTK_WIDGET(tool_button);
  gtk_widget_set_sensitive(GTK_WIDGET(tool_button), FALSE);


  tmp_arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_ETCHED_OUT);
  gtk_widget_show(tmp_arrow);
  tool_button = gtk_tool_button_new(tmp_arrow, NULL);
  GTK_WIDGET_UNSET_FLAGS(tool_button, GTK_CAN_FOCUS);
  gtk_tool_item_set_tooltip(tool_button,
			    paned_notebook->toolbar->tooltips,
			    _("Go to the Right Page"),
			    "go_right_page");
  gtk_widget_show(GTK_WIDGET(tool_button));
  g_signal_connect(G_OBJECT(tool_button), "clicked",
		   G_CALLBACK(go_right_button_cb), paned_notebook);
  gtk_toolbar_insert(paned_notebook->toolbar, tool_button, -1);
  paned_notebook->right_arrow = GTK_WIDGET(tool_button);
  gtk_widget_set_sensitive(GTK_WIDGET(tool_button), FALSE);


  tmp_item = gtk_tool_item_new();
  gtk_widget_show(GTK_WIDGET(paned_notebook->contents_menu));
  gtk_container_add(GTK_CONTAINER(tmp_item),
		    GTK_WIDGET(paned_notebook->contents_menu));
  gtk_tool_item_set_tooltip(tmp_item, paned_notebook->toolbar->tooltips,
			    _("Select the Page to be Shown"),
			    "item_selection_menu");
  gtk_widget_show(GTK_WIDGET(tmp_item));
  gtk_toolbar_insert(paned_notebook->toolbar, tmp_item, -1);
#endif

#if 0	/* GTK+-2.4.1ǤȤʤΤǤ */
  gtk_widget_set_sensitive(GTK_WIDGET(paned_notebook->contents_menu), FALSE);
#endif

  paned_notebook->item_contents = MOD_NOTEBOOK(mod_notebook_new());
  mod_notebook_set_scrollable(paned_notebook->item_contents, TRUE);
  mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);
  mod_notebook_close_button_enable(paned_notebook->item_contents);

  g_signal_connect(G_OBJECT(paned_notebook->item_contents),
		   "switch_page",
		   G_CALLBACK(switch_page_cb), paned_notebook);
  g_signal_connect(G_OBJECT(paned_notebook->item_contents),
		   "button_press_event",
		   G_CALLBACK(button_press_event_cb), paned_notebook);
  g_signal_connect(G_OBJECT(paned_notebook->item_contents),
		   "close_button_clicked",
		   G_CALLBACK(close_button_clicked_cb), paned_notebook);
  gtk_box_pack_start(GTK_BOX(paned_notebook->contents_pane),
		     GTK_WIDGET(paned_notebook->item_contents),
		     TRUE, TRUE, 0);

  paned_notebook->number_of_items_shown = 0;
  paned_notebook->current_item = NULL;
  paned_notebook->current_item_view = NULL;
#if GTK_MINOR_VERSION <= 2
  paned_notebook->items_shown = NULL;
  paned_notebook->active_menu_item = NULL;
#endif

  paned = GTK_PANED(paned_notebook);
  paned->cursor_type = GDK_SB_H_DOUBLE_ARROW;
  paned->orientation = GTK_ORIENTATION_VERTICAL;

  paned_notebook->paned_style = HPANED_STYLE;
  paned_notebook->tab_policy = GTK_POLICY_AUTOMATIC;
  paned_notebook->always_show_contents_pane = FALSE;

  g_object_ref(paned_notebook->contents_pane);
  g_signal_connect(G_OBJECT(paned_notebook->contents_pane),
		   "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &paned_notebook->contents_pane);
  gtk_object_sink(GTK_OBJECT(paned_notebook->contents_pane));
}


static void
paned_notebook_finalize(GObject *object)
{
#if 0
  PanedNotebook *paned_notebook = (PanedNotebook *)object;

  fprintf(stderr, "paned_notebook_finalize(%p)\n", paned_notebook);
#endif

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


#if GTK_MINOR_VERSION <= 2
static void
fake_close_signal(ItemViewRelation *relation, PanedNotebook *paned_notebook)
{
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL],
		0,
		relation->item,
		relation->item_view);
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL],
		0,
		relation->item,
		relation->item_view);
  gtk_widget_hide(relation->item_view);
  gtk_widget_unrealize(relation->item_view);
}
#else
static gboolean
fake_close_signal(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
		  PanedNotebook *paned_notebook)
{
  gpointer item;
  GtkWidget *item_view;
  gtk_tree_model_get(model, iter,
		     NOTEBOOK_ITEM, &item,
		     NOTEBOOK_ITEM_VIEW, &item_view, -1);
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL],
		0,
		item,
		item_view);

  gtk_widget_hide(item_view);
  gtk_widget_unrealize(item_view);

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL],
		0,
		item,
		item_view);

  return FALSE;
}
#endif


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
paned_notebook_destroy(GtkObject *object)
{
  PanedNotebook *paned_notebook = (PanedNotebook *)object;

#if 0
  fprintf(stderr, "paned_notebook_destroy(%p)\n", paned_notebook);
#endif

#if GTK_MINOR_VERSION <= 2
  if (paned_notebook->items_shown != NULL)
    {
      g_slist_foreach(paned_notebook->items_shown,
		      (GFunc)fake_close_signal, paned_notebook);
      g_slist_foreach(paned_notebook->items_shown,
		      (GFunc)TRACE_FREE, NULL);
      g_slist_free(paned_notebook->items_shown);
      paned_notebook->items_shown = NULL;
    }
#else
  if (paned_notebook->contents_list != NULL)
    {
      gtk_tree_model_foreach(GTK_TREE_MODEL(paned_notebook->contents_list),
			     (GtkTreeModelForeachFunc)fake_close_signal,
			     paned_notebook);
      g_object_unref(paned_notebook->contents_list);
      paned_notebook->contents_list = NULL;
    }

  if (paned_notebook->ui_manager != NULL)
    {
      g_object_unref(paned_notebook->ui_manager);
      paned_notebook->ui_manager = NULL;
    }
#endif

  if (paned_notebook->contents_pane != NULL)
    {
      g_object_unref(paned_notebook->contents_pane);
      paned_notebook->contents_pane = NULL;
    }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


void
paned_notebook_set_paned_style(PanedNotebook *paned_notebook,
			       PanedStyleType paned_style)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "paned_notebook_set_paned_style: %d\n", paned_style);
#endif

  if (paned_notebook->paned_style != paned_style)
    {
      GtkPaned *paned = GTK_PANED(paned_notebook);
      paned_notebook->paned_style = paned_style;

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  paned->cursor_type = GDK_SB_H_DOUBLE_ARROW;
	  paned->orientation = GTK_ORIENTATION_VERTICAL;
	}
      else
	{
	  paned->cursor_type = GDK_SB_V_DOUBLE_ARROW;
	  paned->orientation = GTK_ORIENTATION_HORIZONTAL;
	}

      if (GDK_IS_WINDOW(paned->handle))
	{
	  GdkCursor *cursor = gdk_cursor_new_for_display(gtk_widget_get_display(GTK_WIDGET(paned)),
							 paned->cursor_type);
	  gdk_window_set_cursor(paned->handle, cursor);
	  gdk_cursor_unref(cursor);
	}

      if (GTK_WIDGET_VISIBLE(paned_notebook))
	gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
    }

  g_object_notify(G_OBJECT(paned_notebook), "paned_style");
}


PanedStyleType
paned_notebook_get_paned_style(PanedNotebook *paned_notebook)
{
  return paned_notebook->paned_style;
}


void
paned_notebook_set_tab_policy(PanedNotebook *paned_notebook,
			      GtkPolicyType tab_policy)
{
  g_return_if_fail(tab_policy >= GTK_POLICY_ALWAYS
		   && tab_policy <= GTK_POLICY_NEVER);

#if DEBUG_WIDGET
  fprintf(stderr, "set_tab_policy: %d\n", tab_policy);
#endif

  paned_notebook->tab_policy = tab_policy;
  switch (tab_policy)
    {
    case GTK_POLICY_ALWAYS:
      mod_notebook_set_show_tabs(paned_notebook->item_contents, TRUE);
      break;

    case GTK_POLICY_AUTOMATIC:
      if (paned_notebook->number_of_items_shown > 1)
	mod_notebook_set_show_tabs(paned_notebook->item_contents, TRUE);
      else
	mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);
      break;

    case GTK_POLICY_NEVER:
      mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);
      break;
    }

  gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
}


void
paned_notebook_set_tab_shrinkable(PanedNotebook *paned_notebook,
				  gboolean shrinkable)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

  mod_notebook_set_tab_shrinkable(paned_notebook->item_contents, shrinkable);
}


gboolean
paned_notebook_get_tab_shrinkable(PanedNotebook *paned_notebook)
{
  g_return_val_if_fail(IS_PANED_NOTEBOOK(paned_notebook), FALSE);

  return mod_notebook_get_tab_shrinkable(paned_notebook->item_contents);
}


void
paned_notebook_set_minimum_tab_label_size(PanedNotebook *paned_notebook,
					  guint size)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

  mod_notebook_set_minimum_tab_label_size(paned_notebook->item_contents, size);
}


guint
paned_notebook_get_minimum_tab_label_size(PanedNotebook *paned_notebook)
{
  g_return_val_if_fail(IS_PANED_NOTEBOOK(paned_notebook), 0);

  return mod_notebook_get_minimum_tab_label_size(paned_notebook->item_contents);
}


void
paned_notebook_set_tooltips(PanedNotebook *paned_notebook, gboolean enable)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

  if (enable)
    mod_notebook_tooltips_enable(paned_notebook->item_contents);
  else
    mod_notebook_tooltips_disable(paned_notebook->item_contents);
}


void
paned_notebook_set_always_show_contents_pane(PanedNotebook *paned_notebook,
					     gboolean show_always)
{
  if (paned_notebook->always_show_contents_pane == show_always)
    return;

  paned_notebook->always_show_contents_pane = show_always;

  if (paned_notebook->number_of_items_shown > 0)
    return;

  if (show_always)
    {
      gtk_widget_show_all(GTK_WIDGET(paned_notebook->contents_pane));
      gtk_paned_add2(GTK_PANED(paned_notebook),
		     GTK_WIDGET(paned_notebook->contents_pane));
    }
  else
    {
      if (gtk_paned_get_position(GTK_PANED(paned_notebook)) == 0)
	paned_notebook->position_zero_hack = TRUE;

      gtk_container_remove(GTK_CONTAINER(paned_notebook),
			   GTK_WIDGET(paned_notebook->contents_pane));
    }
}


gboolean
paned_notebook_get_always_show_contents_pane(PanedNotebook *paned_notebook)
{
  return paned_notebook->always_show_contents_pane;
}


GtkPolicyType
paned_notebook_get_tab_policy(PanedNotebook *paned_notebook)
{
  return paned_notebook->tab_policy;
}


static void
paned_notebook_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
  PanedNotebook *paned_notebook = PANED_NOTEBOOK(widget);
  GtkPaned *paned = GTK_PANED(widget);
  GtkRequisition child_requisition;

  requisition->width = 0;
  requisition->height = 0;

  if (paned->child1 && GTK_WIDGET_VISIBLE(paned->child1))
    {
      gtk_widget_size_request(paned->child1, &child_requisition);

      requisition->height = child_requisition.height;
      requisition->width = child_requisition.width;
    }

  if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
    {
      gtk_widget_size_request(paned->child2, &child_requisition);

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  requisition->height = MAX(requisition->height,
				    child_requisition.height);
	  requisition->width += child_requisition.width;
	}
      else
	{
	  requisition->width = MAX(requisition->width,
				   child_requisition.width);
	  requisition->height += child_requisition.height;
	}
    }

  requisition->height += GTK_CONTAINER(paned)->border_width * 2;
  requisition->width += GTK_CONTAINER(paned)->border_width * 2;
  
  if (paned->child1 && GTK_WIDGET_VISIBLE(paned->child1) &&
      paned->child2 && GTK_WIDGET_VISIBLE(paned->child2))
    {
      int handle_size;

      gtk_widget_style_get(widget, "handle_size", &handle_size, NULL);
      if (paned_notebook->paned_style == HPANED_STYLE)
	requisition->width += handle_size;
      else
	requisition->height += handle_size;
    }
}


static void
paned_notebook_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
  PanedNotebook *paned_notebook = PANED_NOTEBOOK(widget);
  GtkPaned *paned = GTK_PANED (widget);
  int border_width = GTK_CONTAINER (paned)->border_width;

  widget->allocation = *allocation;

  if (paned->child1 && GTK_WIDGET_VISIBLE(paned->child1) &&
      paned->child2 && GTK_WIDGET_VISIBLE(paned->child2))
    {
      GtkAllocation child1_allocation;
      GtkAllocation child2_allocation;
      GtkRequisition child1_requisition;
      GtkRequisition child2_requisition;
      int handle_size;
      
      gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);

      gtk_widget_get_child_requisition(paned->child1, &child1_requisition);
      gtk_widget_get_child_requisition(paned->child2, &child2_requisition);

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  gtk_paned_compute_position(paned,
				     MAX(1, widget->allocation.width
					 - handle_size
					 - 2 * border_width),
				     child1_requisition.width,
				     child2_requisition.width);

	  paned->handle_pos.x = widget->allocation.x + paned->child1_size + border_width;
	  paned->handle_pos.y = widget->allocation.y + border_width;
	  paned->handle_pos.width = handle_size;
	  paned->handle_pos.height = MAX(1, widget->allocation.height - 2 * border_width);
	}
      else
	{
	  gtk_paned_compute_position(paned,
				     MAX(1, widget->allocation.height
					 - handle_size
					 - 2 * border_width),
				     child1_requisition.height,
				     child2_requisition.height);

	  paned->handle_pos.x = widget->allocation.x + border_width;
	  paned->handle_pos.y = widget->allocation.y + paned->child1_size + border_width;
	  paned->handle_pos.width = MAX(1, (int) widget->allocation.width - 2 * border_width);
	  paned->handle_pos.height = handle_size;
	}
      
      
      if (GTK_WIDGET_REALIZED(widget))
	{
	  if (GTK_WIDGET_MAPPED(widget))
	    gdk_window_show(paned->handle);
	  if (paned_notebook->paned_style == HPANED_STYLE)
	    {
	      gdk_window_move_resize(paned->handle,
				     paned->handle_pos.x,
				     paned->handle_pos.y,
				     handle_size,
				     paned->handle_pos.height);
	    }
	  else
	    {
	      gdk_window_move_resize(paned->handle,
				     paned->handle_pos.x,
				     paned->handle_pos.y,
				     paned->handle_pos.width,
				     handle_size);
	    }
	}

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  child1_allocation.height = child2_allocation.height = MAX(1, (int) allocation->height - border_width * 2);
	  child1_allocation.width = paned->child1_size;
	  child1_allocation.x = widget->allocation.x + border_width;
	  child1_allocation.y = child2_allocation.y = widget->allocation.y + border_width;

	  child2_allocation.x = child1_allocation.x + child1_allocation.width +  paned->handle_pos.width;
	  child2_allocation.width = MAX(1, widget->allocation.x + widget->allocation.width - child2_allocation.x - border_width);

	}
      else
	{
	  child1_allocation.width = child2_allocation.width = MAX(1, (int) allocation->width - border_width * 2);
	  child1_allocation.height = paned->child1_size;
	  child1_allocation.x = child2_allocation.x = widget->allocation.x + border_width;
	  child1_allocation.y = widget->allocation.y + border_width;

	  child2_allocation.y = child1_allocation.y + child1_allocation.height + paned->handle_pos.height;
	  child2_allocation.height = MAX(1, widget->allocation.y + widget->allocation.height - child2_allocation.y - border_width);
	}
      
      
      /* Now allocate the childen, making sure, when resizing not to
       * overlap the windows
       */
      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  if (GTK_WIDGET_MAPPED(widget) &&
	      paned->child1->allocation.width < child1_allocation.width)
	    {
	      gtk_widget_size_allocate(paned->child2, &child2_allocation);
	      gtk_widget_size_allocate(paned->child1, &child1_allocation);
	    }
	  else
	    {
	      gtk_widget_size_allocate(paned->child1, &child1_allocation);
	      gtk_widget_size_allocate(paned->child2, &child2_allocation);
	    }
	}
      else
	{
	  if (GTK_WIDGET_MAPPED(widget) &&
	      paned->child1->allocation.height < child1_allocation.height)
	    {
	      gtk_widget_size_allocate(paned->child2, &child2_allocation);
	      gtk_widget_size_allocate(paned->child1, &child1_allocation);
	    }
	  else
	    {
	      gtk_widget_size_allocate(paned->child1, &child1_allocation);
	      gtk_widget_size_allocate(paned->child2, &child2_allocation);
	    }
	}
    }
  else
    {
      GtkAllocation child_allocation;

      if (GTK_WIDGET_REALIZED (widget))      
	gdk_window_hide(paned->handle);
	  
      child_allocation.x = widget->allocation.x + border_width;
      child_allocation.y = widget->allocation.y + border_width;
      child_allocation.width = MAX(1, allocation->width - 2 * border_width);
      child_allocation.height = MAX(1, allocation->height - 2 * border_width);

      if (paned->child1 && GTK_WIDGET_VISIBLE(paned->child1))
	{
	  gtk_widget_size_allocate(paned->child1, &child_allocation);
	  if (paned_notebook->position_zero_hack)
	    {
	      gtk_widget_unmap(paned->child1);
	      gtk_widget_set_child_visible(paned->child1, TRUE);
	      gtk_widget_map(paned->child1);
	      paned_notebook->position_zero_hack = FALSE;
	    }
	}
      else if (paned->child2 && GTK_WIDGET_VISIBLE(paned->child2))
	gtk_widget_size_allocate(paned->child2, &child_allocation);
    }
}


GtkWidget *
paned_notebook_new(PanedStyleType paned_style)
{
  GtkWidget *widget;

  widget = GTK_WIDGET(g_object_new(PANED_NOTEBOOK_TYPE,
				   "paned_style", paned_style,
				   NULL));

  return widget;
}


GtkWidget *
paned_notebook_new_with_selector(PanedStyleType paned_style,
				 GtkWidget *selector)
{
  GtkWidget *widget = paned_notebook_new(paned_style);

  g_return_val_if_fail((widget != NULL), NULL);

  paned_notebook_set_selector(PANED_NOTEBOOK(widget), selector);

  return widget;
}


GtkWidget *
paned_notebook_get_selector(PanedNotebook *paned_notebook)
{
  return paned_notebook->selector;
}


void
paned_notebook_set_selector(PanedNotebook *paned_notebook, GtkWidget *selector)
{
  if (paned_notebook->selector == selector)
    return;	/* ⤷ʤ */

  if (paned_notebook->selector != NULL)
    gtk_container_remove(GTK_CONTAINER(paned_notebook),
			 paned_notebook->selector);

  if (selector != NULL)
    {
      gtk_paned_add1(GTK_PANED(paned_notebook), selector);
    }

  paned_notebook->selector = selector;

  return;
}


#if GTK_MINOR_VERSION <= 2
static ItemViewRelation *
item_view_relation_new(gpointer item, GtkWidget *item_view,
		       GtkWidget *menu_item, GtkWidget *tab_label)
{
  ItemViewRelation *relation
    = (ItemViewRelation *)G_MALLOC(sizeof(ItemViewRelation));
  relation->item = item;
  relation->item_view = item_view;
  relation->menu_item = menu_item;
  relation->tab_label = tab_label;
  return relation;
}


static int
compare_items(gconstpointer list_data, gconstpointer user_data)
{
  return (((ItemViewRelation *)list_data)->item != user_data
	  && ((ItemViewRelation *)list_data)->item_view != user_data
	  && ((ItemViewRelation *)list_data)->menu_item != user_data
	  && ((ItemViewRelation *)list_data)->tab_label != user_data);
}


static ItemViewRelation *
find_item_view_relation(PanedNotebook *paned_notebook, gconstpointer key)
{
  GSList *entry = g_slist_find_custom(paned_notebook->items_shown, key,
				      compare_items);
  return (ItemViewRelation *)(entry != NULL ? entry->data : NULL);
}
#else
typedef struct _FindItemArgs
{
  gconstpointer key;

  ItemViewRelation *relation;
} FindItemArgs;


static gboolean
find_item_func(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
	       FindItemArgs *args)
{
  gconstpointer key = args->key;
  gpointer item;
  GtkWidget *item_view;
  GtkWidget *tab_label;
  gtk_tree_model_get(model, iter,
		     NOTEBOOK_ITEM, &item,
		     NOTEBOOK_ITEM_VIEW, &item_view,
		     NOTEBOOK_ITEM_TAB_LABEL, &tab_label, -1);
  if (item == key || item_view == key || tab_label == key)
    {
      ItemViewRelation *relation = args->relation;
      relation->item = item;
      relation->item_view = item_view;
      relation->tab_label = tab_label;
      if (relation->iter != NULL)
	gtk_tree_model_get_iter(model, relation->iter, path);
      return TRUE;
    }
  return FALSE;
}


static ItemViewRelation *
find_item_view_relation(PanedNotebook *paned_notebook, gconstpointer key,
			ItemViewRelation *relation)
{
  FindItemArgs args = { key, relation };
  gtk_tree_model_foreach(GTK_TREE_MODEL(paned_notebook->contents_list),
			 (GtkTreeModelForeachFunc)find_item_func, &args);
  return relation->item != NULL ? relation : NULL;
}
#endif


GtkToolbar *
paned_notebook_get_toolbar(PanedNotebook *paned_notebook)
{
  return paned_notebook->toolbar;
}


gpointer
paned_notebook_get_current_item(PanedNotebook *paned_notebook)
{
  return paned_notebook->current_item;
}


GtkWidget *
paned_notebook_get_current_item_view(PanedNotebook *paned_notebook)
{
  return paned_notebook->current_item_view;
}


gpointer
paned_notebook_get_item(PanedNotebook *paned_notebook, GtkWidget *item_view)
{
#if GTK_MINOR_VERSION <= 2
  ItemViewRelation *relation = find_item_view_relation(paned_notebook,
						       item_view);

  if (relation == NULL)
    return NULL;
  return relation->item;
#else
  ItemViewRelation relation = { NULL, NULL, NULL, NULL };
  if (find_item_view_relation(paned_notebook, item_view, &relation) != NULL)
    return relation.item;
  return NULL;
#endif
}


GtkWidget *
paned_notebook_get_item_view(PanedNotebook *paned_notebook, gpointer item)
{
#if GTK_MINOR_VERSION <= 2
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);

  if (relation == NULL)
    return NULL;
  return relation->item_view;
#else
  ItemViewRelation relation = { NULL, NULL, NULL, NULL };
  if (find_item_view_relation(paned_notebook, item, &relation) != NULL)
    return relation.item_view;
  return NULL;
#endif
}


GtkWidget *
paned_notebook_get_tab_label(PanedNotebook *paned_notebook, gpointer item)
{
#if GTK_MINOR_VERSION <= 2
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);

  if (relation == NULL)
    return NULL;
  return relation->tab_label;
#else
  ItemViewRelation relation = { NULL, NULL, NULL, NULL };
  if (find_item_view_relation(paned_notebook, item, &relation) != NULL)
    return relation.tab_label;
  return NULL;
#endif
}


#if GTK_MINOR_VERSION <= 2
typedef struct _ForeachItemArgs
{
  GFunc func;
  gpointer user_data;
} ForeachItemArgs;


static void
foreach_item_func(gpointer data, gpointer user_data)
{
  ItemViewRelation *relation = (ItemViewRelation *)data;
  ForeachItemArgs *args = (ForeachItemArgs *)user_data;
  (*args->func)(relation->item, args->user_data);
}


void
paned_notebook_foreach_item(PanedNotebook *paned_notebook, GFunc func,
			    gpointer user_data)
{
  ForeachItemArgs args =
    {
      func,
      user_data
    };
  g_slist_foreach(paned_notebook->items_shown, foreach_item_func, &args);
}


static void
foreach_item_view_func(gpointer data, gpointer user_data)
{
  ItemViewRelation *relation = (ItemViewRelation *)data;
  ForeachItemArgs *args = (ForeachItemArgs *)user_data;
  (*args->func)(relation->item_view, args->user_data);
}


void
paned_notebook_foreach_item_view(PanedNotebook *paned_notebook, GFunc func,
				 gpointer user_data)
{
  ForeachItemArgs args =
    {
      func,
      user_data
    };
  g_slist_foreach(paned_notebook->items_shown, foreach_item_view_func, &args);
}
#else
struct _ForeachItemArgs
{
  GFunc func;
  gpointer user_data;
};


static gboolean
foreach_item_func(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
		  ForeachItemArgs *args)
{
  gpointer item;
  gtk_tree_model_get(model, iter, NOTEBOOK_ITEM, &item, -1);
  (*args->func)(item, args->user_data);
  return FALSE;
}


void
paned_notebook_foreach_item(PanedNotebook *paned_notebook, GFunc func,
			    gpointer user_data)
{
  ForeachItemArgs args = { func, user_data };
  gtk_tree_model_foreach(GTK_TREE_MODEL(paned_notebook->contents_list),
			 (GtkTreeModelForeachFunc)foreach_item_func, &args);
}


static gboolean
foreach_item_view_func(GtkTreeModel *model, GtkTreePath *path,
		       GtkTreeIter *iter, ForeachItemArgs *args)
{
  gpointer item_view;
  gtk_tree_model_get(model, iter, NOTEBOOK_ITEM_VIEW, &item_view, -1);
  (*args->func)(item_view, args->user_data);
  return FALSE;
}


void
paned_notebook_foreach_item_view(PanedNotebook *paned_notebook, GFunc func,
				 gpointer user_data)
{
  ForeachItemArgs args = { func, user_data };
  gtk_tree_model_foreach(GTK_TREE_MODEL(paned_notebook->contents_list),
			 (GtkTreeModelForeachFunc)foreach_item_view_func,
			 &args);
}
#endif


/*
 * itemNotebookɽ롣
 *
 * itemɽǤ(ModNotebook˴ޤޤƤ)硢in_tabǤС
 * itemɽƤڡ򤹤롣in_tabξ硢select_page
 * ǤХ֤򤷡ξˤϲ⤷ʤ롣
 *
 * itemɽǤʤ硢item_view_requiredʥȯԤ
 * ηitemɽ륦åȤͿ줿ˤϤΥ
 * ȤModNotebookɲä롣λin_tabξˤϸ򤵤Ƥ
 * ڡɽƤitemĤǤΰ֤˥åȤ
 * Υڡ򤹤롣in_tabġselect_page⿿ǤХ֤
 * 롣
 */
void
paned_notebook_open_item_view(PanedNotebook *paned_notebook, gpointer item,
			      gboolean in_tab, gboolean select_page)
{
#if GTK_MINOR_VERSION <= 2
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);
  GtkWidget *menu_item;
#else
  ItemViewRelation tmp_relation = { NULL, NULL, NULL, NULL };
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item,
						       &tmp_relation);
  GtkTreeIter iter;
#endif
  GtkWidget *item_view = relation != NULL ? relation->item_view : NULL;
  const gchar *item_title = NULL;
  GtkWidget *tab_label = NULL;
  int page;

  if (item_view != NULL)
    {
      if (in_tab && !select_page)
	return;

      page = mod_notebook_page_num(paned_notebook->item_contents, item_view);
      if (page == -1)
	{
	  /* ɽƤϤΥåȤĤʤ */
	  fprintf(stderr, "Something wrong happen.\n");
	  return;
	}

      mod_notebook_set_current_page(paned_notebook->item_contents, page);

      return;
    }

  /* itemɽƤʤ */
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_REQUIRED_SIGNAL],
		0,
		item,
		&item_view,
		&item_title,
		&tab_label);

  /* TODO: item_title == NULLξŬ˽٤ */
  if (item_view == NULL || item_title == NULL)
    {
      return;	/* itemɽǽ */
    }

  if (tab_label == NULL)
    tab_label = gtk_label_new(item_title);

#if GTK_MINOR_VERSION <= 2
  menu_item = gtk_menu_item_new_with_label(item_title);
  paned_notebook->items_shown
    = g_slist_append(paned_notebook->items_shown,
		     item_view_relation_new(item, item_view, menu_item,
					    tab_label));

  g_signal_connect(G_OBJECT(menu_item), "activate",
		   G_CALLBACK(menu_item_activate_cb), paned_notebook);

  gtk_widget_show(menu_item);
  gtk_menu_shell_append(GTK_MENU_SHELL(paned_notebook->contents_list),
			menu_item);
#else
  gtk_list_store_append(paned_notebook->contents_list, &iter);
  gtk_list_store_set(paned_notebook->contents_list, &iter,
		     NOTEBOOK_ITEM, item,
		     NOTEBOOK_ITEM_VIEW, item_view,
		     NOTEBOOK_ITEM_TITLE, item_title,
		     NOTEBOOK_ITEM_TAB_LABEL, tab_label,
		     -1);
#endif

  if (paned_notebook->number_of_items_shown == 0
      && !paned_notebook->always_show_contents_pane)
    {
      gtk_widget_show_all(GTK_WIDGET(paned_notebook->contents_pane));
      gtk_paned_add2(GTK_PANED(paned_notebook),
		     GTK_WIDGET(paned_notebook->contents_pane));
    }

#if 0	/* GTK+-2.4.1ǤȤʤΤǤ */
  if (paned_notebook->number_of_items_shown >= 0)
    gtk_widget_set_sensitive(GTK_WIDGET(paned_notebook->contents_menu), TRUE);
#endif

  if (paned_notebook->number_of_items_shown++ == 0 || in_tab)
    {
      mod_notebook_append_page(paned_notebook->item_contents, item_view,
			       tab_label);
      mod_notebook_set_tab_label_packing(paned_notebook->item_contents,
					 item_view, FALSE, FALSE,
					 GTK_PACK_START);
      if (paned_notebook->number_of_items_shown == 2
	  && paned_notebook->tab_policy == GTK_POLICY_AUTOMATIC)
	{
	  gtk_widget_show_all(GTK_WIDGET(paned_notebook->item_contents));
	  mod_notebook_set_show_tabs(paned_notebook->item_contents, TRUE);
	  gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
	}

      if (paned_notebook->number_of_items_shown > 1 && select_page)
	{
	  page = mod_notebook_page_num(paned_notebook->item_contents,
				       item_view);
	  mod_notebook_set_current_page(paned_notebook->item_contents, page);
	}

      page = mod_notebook_get_current_page(paned_notebook->item_contents);
      gtk_widget_set_sensitive(paned_notebook->left_arrow, page > 0);
      gtk_widget_set_sensitive(paned_notebook->right_arrow,
			       (page
				< paned_notebook->number_of_items_shown - 1));
      return;
    }

  /* 򤵤Ƥڡ֤*/
  page = mod_notebook_get_current_page(paned_notebook->item_contents);
  if (page == -1)
    {
      /* */
      mod_notebook_append_page(paned_notebook->item_contents, item_view,
			       tab_label);
      mod_notebook_set_tab_label_packing(paned_notebook->item_contents,
					 item_view, FALSE, FALSE,
					 GTK_PACK_START);
      return;
    }

  close_nth_page(paned_notebook, page);
  mod_notebook_insert_page(paned_notebook->item_contents, item_view,
			   tab_label, page);
  mod_notebook_set_tab_label_packing(paned_notebook->item_contents,
				     item_view, FALSE, FALSE,
				     GTK_PACK_START);
  mod_notebook_set_current_page(paned_notebook->item_contents, page);
}


/*
 * itemɽǤä餽ΥڡĤ롣
 */
void
paned_notebook_close_item_view(PanedNotebook *paned_notebook, gpointer item)
{
#if GTK_MINOR_VERSION <= 2
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);
#else
  ItemViewRelation tmp_relation = { NULL, NULL, NULL, NULL };
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item,
						       &tmp_relation);
#endif
  int page;

  if (relation == NULL)
    return;

  page = mod_notebook_page_num(paned_notebook->item_contents,
			       relation->item_view);
  if (page == -1)
    {
      /* ɽƤϤΥåȤĤʤ */
      fprintf(stderr, "Something wrong happen.\n");
      return;
    }

  close_nth_page(paned_notebook, page);
}


/*
 * viewɽǤä餽ΥڡĤ롣
 */
void
paned_notebook_close_view(PanedNotebook *paned_notebook, GtkWidget *view)
{
#if GTK_MINOR_VERSION <= 2
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, view);
#else
  ItemViewRelation tmp_relation = { NULL, NULL, NULL, NULL };
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, view,
						       &tmp_relation);
#endif
  int page;

  if (relation == NULL)
    return;

  page = mod_notebook_page_num(paned_notebook->item_contents,
			       relation->item_view);
  if (page == -1)
    {
      /* ɽƤϤΥåȤĤʤ */
      fprintf(stderr, "Something wrong happen.\n");
      return;
    }

  close_nth_page(paned_notebook, page);
}


void
paned_notebook_close_current_item_view(PanedNotebook *paned_notebook)
{
  int page = mod_notebook_get_current_page(paned_notebook->item_contents);
  if (page == -1)
    return;

  close_nth_page(paned_notebook, page);
}


/*
 * ꤷڡĤΥڡбitemˤĤ
 * item_view_closedʥȯԤ롣
 */
static void
close_nth_page(PanedNotebook *paned_notebook, int page)
{
  ItemViewRelation *relation;
#if GTK_MINOR_VERSION <= 2
  GtkWidget *menu_item;
#else
  GtkTreeIter iter;
  ItemViewRelation tmp_relation = { NULL, NULL, NULL, &iter };
#endif
  gpointer item;
  GtkWidget *widget = mod_notebook_get_nth_page(paned_notebook->item_contents,
						page);
  g_return_if_fail(widget != NULL);

#if GTK_MINOR_VERSION <= 2
  relation = find_item_view_relation(paned_notebook, widget);
#else
  relation = find_item_view_relation(paned_notebook, widget, &tmp_relation);
#endif

#if DEBUG_WIDGET
  g_return_if_fail(relation != NULL && relation->item_view == widget);
#else
  if (relation == NULL || relation->item_view != widget)
    return;
#endif

  item = relation->item;
#if GTK_MINOR_VERSION <= 2
  menu_item = relation->menu_item;
  paned_notebook->items_shown = g_slist_remove(paned_notebook->items_shown,
					       relation);
  G_FREE(relation);
#else
  gtk_list_store_remove(paned_notebook->contents_list, relation->iter);
#endif

  if (paned_notebook->current_item_view == widget)
    {
      paned_notebook->current_item = NULL;
      paned_notebook->current_item_view = NULL;
    }

  OCHU_OBJECT_REF(widget);

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL],
		0,
		item,
		widget);

  mod_notebook_remove_page(paned_notebook->item_contents, page);
  gtk_widget_hide(widget);
  gtk_widget_unrealize(widget);

  paned_notebook->number_of_items_shown--;

  if (paned_notebook->number_of_items_shown == 0)
    {
#if 0	/* GTK+-2.4.1ǤȤʤΤǤ */
      gtk_widget_set_sensitive(GTK_WIDGET(paned_notebook->contents_menu),
			       FALSE);
#endif
      if (!paned_notebook->always_show_contents_pane)
	{
	  if (gtk_paned_get_position(GTK_PANED(paned_notebook)) == 0)
	    paned_notebook->position_zero_hack = TRUE;
	  
	  gtk_container_remove(GTK_CONTAINER(paned_notebook),
			       GTK_WIDGET(paned_notebook->contents_pane));
	}
    }

  if (paned_notebook->number_of_items_shown == 1
      && paned_notebook->tab_policy == GTK_POLICY_AUTOMATIC)
    {
      gtk_widget_show_all(GTK_WIDGET(paned_notebook->item_contents));
      mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);
      gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
    }

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL],
		0,
		item,
		widget);

  OCHU_OBJECT_UNREF(widget);

  page = mod_notebook_get_current_page(paned_notebook->item_contents);
  gtk_widget_set_sensitive(paned_notebook->left_arrow, page > 0);
  gtk_widget_set_sensitive(paned_notebook->right_arrow,
			   page < paned_notebook->number_of_items_shown - 1);

#if GTK_MINOR_VERSION <= 2
  gtk_container_remove(GTK_CONTAINER(paned_notebook->contents_list),
		       menu_item);
  gtk_option_menu_set_history(paned_notebook->contents_menu, 0);
#endif
}


static void
switch_page_cb(ModNotebook *notebook, ModNotebookPage *page, guint page_num,
	       PanedNotebook *paned_notebook)
{
  ItemViewRelation *relation;
#if GTK_MINOR_VERSION > 2
  GtkTreeIter iter;
  ItemViewRelation tmp_relation = { NULL, NULL, NULL, &iter };
#endif
  GtkWidget *new_current_page = mod_notebook_get_nth_page(notebook, page_num);
  GtkWidget *previous_page = paned_notebook->current_item_view;

  gtk_widget_set_sensitive(paned_notebook->left_arrow, page_num > 0);
  gtk_widget_set_sensitive(paned_notebook->right_arrow,
			   (page_num
			    < paned_notebook->number_of_items_shown - 1));

  if (previous_page == new_current_page)
    return;

#if GTK_MINOR_VERSION <= 2
  relation = find_item_view_relation(paned_notebook, new_current_page);
#else
  relation = find_item_view_relation(paned_notebook, new_current_page,
				     &tmp_relation);
#endif
  if (relation == NULL)
    return;

  paned_notebook->current_item = relation->item;
  paned_notebook->current_item_view = new_current_page;

#if GTK_MINOR_VERSION <= 2
  gtk_menu_reorder_child(paned_notebook->contents_list,
			 relation->menu_item, 0);
  gtk_option_menu_set_history(paned_notebook->contents_menu, 0);
#else
  gtk_combo_box_set_active_iter(paned_notebook->contents_menu, relation->iter);
#endif

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[PAGE_SWITCHED_SIGNAL],
		0,
		previous_page,
		new_current_page);

  gtk_widget_queue_resize(new_current_page);
}


static gboolean
button_press_event_cb(ModNotebook *notebook, GdkEventButton *event,
		      PanedNotebook *paned_notebook)
{
  int page;
  g_return_val_if_fail(notebook == paned_notebook->item_contents, FALSE);

  if (event->button == 2)
    {
      page = mod_notebook_get_mouse_event_page_num(notebook,
						   (GdkEventAny*)event);
      if (page != -1)
	{
	  close_nth_page(paned_notebook, page);
	  return TRUE;
	}
    }
  else if (event->button == 1)
    {
      if (event->type == GDK_2BUTTON_PRESS)
	{
	  /* ZZZ: ֥륯å٥ */
	  page = mod_notebook_get_mouse_event_page_num(notebook,
						       (GdkEventAny*)event);
	  if (page != -1)
	    {
	      ItemViewRelation *relation;
#if GTK_MINOR_VERSION > 2
	      ItemViewRelation tmp_relation = { NULL, NULL, NULL, NULL };
#endif
	      GtkWidget *widget
		= mod_notebook_get_nth_page(paned_notebook->item_contents,
					    page);
	      if (widget == NULL)
		return TRUE;

#if GTK_MINOR_VERSION <= 2
	      relation = find_item_view_relation(paned_notebook, widget);
#else
	      relation = find_item_view_relation(paned_notebook, widget,
						 &tmp_relation);
#endif
	      if (relation == NULL || relation->item_view != widget)
		return TRUE;

	      g_signal_emit(G_OBJECT(paned_notebook),
			    paned_notebook_signals[PAGE_DOUBLE_CLICKED_SIGNAL],
			    0,
			    relation->item,
			    relation->item_view);

	      return FALSE;
	    }
	}
    }

  return FALSE;
}


static void
close_button_clicked_cb(ModNotebook *notebook, PanedNotebook *paned_notebook)
{
  paned_notebook_close_current_item_view(paned_notebook);
}


#if GTK_MINOR_VERSION <= 2
static void
menu_changed_cb(GtkOptionMenu *option_menu, PanedNotebook *paned_notebook)
{
  ItemViewRelation *relation;
  GtkMenuItem *menu_item = paned_notebook->active_menu_item;
  int page;

  if (menu_item == NULL)
    return;

  paned_notebook->active_menu_item = NULL;

  relation = find_item_view_relation(paned_notebook, menu_item);
  if (relation == NULL)
    return;

  page = mod_notebook_page_num(paned_notebook->item_contents,
			       relation->item_view);
  if (page != -1)
    mod_notebook_set_current_page(paned_notebook->item_contents, page);
}


static void
menu_item_activate_cb(GtkMenuItem *menu_item, PanedNotebook *paned_notebook)
{
  paned_notebook->active_menu_item = menu_item;
}
#else
static void
menu_changed_cb(GtkComboBox *combo_box, PanedNotebook *paned_notebook)
{
  GtkWidget *item_view = NULL;
  GtkTreeIter iter;
  int page;

  if (gtk_combo_box_get_active_iter(combo_box, &iter) == FALSE)
    return;

  gtk_tree_model_get(GTK_TREE_MODEL(paned_notebook->contents_list), &iter,
		     NOTEBOOK_ITEM_VIEW, &item_view, -1);

  g_return_if_fail(item_view != NULL && GTK_IS_WIDGET(item_view));

  page = mod_notebook_page_num(paned_notebook->item_contents, item_view);
  if (page != -1)
    mod_notebook_set_current_page(paned_notebook->item_contents, page);
}
#endif


static void
go_left_button_cb(gpointer unused, PanedNotebook *paned_notebook)
{
  mod_notebook_prev_page(paned_notebook->item_contents);
}


static void
go_right_button_cb(gpointer unused, PanedNotebook *paned_notebook)
{
  mod_notebook_next_page(paned_notebook->item_contents);
}
