﻿using System;
using System.Collections.Generic;
using System.Text;

namespace NT2chCtrl.html.js
{


    abstract class JsFunction : JsVariant
    {
        public const int FUNC_PRIORITY_INVALID = -2;
        public const int FUNC_PRIORITY_ROOT = -1;
        public const int FUNC_PRIORITY_LOAD = 1;

        public const int FUNC_PRIORITY_COMMA = 10;
        public const int FUNC_PRIORITY_SEMICOLON = 11;
        
        public const int FUNC_PRIORITY_COMPARE = 100;

        public const int FUNC_PRIORITY_ADD = 1000;
        public const int FUNC_PRIORITY_SUB = 1000;
        public const int FUNC_PRIORITY_DIV = 1001;
        public const int FUNC_PRIORITY_MUL = 1001;
        public const int FUNC_PRIORITY_MINUS = 1002;
        public const int FUNC_PRIORITY_PLUS = 1002;

        public const int FUNC_PRIORITY_NOT = 1002;
        public const int FUNC_PRIORITY_INC = 1003;
        public const int FUNC_PRIORITY_DEC = 1003;

        public const int FUNC_PRIORITY_FUNCTION = int.MaxValue - 2;
        public const int FUNC_PRIORITY_L_BRACKET = int.MaxValue - 1;
        public const int FUNC_PRIORITY_R_BRACKET = int.MaxValue;

        protected static JsUndefValue JS_UNDEF_VALUE = new JsUndefValue();
        protected static JsNullValue JS_NULL_VALUE = new JsNullValue();

        protected bool mArranged = false;
        public virtual bool IsArranged() { return mArranged; }
        public virtual bool IsAssignable() { return false; }

        protected List<List<JsVariant>> mInstructions = new List<List<JsVariant>>();

        public abstract int getPriority();
        public virtual bool rearegeSquence(List<JsVariant> insList, int currIdx) { return false; }

        protected int mSourceLine;
        protected int mSourceIdx;

        public JsFunction(int lineNo, int srcIdx)
        {
            mSourceIdx = srcIdx;
            mSourceLine = lineNo;
        }

        public void setDebugContext(DebugContext dCtx)
        {
            dCtx.mCurentIndex = mSourceIdx;
            dCtx.mCurrentLine = mSourceLine;
        }

        public virtual JsVariant Execute(JsFunctionContext ctx, HtmlElement rootElement)
        {  
            DebugContext dCtx = ctx.getDbgCtx();
            if (dCtx != null)
            {
                dCtx.mCurrentLine = mSourceLine;
                dCtx.mCurentIndex = mSourceIdx;
            }
            return null;
        }

        public virtual JsVariant Execute(JsFunctionContext ctx, HtmlElement rootElement, JsVariant assignVal){return null;}

        protected static bool rearangeInstructions(List<JsVariant> instructions)
        {
            JsFunction func;
            int maxPriority = -1;
            int maxIdx = -1;
            int count = instructions.Count;
            for (int i = 0; i < count; i++)
            {
                func = instructions[i] as JsFunction;
                if (func == null)
                    continue;
                if (func.IsArranged())
                    continue;

                int priority = func.getPriority();
                if (maxPriority < priority)
                {
                    maxPriority = priority;
                    maxIdx = i;
                }
            }
            if (maxIdx < 0)
                return true;
            func = instructions[maxIdx] as JsFunction;
            if (func == null)
                return false;
            if (!func.rearegeSquence(instructions, maxIdx))
                return false;

            return rearangeInstructions(instructions);
        }

    }

    class JsEmptyFunction : JsFunction
    {
        public JsEmptyFunction(int lineNo, int srcIdx)
            : base(lineNo, srcIdx)
        {
            mArranged = true;
        }
        public override int getPriority()
        {
            return FUNC_PRIORITY_INVALID;
        }
        public override JsVariant Execute(JsFunctionContext ctx, HtmlElement rootElement)
        {
            base.Execute(ctx, rootElement);
            return JsUndefValue.getConstant();
        }

    }

    class JsDebuggerFunction : JsFunction
    {
        public JsDebuggerFunction(int lineNo, int srcIdx)
            : base(lineNo, srcIdx)
        {
            mArranged = true;
        }
        public override int getPriority()
        {
            return FUNC_PRIORITY_INVALID;
        }

    }

    class JsLoopBreak : JsFunction
    {
        public JsLoopBreak(int lineNo, int srcIdx)
            : base(lineNo, srcIdx)
        {
            mArranged = true;
        }
        public override int getPriority()
        {
            return FUNC_PRIORITY_INVALID;
        }
    }
    class JsLoopContinue : JsFunction
    {
        public JsLoopContinue(int lineNo, int srcIdx)
            : base(lineNo, srcIdx)
        {
            mArranged = true;
        }
        public override int getPriority()
        {
            return FUNC_PRIORITY_INVALID;
        }
    }


    partial class JsRootFunction : JsFunction
    {

        public JsRootFunction(int lineNo, int srcIdx)
            : base(lineNo, srcIdx)
        {
        }

        public override int getPriority()
        {
            return FUNC_PRIORITY_ROOT;
        }


        public bool Parse(DebugContext ctx, string source)
        {
            int startIdx = 0;
            JsTokenStruct sToken;
            int length = source.Length;
            int endIdx;
            int bracket = 0;
            bool inside_if = false;
            bool inside_for = false;
            
            List<JsVariant> instruction = new List<JsVariant>();

            int state = 0;

            do
            {
                endIdx = JsParser.getJsToken(ctx, source, startIdx, out sToken);
                if (endIdx < 0)
                    break;
                startIdx = endIdx;


                switch (sToken.type)
                {
                    case JsToken.TYPE_ERROR:
                        return false;
                    case JsToken.TYPE_KEYWORD:
                        switch (sToken.keyword)
                        {
                            case JsToken.KEY_VAR:
                                if (state != 0 && state != 9 && state != 11)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                state = 1;
                                break;
                            case JsToken.KEY_TRUE:
                                if (state == 9 || state == 4)
                                {
                                    instruction.Add(new JsBooleanValue(true));
                                    state = 5;
                                }
                                else
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                break;
                            case JsToken.KEY_FALSE:
                                if (state == 9 || state == 4)
                                {
                                    instruction.Add(new JsBooleanValue(false));
                                    state = 5;
                                }
                                else
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                break;
                            case JsToken.KEY_NULL:
                                if (state == 9 || state == 4)
                                {
                                    instruction.Add(new JsNullValue());
                                    state = 5;
                                }
                                else
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                break;
                            case JsToken.KEY_UNDEFINED:
                                if (state == 9 || state == 4)
                                {
                                    instruction.Add(new JsUndefValue());
                                    state = 5;
                                }
                                else
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;

                        }
                        break;
                    case JsToken.TYPE_OP_KEYWORD:
                        switch (sToken.keyword)
                        {
                            case JsToken.KEY_OP_FOR:
                                if (state != 0 && state != 11)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (state == 0 && instruction.Count > 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsOpFunction_for(ctx.mCurrentLine, startIdx));
                                state = 8;
                                inside_for = true;
                                break;
                            case JsToken.KEY_OP_IF:
                                if (state != 0 && state != 11)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (state == 0 && instruction.Count > 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsOpFunction_if(ctx.mCurrentLine, startIdx));
                                state = 8;
                                inside_if = true;
                                break;
                            case JsToken.KEY_OP_ELSE:
                                if (state != 0)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (instruction.Count > 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    } 
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsOpFunction_else(ctx.mCurrentLine, startIdx));
                                mInstructions.Add(instruction);
                                instruction = new List<JsVariant>();
                                state = 11;
                                inside_if = true;
                                break;
                            case JsToken.KEY_OP_LOOP_BREAK:
                                if (state != 0 && state != 11 && state != 15)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (instruction.Count > 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    } 
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsLoopBreak(ctx.mCurrentLine, startIdx));
                                mInstructions.Add(instruction);
                                instruction = new List<JsVariant>();
                                state = 16;
                                break;
                            case JsToken.KEY_OP_LOOP_CONTINUE:
                                if (state != 0 && state != 11 && state != 15)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (instruction.Count > 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsLoopContinue(ctx.mCurrentLine, startIdx));
                                mInstructions.Add(instruction);
                                instruction = new List<JsVariant>();
                                state = 16;
                                break;
                            case JsToken.KEY_OP_DEBUGGER:
                                if (state != 0 && state != 11 && state != 15)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (instruction.Count > 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsDebuggerFunction(ctx.mCurrentLine, startIdx));
                                mInstructions.Add(instruction);
                                instruction = new List<JsVariant>();
                                state = 16;
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;
                        }
                        break;
                    case JsToken.TYPE_DOM_KEYWORD:
                        switch (sToken.keyword)
                        {
                            case JsToken.KEY_DOM_DOCUMENT:
                                if (state == 0 || state == 4 || state ==8 || state == 10 || state == 11)
                                {
                                    state = 6;
                                }
                                else
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                break;
                            case JsToken.KEY_DOM_getElementById:
                                if (state != 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsDomFunction(sToken.keyword, ctx.mCurrentLine, startIdx));
                                state = 8;
                                break;
                            case JsToken.KEY_DOM_innerHTML:
                                if (state != 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsDomFunction(sToken.keyword, ctx.mCurrentLine, startIdx));
                                state = 2;
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;
                        }
                        break;
                    case JsToken.TYPE_DOM_STR_KEYWORD:
                        switch (sToken.keyword)
                        {
                            case JsToken.KEY_DOM_STR_indexOf:
                                if (state != 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsDomStrFunc_indexOf(ctx.mCurrentLine, startIdx));
                                state = 8;
                                break;
                            case JsToken.KEY_DOM_STR_substring:
                                if (state != 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsDomStrFunc_substring(ctx.mCurrentLine, startIdx));
                                state = 8;
                                break;
                            case JsToken.KEY_DOM_STR_replace:
                                if (state != 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsDomStrFunc_replace(ctx.mCurrentLine, startIdx));
                                state = 8;
                                break;
                            case JsToken.KEY_DOM_STR_length:
                                if (state != 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsDomStrFunc_length(ctx.mCurrentLine, startIdx));
                                state = 5;
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;
                        }
                        break;
                    case JsToken.TYPE_VARIABLENAME:
                        switch (state)
                        {
                            case 0:
                            case 4:
                            //case 8:
                            case 9:
                            case 10:
                            case 11:
                            case 12:
                            case 13:
                            case 15:
                                instruction.Add(new JsIdentifier(sToken.sValue, false));
                                break;
                            case 1:
                                instruction.Add(new JsIdentifier(sToken.sValue, true));
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;
                        }
                        state = 2;
                        break;
                    case JsToken.TYPE_PUNCTUATOR:
                        switch (sToken.keyword)
                        {
                            case JsToken.KEY_OP_CMP_LT:
                            case JsToken.KEY_OP_CMP_GT:
                            case JsToken.KEY_OP_CMP_LT_EQUAL:
                            case JsToken.KEY_OP_CMP_GT_EQUAL:
                            case JsToken.KEY_OP_CMP_EQUAL:
                            case JsToken.KEY_OP_CMP_STRICT_EQUAL:
                            case JsToken.KEY_OP_CMP_NOT_EQUAL:
                            case JsToken.KEY_OP_CMP_STRICT_NOT_EQUAL:
                                if ((state != 0 && state != 2 && state != 5) || instruction.Count == 0)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsCmpFunction(ctx.mCurrentLine, startIdx,
                                                            sToken.keyword));
                                state = 10;
                                break;
                            case JsToken.PUNC_L_CURLY_BRACKET:
                                if (state == 4 || state == 5 || state == 6 || state == 7 || state == 8)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                else if (!rearangeInstructions(instruction))
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (instruction.Count > 0)
                                {
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsL_CURLY_Bracket(ctx.mCurrentLine, startIdx));
                                mInstructions.Add(instruction);
                                instruction = new List<JsVariant>();
                                state = 0;
                                break;
                            case JsToken.PUNC_R_CURLY_BRACKET:
                                if (state == 4 || state == 6 || state == 7)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                else if (!rearangeInstructions(instruction))
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (instruction.Count > 0)
                                {
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                }
                                instruction.Add(new JsR_CURLY_Bracket(ctx.mCurrentLine, startIdx));
                                mInstructions.Add(instruction);
                                instruction = new List<JsVariant>();
                                state = 0;
                                break;
                            case JsToken.PUNC_SEMICOL0N:
                                if (bracket > 0)
                                {
                                    if (state != 2 && state != 5 && state != 9 && state != 15)
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    instruction.Add(new JsSemiColonFunction(ctx.mCurrentLine, startIdx));
                                    state = 9;
                                }
                                else
                                {
                                    if (state == 4 || state == 6 || state == 7)
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    else if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    if (instruction.Count > 0)
                                    {
                                        mInstructions.Add(instruction);
                                        instruction = new List<JsVariant>();
                                    }
                                    state = 0;
                                }
                                break;
                            case JsToken.PUNC_EQUAL:
                                switch (state)
                                {
                                    case 0:
                                    case 1:
                                    case 5:
                                    case 6:
                                    case 7:
                                    //case 8: document.
                                    case 9:
                                    case 10:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    case 2:
                                        instruction.Add(new JsLoadFunction(ctx.mCurrentLine, startIdx));
                                        state = 4;
                                        break;
                                }
                                break;
                            case JsToken.PUNC_PERIOD:
                                if (state != 2 && state != 6)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                state = 7;
                                break;
                            case JsToken.PUNC_COMMA:
                                if (state != 2 && state != 5)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsCommaFunction(ctx.mCurrentLine, startIdx));
                                state = 9;
                                break;
                            case JsToken.PUNC_L_BRACKET:
                                if (state != 4 && state != 8 && state != 12)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsLBracket(ctx.mCurrentLine, startIdx));
                                bracket++;
                                state = 9;
                                break;
                            case JsToken.PUNC_R_BRACKET:
                                bracket--;
                                if (bracket < 0)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                if (state == 4)
                                {
                                    ctx.setCurrentIdx(startIdx);
                                    return false;
                                }
                                instruction.Add(new JsRBracket(ctx.mCurrentLine, startIdx));
                                if ((inside_if || inside_for) && bracket == 0)
                                {
                                    if (!rearangeInstructions(instruction))
                                    {
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                    }
                                    mInstructions.Add(instruction);
                                    instruction = new List<JsVariant>();
                                    inside_if = false;
                                    inside_for = false;
                                }
                                state = 15;
                                break;
                            case JsToken.PUNC_INC:
                                switch (state)
                                {
                                    case 0:
                                    case 4:
                                    case 9:
                                    case 10:
                                    case 12:
                                        instruction.Add(new JsIncrementFunction(ctx.mCurrentLine, startIdx));
                                        state = 14;
                                        break;
                                    case 2:
                                        instruction.Add(new JsIncrementFunction(ctx.mCurrentLine, startIdx));
                                        state = 2;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                            case JsToken.PUNC_DEC:
                                switch (state)
                                {
                                    case 0:
                                    case 4:
                                    case 9:
                                    case 10:
                                    case 12:
                                        instruction.Add(new JsDecrementFunction(ctx.mCurrentLine, startIdx));
                                        state = 14;
                                        break;
                                    case 2:
                                        instruction.Add(new JsDecrementFunction(ctx.mCurrentLine, startIdx));
                                        state = 2;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                            case JsToken.PUNC_OP_NOT:
                                switch (state)
                                {
                                    case 0:
                                    case 4:
                                    case 9:
                                    case 10:
                                    case 12:
                                        instruction.Add(new JsNotFunction(ctx.mCurrentLine, startIdx));
                                        state = 14;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                            case JsToken.PUNC_ADD:
                                switch (state)
                                {
                                    case 0:
                                    case 4:
                                    case 9:
                                    case 10:
                                    case 12:
                                        instruction.Add(new JsPlusFunction(ctx.mCurrentLine, startIdx));
                                        state = 13;
                                        break;
                                    case 2:
                                    case 5:
                                    case 15:
                                        instruction.Add(new JsAddFunction(ctx.mCurrentLine, startIdx));
                                        state = 12;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                            case JsToken.PUNC_SUB:
                                switch (state)
                                {
                                    case 0:
                                    case 4:
                                    case 9:
                                    case 10:
                                    case 12:
                                        instruction.Add(new JsMinusFunction(ctx.mCurrentLine, startIdx));
                                        state = 13;
                                        break;
                                    case 2:
                                    case 5:
                                    case 15:
                                        instruction.Add(new JsSubFunction(ctx.mCurrentLine, startIdx));
                                        state = 12;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                            case JsToken.PUNC_MUL:
                                switch (state)
                                {
                                    case 2:
                                    case 5:
                                    case 15:
                                        instruction.Add(new JsMultipleFunction(ctx.mCurrentLine, startIdx));
                                        state = 12;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                            case JsToken.PUNC_DIV:
                                switch (state)
                                {
                                    case 2:
                                    case 5:
                                    case 15:
                                        instruction.Add(new JsDivideFunction(ctx.mCurrentLine, startIdx));
                                        state = 12;
                                        break;
                                    default:
                                        ctx.setCurrentIdx(startIdx);
                                        return false;
                                }
                                break;
                        }
                        break;
                    case JsToken.TYPE_NUMBER_LITERAL:
                        switch (state)
                        {
                            case 4:
                            case 9:
                            case 10:
                            case 12:
                            case 13:
                                instruction.Add(new JsNumberValue(sToken.dValue));
                                state = 5;
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;
                        }
                        break;
                    case JsToken.TYPE_STRING_LITERAL:
                        switch (state)
                        {
                            case 4:
                            case 9:
                            case 10:
                            case 12:
                            case 13:
                                instruction.Add(new JsStringValue(sToken.sValue));
                                state = 5;
                                break;
                            default:
                                ctx.setCurrentIdx(startIdx);
                                return false;
                        }
                        break;
                    case JsToken.TYPE_END_SRC:
                        endIdx = length; 
                        break;
                }
            } while (length > endIdx);


            switch (state)
            {
                case 1:
                case 4:
                case 6:
                case 7:
                case 8:
                case 9:
                case 10:
                case 11:
                    ctx.setCurrentIdx(startIdx);
                    return false;
            }
            if (bracket > 0)
            {
                ctx.setCurrentIdx(startIdx);
                return false;
            }
            if (instruction.Count > 0)
            {
                if (!rearangeInstructions(instruction))
                {
                    ctx.setCurrentIdx(startIdx);
                    return false;
                }
                mInstructions.Add(instruction);
            }
            return true;
        }

        public int getInstaructionCount()
        {
            return mInstructions.Count;
        }


    }
 

}
