//---------------------------------------------------------------------
// I2C for EDU2(ATmega64)
// by takuya matsubara
// http://nicotak.com

#include <avr/io.h>

char i2c_wait(void);
void i2c_start(void);
void i2c_stop(void);
void i2c_writebyte(unsigned char dat);
unsigned char i2c_readbyte(char noack);

#ifndef F_CPU
	#define F_CPU    8000000    //CPUNbNg 8MHz
#endif

#define BITRATE 100000		// [bit/sec] @2.7V
#define SOFTI2C 0
// 1=\tgEFAI2CʐM
// 0=pWX^g܂B

#if SOFTI2C

	#define I2C_PORT   PORTD
	#define I2C_PIN    PIND
	#define I2C_DDR	DDRD

	#define SCL		0
	#define SDA		1

//------------------------------------EGCg
void i2c_delay(void)
{
	int i=20;

	while(i-- > 0){
	}
}

//------------------------------------SCLHigh
void i2c_scl_high(void)
{
	I2C_PORT |= (1<<SCL);
	i2c_delay();
}

//------------------------------------SCLLow
void i2c_scl_low(void)
{
	I2C_PORT &= ~(1<<SCL);
	i2c_delay();
}

//------------------------------------SDAHigh
void i2c_sda_high(void)
{
	I2C_PORT |= (1<<SDA);
	I2C_DDR |= (1<<SDA);
	i2c_delay();
}

//------------------------------------SCLLow
void i2c_sda_low(void)
{
	I2C_PORT &= ~(1<<SDA);
	I2C_DDR |= (1<<SDA);
	i2c_delay();
}


//------------------------------------M(:Mf[^CMrbg)
void i2c_write_bit(unsigned char data,char cnt)
{
	while(cnt > 0)
	{
		if(data & 0x80)
			i2c_sda_high();
		else
			i2c_sda_low();

		i2c_scl_high();
		i2c_scl_low();
		data <<= 1;
		cnt--;
	}
}


//------------------------------------M(:Mrbg)
unsigned char i2c_read_bit(char cnt)
{
	unsigned char data=0;

	I2C_DDR &= ~(1<<SDA);
	while(cnt > 0)
	{
		data <<=1;
		i2c_scl_high();
		if (I2C_PIN & (1<<SDA))
			data |= 1;

		i2c_scl_low();
		cnt--;
	}
	return(data);
}
#else
//---------------------------------------------------------------------
char i2c_wait(void)
{
	unsigned int timeout=30000;

	while((TWCR & (1 << TWINT))==0){
		timeout--;
		if(timeout==0)
			return(-1);	//timeout error
	}

	switch(TWSR)
	{
	case 0x20:	//TW_MT_SLA_NACK
	case 0x58:	//TW_MR_DATA_NACK
    case 0x38:	//TW_MT_ARB_LOST
		return(-1);
		break;
	default:
		break;
	}
	return(0);
}


#endif //SOFTI2C
//---------------------------------------------------------------------
void i2c_init(void)
{
#if SOFTI2C
	i2c_sda_high();
	i2c_scl_high();
	I2C_DDR |= (1<<SCL);// SCL direction = output
#else
	TWSR = 0;	//vXP[l CLK/1 
	TWBR = ((F_CPU / BITRATE) - 16) / (2*1);	//  
	TWCR = (1 << TWEN);
#endif
}


//---------------------------------------------------------------------
void i2c_start(void)
{
#if SOFTI2C
	i2c_sda_high();
	i2c_sda_low();
	i2c_scl_low();
#else
	TWCR = (1<<TWINT) | (1<<TWSTA) | (1<<TWEN);	//start
	i2c_wait();
#endif
}


//---------------------------------------------------------------------
void i2c_stop(void)
{
#if SOFTI2C
	i2c_sda_low();
	i2c_scl_high();
	i2c_sda_high();
#else
	TWCR = (1<<TWINT) | (1<<TWSTO) | (1<<TWEN);	//stop
#endif
}


//---------------------------------------------------------------------
void i2c_writebyte(unsigned char dat)
{
#if SOFTI2C
	// 1oCgM+ACKM
	i2c_write_bit(dat,8);

	// read ack.
	i2c_read_bit(1);
#else
	TWDR = dat;
	TWCR = (1<<TWINT) | (1<<TWEN);	//
	i2c_wait();
#endif
}



//---------------------------------------------------------------------
unsigned char i2c_readbyte(char noack)
{
#if SOFTI2C
	unsigned char rxdata;

	rxdata = i2c_read_bit(8);

	if(noack)
		i2c_write_bit(0xff,1);	// NACK
	else
		i2c_write_bit(0x00,1);	// ACK

	return(rxdata);
#else
	if (noack) {
	   TWCR = (1<<TWINT)|(1<<TWEN);	//NO ACK
	} else {
	   TWCR = (1<<TWINT)|(1<<TWEN)|(1<<TWEA);	//ACK
	}
	i2c_wait();

	return (TWDR);
#endif
}



//---------------------------------------------------------------------
//
//20msec炢҂
void i2c_eeprom_write(
	unsigned char i2c_addr,
	unsigned long addr,
	unsigned char* data,
	unsigned char bytecnt)
{
	if(addr & 0x10000)	i2c_addr |= 0x2;
	i2c_start();
	i2c_writebyte(i2c_addr);	// SLA + W
	i2c_writebyte(addr >> 8);	// address(high 8bits)
	i2c_writebyte(addr & 0xff);	// address(low 8bits)
	while(bytecnt > 0){
		i2c_writebyte(*data++);
		bytecnt--;
	}
	i2c_stop();
}


//---------------------------------------------------------------------
void i2c_eeprom_read(
	unsigned char i2c_addr,
	unsigned long addr,
	unsigned char* data,
	unsigned char bytecnt)
{
	if(addr & 0x10000)	i2c_addr |= 0x2;

	i2c_start();
	i2c_writebyte(i2c_addr);	// SLA + W 
	i2c_writebyte(addr >> 8);	// address(high 8bits)
	i2c_writebyte(addr & 0xff);	// address(low 8bits)
#if SOFTI2C
	i2c_sda_high();
	i2c_scl_high();
	i2c_delay();
#endif
	i2c_start();	// Đڑ
	i2c_writebyte(i2c_addr | 0x01);	// SLA + R 
	while(bytecnt > 0){
		*data++ = i2c_readbyte(bytecnt == 1);
		bytecnt--;
	}
	i2c_stop();
};


//                   b7    b6    b5    b4    b3    b2    b1    b0  
// 00 Control1       TEST  0     STOP  0     TEST  0     0     0   
// 01 Control2       0     x     0     TI/TP AF    TF    AIE   TIE 
// 02 Seconds        VL    S40   S20   S10   S8    S4    S2    S1  
// 03 Minutes        x     min40 min20 min10 min8  min4  min2  min1
// 04 Hours          x     x     h20   h10   h8    h4    h2    h1  
// 05 Days           x     x     d20   d10   d8    d4    d2    d1  
// 06 DayofWeek      x     x     x     x     x     W4    W2    W1  
// 07 Months/Century C     x     x     Mon10 Mon8  Mon4  Mon2  Mon1
// 08 Years          Y80   Y40   Y20   Y10   Y8    Y4    Y2    Y1  
// 09 Min. Alarm     AE    m40   m20   m10   m8    m4    m2    m1  
// 0A Hour Alarm     AE    x     h20   h10   h8    h4    h2    h1  
// 0B Day            AE    x     d20   d10   d8    d4    d2    d1  
// 0C Weekday        AE    x     x     x     x     w4    w2    w1  
// 0D CLKOUT freq.   FE    x     x     x     x     x     FD1   FD0 
// 0E Timer Control  TE    x     x     x     x     x     TD1   TD0 
// 0F Timer          128   64    32    16    8     4     2     1   

#if USE_RTC

void rtc_write(
	unsigned char i2c_addr,
	unsigned char addr,
	unsigned char* data,
	unsigned char bytecnt)
{
	i2c_start();
	i2c_writebyte(i2c_addr);	// SLA + W
	i2c_writebyte(addr);
	while(bytecnt > 0){
		i2c_writebyte(*data++);
		bytecnt--;
	}
	i2c_stop();
}

void rtc_read(
	unsigned char i2c_addr,
	unsigned char addr,
	unsigned char* data,
	unsigned char bytecnt)
{
	i2c_start();
	i2c_writebyte(i2c_addr);		// SLA + W
	i2c_writebyte(addr);	// address(8bits)

	i2c_start();
	i2c_writebyte(i2c_addr | 0x01);	// SLA + R
	while(bytecnt > 0){
		*data++ = i2c_readbyte(bytecnt == 1);
		bytecnt--;
	}
	i2c_stop();
};

#endif  //USE_RTC
