/*
 * time.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"types.h"
#include"lib.h"
#include"errno.h"
#include"interrupt.h"
#include"proc.h"
#include"sp.h"
#include"mp.h"
#include"lock.h"
#include"time.h"


enum{
	MAX_TIMER=4,			/* ݻޡ */

	/* Timer type */
	TYP_INTERVAL=1,			/* Interval timer */
	TYP_REAL=2,				/* Real-clock timer */
};


typedef struct{
	PROC *wait_next;	/* ȥ󥯤μΥץ,PROCnextƱ */
	PROC *wait_prev;	/* ȥ󥯤Υץ,PROCprevƱ */
	int timer_count;	/* Counter for timer */
	int lock_gate;
}TIMER_LINK;


int clock_1micro;						/* 1micros */
int clock_1m;							/* 1ms */


/* ʿʿѴ롣 */
extern inline uchar bcd_to_bin(uchar value)
{
	return value-6*(value>>4);
}



/***************************************************************************************************
 *
 * 8254 interval timer
 *
 ***************************************************************************************************/

static uint tasking_count=				/* 󥰥ץåѥޡ */
	PIT_HZ/(1000/TASK_TIME);
static TIMER_LINK interval_timer=		/* Interval timer link queue */
	{(PROC*)&interval_timer,(PROC*)&interval_timer,0,0};
static int interval_timer_num=0;		/* Number of used interval timer */
static int interval_current_count=0;	/* Interval timer current count */


/*
 * 8254 interval timer handler
 */
static int interval_timer_handler()
{
	PROC *p;

/****************************************************************/
	printk("Interrupt IRQ0\n");
/*****************************************************************/
	enter_spinlock(&interval_timer.lock_gate);
	{
		/* ॢȤΥץ褵 */
		for(p=interval_timer.wait_next;p->timer_count==interval_current_count;p=p->wait_next)
		{
			p->timer_count=0;
			p->wait_prev->wait_next=p->wait_next;
			p->wait_next->wait_prev=p->wait_prev;
			add_to_schedule(p);
			--interval_timer_num;
		}

		/* No timer */
		if(interval_timer_num==0)
		{
			if(MFPS_addres==0)
			{
				tasking_count=PIT_HZ/(1000/TASK_TIME);
				init_sp_task();
			}
		}
		else
		{
			/* ƥץΥȤ򸺤餹 */
			for(p=interval_timer.wait_next;p!=(PROC*)&interval_timer;p=p->wait_next)
				p->timer_count-=interval_current_count;

			if(interval_timer.wait_prev->timer_count>0x10000)interval_current_count=0x10000;
			else interval_current_count=interval_timer.wait_prev->timer_count;

			if(MFPS_addres==0)
			{
				if((tasking_count-=interval_current_count)==0)
					tasking_count=PIT_HZ/(1000/TASK_TIME);

				if(tasking_count<interval_current_count)interval_current_count=tasking_count;
			}

			/* ޡꤹ */
			outb(PIT_CNR0,interval_current_count&0xff);
			outb(PIT_CNR0,interval_current_count>>8);
		}
	}
	exit_spinlock(&interval_timer.lock_gate);

	return 1;
}


/*
 * Set interval timer
 * parameters : time(msñ)
 * rturn : Task switch flag or error number
 */
static int set_interval_timer(int time)
{
	uint count;
	PROC *proc;
	PROC *p;


	if(interval_timer_num==MAX_TIMER)return -ENOTIMER;

	proc=cputask[get_current_cpu()].current_task;

	time=PIT_HZ*time/1000;				/* åѴ */

	cli();
	enter_spinlock(&interval_timer.lock_gate);
	{
		if(interval_timer_num==0)
		{
			proc->timer_count=time;
			interval_timer.wait_next=proc;
			interval_timer.wait_prev=proc;
			proc->wait_next=(PROC*)&interval_timer;
			proc->wait_prev=(PROC*)&interval_timer;

			if(MFPS_addres==0)
			{
				irq_entry[IRQ0]=interval_timer_handler;
				outb(PIT_CTR,0x30);					/* One shot timer(mode 0) */
				if(time>PIT_HZ/(1000/TASK_TIME))
					time=PIT_HZ/(1000/TASK_TIME);
			}

			/* ޡ */
			if(time>0x10000)interval_current_count=0x10000;
			else interval_current_count=time;
			outb(PIT_CNR0,interval_current_count&0xff);
			outb(PIT_CNR0,interval_current_count>>8);
		}
		else
		{
			count=inb(PIT_CNR0);
			count+=inb(PIT_CNR0)<<8;
			proc->timer_count=time+count;

			/* ޡ󥯤³ */
			for(p=interval_timer.wait_next;p!=(PROC*)&interval_timer;p=p->wait_next)
				if(p->timer_count>=proc->timer_count)break;
			proc->wait_next=p->wait_next;
			proc->wait_prev=p;
			p->wait_next->wait_prev=proc;
			p->wait_next=proc;

			if(proc->timer_count<interval_current_count)
			{
				/* ޡ */
				interval_current_count=proc->timer_count;
				outb(PIT_CNR0,time&0xff);
				outb(PIT_CNR0,time>>8);
			}
		}

		++interval_timer_num;
	}
	exit_spinlock(&interval_timer.lock_gate);
	sti();

	proc->timer_type=TYP_INTERVAL;

	return 1;
}


/*
 * Delete interval timer from link
 * parameters : Process number
 */
static void del_interval_timer(PROC *proc)
{
	if(interval_timer_num==0)return;

	cli();
	enter_spinlock(&interval_timer.lock_gate);
	{
		proc->timer_count=0;
		proc->wait_prev->wait_next=proc->wait_next;
		proc->wait_next->wait_prev=proc->wait_prev;

		if(--interval_timer_num==0)
		{
			if(MFPS_addres==0)
			{
				tasking_count=PIT_HZ/(1000/TASK_TIME);
				init_sp_task();
			}
			else outb(PIT_CTR,0x30);		/* ޡȥå */
		}
	}
	exit_spinlock(&interval_timer.lock_gate);
	sti();
}


/***************************************************************************************************
 *
 * MC146818 real-clock timer
 *
 ***************************************************************************************************/

enum{
	REAL_TIMER_RATE=500,	/* Real-clock timer rate(ms) */
};


static TIMER_LINK real_timer=					/* Real timer link queue */
	{(PROC*)&real_timer,(PROC*)&real_timer,0,0};
static int real_timer_num=0;					/* Number of used real timer */
static uint utc;								/* current UTC(number of seconds since 00:00:00.1.1.1970). */


/*
 * MC146818 real timer handler
 * return : ץåʤ
 */
static int real_timer_handler()
{
	static int half=0;
	PROC *p;


/*********************************************
	printk("Interrupt IRQ8\r");
**********************************************/

	/* Status register C ɤǳߤꥻåȤ롣 */
	read_cmos(CMOS_STRC);

	/* ॢȥץĴ٤ */
	if(real_timer_num>0)
	{
	/*	cli();*/
		enter_spinlock(&real_timer.lock_gate);
		{
			for(p=real_timer.wait_next;p!=(PROC*)&real_timer;p=p->wait_next)
				if((p->timer_count-=REAL_TIMER_RATE)<=0)		/* ॢ */
				{
					p->wait_prev->wait_next=p->wait_next;
					p->wait_next->wait_prev=p->wait_prev;
					--real_timer_num;
				}

			/*if(real_timer_num==0)
				write_cmos(CMOS_STRB,0x2);*/	/* ߥ */
		}
		exit_spinlock(&real_timer.lock_gate);
	/*	sti();*/
	}

	/* UTC򹹿롣 */
	half^=1;				/* 500msȹ */
	utc+=half;

	return 0;
}


/*
 * Set real_clock timer
 * parameters : time(msñ)
 * rturn : Task switch flag or error number
 */
static int set_real_timer(int time)
{
	PROC *proc;


	if(real_timer_num==MAX_TIMER)return -ENOTIMER;

	proc=cputask[get_current_cpu()].current_task;

	proc->timer_count=time;

	/* 󥯤³ */
/*	cli();*/
	enter_spinlock(&real_timer.lock_gate);
	{
		proc->wait_next=real_timer.wait_next;
		proc->wait_prev=(PROC*)&real_timer;
		real_timer.wait_next->wait_prev=proc;
		real_timer.wait_next=proc;

		++real_timer_num;
/*		if(++real_timer_num==1)
			write_cmos(CMOS_STRB,0x42);*/		/* ߥ */
	}
	exit_spinlock(&real_timer.lock_gate);
/*	sti();*/

	proc->timer_type=TYP_REAL;

	return 1;
}


/*
 * Delete real-clock timer
 * prameters : process
 */
static void del_real_timer(PROC *proc)
{
	if(real_timer_num==0)return;

/*	cli();*/
	enter_spinlock(&real_timer.lock_gate);
	{
		proc->wait_prev->wait_next=proc->wait_next;
		proc->wait_next->wait_prev=proc->wait_prev;

		--real_timer_num;
/*		if(--real_timer_num==0)
			write_cmos(CMOS_STRB,0x2);*/		/* ߥ */
	}
	exit_spinlock(&real_timer.lock_gate);
/*	sti();*/
}


/*
 * get current time from cmos
 * parameters : time structure
 */
static void get_time_from_cmos()
{
	static ushort sumMonthDay[13]=				/* ιס */
		{0,0,31,59,90,120,151,181,212,243,273,304,334};
	
	struct TIME{
		uchar second;		/* seconds */
		uchar minute;		/* minutes */
		uchar hour;			/* hours */
		uchar day;			/* days */
		uchar month;		/* months */
		ushort year;		/* years */
	}time;
	int days;


	do
	{
		time.second=bcd_to_bin(read_cmos(CMOS_SCD));
		time.minute=bcd_to_bin(read_cmos(CMOS_MNT));
		time.hour=bcd_to_bin(read_cmos(CMOS_HOR));
		time.day=bcd_to_bin(read_cmos(CMOS_DAY));
		time.month=bcd_to_bin(read_cmos(CMOS_MTH));
		time.year=bcd_to_bin(read_cmos(CMOS_YER));
	}while(bcd_to_bin(read_cmos(CMOS_SCD))<time.second);

	time.year+=(time.year<70)?2000:1900;

	/* UTCη׻ */
	days=(time.year-1970)*365;
	days+=(time.year-1-1970)/4;
	days+=sumMonthDay[time.month]+time.day-1;
	if((time.year%4==0)&&(time.month>2))++days;
	utc=((days*24+time.hour)*60+time.minute)*60+time.second;
}


/***************************************************************************************************
 *
 * misc functions
 *
 ***************************************************************************************************/

/*
 * cpuå¬
 * ֤ : å
 */
int count_cpu_clock()
{
	uint cpu_clock;
	int low,high;
	long long r_beg,r_end;
	double time;


 	outb(PIT_CTR,0x30);	/* CLK0,LSB.MSB,⡼0,binary */
	outb(PIT_CNR0,0);	/* LSB */
	outb(PIT_CNR0,0);	/* MSB */

 	r_beg=rdtsc();		/* clock counter */

	for(;;)if(rdtsc()-r_beg>0x400000)break;		/* ٱ */

	r_end=rdtsc();

	low=inb(PIT_CNR0);
	high=inb(PIT_CNR0);

	time=(0xffff-low-(high<<8))/(double)PIT_HZ;
	cpu_clock=(r_end-r_beg)/time;

 	clock_1micro=cpu_clock/1000000+1;		/* 1micros */
 	clock_1m=cpu_clock/1000+1;				/* 1ms */

	return cpu_clock;
}


/*
 * Set timer
 * parameters : time(msñ)
 * rturn : Task switch flag or error number
 */
int set_timer(int time)
{
	if(time==0)return 0;

	if(time<REAL_TIMER_RATE)return set_interval_timer(time);
	else return set_real_timer(time);
}


/*
 * Delete timer from timer link
 * parameters : Process number
 */
void del_timer(PROC *proc)
{
	if(proc->timer_type==TYP_INTERVAL)del_interval_timer(proc);
	if(proc->timer_type==TYP_REAL)del_real_timer(proc);
}


/***************************************************************************************************
 *
 * < ƥॳ >
 *
 ***************************************************************************************************/

/*
 * get Coordinated Universal Time(number of seconds since 00:00:00.1.1.1970)
 */
int sys_time()
{
	return (int)utc;
}


/***************************************************************************************************
 *
 * 
 *
 ***************************************************************************************************/


/*
 * Init time
 */
void init_time()
{
	/* ޥץå */
	if(MFPS_addres)
	{
		irq_entry[IRQ0]=interval_timer_handler;
		outb(PIT_CTR,0x30);						/* One shot timer(mode 0) */
		release_irq_mask(0);
	}

	/* Init MC146818 real time clock */
	irq_entry[IRQ8]=real_timer_handler;
	get_time_from_cmos();				/* get current time from cmos */
	write_cmos(CMOS_STRA,0x2f);			/* time bese 0x010(default)|ޡ졼65536/(20xf)=2HZ(500ms) */
	read_cmos(CMOS_STRC);				/* Status register C ɤǳߤꥻåȤ롣 */
	write_cmos(CMOS_STRB,0x42);			/* 24hour,ߥ */
	release_irq_mask(8);
}
/*************************************************************************************/
void test_time()
{
	
}
/**************************************************************************************/
