/*
 * Copyright (c) 1989, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Michael Fischbein.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * 2004: modified by minoru murashma.
 */


#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <limits.h>
#include "../fts.h"
#include "../tzfile.h"
#include "../err.h"
#include "../command.h"
#include "ls.h"
#include "extern.h"


#define	IS_NOPRINT(p)	((p)->fts_number == NO_PRINT)


static void printlink(FTSENT *p)
{
	int lnklen;
	char name[PATH_MAX + 1], path[PATH_MAX + 1];


	if (p->fts_level == FTS_ROOTLEVEL)
		snprintf(name, sizeof(name), "%s", p->fts_name);
	else
		snprintf(name, sizeof(name),"%s/%s", p->fts_parent->fts_accpath, p->fts_name);
	if ((lnklen = readlink(name, path, sizeof(path) - 1)) == -1)
	{
		fprintf(stderr, "\nls: %s: %s\n", name, strerror(errno));
		return;
	}
	path[lnklen] = '\0';
	printf(" -> %s", path);
}


static int printtype(u_int mode)
{
	switch (mode & S_IFMT)
	{
		case S_IFDIR:
			putchar('/');
			return (1);
		case S_IFIFO:
			putchar('|');
			return (1);
		case S_IFLNK:
			putchar('@');
			return (1);
		case S_IFSOCK:
			putchar('=');
			return (1);
		case S_IFWHT:
			putchar('%');
			return (1);
	}

	if (mode & (S_IXUSR | S_IXGRP | S_IXOTH))
	{
		putchar('*');
		return (1);
	}

	return (0);
}


static void printtime(time_t ftime)
{
	int i;
	char *longstring;


	longstring = ctime(&ftime);
	for (i = 4; i < 11; ++i)
		putchar(longstring[i]);

#define	SIXMONTHS	((DAYSPERNYEAR / 2) * SECSPERDAY)
	if (f_sectime)
		for (i = 11; i < 24; i++)
			putchar(longstring[i]);
	else if (ftime + SIXMONTHS > time(NULL))
		for (i = 11; i < 16; ++i)
			putchar(longstring[i]);
	else
	{
		putchar(' ');
		for (i = 20; i < 24; ++i)
			putchar(longstring[i]);
	}

	putchar(' ');
}


/*
 * print [inode] [size] name
 * return # of characters printed, no trailing characters.
 */
static int printaname(FTSENT *p,u_long sizefield,u_long inodefield)
{
	struct stat *sp;
	int chcnt;


	sp = p->fts_statp;
	chcnt = 0;

	if (f_inode)
		chcnt += printf("%*lu ", (int)inodefield, sp->st_ino);
	if (f_size)
		chcnt += printf("%*d ",(int)sizefield, howmany(sp->st_blocks, blocksize));
	chcnt += printf("%s", p->fts_name);
	if (f_type)
		chcnt += printtype(sp->st_mode);

	return (chcnt);
}


void printscol(DISPLAY *dp)
{
	FTSENT *p;


	for (p = dp->list; p; p = p->fts_link)
	{
		if (IS_NOPRINT(p))continue;
		printaname(p, dp->s_inode, dp->s_block);
		putchar('\n');
	}
}


void printlong(DISPLAY *dp)
{
	struct stat *sp;
	FTSENT *p;
	NAMES *np;
	char buf[20];


	if (dp->list->fts_level != FTS_ROOTLEVEL && (f_longform || f_size))
		printf("total %lu\n", howmany(dp->btotal, blocksize));

	for (p = dp->list; p; p = p->fts_link)
	{
		if (IS_NOPRINT(p))continue;
		sp = p->fts_statp;
		if (f_inode)
			printf("%*lu ", dp->s_inode, sp->st_ino);
		if (f_size)
			printf("%*d ",dp->s_block, howmany(sp->st_blocks, blocksize));
		strmode(sp->st_mode, buf);
		np = p->fts_pointer;
		printf("%s %*u %-*s  %-*s  ",
			buf, dp->s_nlink,sp->st_nlink, dp->s_user, np->user, dp->s_group,np->group);
		if (f_flags)
			printf("%-*s ", dp->s_flags, np->flags);

		if (S_ISCHR(sp->st_mode) || S_ISBLK(sp->st_mode))
			printf("%3d, %3d ",major(sp->st_rdev), minor(sp->st_rdev));
		else if (dp->bcfile)
			printf("%*s%*d ",8 - dp->s_size, "", dp->s_size, sp->st_size);
		else
			printf("%*d ", dp->s_size, sp->st_size);

		if (f_accesstime)
			printtime(sp->st_atime);
		else if (f_statustime)
			printtime(sp->st_ctime);
		else
			printtime(sp->st_mtime);

		printf("%s", p->fts_name);

		if (f_type)
			printtype(sp->st_mode);
		if (S_ISLNK(sp->st_mode))
			printlink(p);

		putchar('\n');
	}
}


#define	TAB	8

void printcol(DISPLAY *dp)
{
	extern int termwidth;
	static FTSENT **array;
	static int lastentries = -1;
	FTSENT *p;
	int base, chcnt, cnt, col, colwidth, num;
	int endcol, numcols, numrows, row;


	/*
	 * Have to do random access in the linked list -- build a table
	 * of pointers.
	 */
	if (dp->entries > lastentries)
	{
		lastentries = dp->entries;
		if ((array =realloc(array, dp->entries * sizeof(FTSENT *))) == NULL)
		{
			warn(NULL);
			printscol(dp);
		}
	}
	for (p = dp->list, num = 0; p; p = p->fts_link)
		if (p->fts_number != NO_PRINT)
			array[num++] = p;

	colwidth = dp->maxlen;
	if (f_inode)
		colwidth += dp->s_inode + 1;
	if (f_size)
		colwidth += dp->s_block + 1;
	if (f_type)
		colwidth += 1;

	colwidth = (colwidth + TAB) & ~(TAB - 1);
	if (termwidth < 2 * colwidth)
	{
		printscol(dp);
		return;
	}

	numcols = termwidth / colwidth;
	numrows = num / numcols;
	if (num % numcols)++numrows;

	if (dp->list->fts_level != FTS_ROOTLEVEL && (f_longform || f_size))
		printf("total %lu\n", howmany(dp->btotal, blocksize));
	for (row = 0; row < numrows; ++row)
	{
		endcol = colwidth;
		for (base = row, chcnt = col = 0; col < numcols; ++col)
		{
			chcnt += printaname(array[base], dp->s_inode,dp->s_block);
			if ((base += numrows) >= num)break;
			while ((cnt = ((chcnt + TAB) & ~(TAB - 1))) <= endcol)
			{
				putchar('\t');
				chcnt = cnt;
			}
			endcol += colwidth;
		}
		putchar('\n');
	}
}
