/*
 * Copyright (c) 1980, 1990, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * 2004: modified by minoru murashma.
 */


#include <sys/stat.h>
#include <sys/statvfs.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "../command.h"
#include "../err.h"


/*
 * Convert statfs returned filesystem size into BLOCKSIZE units.
 * Attempts to avoid overflow for large filesystems.
 */
#define fsbtoblk(num, fsbs, bs) \
	(((fsbs) != 0 && (fsbs) < (bs)) ? (num) / ((bs) / (fsbs)) : (num) * ((fsbs) / (bs)))
#define sblock sb.iu_fs

/*
union {
	struct fs iu_fs;
	char dummy[SBSIZE];
} sb;
*/

extern int getmntinfo (struct statfs **, int);
extern int checkvfsname (const char *, char **);
extern char **makevfslist (char *);


int	iflag, nflag;
/*struct ufs_args mdev;*/
int	rfd;


void usage()
{
	fprintf(stderr,"usage: df [-in] [-t type] [file | file_system ...]\n");
	exit(1);
}


int bread(off_t off,void *buf,int cnt)
{
	int nr;


	lseek(rfd, off, SEEK_SET);
	if ((nr = read(rfd, buf, cnt)) != cnt)
	{
		/* Probably a dismounted disk if errno == EIO. */
		if (errno != EIO)
			fprintf(stderr, "\ndf: %qd: %s\n",off, strerror(nr > 0 ? EIO : errno));
		return (0);
	}

	return (1);
}


/*
 * Print out status about a filesystem.
 */
void prtstat(struct statfs *sfsp,int maxwidth)
{
	static long blocksize;
	static int headerlen, timesthrough;
	static char *header;
	long used, availblks, inodes;


	if (maxwidth < 11)
		maxwidth = 11;
	if (++timesthrough == 1)
	{
		header = getbsize(&headerlen, &blocksize);
		printf("%-*.*s %s     Used    Avail Capacity",maxwidth, maxwidth, "Filesystem", header);
		if (iflag)
			printf(" iused   ifree  %%iused");
		printf("  Mounted on\n");
	}
	printf("%-*.*s", maxwidth, maxwidth, sfsp->f_mntfromname);
	used = sfsp->f_blocks - sfsp->f_bfree;
	availblks = sfsp->f_bavail + used;
	printf(" %*ld %8ld %8ld", headerlen,
	    fsbtoblk(sfsp->f_blocks, sfsp->f_bsize, blocksize),
	    fsbtoblk(used, sfsp->f_bsize, blocksize),
	    fsbtoblk(sfsp->f_bavail, sfsp->f_bsize, blocksize));
	printf(" %5.0f%%",availblks == 0 ? 100.0 : (double)used / (double)availblks * 100.0);
	if (iflag)
	{
		inodes = sfsp->f_files;
		used = inodes - sfsp->f_ffree;
		printf(" %7ld %7ld %5.0f%% ", used, sfsp->f_ffree,inodes == 0 ? 100.0 : (double)used / (double)inodes * 100.0);
	}
	else printf("  ");
	printf("  %s\n", sfsp->f_mntonname);
}


/*
 * Make a pass over the filesystem info in ``mntbuf'' filtering out
 * filesystem types not in vfslist and possibly re-stating to get
 * current (not cached) info.  Returns the new count of valid statfs bufs.
 */
long regetmntinfo(struct statfs **mntbufp,long mntsize,char **vfslist)
{
	int i, j;
	struct statfs *mntbuf;


	if (vfslist == NULL)
		return (nflag ? mntsize : getmntinfo(mntbufp, MNT_WAIT));

	mntbuf = *mntbufp;
	for (j = 0, i = 0; i < mntsize; i++)
	{
		if (checkvfsname(mntbuf[i].f_fstypename, vfslist))
			continue;
		if (!nflag)
			statfs(mntbuf[i].f_mntonname,&mntbuf[j]);
		else if (i != j)
			mntbuf[j] = mntbuf[i];
		j++;
	}

	return (j);
}


char *getmntpt(char *name)
{
	long mntsize, i;
	struct statfs *mntbuf;


	mntsize = getmntinfo(&mntbuf, MNT_NOWAIT);
	for (i = 0; i < mntsize; i++)
	{
		if (!strcmp(mntbuf[i].f_mntfromname, name))
			return (mntbuf[i].f_mntonname);
	}

	return (0);
}


/*
 * This code constitutes the pre-system call Berkeley df code for extracting
 * information from filesystem superblocks.
 */
/*void ufs_df(char *file,int maxwidth)
{
	struct statfs statfsbuf;
	struct statfs *sfsp;
	char *mntpt;
	static int synced;


	if (synced++ == 0)sync();

	if ((rfd = open(file, O_RDONLY)) < 0)
	{
		warn("%s", file);
		return;
	}
	if (bread((off_t)SBOFF, &sblock, SBSIZE) == 0)
	{
		close(rfd);
		return;
	}

	sfsp = &statfsbuf;
	sfsp->f_type = 1;
	strcpy(sfsp->f_fstypename, "ufs");
	sfsp->f_flags = 0;
	sfsp->f_bsize = sblock.fs_fsize;
	sfsp->f_iosize = sblock.fs_bsize;
	sfsp->f_blocks = sblock.fs_dsize;
	sfsp->f_bfree = sblock.fs_cstotal.cs_nbfree * sblock.fs_frag + sblock.fs_cstotal.cs_nffree;
	sfsp->f_bavail = (sblock.fs_dsize * (100 - sblock.fs_minfree) / 100) - (sblock.fs_dsize - sfsp->f_bfree);
	if (sfsp->f_bavail < 0)
		sfsp->f_bavail = 0;
	sfsp->f_files =  sblock.fs_ncg * sblock.fs_ipg;
	sfsp->f_ffree = sblock.fs_cstotal.cs_nifree;
	sfsp->f_fsid.val[0] = 0;
	sfsp->f_fsid.val[1] = 0;
	if ((mntpt = getmntpt(file)) == 0)
		mntpt = "";
	memmove(&sfsp->f_mntonname[0], mntpt, MNAMELEN);
	memmove(&sfsp->f_mntfromname[0], file, MNAMELEN);
	prtstat(sfsp, maxwidth);

	close(rfd);
}
*/

int main(int argc,char *argv[])
{
	struct stat stbuf;
	struct statfs statfsbuf, *mntbuf;
	long mntsize;
	int ch, err, i, maxwidth, width;
	char *mntpt, **vfslist;


	vfslist = NULL;
	while ((ch = getopt(argc, argv, "int:")) != EOF)
		switch (ch)
		{
			case 'i':
				iflag = 1;
				break;
/*			case 'n':
				nflag = 1;
				break;
*/			case 't':
				if (vfslist != NULL)
					errx(1, "only one -t option may be specified.");
				vfslist = makevfslist(optarg);
				break;
			case '?':
			default:
				usage();
		}
	argc -= optind;
	argv += optind;

	mntsize = getmntinfo(&mntbuf, MNT_NOWAIT);
	maxwidth = 0;
	for (i = 0; i < mntsize; i++)
	{
		width = strlen(mntbuf[i].f_mntfromname);
		if (width > maxwidth)
			maxwidth = width;
	}

	/* եλʤ */
	if (!*argv)
	{
		mntsize = regetmntinfo(&mntbuf, mntsize, vfslist);
		if (vfslist != NULL)
		{
			maxwidth = 0;
			for (i = 0; i < mntsize; i++)
			{
				width = strlen(mntbuf[i].f_mntfromname);
				if (width > maxwidth)
					maxwidth = width;
			}
		}
		for (i = 0; i < mntsize; i++)
			prtstat(&mntbuf[i], maxwidth);

		exit(0);
	}

	/* եλꤢꡣ */
	for (; *argv; argv++)
	{
		if (stat(*argv, &stbuf) < 0)
		{
			err = errno;
			if ((mntpt = getmntpt(*argv)) == 0)
			{
				warn("%s", *argv);
				continue;
			}
		}

		if ((stbuf.st_mode & S_IFMT) == S_IFCHR)
		{
/*			ufs_df(*argv, maxwidth);
*/			continue;
		}
		else if ((stbuf.st_mode & S_IFMT) == S_IFBLK)
		{
			if ((mntpt = getmntpt(*argv)) == 0)
			{
/*				mntpt = mktemp(strdup("/tmp/df.XXXXXX"));
				mdev.fspec = *argv;
				if (mkdir(mntpt, DEFFILEMODE) != 0)
				{
					warn("%s", mntpt);
					continue;
				}

				if (mount("ufs", mntpt, MNT_RDONLY, &mdev) != 0)
				{
					ufs_df(*argv, maxwidth);
					rmdir(mntpt);
					continue;
				}
				else if (statfs(mntpt, &statfsbuf))
				{
					statfsbuf.f_mntonname[0] = '\0';
					prtstat(&statfsbuf, maxwidth);
				}
				else warn("%s", *argv);

				unmount(mntpt, 0);
				rmdir(mntpt);
*/
				continue;
			}
		}
		else mntpt = *argv;

		/*
		 * Statfs does not take a `wait' flag, so we cannot
		 * implement nflag here.
		 */
		if (statfs(mntpt, &statfsbuf) < 0)
		{
			warn("%s", mntpt);
			continue;
		}
		if (argc == 1)
			maxwidth = strlen(statfsbuf.f_mntfromname) + 1;

		prtstat(&statfsbuf, maxwidth);
	}

	return (0);
}
