/*
 * time.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include<types.h>
#include<lib.h>
#include<errno.h>
#include<interrupt.h>
#include<proc.h>
#include<sp.h>
#include<mp.h>
#include<lock.h>
#include<mm.h>
#include<signal.h>
#include<time.h>


/**************************************************************/
#define STATIC static
/**************************************************************/


/***************************************************************************************************
 *
 * 
 *
 ***************************************************************************************************/


enum{
	/* Timer flags. */
	INTERVAL=	1<<1,	/* Interval timer. */
	REAL=		1<<2,	/* Realclock timer. */
	TIMER_DEL=	1<<3,	/* Delete timer. */
};


#define GATE flag 		/* TIMER¤ѡflagС򥹥ԥåȤɤѤ롣 */


/***************************************************************************************************
 *
 * 8254 interval timer
 *
 ***************************************************************************************************/


static uint tasking_count=				/* 󥰥ץåѥޡ */
	PIT_HZ/(1000/TASK_TIME);
static TIMER intervalTimer=				/* Interval timer link. */
	{&intervalTimer,&intervalTimer,0,0};
static TIMER intervalAlarm=				/* alarm timer link.  */
	{&intervalAlarm,&intervalAlarm,0,0};
static int interval_current_count;		/* ޡθߥ५. */


/*
 * GLOBAL
 * 8254 interval timer handler
 */
STATIC int interval_timer_handler()
{
	TIMER *p;


	enter_spinlock(&intervalTimer.GATE);
	{
		if(intervalTimer.count>0)
		{
			p=intervalTimer.next;

			if(p->count==interval_current_count)
			{
				p->next->prev=p->prev;
				p->prev->next=p->next;
				intervalTimer.count-=1;

				/*  */
				if((p->flag&TIMER_DEL)==0)
					add_to_schedule(getProcAddr(p,timer),TASK_DEVICE_WAIT|TASK_SIGNAL_WAIT);
				else p->count=0;
				p->flag=0;
			}
			else p->count-=interval_current_count;
		}

		/* 顼ࡣ */
		if(intervalAlarm.count>0)
		{
			p=intervalAlarm.next;

			if(p->count==interval_current_count)
			{
				p->next->prev=p->prev;
				p->prev->next=p->next;
				intervalAlarm.count-=1;

				/*  */
				if((p->flag&TIMER_DEL)==0)
					sendSignal(getProcAddr(p,alarm),SIGALRM);
				else p->count=0;
				p->flag=0;
			}
			else p->count-=interval_current_count;
		}

		/* No timer */
		if(intervalTimer.count==0)
		{
			if(MFPS_addres==0)
			{
				tasking_count=PIT_HZ/(1000/TASK_TIME);
				init_sp_task();
			}
		}
		else
		{
			interval_current_count=intervalTimer.next->count;

			if(MFPS_addres==0)
			{
				if((tasking_count-=interval_current_count)==0)
					tasking_count=PIT_HZ/(1000/TASK_TIME);

				if(tasking_count<interval_current_count)
					interval_current_count=tasking_count;
			}

			/* ޡꤹ */
			outb(PIT_CNR0,interval_current_count&0xff);
			outb(PIT_CNR0,interval_current_count>>8);
		}
	}
	exit_spinlock(&intervalTimer.GATE);

	return 1;
}


/*
 * PUBLIC
 * åѴ
 * parameters : time(ms)
 * return : å
 */
STATIC inline uint clockTime(uint time)
{
	return PIT_HZ*time/1000;
}


/*
 * PUBLIC
 * Set interval timer.
 * parameters : time(å)
 * rturn : new time
 */
STATIC uint setIntervalTimer1(uint time)
{
	uint count;


	if(intervalTimer.count==0)
	{
		count=time;

		if(MFPS_addres==0)
		{
			irq_entry[IRQ0]=interval_timer_handler;
			outb(PIT_CTR,0x30);					/* One shot timer(mode 0) */
			if(time>PIT_HZ/(1000/TASK_TIME))
				count=PIT_HZ/(1000/TASK_TIME);
		}

		/* ޡ */
		if(count>0x10000)interval_current_count=0x10000;
		else interval_current_count=count;
		outb(PIT_CNR0,interval_current_count&0xff);
		outb(PIT_CNR0,interval_current_count>>8);
	}
	else
	{
		count=inb(PIT_CNR0);
		count+=inb(PIT_CNR0)<<8;
		time+=count;
	}

	return time;
}

/*
 * parameters : time(å),setIntervalTimer1()ΰtime
 */
STATIC void setIntervalTimer2(uint time,uint first_time)
{
	if((time!=first_time)&&(time<interval_current_count))
	{
		/* ޡ */
		interval_current_count=time;
		outb(PIT_CNR0,first_time&0xff);
		outb(PIT_CNR0,first_time>>8);
	}
}


/***************************************************************************************************
 *
 * MC146818 realclock timer
 *
 ***************************************************************************************************/


enum{
	REAL_TIMER_RATE=500,	/* Real-clock timer rate(ms) */
};


static TIMER realTimer=			/* Real timer link queue */
	{&realTimer,&realTimer,0,0};
static TIMER realAlarm=			/* alarm timer link.  */
	{&realAlarm,&realAlarm,0,0};
static uint utc;				/* current UTC(number of seconds since 00:00:00.1.1.1970). */
static uint64 curRdtsc;			/* ߤRDTSC͡ */


/*
 * PRIVATE
 * ʿʿѴ롣
 */
STATIC inline uchar bcd_to_bin(uchar value)
{
	return value-6*(value>>4);
}


/*
 * PUBLIC
 * UTCꤹ롣
 * parameters : seconds
 */
static void inline setUtc(uint seconds)
{
	utc=seconds;
}


/*
 * GLOBAL
 * MC146818 real timer handler
 * return : ץåʤ
 */
STATIC int real_timer_handler()
{
	static int half = 0;
	uint64 half64;
	int tm;
	TIMER *p,*q;


	/* Status register C ɤǳߤꥻåȤ롣 */
	read_cmos(CMOS_STRC);

	/* ॢȥץĴ٤ */
	enter_spinlock(&realTimer.GATE);
	{
		/* ȥޡ */
		if(realTimer.count>0)
		{
			tm=REAL_TIMER_RATE;
			for(p=realTimer.next;p!=&realTimer;)
			{

				if(p->count<=tm)
				{
					p->next->prev=p->prev;
					p->prev->next=p->next;
					realTimer.count-=1;
					tm-=p->count;

					/*  */
					if((p->flag&TIMER_DEL)==0)
						add_to_schedule(getProcAddr(p,timer),TASK_DEVICE_WAIT|TASK_SIGNAL_WAIT);
					else p->count=0;

					q=p;			/* flagꥢ¾Υޡ˹Ǥ뤬        */
					p=p->next;		/* ꥢ¾Υޡꤵƥ󥯤Ѥɤ */
					q->flag=0;
				}
				else
				{
					p->count-=tm;
					break;
				}
			}
		}
	}
	exit_spinlock(&realTimer.GATE);

	enter_spinlock(&realAlarm.GATE);
	{
		/* 顼ࡣ */
		if(realAlarm.count>0)
		{
			tm=REAL_TIMER_RATE;
			for(p=realAlarm.next;p!=&realAlarm;)
			{

				if(p->count<=tm)
				{
					p->next->prev=p->prev;
					p->prev->next=p->next;
					realAlarm.count-=1;
					tm-=p->count;
					
					/* Send alarm. */
					if((p->flag&TIMER_DEL)==0)
						sendSignal(getProcAddr(p,alarm),SIGALRM);
					else p->count=0;

					q=p;			/* flagꥢ¾Υޡ˹Ǥ뤬        */
					p=p->next;		/* ꥢ¾Υޡꤵƥ󥯤Ѥɤ */
					q->flag=0;
				}
				else
				{
					p->count-=tm;
					break;
				}
			}
		}
	}
	exit_spinlock(&realAlarm.GATE);

	/* UTC򹹿롣 */
	half ^= 1;				/* 500msȹ */
	utc += half;
	
	/* ߤRDTSCͤ */
	half64 = half;
	curRdtsc *= half64 ^ 1;
	curRdtsc += rdtsc() * half64;
	
	return 0;
}


/*
 * GLOBAL
 * get current time from cmos
 * parameters : time structure
 */
STATIC void get_time_from_cmos()
{
	static ushort sumMonthDay[13]=				/* ιס */
		{0,0,31,59,90,120,151,181,212,243,273,304,334};

	struct TIME{
		uchar second;		/* seconds */
		uchar minute;		/* minutes */
		uchar hour;			/* hours */
		uchar day;			/* days */
		uchar month;		/* months */
		ushort year;		/* years */
	}time;
	int days;


	do
	{
		time.second=bcd_to_bin(read_cmos(CMOS_SCD));
		time.minute=bcd_to_bin(read_cmos(CMOS_MNT));
		time.hour=bcd_to_bin(read_cmos(CMOS_HOR));
		time.day=bcd_to_bin(read_cmos(CMOS_DAY));
		time.month=bcd_to_bin(read_cmos(CMOS_MTH));
		time.year=bcd_to_bin(read_cmos(CMOS_YER));
	}while(bcd_to_bin(read_cmos(CMOS_SCD))<time.second);

	time.year+=(time.year<70)?2000:1900;

	/* UTCη׻ */
	days=(time.year-1970)*365;
	days+=(time.year-1-1970)/4;
	days+=sumMonthDay[time.month]+time.day-1;
	if((time.year%4==0)&&(time.month>2))++days;
	utc=((days*24+time.hour)*60+time.minute)*60+time.second;
	utc+=LOCAL_TIME_TOKYO;								/* 륿֤Ĵ */
}


/***************************************************************************************************
 *
 * ޡ
 *
 ***************************************************************************************************/


/*
 * PUBLIC
 * Delelte timer from link.
 *
 */
STATIC void delTimer(TIMER *timer,TIMER *interval,TIMER *real)
{
	TIMER *link;


	if(timer->flag!=0)
	{
		if(timer->flag&INTERVAL)link=interval;
		else                    link=real;

		enter_spinlock(&link->GATE);
		{
			if(timer->flag!=0)
			{
				if(timer->next!=link)
					timer->next->count+=timer->count;
				timer->next->prev=timer->prev;
				timer->prev->next=timer->next;

				link->count-=1;
				timer->flag=0;
			}
		}
		exit_spinlock(&link->GATE);
	}
}


/*
 * PUBLIC
 * Set timer
 * ա
 *  δؿɬ del_from_schedule() θǻȤȡ⤷˥ॢȤ
 *  ץʤ롣
 * parameters : time(msñ),flag
 * rturn : 0 or error number
 */
STATIC int setTimer(TIMER *timer,uint time,TIMER *interval,TIMER *real)
{
	uint tm,first_time=0;
	TIMER *link;
	TIMER *p;

	
	if(time==0)return 0;

	if(time<REAL_TIMER_RATE)
	{
		link=interval;
		first_time=time=clockTime(time);
		time=setIntervalTimer1(time);
	}
	else link=real;

	/* 󥯤 */
	delTimer(timer,interval,real);

	/* 󥯤³ */
	tm=time;
	enter_spinlock(&link->GATE);
	{
		for(p=link->next;p!=link;p=p->next)
		{
			if(tm<p->count)
			{
				p->count-=tm;
				break;
			}
			else tm-=p->count;
		}
		timer->prev=p->prev;
		timer->next=p;
		p->prev->next=timer;
		p->prev=timer;
		timer->count=tm;
		if(link==interval)
		{
			timer->flag=INTERVAL;
			setIntervalTimer2(time,first_time);
		}
		else timer->flag=REAL;

		link->count+=1;
    }
	exit_spinlock(&link->GATE);

	return 0;
}


/*
 * GLOBAL
 * Set timer.
 * parameters : time(ms)
 * 0 or error number
 */
int set_timer(uint time)
{
	return setTimer(&get_current_task()->timer,time,&intervalTimer,&realTimer);
}


/*
 * GLOBAL
 * Delete timer from timer link.
 * parameters : Process number
 */
void del_timer(TIMER *timer)
{
	 timer->flag|=TIMER_DEL;
}


/*
 * GLOBAL
 * set time struct.
 * return : time struct address or error=NULL
 */
void setTimerStruct(void *proc)
{
	memset(&((PROC*)proc)->timer,0,sizeof(TIMER));
	memset(&((PROC*)proc)->alarm,0,sizeof(TIMER));
}


/*
 * GLOBAL
 * release time struct.
 */
void releaseTimer(TIMER *timer,TIMER *alarm)
{
	delTimer(timer,&intervalTimer,&realTimer);
	delTimer(alarm,&intervalAlarm,&realAlarm);
}


/***************************************************************************************************
 *
 * å¬
 *
 ***************************************************************************************************/


int clock_1micro;	/* 1micros */
int clock_1m;		/* 1ms */


/*
 * GLOBAL
 * cpuå¬
 * ֤ : å
 */
int count_cpu_clock()
{
	uint cpu_clock;
	int low,high;
	long long r_beg,r_end;
	double time;


 	outb(PIT_CTR,0x30);	/* CLK0,LSB.MSB,⡼0,binary */
	outb(PIT_CNR0,0);	/* LSB */
	outb(PIT_CNR0,0);	/* MSB */

 	r_beg=rdtsc();		/* clock counter */

	for(;;)if(rdtsc()-r_beg>0x400000)break;		/* ٱ */

	r_end=rdtsc();

	low=inb(PIT_CNR0);
	high=inb(PIT_CNR0);

	time=(0xffff-low-(high<<8))/(double)PIT_HZ;
	cpu_clock=(r_end-r_beg)/time;

 	clock_1micro = cpu_clock / 1000000 + 1;		/* 1micros */
 	clock_1m = cpu_clock / 1000 + 1;			/* 1ms */

	return cpu_clock;
}


/***************************************************************************************************
 *
 * < ƥॳ >
 *
 ***************************************************************************************************/


/*
 * get Coordinated Universal Time(number of seconds since 00:00:00.1.1.1970)
 */
int sys_time(int *tp)
{
	if(tp!=NULL)
	{
		if(checkMem(tp)==-1)return -EFAULT;
		*tp=utc;
	}

	return (int)utc;
}


int sys_sleep(uint seconds)
{
	int time;
	PROC *proc = get_current_task();

	setTimer(&proc->timer,seconds*1000,&intervalTimer,&realTimer);
	sigWait();
	time = proc->timer.count/1000;
	del_timer(&proc->timer);
	
	return time;
}


void sys_wake(void *proc)
{
	forceSendSignal(proc,SIGCONT);
}


int sys_alarm(uint seconds)
{
	TIMER *alarm;

	alarm=&get_current_task()->alarm;
	setTimer(&get_current_task()->alarm,seconds*1000,&intervalAlarm,&realAlarm);
	
	return 0;
}


int sys_settime(uint seconds)
{
	if(get_current_task()->uid!=0)return -EPERM;

	setUtc(seconds);

	return 0;
}


int sys_gettimeofday(struct timeval *tp, void *tzp)
{
	if (checkMem(tp) == -1)
		return 0;

	tp->tv_sec = utc;
	tp->tv_usec = (rdtsc() - curRdtsc) / clock_1micro;
	if (tp->tv_usec > 1000000)
	{
		tp->tv_sec += 1;
		tp->tv_usec -= 1000000;
	}

	return 0;	
}

/***************************************************************************************************
 *
 * 
 *
 ***************************************************************************************************/

/*
 * GLOBAL
 * Init time
 */
void init_time()
{
	/* ޥץå */
	if(MFPS_addres)
	{
		irq_entry[IRQ0]=interval_timer_handler;
		outb(PIT_CTR,0x30);						/* One shot timer(mode 0). */
		release_irq_mask(0);
	}

	/* Init MC146818 real time clock */
	irq_entry[IRQ8]=real_timer_handler;
	get_time_from_cmos();				/* get current time from cmos. */
	write_cmos(CMOS_STRA,0x2f);			/* time bese 0x010(default)|ޡ졼65536/(20xf)=2HZ(500ms) */
	read_cmos(CMOS_STRC);				/* Status register C ɤǳߤꥻåȤ롣 */
	write_cmos(CMOS_STRB,0x42);			/* 24hour,ߥ */
	release_irq_mask(8);
}

/*************************************************************************************/
void test_time(uint signal)
{
	if(get_current_task()==(PROC*)0x150400)
	{
		printk("dosignal signal=%x ",signal);
	}	
}
/**************************************************************************************/
