/*
 * tcp.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * Transmission control Protocl.
 */


#include<config.h>
#include<types.h>
#include<lib.h>
#include<errno.h>
#include<mm.h>
#include<proc.h>
#include<time.h>
#include<lock.h>
#include<interrupt.h>
#include<signal.h>
#include<fs.h>
#include<net/net.h>
#include<net/ether.h>
#include<net/ip.h>
#include<net/tcp.h>
#include<test.h>


/*#define DEBUG_TCP 1
*/
#ifdef DEBUG_TCP
	#define STATIC
#else
	#define STATIC static
#endif


/* PUBLIC Functions. */
STATIC void closeConnect(SOCKET*,TCP_HEADER*);


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


enum{
	/* TCP flag. */
	TCP_FLG_FIN=1<<0,
	TCP_FLG_SYN=1<<1,
	TCP_FLG_RST=1<<2,
	TCP_FLG_PSH=1<<3,
	TCP_FLG_ACK=1<<4,
	TCP_FLG_URG=1<<5,

	TCP_WIN_SIZE=0xffff,	/* Window size. */
	TCP_MSS_SIZE=1460,		/* Ⱥ祵 */
};


typedef struct{
	uint8_t  kind;
	uint8_t  len;
	uint16_t mss;
}MSS;


/*
 * PUBLIC
 * إåꡣ
 * parameters : dectination IP,pseudo header address,data size
 */
static inline void setPseudoHead(in_addr_t dstip,in_addr_t srcip,uint16_t size,PSEUDO_HEADER *pshead)
{
	pshead->dstip=dstip;
	pshead->srcip=srcip;
	pshead->tmp=0;
	pshead->prot=IPPROTO_TCP;
	pshead->len=swapWord(size);
}


/*
 * PUBLIC
 * Get sequence number.
 * return : sequence number
 */
static inline uint32_t getSeqNumber()
{
	/* 4msǣ䤹RFC793ˡ */
	return rdtsc()/clock_1m*4;
}


/*
 * PUBLIC
 * IPХåե󥯥ꥹȤγ
 */
STATIC void releaseIpBuf(IP_BUFFER *ip_buf)
{
	IP_BUFFER *p = ip_buf,*q;
	
	
	while (p != NULL)
	{
		q = p->next;
		kfree(p);
		p = q;
	}
}


/*
 * PRIVATE
 * TCPХȿ֤
 */
static inline int getTcpSize(IP_HEADER *ip)
{
	return swapWord(ip->len) - (ip->verhead & 0xf) * 4;
}


/*
 * PRIVATE
 * TCPǡХȿ֤
 */
STATIC int getTcpDataSize(IP_HEADER *ip)
{
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;


	return getTcpSize(ip) - (tcp->headlen >> 4) * 4;
}


/**************************************************************************
 *
 * إåץ
 *
 **************************************************************************/


enum{
	/* Header option. */
	TCP_KIND_END=  0,
	TCP_KIND_NOP=  1,
	TCP_KIND_MSS=  2,
	TCP_KIND_WSC=  3,
	TCP_KIND_TSTMP=8,

	TCP_OPT_MAXSIZE=32,		/* TCP ץκ祵 */
};


typedef struct{
	uint32_t timeStamp;
	uint32_t timeEcho;
	uint16_t mss;
	uint8_t  winscal;
}OPTIONS;


/*
 * PUBLIC
 * Get TCP options.
 * parameters : option address
 */
STATIC void getTcpOptions(uchar *c,int size,OPTIONS *optst)
{
	uchar *last;


	last=c+size;
	while(c<last)
	{
		switch(*c)
		{
			case TCP_KIND_MSS:
				optst->mss= *(uint16_t*)(c+2);
				c+=4;
				break;
			case TCP_KIND_WSC:
				optst->winscal=*(uint8_t*)(c+2);
				c+=3;
				break;
			case TCP_KIND_TSTMP:
				optst->timeStamp=*(uint32_t*)(c+2);
				optst->timeEcho=*(uint32_t*)(c+6);
				c+=10;
				break;
			case TCP_KIND_NOP:
				c+=1;
				break;
			case TCP_KIND_END:
				break;
			default:
				c+=*(c+1);
		}
	}
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/

/*
 * PUBLIC
 * Transmit a tcp segment.
 * Note!
 *  ץǡɬintǳڤ륵ˤ뤳ȡ
 * parameters : socket,TCP flag,option address,option size,data address,data size,fragment flag
 */
STATIC void transSegment(SOCKET *sc,uint8_t flag,void *opt,int opt_size,void *data,int data_size,int fragment)
{
	char head[sizeof(PSEUDO_HEADER) + sizeof(TCP_HEADER) + TCP_OPT_MAXSIZE];
	TRANS_BUF_INFO tbi;
	TCP_HEADER *send_tcp = (TCP_HEADER*)(head+sizeof(PSEUDO_HEADER));


	/* إåꡣ */
	setPseudoHead(sc->dstip,getSrcip(sc->dstip),sizeof(TCP_HEADER)+opt_size+data_size,(PSEUDO_HEADER*)head);

	/* Хåեơ֥ */
	tbi.data[2] = data;
	tbi.size[2] = data_size;
	tbi.data[1] = (char*)send_tcp;
	tbi.size[1] = sizeof(TCP_HEADER)+opt_size;
	tbi.ipType = IPPROTO_TCP;

	/* Make TCP header. */
	send_tcp->srcport = sc->srcport;
	send_tcp->dstport = sc->dstport;
	send_tcp->seqnum =  swapInt32(sc->seqnum);
	send_tcp->acknum =  swapInt32(sc->acknum);
	send_tcp->headlen = ((sizeof(TCP_HEADER)+opt_size)/sizeof(uint32_t))<<4;
	send_tcp->flag =    flag;
	send_tcp->wndsize = swapWord(TCP_WIN_SIZE);
	send_tcp->chksum =  0;
	send_tcp->urgpoint = 0;
	memcpy(send_tcp->option,opt,opt_size);
	send_tcp->chksum = calcSumM((uint*)head,data,sizeof(PSEUDO_HEADER) + sizeof(TCP_HEADER) + opt_size,data_size);
/************************************************************************************************************************
{
	union{
		uint ui;
		uchar uc[4];
	}src_ip;

	src_ip.ui = sc->dstip;
	printk("send srcport=%x,dstport=%x,seqnum=%x,acknum=%x,flag=%x,ip=%u.%u.%u.%u\n",
		swapWord(send_tcp->srcport),swapWord(send_tcp->dstport),swapInt32(send_tcp->seqnum),swapInt32(send_tcp->acknum),
		send_tcp->flag,src_ip.uc[0],src_ip.uc[1],src_ip.uc[2],src_ip.uc[3]);
}
************************************************************************************************************************/
	/*  */
	transIp(&tbi,sc->dstip,0,fragment);
}


/*
 * PUBLIC
 * ͥ󤬤ʤΥꥻåȥȤ
 * parameters : receive IP header
 */
STATIC void transReset(IP_HEADER *ip)
{
	SOCKET sc;
	TCP_HEADER *tcp;


	tcp=(TCP_HEADER*)ip->data;
	sc.seqnum= 0;
	sc.acknum= swapInt32(tcp->seqnum)+1;
	sc.srcport=tcp->dstport;
	sc.dstport=tcp->srcport;
	sc.dstip=  ip->srcip;
	transSegment(&sc,TCP_FLG_RST,NULL,0,NULL,0,1);
}


/*
 * PBLIC
 * Send FIN ack.
 * parameters : socket
 */
STATIC void sendFinAck(SOCKET *sc)
{
	sc->stat|=SOCK_CNCT_RECV_FIN;
	sc->acknum+=1;
	transSegment(sc,TCP_FLG_ACK,NULL,0,NULL,0,1);
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/

enum{
	/* ACKȥåؿͤΥӥåȥޥåס */
	CHKACK_EQL=1,			/* ACKͤSEQͤϰƱ */
	CHKACK_SML=1<<1,		/* ACKͤ꾮 */
	CHKACK_LRG=1<<2,		/* SEQͤ礭 */
	
	LISTEN_WAIT_MAX = 5,	/* listenԤॢȻ֡á */
};


/* PUBLIC */
static SOCKET waitCnctSc;	/* Ԥ󥯡 */
static SOCKET waitLstnSc;	/* å󥯡 */
static int waitCnctLock;	/* Ԥѥåȡ */


/*
 * PRIVATE
 * ԤåȤõ
 * return : 0 = ̾ｪλ 1 = 򵯤 -1 = 顼
 */
STATIC int searchSocket(IP_HEADER *ip)
{
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;
	SOCKET *sc;
	IP_BUFFER *ip_buf;


/****************************************************************************************
{
	union{
		uint ui;
		uchar uc[4];
	}src_ip;

	src_ip.ui = ip->srcip;
	printk("recieve srcport=%x,dstport=%x,seqnum=%x,acknum=%x,flag=%x,ip=%u.%u.%u.%u\n",
		swapWord(tcp->srcport),swapWord(tcp->dstport),
		swapInt32(tcp->seqnum),swapInt32(tcp->acknum),
		tcp->flag,src_ip.uc[0],src_ip.uc[1],src_ip.uc[2],src_ip.uc[3]);
}
****************************************************************************************/
	enter_spinlock(&waitCnctLock);

	/* ͥ󥯤õ */
	for (sc = waitCnctSc.next; sc != &waitCnctSc; sc = sc->next)
		if ((sc->srcport == tcp->dstport) && (sc->dstport == tcp->srcport) && (sc->dstip == ip->srcip))
		{
			/* åȤϴǤƤ롣 */
			if (sc->stat & SOCK_DISCON_RECV)
			{
				exit_spinlock(&waitCnctLock);

				closeConnect(sc,tcp);

				return 0;
			}

			/* ǡ򥽥åȤϤ */
			if (allocIpBuf(ip,&ip_buf) != 0)
			{
				exit_spinlock(&waitCnctLock);

				transReset(ip);

				return 0;
			}
			addRecvBuf(sc,ip_buf);
			if (sc->waitProc != NULL)
				sys_wake(sc->waitProc);

			exit_spinlock(&waitCnctLock);

			return 1;
		}

	/* å󥯤õ */
	for (sc = waitLstnSc.next; sc != &waitLstnSc; sc = sc->next)
		if ((sc->srcport == tcp->dstport) && ((sc->srcip == INADDR_ANY) || (sc->srcip == ip->dstip)))
		{
			/* ǡåԤϤ */
			if (allocIpBuf(ip,&ip_buf) != 0)
			{
				exit_spinlock(&waitCnctLock);

				transReset(ip);

				return 0;
			}
			addRecvBuf(sc,ip_buf);
			sys_wake(sc->listenThread);

			exit_spinlock(&waitCnctLock);

			return 1;
		}
	
	exit_spinlock(&waitCnctLock);
	
	/* Ĥ餺 */
	transReset(ip);

	return 0;
}


/*
 * PUBLIC
 * Get data from receive buffer.
 * return : 0 or error number,time out seconds
 */
STATIC int waitReceive(SOCKET *sc,uint timeout)
{
	for (;;)
	{
		/* SIGINTߤ */
		if(isSigint())
			return -EINTR;

		if (sc->recvBuf == NULL)
		{
			sc->waitProc = get_current_task();
			if (sys_sleep(timeout) == 0)		/* Time out. */
			{
				if (sc->recvBuf == NULL)
					return -ETIMEDOUT;
			}
		}
		else
			break;
	}

	return 0;
}


/*
 * PUBLIC
 * Хåեγǧ
 * return : NULL to IP_BUFFER
 */
static inline IP_BUFFER *isExistReceiveBuf(SOCKET *sc)
{
	if (sc->recvBuf != NULL)
		return removeRecvBuf(sc);
	else
		return NULL;
}


/*
 * PUBLIC
 * Ȥγǧ
 * parameters : TCP header,ӤACKֹ,ӤSEQֹ,ӤFLAG
 * return : ӥåȥޥå or failed=-1
 */
STATIC int isAck(TCP_HEADER *tcp,uint32_t comp_ack,uint32_t comp_seq)
{
	int rest = 0;
	uint32_t ack = swapInt32(tcp->acknum);
	uint32_t seq = swapInt32(tcp->seqnum);


	/* ե饰ӡ */
	if((tcp->flag&TCP_FLG_ACK)==0)return -1;

	/* ֹӡ */
	if (ack == comp_ack)
	{
		if (seq == comp_seq)
			rest = CHKACK_EQL;
		else if (seq > comp_seq)
			rest = CHKACK_LRG;
		else 
			return -1;
	}
	else if (ack < comp_ack)
	{
		if(seq==comp_seq)
			rest = CHKACK_SML;
		else
			return -1;
	}

	return rest;
}


/*
 * PUBLIC
 * Add to connection link.
 * parameters : socket
 */
STATIC void addWaitCnctLink(SOCKET *sc)
{
	cli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next=waitCnctSc.next;
		sc->prev=&waitCnctSc;
		waitCnctSc.next->prev=sc;
		waitCnctSc.next=sc;
	}
	exit_spinlock(&waitCnctLock);
	sti();
}


/*
 * PUBLIC
 * Add to connection link.
 * parameters : socket
 */
STATIC void addWaitLstnLink(SOCKET *sc)
{
	cli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next=waitLstnSc.next;
		sc->prev=&waitLstnSc;
		waitLstnSc.next->prev=sc;
		waitLstnSc.next=sc;
	}
	exit_spinlock(&waitCnctLock);
	sti();
}


/*
 * PUBLIC
 * Delete from connection link.
 * parameters : socket
 */
STATIC void delWaitLink(SOCKET *sc)
{
	cli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next->prev = sc->prev;
		sc->prev->next = sc->next;
		sc->next = sc->prev = sc;
	}
	exit_spinlock(&waitCnctLock);
	sti();
}


/*
 * PUBLIC
 * å󥽥åȤݡȤõ
 * parameters : source port
 * return : socket address or NULL
 */
STATIC SOCKET *searchListenSocket(in_port_t srcport)
{
	SOCKET *sc,*search_sc = NULL;
	
	
	cli();
	enter_spinlock(&waitCnctLock);
	{
		for (sc = waitLstnSc.next; sc != &waitLstnSc; sc = sc->next)
			if (sc->srcport == srcport)
			{
				search_sc = sc;
				break;
			}
	}
	exit_spinlock(&waitCnctLock);
	sti();
	
	return search_sc;
}


/*
 * PUBLIC
 * 㤦Ȥ줿Ȥν
 * parameters : socket,receive TCP header
 * return : error number
 */
STATIC int recvBadSegment(SOCKET *sc,TCP_HEADER *tcp)
{
	if (tcp->flag & TCP_FLG_RST)
		return -ECONNREFUSED;

	/* ꥻåȥȤ */
	sc->acknum = swapInt32(tcp->seqnum) + 1;
	sc->seqnum = 0;
	transSegment(sc,TCP_FLG_RST,NULL,0,NULL,0,1);

	return -EAGAIN;
}


/*
 * GROBAL
 * Receive TCP datagram.
 * parameters : IP data
 * return : 0 or task switch=1
 */
int receiveTcp(IP_HEADER *ip)
{
	int tcp_size;
	PSEUDO_HEADER pshead;


	/* åγǧ */
	tcp_size = getTcpSize(ip);
	setPseudoHead(ip->dstip,ip->srcip,tcp_size,&pshead);
	if (calcSumM((uint*)&pshead,(uint*)ip->data,sizeof(PSEUDO_HEADER),tcp_size))
		return 0;

	return searchSocket(ip);
}


/**************************************************************************
 *
 * å󥹥å
 *
 **************************************************************************/

/*
 * PUBLIC
 * å󥳥ͥȺѥåȤå󥽥åȤ³
 * parameters : å󥽥å,ͥȺѥå
 */
STATIC void addNewSocket(SOCKET *sc,SOCKET *newsc)
{
	SOCKET **p;
	
	
	enter_spinlock(&sc->lockGate);
	{
		newsc->newsc_next = NULL;
		for (p = &sc->newsc_next;; p = &(*p)->newsc_next)
			if (*p == NULL)
			{
				*p = newsc;
				break;
			}
	}
	exit_spinlock(&sc->lockGate);
}


/*
 * PUBLIC
 * å󥳥ͥȺѥåȤå󥽥åȤƺ
 * parameters : å󥽥å
 */
STATIC SOCKET *removeNewSocket(SOCKET *sc)
{
	SOCKET *newsc;
	
	
	if (sc->newsc_next == NULL)
		return NULL;

	enter_spinlock(&sc->lockGate);
	{
		newsc = sc->newsc_next;
		sc->newsc_next = newsc->newsc_next;
	}
	exit_spinlock(&sc->lockGate);
	
	return newsc;
}


/*
 * PRIVATE
 * åȤ
 * return :  0 or error number
 */
STATIC int makeNewSocket(SOCKET *sc,IP_BUFFER *ip_buf,SOCKET **pnewsc)
{
	SOCKET *newsc;
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	OPTIONS optst;
	int rest;


	rest = makeSocket(sc->sockType,IPPROTO_TCP,&newsc);
	if (rest < 0)
		return rest;

	ip = &ip_buf->ip;
	tcp = (TCP_HEADER*)ip->data;

	/* ץ(MSSΤ߼) */
	optst.mss = TCP_MSS_SIZE;
	getTcpOptions(tcp->option,getTcpDataSize(ip),&optst);

	/*
	 * åȤꡣ
	 */
	memcpy(&newsc->fs,&sc->fs,sizeof(SOCKET)-OFFSETOF(SOCKET,fs));
	newsc->dstip = ip->srcip;
	newsc->dstport = tcp->srcport;
	newsc->sc = newsc;
	newsc->acknum = swapInt32(tcp->seqnum);
	newsc->seqnum = swapInt32(tcp->acknum);
	newsc->next = newsc->prev = newsc;
	newsc->stat = SOCK_LINK_ON | SOCK_CNCT_ON;
	newsc->mss = optst.mss;
	newsc->window = swapWord(tcp->wndsize);

	*pnewsc = newsc;
	
	return 0;
}


/*
 * PRIVATE
 * listenԤǽ
 * return :  0 = acceptԤϤʤ
 *          -1 = error
 */
STATIC int listenFirst(SOCKET *sc,IP_BUFFER *ip_buf)
{
	IP_HEADER *ip = &ip_buf->ip;
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;
	MSS mss;
	SOCKET trans_sc;
	IP_BUFFER **p;
	

	/* ³׵᥻Ȥγǧ */
	if ((tcp->flag & TCP_FLG_SYN) && (tcp->acknum == 0))
		;
	else
	{
		transReset(&ip_buf->ip);
		return -1;
	}

	/* إåץꡣ */
	mss.kind=TCP_KIND_MSS;
	mss.len= 4;
	mss.mss= swapWord(TCP_MSS_SIZE);

	/* SYNȤ */
	trans_sc.srcport = tcp->dstport;
	trans_sc.dstip = ip->srcip;
	trans_sc.dstport = tcp->srcport;
	trans_sc.acknum = swapInt32(tcp->seqnum) + 1;
	trans_sc.seqnum = getSeqNumber();
	transSegment(&trans_sc,TCP_FLG_SYN|TCP_FLG_ACK,&mss,sizeof(mss),NULL,0,1);

	/*
	 * å󥳥ͥ󥯤³
	 * ա
	 * 	ip.acknumip.seqnumˤäacknumseqnum򤽤줾
	 */
	tcp->acknum = trans_sc.acknum;
	tcp->seqnum = trans_sc.seqnum;
	ip_buf->acktime = sys_time(NULL);
	ip_buf->next = NULL;
	for (p = &sc->connect;; p = &(*p)->next)
		if (*p == NULL)
		{
			*p = ip_buf;
			break;
		}
	
	return 0;
}


/*
 * PRIVATE
 * listenԤ
 * return :  0 = acceptԤ򵯤
 *          -1 = error
 */
STATIC int listenNext(SOCKET *sc,IP_BUFFER *ip_buf,IP_BUFFER **pip_connect)
{
	int rest;
	SOCKET *newsc;
	IP_HEADER *ip = &ip_buf->ip,
	          *ip_connect = &(*pip_connect)->ip;
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data,
	           *tcp_connect = (TCP_HEADER*)ip_connect->data;
	IP_BUFFER *del_buf;


	/*
	 * Ȥγǧ
	 * ip_connectacknumseqnumˤϱäacknumseqnum򤽤줾ꤵƤ
	 */
	rest = isAck(tcp,tcp_connect->seqnum+1,tcp_connect->acknum);
	if ((rest == -1) || ((rest & CHKACK_EQL)==0))
	{
		transReset(&ip_buf->ip);
		return -1;
	}
	
	/* ׵ */
	if (tcp->flag & TCP_FLG_RST)
	{
		rest = -1;
		goto END;
	}

	/* åȤκ */
	if (makeNewSocket(sc,ip_buf,&newsc) < 0)
	{
		rest = -1;
		goto END;
	}
	
	/* ͥ󥯤³ */
	addWaitCnctLink(newsc);

	/* å󥽥åȤ³ */	
	addNewSocket(sc,newsc);

	rest = 0;
END:
	/* listenԤ鳫 */
	del_buf = *pip_connect;
	*pip_connect = del_buf->next;
	kfree(del_buf);

	return rest;
}


/*
 * PRIVATE
 * listenԤ³
 */
STATIC void listenConnect(SOCKET *sc)
{
	int cur_time = sys_time(NULL);
	IP_BUFFER *ip_buf = removeRecvBuf(sc);
	IP_HEADER *ip = &ip_buf->ip;
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;
	IP_BUFFER **p,*del_buf;
	
	
	/* å󥳥ͥХåե򸡺 */
	for (p = &sc->connect;;)
	{
		if (*p == NULL)
		{
			if (listenFirst(sc,ip_buf) == -1)
				kfree(ip_buf);
			return;
		}

		if (((*p)->ip.srcip == ip->srcip) && (((TCP_HEADER*)(*p)->ip.data)->srcport == tcp->srcport))
		{
			if (listenNext(sc,ip_buf,p) == 0)
				sys_wake(sc->waitProc);
			kfree(ip_buf);
			return;
		}

		/* ॢ */
		if (((*p)->acktime - cur_time) > LISTEN_WAIT_MAX)
		{
			del_buf = *p;
			p = &(*p)->next;
			kfree(del_buf);
		}
		else
			p = &(*p)->next;
	}
}


/*
 * PRIVATE
 * åɽλ롼
 * parameters : å,եǥץ
 */
STATIC void listenRoutin(SOCKET *sc,int fd)
{
	sc->listenThread = get_current_task();
	
	/*
	 * СåȤǤ褦
	 * եǥץȥȤ򸺤餹
	 */
	sys_close(fd);

	/* ͥγ */
	addWaitLstnLink(sc);

	for (;;)
	{
		if (sc->recvBuf == NULL)
		{
			/* ꡼ */
			sys_sleep(-1);

			/* å֤ǤʤХåɽλ */
			if ((sc->stat & SOCK_LISTEN) == 0)
				break;
		}
		else
			listenConnect(sc);
	}

	/*  */
	releaseIpBuf(sc->connect);
	kfree(sc);
	sys_exit(0);
}


/*
 * PUBLIC
 * å󥹥åɤκ
 * parameters : socket,file discripter
 * return     : 0 or error number
 */
STATIC int makeListenThread(SOCKET *sc,int fd)
{
	/* listenѥץư */
	switch(sys_fork())
    {
		case -1:
			return -ENOBUFS;
		case 0:
			listenRoutin(sc,fd);
			/* Not reached */
	}

	return 0;
}


/*
 * PUBLIC
 * å󥹥åɤνλ
 * parameters : socket
 */
STATIC void endListenThread(SOCKET *sc)
{
	sc->stat &= ~SOCK_LISTEN;
	sys_wake(sc->waitProc);
}


/**************************************************************************
 *
 * ͥ
 *
 **************************************************************************/


enum{
	CNCT_CLOSE_WAIT_TIME=5000,		/* ǽॢȻms */
	REPLY_TIMEOUT=5,				/* ॢȥߥá */
};


/* PRIVATE */
static SOCKET *finSocket;		/* 楽åȥ󥯡 */


/*
 * PRIVATE
 * ǳϥåȤ󥯤³롣
 * parameters : socket
 */
STATIC void addCloseSocket(SOCKET *sc)
{
	sc->waitNext = finSocket;
	finSocket = sc;
	sc->waitTime = sys_time();
}


/*
 * PRIVATE
 * ǳϥåȤǡॢȤõ
 */
STATIC void delCloseTimeout()
{
	uint time;
	SOCKET *p;


	time = sys_time();
	while (finSocket != NULL)
	{
		if (CNCT_CLOSE_WAIT_TIME <= time - finSocket->waitTime)
		{
			p = finSocket->waitNext;
			delWaitLink(finSocket);
			releaseIpBuf(finSocket->recvBuf);
			kfree(finSocket);
			finSocket = p;
		}
		else
			break;
	}
}


/*
 * PUBLIC
 * transmit fin segment.
 * parameters :
 */
STATIC void sendFin(SOCKET *sc)
{
	IP_BUFFER *ip_buf;
	
	
	/* FINȤ */
	transSegment(sc,TCP_FLG_FIN|TCP_FLG_ACK,NULL,0,NULL,0,1);
	sc->seqnum += 1;
	sc->stat |= SOCK_CNCT_TRANS_FIN;

	/* Хåե˻Ĥꤢ */
	while ((ip_buf = removeRecvBuf(sc)) != NULL)
	{
		closeConnect(sc,(TCP_HEADER*)ip_buf->ip.data);
		kfree(ip_buf);
	}

	delCloseTimeout();		/* ॢȥåȤ롣 */
	addCloseSocket(sc);		/* åȥ󥯤³롣 */
}


/*
 * PUBLIC
 * Close connection.
 */
STATIC void closeConnect(SOCKET *sc,TCP_HEADER *tcp)
{
	if (tcp->flag & TCP_FLG_FIN)													/* FINȤ */
	{
		if (sc->stat & SOCK_CNCT_RECV_FINACK)
		{
			if (swapInt32(tcp->acknum) == sc->seqnum)
				sendFinAck(sc);
		}
		else
		{
			if (swapInt32(tcp->acknum) == sc->seqnum)
			{
				sc->stat |= SOCK_CNCT_RECV_FINACK;
				sendFinAck(sc);
			}
			else if (swapInt32(tcp->acknum) == (sc->seqnum - 1))
				sendFinAck(sc);
		}
	}
	else if ((tcp->flag & TCP_FLG_ACK) && (swapInt32(tcp->acknum) == sc->seqnum))	/* ACKȤ */
		sc->stat |= SOCK_CNCT_RECV_FINACK;

	sc->waitTime = sys_time();
}


/*
 * PUBLIC
 * ͥΩ
 * parameters : socket
 * return : 0 or error number
 */
STATIC int connectTo(SOCKET *sc)
{
	int rest;
	MSS mss;
	IP_BUFFER *ip_buf = NULL;
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	OPTIONS optst;
	

	/* ͥ󥯤³ */
	addWaitCnctLink(sc);

	/* إåץꡣ */
	mss.kind=TCP_KIND_MSS;
	mss.len= 4;
	mss.mss= swapWord(TCP_MSS_SIZE);

	/* SYNȤ */
	transSegment(sc,TCP_FLG_SYN,&mss,sizeof(mss),NULL,0,1);
	sc->seqnum += 1;

	/* Ȥμ */
	rest = waitReceive(sc,REPLY_TIMEOUT);
	if (rest < 0)
		goto ERR;
	ip_buf = removeRecvBuf(sc);
	ip = &ip_buf->ip;
	tcp = (TCP_HEADER*)ip->data;
	if(((tcp->flag & TCP_FLG_SYN) == 0) ||
	   ((rest = isAck(tcp,sc->seqnum,0)) == -1) ||
	   ((rest & CHKACK_LRG) == 0))
	{
		rest = recvBadSegment(sc,tcp);
		goto ERR;
	}

	/* ץγǧ(MSSΤ߼) */
	optst.mss=TCP_MSS_SIZE;
	getTcpOptions(tcp->option,getTcpDataSize(ip),&optst);
	sc->mss = optst.mss;

	/* ǧ */
	sc->window = swapWord(tcp->wndsize);
	sc->acknum = swapInt32(tcp->seqnum)+1;
	sc->stat |= SOCK_CNCT_ON;
	transSegment(sc,TCP_FLG_ACK,NULL,0,NULL,0,1);

	kfree(ip_buf);

	return 0;
ERR:
	kfree(ip_buf);

	/* ͥ󥯤 */
	delWaitLink(sc);
	releaseIpBuf(sc->recvBuf);

	return rest;
}


/**************************************************************************
 *
 * Socket interface.
 *
 **************************************************************************/


enum{
	RECEIVE_TIMEOUT=5,		/* ॢȥߥá */
	SEND_TIMEOUT=   5,		/* Ԥॢȥߥá */
};


/*
 * GLOBAL
 * åȤ򥳥ͥ󤫤鳫롣
 * parameters : socket
 * return ; 0 or 1(åȤƤϤʤ)
 */
int releaseSckFromConect(SOCKET *sc)
{
	if (sc->stat & SOCK_CNCT_ON)
	{
		sc->stat |= SOCK_DISCON_RECV | SOCK_DISCON_SEND;
		if ((sc->stat & SOCK_CNCT_TRANS_FIN) == 0)
			sendFin(sc);

		return 1;
	}

	if (sc->stat & SOCK_LISTEN)
	{
		delWaitLink(sc);
		releaseIpBuf(sc->recvBuf);
		endListenThread(sc);

		return 1;
	}

	if (sc->stat & SOCK_LINK_ON)
	{
		delWaitLink(sc);
		releaseIpBuf(sc->recvBuf);
	}

	return 0;
}


STATIC int send(SOCKET *sc,const void *msg,size_t len,int flags,struct sockaddr *to)
{
	enum{
		RESEND_CNT=3,		/*  */
	};

	char *buf;
	int resend;				/*  */
	int rest;
	int window;				/* 襦ɥ */
	int data_size;			/* TCPǡ */
	uint32_t acknum;		/* ֹ档 */
	uint32_t lastnum;		/* ֹ档 */
	uint32_t sendnum;		/* ǽֹ档 */
	IP_BUFFER *ip_buf;
	TCP_HEADER *tcp;
	int l;


	/* γǧ */
	if ((sc->stat & SOCK_CNCT_ON) == 0)
		return -ENOTCONN;
	if (sc->stat & SOCK_CNCT_RECV_FIN)
	{
		sendFin(sc);
		return -EPIPE;
	}

	window = sc->window;
	acknum = sc->seqnum;
	buf = (char*)msg;
	resend = 0;
	lastnum = acknum + len;
	for (;;)
	{
		/*  */
		sendnum = ((acknum + window) < lastnum)? (acknum + window) : lastnum;
		for (;sc->seqnum < sendnum; sc->seqnum += l)
		{
			l = ((sendnum - sc->seqnum) < sc->mss)? (sendnum - sc->seqnum) : sc->mss;
			transSegment(sc,TCP_FLG_PSH|TCP_FLG_ACK,NULL,0,buf,l,0);
			buf += l;

			/* γǧ */
			ip_buf = isExistReceiveBuf(sc);
			if (ip_buf != NULL)
			{
				
				tcp = (TCP_HEADER*)ip_buf->ip.data;
				rest = isAck(tcp,sc->seqnum,sc->acknum);
				if ((rest == -1) || (rest & CHKACK_LRG))
				{
					kfree(ip_buf);
					return len-(lastnum-(sc->seqnum+l));
				}
				if (acknum < swapInt32(tcp->acknum))
				{
					acknum = swapInt32(tcp->acknum);
					window = swapWord(tcp->wndsize);
					sendnum = ((acknum + window) < lastnum)? (acknum + window) : lastnum;
				}
				kfree(ip_buf);
			}
		}

		/* Ԥ */
		for (;;)
		{
			/* ԡ */
			rest = waitReceive(sc,SEND_TIMEOUT);
			if (rest < 0)
			{
				if (rest == -EINTR)
					return -EINTR;
				if (RESEND_CNT < ++resend)
					return rest;
				sc->seqnum = acknum;
				buf -= sendnum - acknum;
				break;				/*  */
			}

			/* TCP˥ǡip_bufĤƤ */
			ip_buf = referRecvBuf(sc);
			data_size = getTcpDataSize(&ip_buf->ip);
			tcp = (TCP_HEADER*)ip_buf->ip.data;

			/* ǡǧ */
			rest = isAck(tcp,sc->seqnum,sc->acknum);
			if ((rest == -1) || (rest & CHKACK_LRG))
			{
				removeRecvBuf(sc);
				kfree(ip_buf);
				return len-(lastnum - sc->seqnum);
			}

			/* TCP˥ǡ¨λμˤޤ */
			if (0 < data_size)
				return len;

			removeRecvBuf(sc);
			if (acknum < swapInt32(tcp->acknum))
			{
				acknum = swapInt32(tcp->acknum);
				if (acknum == lastnum)
				{
					kfree(ip_buf);
					return len;
				}
				window = swapWord(tcp->wndsize);
				resend = 0;
				if ((sendnum < lastnum) && (sendnum < acknum+window))
					break;
			}
		}
	}
}


/*
 * return : copy data size or error number
 */
STATIC int recv(SOCKET *sc,void *buf,size_t len,int flags)
{
	char *data;
	int tcp_size,data_size,all_size;
	int rest;
	IP_BUFFER *ip_buf,*res_ip_buf = NULL;
	IP_HEADER *ip;
	TCP_HEADER *tcp;


	/* ͥγǧ */
	if ((sc->stat & SOCK_CNCT_ON) == 0)
		return -ENOTCONN;

	if (sc->stat & SOCK_CNCT_RECV_FIN)
		return 0;				/* ǤFINȤƤ롣 */

	all_size = 0;
	for (;;)
	{
		/*  */
		rest = waitReceive(sc,RECEIVE_TIMEOUT);
		if (rest < 0)
			return rest;
		ip_buf = removeRecvBuf(sc);
		ip = &ip_buf->ip;
		tcp = (TCP_HEADER*)ip->data;
		tcp_size = getTcpSize(ip);
		
		/* γǧ */
RET:	rest = isAck(tcp,sc->seqnum,sc->acknum);
        switch (rest)
        {
        	case -1:
        	case CHKACK_SML:
				rest = recvBadSegment(sc,tcp);
				kfree(ip_buf);
				return rest;
			case CHKACK_LRG:
				/*
				 * ȤΥֹ椬礭ȤϡΥȤ夷ǽΤ
				 * αƤ
				 */
				res_ip_buf = ip_buf;
				continue;
        }

		if (tcp->flag & TCP_FLG_FIN)
		{
			sendFinAck(sc);
			kfree(ip_buf);
			return all_size;
		}
		if (tcp->flag & TCP_FLG_RST)
		{
			kfree(ip_buf);
			return -ECONNRESET;
		}

		/* 桼Хåե˥ԡ */
		data_size = getTcpDataSize(ip);
		data = (char*)tcp + (tcp_size - data_size);
		all_size += data_size;
		sc->acknum += data_size;
		if (len < all_size)
		{
			data_size -= all_size - len;
			all_size = len;
			memcpy(buf,data,data_size);
			break;
		}
		memcpy(buf,data,data_size);
		buf = (char*)buf + data_size;

		/* αȤ硣 */
		if (res_ip_buf != NULL)
		{
			ip_buf = res_ip_buf;
			ip = &ip_buf->ip;
			tcp = (TCP_HEADER*)ip->data;
			tcp_size = getTcpSize(ip);
			res_ip_buf = NULL;
			goto RET;
		}

		if(sc->recvBuf == NULL)
			break;
	}
	kfree(ip_buf);	
	
	/* Ȥ */
	transSegment(sc,TCP_FLG_ACK,NULL,0,NULL,0,1);
	
	return all_size;
}


STATIC int shutdown(SOCKET *sc,int flag)
{
	if((sc->stat&SOCK_CNCT_ON)==0)return -ENOTCONN;


	switch(flag)
	{
		case SHUT_RD:
			sc->stat|=SOCK_DISCON_RECV;
			break;
		case SHUT_WR:
			sc->stat|=SOCK_DISCON_SEND;
			break;
		case SHUT_RDWR:
			sc->stat|=SOCK_DISCON_RECV|SOCK_DISCON_SEND;
			break;
	}
	if((sc->stat&SOCK_CNCT_TRANS_FIN)==0)
		sendFin(sc);

	return 0;
}


STATIC int connect(SOCKET *sc)
{
	if (sc->stat & SOCK_CNCT_ON)
		return -EISCONN;

	/* Set connect structure. */
	sc->next = sc->prev = sc;
	sc->stat = SOCK_LINK_ON;
	sc->seqnum = getSeqNumber();
	sc->acknum = 0;

	/* ͥγΩ */
	return connectTo(sc);
}


STATIC int listen(SOCKET *sc,int fd)
{
	if (sc->stat & SOCK_LISTEN)
		return -EADDRINUSE;

	if (searchListenSocket(sc->srcport) != NULL)
		return -EADDRINUSE;

	sc->next = sc->prev = sc;
	sc->stat = SOCK_LINK_ON | SOCK_LISTEN;
	sc->waitProc = get_current_task();
	
	/* å󥹥åɤγ */
	makeListenThread(sc,fd);
	
	return 0;
}


STATIC int accept(SOCKET *sc,SOCKET **pnewsc,uint32_t *srcip)
{
	SOCKET *newsc;


	for (;;)
	{
		/* ͥѤߥåȤõ */
		newsc = removeNewSocket(sc);
		if (newsc != NULL)
			break;

		/* å󥹥åԤ */
		if (sys_sleep(-1) == 0)		/* Time out. */
			return -ETIMEDOUT;
	}

	*srcip = newsc->dstip;
	*pnewsc = newsc;

	return 0;
}


STATIC int select(SOCKET *sc,int flag,int polling)
{
		/* ͥγǧ */
	if (((sc->stat & SOCK_CNCT_ON) == 0) && ((sc->stat & SOCK_LISTEN) == 0))
		return -ENOTCONN;

	if (flag != FD_READ)
		return 1;
	if (sc->stat & SOCK_LISTEN)
	{
		if (sc->newsc_next != NULL)
			return 1;
	}
	else
		if (sc->recvBuf != NULL)
			return 1;
	if (polling)
		return 0;

	sc->waitProc = get_current_task();

	return 0;
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


static SOCKET_INFO sockInfo={send,recv,connect,shutdown,listen,accept,select};


/*
 * GLOBAL
 * Init UDP.
 */
int initTcp()
{
	registSocket(&sockInfo,IPPROTO_TCP);
	waitCnctSc.next=waitCnctSc.prev=&waitCnctSc;
	waitLstnSc.next=waitLstnSc.prev=&waitLstnSc;

	return 0;
}

/***************************************************************************/
void test_tcp()
{
	printk("waitNext=%x\n",*(uint*)0x16be64);
}
/****************************************************************************/
