/*
 * ext2_fs.c
 *
 * Copyright 2003, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * EXT2ե륷ƥɥ饤С
 */


#include<types.h>
#include<lib.h>
#include<mm.h>
#include<fs.h>
#include<device.h>
#include<proc.h>
#include<time.h>
#include<errno.h>
#include<test.h>
#include<sys/statvfs.h>


enum{
	INT_LOG=2,		/* intΥХȿΣ򺬤Ȥп */
};


/************************************************************************************
 *
 * < Common class >
 *
 * PUBLIC functions.
 *  blkToSct():֥å饻򻻽Ф롣
 *  writeBackSbGd():ѡ֥åȥ롼ץǥץǥ˽᤹
 *
 *************************************************************************************/


enum{
	EXT2_BLOCK_BASE=1024,			/* ֥å׻뤿Υ١Хȿ */
	EXT2_BLOCK_BASE_LOG=10,			/* ١֥åΣ򺬤Ȥп */

	EXT2_SUPER_BLK_POSITION=1024,	/* ѡ֥åΰ֡ƬΥХȿˡ */

	EXT2_GROUP_DESC_BLK=1,			/* group descriptor block number. */

	/* File system states at "s_state". */
	EXT2_VALID_FS=0x0001,			/* Unmounted cleanly */
	EXT2_ERROR_FS=0x0002,			/* Errors detected */

	/* Maximal mount counts. */
	EXT2_DFL_MAX_MNT_COUNT=20,
	EXT2_DFL_CHECKINTERVAL=0,

	/* Magic number at "s_magic". */
	EXT2_SUPER_MAGIC=0xEF53,

	/* Behaviour when detecting errors. */
	EXT2_ERRORS_CONTINUE=1,			/* continue as if nothing happened. */
	EXT2_ERRORS_RO=2,				/* remount read-only. */
	EXT2_ERRORS_PANIC=3,			/* cause a kernel panic. */
	EXT2_ERRORS_DEFAULT=EXT2_ERRORS_CONTINUE,

	/* revision level value. */
	EXT2_GOOD_OLD_REV=0,			/* original format. */
	EXT2_DYNAMIC_REV=1,				/* V2 format with dynamic inode sizes. */

	/* EXT2_*_INO values at "s_first_ino". */
	EXT2_BAD_INO=0x01, 				/* bad blocks inode */
	EXT2_ROOT_INO=0x02, 			/* root directory inode */
	EXT2_ACL_IDX_INO=0x03, 			/* ACL index inode (deprecated?) */
	EXT2_ACL_DATA_INO=0x04, 		/* ACL data inode (deprecated?) */
	EXT2_BOOT_LOADER_INO=0x05,		/* boot loader inode */
	EXT2_UNDEL_DIR_INO=0x06,		/* undelete directory inode */
};

/* super block. */
typedef struct{
	uint s_inodes_count;			/* 00,total number of inodes. */
	uint s_blocks_count;			/* 04,total number of blocks. */
	uint s_r_blocks_count;			/* 08,total number of blocks reserved for the usage of the super user. */
	uint s_free_blocks_count;		/* 0c,total number of free blocks. */
	uint s_free_inodes_count;		/* 10,total number of free inodes. */
	uint s_first_data_block;		/* 14,the block containing the superblock structure.block size larger than 1KB=0,block size of 1KB=1. */
	uint s_log_block_size;			/* 18,number of bits to shift left the value 1024 for computing block size. */
	int s_log_frag_size;			/* 1c,number of bits to shift the value 1024 for computing fragment size. */
	uint s_blocks_per_group;		/* 20,total number of blocks per group. */
	uint s_frags_per_group;			/* 24,total number of fragments per group. */
	uint s_inodes_per_group;		/* 28,total number of inodes per group. */
	uint s_mtime;					/* 2c,Unix time,last time the file system was mounted. */
	uint s_wtime;					/* 30,Unix time,last write access to the file system. */
	ushort s_mnt_count;				/* 34,how many time the file system was mounted. */
	short s_max_mnt_count;			/* 36,the maximum number of mount times. */
	ushort s_magic;					/* 38,magic number 0xEF53. */
	ushort s_state;					/* 3a,file system state. */
	ushort s_errors;				/* 3c,behaviour when an error is detected. */
	ushort s_minor_rev_level;		/* 3e,minor revision level. */
	uint s_lastcheck;				/* 40,Unix time,last file system check. */
	uint s_checkinterval;			/* 44,between file system checks. */
	uint s_creator_os;				/* 48,os that created the file system. */
	uint s_rev_level;				/* 4c,revision level value. */
	ushort s_def_resuid;			/* 4e,default user id for reserved blocks. */
	ushort s_def_resgid;			/* 50,default group id for reserved blocks. */

	/* EXT2_DYNAMIC_REV Specific */
	uint s_first_ino;				/* 54,index to the first inode useable for standard files. */
	ushort s_inode_size;			/* 58,size of the inode structure. */
	ushort s_block_group_nr;		/* 5a,block group number hosting this superblock. */
	uint s_feature_compat;			/* 5c,bitmask of compatible features. */
	uint s_feature_incompat;		/* 60,bitmask of incompatible features. */
	uint s_feature_ro_compat;		/* 64,bitmask of read-only features. */
	uchar s_uuid[16];				/* 68,volume id. */
	char s_volume_name[16];			/* 78,volume name,mostly unusued. */
	char s_last_mounted[64];		/* 88,last mounted directory path,not normally used. */
	uint s_algo_bitmap;				/* compression algorithms. */

	/* Performance Hints */
	uchar s_prealloc_blocks;		/* nmber of blocks to try to preallocate. */
	uchar s_prealloc_dir_blocks;	/* number to preallocate for dirs. */
	ushort s_padding1;

	/* Journaling Support */
	uchar s_journal_uuid[16];		/*  */
	uint s_journal_inum;			/*  */
	uint s_journal_dev;				/*  */
	uint s_last_orphan;				/*  */

	/* ꡼¸ȼ */
	int blockSize;					/* ֥å */
	int sectorSize;					/*  */
	int sectors;					/* 1֥åΥ */
}SUPER_BLOCK;

/* groupe descriptor. */
typedef struct{
	uint bg_block_bitmap;			/* 0,the first block of the "block bitmap". */
	uint bg_inode_bitmap;			/* 4,the first block of the "inode bitmap". */
	uint bg_inode_table;			/* 8,the first block of the "inode table". */
	ushort bg_free_blocks_count;	/* c,total number of free blocks. */
	ushort bg_free_inodes_count;	/* e,total number of free inodes. */
	ushort bg_used_dirs_count;		/* 10,number of inodes allocated to directories. */
	ushort bg_pad;
	uint bg_reserved[3];
}GROUP_DESC;

/* ӥåȥޥå׹¤Ρ */
typedef struct{
	uchar *bitmap;			/* ӥåȥޥåץХåե */
	ushort currentGd;		/* ߤΥ롼 */
	ushort currentByte;		/* ߤΥӥåȥޥåפΥХȥեåȡ */
}BITMAP;


/* PRIVATE,ӥåȥޥåפǥХ֥ͤåΰ֤롣 */
static const uchar emptyBit[]={
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,7,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,0,
};

/* ޥȻ롣 */
static SUPER_BLOCK *superBlock[MAX_DEVICE_OPEN];
static GROUP_DESC *groupDesc[MAX_DEVICE_OPEN];
static int groupDescSize[MAX_DEVICE_OPEN];			/* ֥åñ̤Υ */


/*
 * PUBLIC
 * ֥å饻򻻽Ф롣
 * parameters : device inode,block
 * return : secter number
 */
static inline uint blkToSct(int din,uint blk)
{
	return superBlock[din]->blockSize*blk/superBlock[din]->sectorSize;
}


/*
 * PUBLIC
 * ѡ֥åȥ롼ץǥץǥ˽᤹
 * return : 0 or error=-1
 */
static int writeBackSbGd(int din)
{
	SUPER_BLOCK *sb=superBlock[din];
	int sectors=sb->sectors;
	void *buf;


	/* 롼ץǥץǥ˽᤹ */
	write_cache(din,groupDesc[din],groupDescSize[din]*sectors,blkToSct(din,EXT2_GROUP_DESC_BLK+sb->s_first_data_block));
	write_cache(din,groupDesc[din],groupDescSize[din]*sectors,
		blkToSct(din,EXT2_GROUP_DESC_BLK+sb->s_first_data_block+sb->s_blocks_per_group));

	/* ѡ֥åǥ˽᤹ */
	if((buf=kmalloc(sb->blockSize))==NULL)return -1;
	/*read_cache(din,buf,sectors,blkToSct(din,sb->s_first_data_block));*/
	memset(buf,0,sb->blockSize);
	if(sb->s_first_data_block==1)memcpy(buf,sb,sizeof(SUPER_BLOCK));
	else memcpy((char*)buf+EXT2_SUPER_BLK_POSITION,sb,sizeof(SUPER_BLOCK));
	write_cache(din,buf,sectors,blkToSct(din,sb->s_first_data_block));
	if(sb->s_first_data_block==0)
	{
		memset((char*)buf+EXT2_SUPER_BLK_POSITION,0,sizeof(SUPER_BLOCK));
		memcpy(buf,sb,sizeof(SUPER_BLOCK));
	}
	write_cache(din,buf,sectors,blkToSct(din,sb->s_first_data_block+sb->s_blocks_per_group));
	kfree(buf);

	return 0;
}


/***********************************************************************
 *
 * < Block class >
 *
 * PUBLIC functions.
 *  getBlock():֥å롣
 *  releaseBlock():֥å롣
 *  initBlock():ޥȻν
 *  umountBlock():ޥȻν
 *
 ***********************************************************************/

/* PUBLIC. */
static BITMAP blockBm[MAX_DEVICE_OPEN];
static BITMAP ReleaseBlkBm[MAX_DEVICE_OPEN];


/*
 * PUBLIC
 * ֥å롣
 * parameters : device inode
 * return : empty block or error number;
 */
static int getBlock(int din)
{
	GROUP_DESC *gd=groupDesc[din];
	SUPER_BLOCK *sb=superBlock[din];
	int block;
	int i,last;


	/* ߤΥ롼פ˶֥åʤ */
	if(gd[blockBm[din].currentGd].bg_free_blocks_count==0)
	{
		int sectors=sb->sectors;

		/* ߤΥӥåȥޥåפ᤹ */
		if(write_cache(din,blockBm[din].bitmap,sectors,blkToSct(din,gd[blockBm[din].currentGd].bg_block_bitmap))!=sectors)
			return -EIO;

		/* ֥åΤ륰롼פõ */
		for(i=blockBm[din].currentGd+1;;++i)
		{
			if(gd[i].bg_block_bitmap==0)i=0;
			if(i==blockBm[din].currentGd)return -ENOSPC;
			if(gd[i].bg_free_blocks_count!=0)break;
		}
		if(read_cache(din,blockBm[din].bitmap,sectors,blkToSct(din,gd[i].bg_block_bitmap))!=sectors)return -EIO;
		blockBm[din].currentGd=i;
	}

	/* ӥåȥޥåפθ */
	i=blockBm[din].currentByte;
	last=sb->s_blocks_per_group/BYTE_BIT;
	while(blockBm[din].bitmap[i]==0xff)
		if(++i>=last)i=0;
	blockBm[din].currentByte=i;
	block=i*BYTE_BIT+emptyBit[blockBm[din].bitmap[i]];
	blockBm[din].bitmap[i]|=1<<emptyBit[blockBm[din].bitmap[i]];

	/* ֥åȤ򸺤餹 */
	--gd[blockBm[din].currentGd].bg_free_blocks_count;
	--sb->s_free_blocks_count;

	return block+blockBm[din].currentGd*sb->s_blocks_per_group+sb->s_first_data_block;
}


/*
 * PUBLIC
 * ֥å롣
 * parameters : device inode,block number
 */
static void releaseBlock(int din,uint block)
{
	uchar *bitmap;
	int gd_index;
	SUPER_BLOCK *sb=superBlock[din];
	int sectors=sb->sectors;
	int a;


	/* ֥åӥåȥޥåפθ */
	gd_index=(block-sb->s_first_data_block)/(sb->s_blocks_per_group);
	if(gd_index==blockBm[din].currentGd)bitmap=blockBm[din].bitmap;
	else if(gd_index==ReleaseBlkBm[din].currentGd)bitmap=ReleaseBlkBm[din].bitmap;
	else
	{
		if(write_cache(din,ReleaseBlkBm[din].bitmap,sectors,blkToSct(din,groupDesc[din][ReleaseBlkBm[din].currentGd].bg_block_bitmap))!=
			sectors)return;
		if(read_cache(din,ReleaseBlkBm[din].bitmap,sectors,blkToSct(din,groupDesc[din][gd_index].bg_block_bitmap))!=sectors)
			return;
		ReleaseBlkBm[din].currentGd=gd_index;
		bitmap=ReleaseBlkBm[din].bitmap;
	}

	/* ӥåȥޥåפ˽񤭹ࡣ */
	a=block-gd_index*sb->s_blocks_per_group-sb->s_first_data_block;
	bitmap[a/BYTE_BIT]&=~(1<<a%BYTE_BIT);

	/* ֥åȤ䤹 */
	++groupDesc[din][gd_index].bg_free_blocks_count;
	++sb->s_free_blocks_count;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode
 * return : groupe descriptor number or error=-1
 */
static int initBlock(int din)
{
	SUPER_BLOCK *sb=superBlock[din];
	GROUP_DESC *gd=groupDesc[din];
	int i;


	/* ֥åӥåȥޥåפɤ߹ࡣ */
	if((blockBm[din].bitmap=(uchar*)kmalloc(sb->blockSize))==NULL)return -1;
	blockBm[din].currentGd=0;
	blockBm[din].currentByte=0;

	for(i=0;gd[i].bg_free_blocks_count==0;++i)
		if(gd[i].bg_block_bitmap==0)return 0;
	if(read_direct(din,blockBm[din].bitmap,sb->sectors,blkToSct(din,gd[i].bg_block_bitmap))!=sb->sectors)return -1;
	blockBm[din].currentGd=i;

	for(i=0;blockBm[din].bitmap[i]==0xff;++i);
	blockBm[din].currentByte=i;

	if((ReleaseBlkBm[din].bitmap=(uchar*)kmalloc(sb->blockSize))==NULL)return -1;
	ReleaseBlkBm[din].currentGd=0;
	ReleaseBlkBm[din].currentByte=0;
	if(read_direct(din,blockBm[din].bitmap,sb->sectors,blkToSct(din,gd[0].bg_block_bitmap))!=sb->sectors)return -1;

	return blockBm[din].currentGd;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode
 */
static void umountBlock(int din)
{
	int sectors=superBlock[din]->sectors;


	write_cache(din,ReleaseBlkBm[din].bitmap,sectors,blkToSct(din,groupDesc[din][ReleaseBlkBm[din].currentGd].bg_block_bitmap));
	kfree(ReleaseBlkBm[din].bitmap);
	write_cache(din,blockBm[din].bitmap,sectors,blkToSct(din,groupDesc[din][blockBm[din].currentGd].bg_block_bitmap));
	kfree(blockBm[din].bitmap);
}


/***********************************************************************
 *
 * Inode
 *
 ***********************************************************************/

enum{
	/* at i_mode */
	EXT2_S_IFMT=  0xF000,	/* format mask */
	EXT2_S_IFSOCK=0xC000,	/* socket */
	EXT2_S_IFLNK= 0xA000,	/* symbolic link */
	EXT2_S_IFREG= 0x8000,	/* regular file */
	EXT2_S_IFBLK= 0x6000,	/* block device */
	EXT2_S_IFDIR= 0x4000,	/* directory */
	EXT2_S_IFCHR= 0x2000,	/* character device */
	EXT2_S_IFIFO= 0x1000,	/* fifo */
	EXT2_S_ISUID= 0x0800,	/* SUID */
	EXT2_S_ISGID= 0x0400,	/* SGID */
	EXT2_S_ISVTX= 0x0200,	/* sticky bit */
	EXT2_S_IRWXU= 0x01C0,	/* user access rights mask */
	EXT2_S_IRUSR= 0x0100,	/* read */
	EXT2_S_IWUSR= 0x0080,	/* write */
	EXT2_S_IXUSR= 0x0040,	/* execute */
	EXT2_S_IRWXG= 0x0038,	/* group access rights mask */
	EXT2_S_IRGRP= 0x0020,	/* read */
	EXT2_S_IWGRP= 0x0010,	/* write */
	EXT2_S_IXGRP= 0x0008,	/* execute */
	EXT2_S_IRWXO= 0x0007,	/* others access rights mask */
	EXT2_S_IROTH= 0x0004,	/* read */
	EXT2_S_IWOTH= 0x0002,	/* write */
	EXT2_S_IXOTH= 0x0001,	/* execute */

	/* behaviour control flags at "i_flags". */
	EXT2_SECRM_FL=0x00000001,		/* secure deletion */
	EXT2_UNRM_FL=0x00000002,		/* record for undelete */
	EXT2_COMPR_FL=0x00000004,		/* compressed file */
	EXT2_SYNC_FL=0x00000008,		/* synchronous updates */
	EXT2_IMMUTABLE_FL=0x00000010,	/* immutable file */
	EXT2_APPEND_FL=0x00000020,		/* append only */
	EXT2_NODUMP_FL=0x00000040,		/* do not dump/delete file */
	EXT2_NOATIME_FL=0x00000080,		/* do not update .i_atime */
	EXT2_DIRTY_FL=0x00000100,		/* dirty (file is in use?) */
	EXT2_COMPRBLK_FL=0x00000200,	/* compressed blocks */
	EXT2_NOCOMPR_FL=0x00000400,		/* access raw compressed data */
	EXT2_ECOMPR_FL=0x00000800,		/* compression error */
	EXT2_BTREE_FL=0x00010000,		/* b-tree format directory */
	EXT2_INDEX_FL=0x00010000,		/* Hash indexed directory */

	EXT2_ROOT_INODE_INDEX=2,		/* 롼inodeΥǥå */

	/* indirect data block index. */
	INODE_DATA_INDIRECT1=12,		/* ܻȥ֥å͡ */
	INODE_DATA_INDIRECT2=13,		/* Ŵܻȥ֥å͡ */
	INODE_DATA_INDIRECT3=14,		/* Ŵܻȥ֥å͡ */

	INODE_DATA_BLOCK_NUM=15,		/* ǡ֥å */

	INODE_BLOCK_SIZE=512,			/* inodei_blocksΥ֥åñ̥Хȿ */
};


typedef struct{
	ushort i_mode;			/* 00,format of the described file and the access rights. */
	ushort i_uid;			/* 02,user id. */
	uint i_size;			/* 04,size of the file in bytes. */
	uint i_atime;			/* 08,Unix time,last time this file was accessed. */
	uint i_ctime;			/* 0c,Unix time,when file was created. */
	uint i_mtime;			/* 10,Unix time,last time this file was modified. */
	uint i_dtime;			/* 14,Unix time,when file was deleted. */
	ushort i_gid;			/* 18,group id. */
	ushort i_links_count;	/* 1a,how many times this inode is linked.(ǥ쥯ȥ) */
	uint i_blocks;			/* 1c,both currently in used and currently reserved blocks.number of 512 bytes block. */
	uint i_flags;			/* 20,behave when accessing the data for this inode. */
	uint i_number;			/* 24,OS dependent 1,inode number(open) */

	/*
	 * data block number.
	 * 112 direct block number.
	 * 13 indirect block number.
	 * 14 bi-indirect block number.
	 * 15 tri-indirect block number.
	 * Each indirect block array contains as many entries of 32bit block numbers as possible
	 * (to fill one entire block).
	 */
	uint i_block[INODE_DATA_BLOCK_NUM];		/* 28 */
	uint i_generation;		/* 64,file version (used by NFS). */
	uint i_file_acl;		/* 68,block number containing the extended attributes, */
	uint i_dir_acl;			/* 6c,"high size" of the file. */
	uint i_faddr;			/* 70,location of the last file fragment. */
	uint i_osd2[3];			/* 74,OS dependent 2 */
}INODE;

typedef struct{
	const char *path;
	INODE *dirInode;
	INODE *inodeBuf;
}INODE_PARAM;


/*====================================================================================================*
 *                                       read data                                                    *
 *====================================================================================================*/

typedef int (*READ_DATA)(int,void*,uint*,size_t,size_t,int);


static int readDirectBlock(int,void*,uint*,size_t,size_t,int);
static int readIndirectBlock(int,void*,uint*,size_t,size_t,int);


/* PRIVATE */
static READ_DATA readIndirect[]={readDirectBlock,readIndirectBlock,readIndirectBlock};


/*
 * PRIVATE
 * ǡɹߡ
 * parameters : device inode,buffer,block table,֥åơ֥νߥ,֥åơ֥γϥХȿ,node
 * return : ߥ or error number
 */
static int readDirectBlock(int din,void *buf,uint *block_tbl,size_t size,size_t begin,int node)
{
	char *tmp_buf=NULL;
	int block_size=superBlock[din]->blockSize;
	int sectors=superBlock[din]->sectors;
	int rest_size=size;
	int rest;
	int s,t;


	block_tbl+=begin/block_size;

	/* ϥХ֥ͤåǤʤ硣 */
	if(begin%block_size>0)
	{
		s=begin%block_size;
		t=(rest_size>block_size-s)?block_size-s:rest_size;
		if((tmp_buf=(char*)kmalloc(block_size))==NULL)return -ENOMEM;
		if((rest=read_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		memcpy(buf,tmp_buf+s,t);
		buf=(char*)buf + t;
		rest_size-=t;
		++block_tbl;
	}

	for(;rest_size>=block_size;rest_size-=block_size)
	{
		if((rest=read_cache(din,buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		buf=(char*)buf + block_size;
		++block_tbl;
	}

	/* ǸΥХ֥ͤåǤʤ硣 */
	if(rest_size>0)
	{
		if(tmp_buf==NULL)
			if((tmp_buf=(char*)kmalloc(block_size))==NULL)return -ENOMEM;
		if((rest=read_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		memcpy(buf,tmp_buf,rest_size);
	}

	kfree(tmp_buf);

	return size;

ERR:
	kfree(tmp_buf);

	return rest;
}


/*
 * PRIVATE
 * ɹߡ
 * parameters : ǥХinode,Хåե,֥åơ֥,֥åơ֥ɹߥ,֥åơ֥γϥХȿ,node
 * return : ߥ or error number
 */
static int readIndirectBlock(int din,void *buf,uint *block_tbl,size_t size,size_t begin,int node)
{
	enum{UINT_LOG=2};	/* uintХȥΣ򺬤Ȥп */

	uint *block_buf;
	uint rest_size=size;
	int sectors=superBlock[din]->sectors;
	int block_size=superBlock[din]->blockSize;
	int idrct_block_log=EXT2_BLOCK_BASE_LOG+superBlock[din]->s_log_block_size-UINT_LOG;
	int idrct_size=block_size<<(idrct_block_log*node);
	int rest;
	int s,t;


	if((block_buf=(uint*)kmalloc(block_size))==NULL)return -ENOMEM;

	block_tbl+=begin/idrct_size;

	/* ϥХ֥ͤåǤʤ硣 */
	if(begin%idrct_size>0)
	{
		s=begin%idrct_size;
		t=(rest_size>idrct_size-s)?idrct_size-s:rest_size;
		if((rest=read_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		if((rest=readIndirect[node-1](din,buf,block_buf,t,s,node-1))<0)goto ERR;
		buf=(char*)buf + t;
		rest_size-=t;
		++block_tbl;
	}

	for(;rest_size>=idrct_size;rest_size-=idrct_size)
	{
		if((rest=read_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		if((rest=readIndirect[node-1](din,buf,block_buf,idrct_size,0,node-1))<0)goto ERR;
		buf=(char*)buf + idrct_size;
		++block_tbl;
	}

	if(rest_size>0)
	{
		if((rest=read_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		if((rest=readIndirect[node-1](din,buf,block_buf,rest_size,0,node-1))<0)goto ERR;
	}

	kfree(block_buf);

	return size;
ERR:
	kfree(block_buf);

	return rest;
}


/*
 * PUBLIC
 * եǡɤ߹ࡣ
 * parameters : device inode,inode,buffer,read size,begin byte
 * return : read size or error number
 */
static int readData(int din,INODE *inode,void *buf,size_t size,size_t begin)
{
	uint64 nsize;
	uint *table;
	size_t rest_size=size;
	int sectors=superBlock[din]->sectors;
	int block_size=superBlock[din]->blockSize;
	int nblock=block_size/sizeof(uint);			/* ֥åơ֥Υ֥å */
	int rest;
	int read_size;
	int i;


	nsize=INODE_DATA_INDIRECT1*block_size;	/* ľܥ֥å */
	if(begin<nsize)goto DIRECT;
	begin-=nsize;
	nsize=block_size*nblock;				/* ܥ֥å */
	for(i=INODE_DATA_INDIRECT1;;++i)
	{
		if(begin<nsize)goto INDIRECT;
		begin-=nsize;
		nsize*=nblock;
	}

DIRECT:
	/* ľܥ֥å */
	read_size=(rest_size>nsize-begin)?nsize-begin:rest_size;
	if((rest=readDirectBlock(din,buf,inode->i_block,read_size,begin,0))<0)return rest;
	if((rest_size-=read_size)==0)return size;
	buf=(char*)buf + read_size;
	begin=0;

	/* ܥ֥å */
	nsize=block_size*nblock;				/* ܥ֥å */
	i=INODE_DATA_INDIRECT1;
INDIRECT:
	if((table=(uint*)kmalloc(block_size))==NULL)return -ENOMEM;
	for(;;++i)
	{
		if((rest=read_cache(din,table,sectors,blkToSct(din,inode->i_block[i])))<0)goto ERR;
		read_size=(rest_size>nsize-begin)?nsize-begin:rest_size;
		if((rest=readIndirect[i-INODE_DATA_INDIRECT1](din,buf,table,read_size,begin,i-INODE_DATA_INDIRECT1))<0)goto ERR;
		if((rest_size-=read_size)==0)break;
		buf=(char*)buf + read_size;
		begin=0;
		nsize*=nblock;
	}

	kfree(table);

	return size;
ERR:
	kfree(table);

	return rest;
}


/*====================================================================================================*
 *                                      write data                                                    *
 *====================================================================================================*/

typedef int (*WRITE_DATA)(int,void*,uint*,size_t,size_t,size_t,int);


static int writeDirectBlock(int,void*,uint*,size_t,size_t,size_t,int);
static int writeIndirectBlock(int,void*,uint*,size_t,size_t,size_t,int);


/* PRIVATE */
static WRITE_DATA writeIndirect[]={writeDirectBlock,writeIndirectBlock,writeIndirectBlock};


/*
 * PRIVATE
 * ǡߡ
 * parameters : device inode,buffer,block table,ߥ,ϥХȿ,ե륵,node
 * return : ߥ or error number
 */
static int writeDirectBlock(int din,void *buf,uint *block_tbl,size_t size,size_t begin,size_t file_size,int node)
{
	char *tmp_buf=NULL;
	int block_size=superBlock[din]->blockSize;
	int sectors=superBlock[din]->sectors;
	int rest_size=size;
	int rest;
	int s,t;


	block_tbl+=begin/block_size;

	/*
	 * ƺѥ֥åؤνߡ
	 */
	/* ϥХ֥ͤåǤʤ硣 */
	if(begin%block_size>0)
	{
		s=begin%block_size;
		t=(rest_size>block_size-s)?block_size-s:rest_size;
		if((tmp_buf=(char*)kmalloc(block_size))==NULL)return -ENOMEM;
		if((rest=read_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		memcpy(tmp_buf+s,buf,t);
		if((rest=write_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		buf=(char*)buf + t;
		rest_size-=t;
		++block_tbl;
	}

	for(;rest_size>=block_size;rest_size-=block_size)
	{
		if(*block_tbl==0)goto NEXT;
		if((rest=write_cache(din,buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		buf=(char*)buf + block_size;
		++block_tbl;
	}

	/* ǸΥХ֥ͤåǤʤ硣 */
	if(rest_size>0)
	{
		if(*block_tbl==0)goto NEXT;
		if(tmp_buf==NULL)
			if((tmp_buf=(char*)kmalloc(block_size))==NULL)return -ENOMEM;
		if((rest=read_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		memcpy(tmp_buf,buf,rest_size);
		if((rest=write_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
	}

	kfree(tmp_buf);

	return size;
NEXT:
	/*
	 * ̤ƥ֥åؤνߡ
	 */
	for(;rest_size>=block_size;rest_size-=block_size)
	{
		if((rest=getBlock(din))<0)goto ERR;
		*block_tbl=rest;
		if((rest=write_cache(din,buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		buf=(char*)buf + block_size;
		++block_tbl;
	}

	/* ǸΥХ֥ͤåǤʤ硣 */
	if(rest_size>0)
	{
		if((rest=getBlock(din))<0)goto ERR;
		*block_tbl=rest;
		if(tmp_buf==NULL)
			if((tmp_buf=(char*)kmalloc(block_size))==NULL)return -ENOMEM;
		memcpy(tmp_buf,buf,rest_size);
		if((rest=write_cache(din,tmp_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
	}

	kfree(tmp_buf);

	return size;
ERR:
	kfree(tmp_buf);

	return rest;
}


/*
 * PRIVATE
 * ܽߡ
 * parameters : device inode,buffer,block table,ߥ,ϥХȿ,ե륵,node
 * return : ߥ or error number
 */
static int writeIndirectBlock(int din,void *buf,uint *block_tbl,size_t size,size_t begin,size_t file,int node)
{
	enum{UINT_LOG=2};	/* uintХȥΣ򺬤Ȥп */

	uint *block_buf;
	uint rest_size=size;
	int64 file_size=file;
	int sectors=superBlock[din]->sectors;
	int block_size=superBlock[din]->blockSize;
	int idrct_block_log=EXT2_BLOCK_BASE_LOG+superBlock[din]->s_log_block_size-UINT_LOG;
	int idrct_size=block_size<<(idrct_block_log*node);
	int rest;
	int s,t;


	if((block_buf=(uint*)kmalloc(block_size))==NULL)return -ENOMEM;
	block_tbl+=begin/idrct_size;

	/*
	 * ֥åγ硣
	 */
	if(begin%idrct_size>0)
	{
		s=begin%idrct_size;
		t=(rest_size>idrct_size-s)?idrct_size-s:rest_size;
		if((rest=read_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		if((rest=writeIndirect[node-1](din,buf,block_buf,t,s,file_size,node-1))<0)goto ERR;
		buf=(char*)buf + t;
		rest_size-=t;
		file_size-=t;
		++block_tbl;
	}

	for(;rest_size>=idrct_size;rest_size-=idrct_size)
	{
		if(*block_tbl==0)goto NEXT;
		if((rest=read_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		if((rest=writeIndirect[node-1](din,buf,block_buf,idrct_size,0,file_size,node-1))<0)goto ERR;
		buf=(char*)buf + idrct_size;
		file_size-=idrct_size;
		++block_tbl;
	}

	if(rest_size>0)
	{
		if(*block_tbl==0)goto NEXT;
		if((rest=read_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		if((rest=writeIndirect[node-1](din,buf,block_buf,rest_size,0,file_size,node-1))<0)goto ERR;
		file_size-=rest_size;
	}

	if(file_size<0)
		if((rest=write_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;

	kfree(block_buf);

	return size;
NEXT:
	/*
	 * ֥åγʤ硣
	 */
	/* ֥åơ֥Υ֥å᤹ */
	if(file_size<0)
		if((rest=write_cache(din,block_buf,sectors,blkToSct(din,*(block_tbl-1))))<0)goto ERR;

	for(;rest_size>=idrct_size;rest_size-=idrct_size)
	{
		if((rest=getBlock(din))<0)goto ERR;
		*block_tbl=rest;
		memset(block_buf,0,block_size);
		if((rest=writeIndirect[node-1](din,buf,block_buf,idrct_size,0,0,node-1))<0)goto ERR;
		if((rest=write_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
		buf=(char*)buf + idrct_size;
		++block_tbl;
	}

	if(rest_size>0)
	{
		if((rest=getBlock(din))<0)goto ERR;
		*block_tbl=rest;
		memset(block_buf,0,block_size);
		if((rest=writeIndirect[node-1](din,buf,block_buf,rest_size,0,0,node-1))<0)goto ERR;
		if((rest=write_cache(din,block_buf,sectors,blkToSct(din,*block_tbl)))<0)goto ERR;
	}

	kfree(block_buf);

	return size;
ERR:
	kfree(block_buf);

	return rest;
}


/*
 * PUBLIC
 * եǡ֥å˽񤭹ࡣ
 * parameters : device inode,inode,buffer,read size,begin byte
 * return : write size or error number
 */
static int writeData(int din,INODE *inode,void *buf,size_t size,size_t begin)
{
	uint64 nsize;
	uint *table;
	size_t rest_size=size;
	size_t file_size=inode->i_size-begin;		/* ƺѤ߻ĤΥ */
	int sectors=superBlock[din]->sectors;
	int block_size=superBlock[din]->blockSize;
	int nblock=block_size/sizeof(uint);			/* ֥åơ֥Υ֥å */
	int rest;
	int write_size;
	int i;


	nsize=INODE_DATA_INDIRECT1*block_size;	/* ľܥ֥å */
	if(begin<nsize)goto DIRECT;
	begin-=nsize;
	nsize=block_size*nblock;				/* ܥ֥å */
	for(i=INODE_DATA_INDIRECT1;;++i)
	{
		if(begin<nsize)goto INDIRECT;
		begin-=nsize;
		nsize*=nblock;
	}

DIRECT:
	/* ľܥ֥å */
	write_size=(rest_size>nsize-begin)?nsize-begin:rest_size;
	if((rest=writeDirectBlock(din,buf,inode->i_block,write_size,begin,file_size,0))<0)return rest;
	if((rest_size-=write_size)==0)return size;
	buf=(char*)buf + write_size;
	begin=0;
	file_size-=write_size;

	/* ܥ֥å */
	nsize=block_size*nblock;				/* ܥ֥å */
	i=INODE_DATA_INDIRECT1;
INDIRECT:
	if((table=(uint*)kmalloc(block_size))==NULL)return -ENOMEM;
	for(;;++i)
	{
		if(inode->i_block[i]==0)goto EMPTY_INDIRECT;

		if((rest=read_cache(din,table,sectors,blkToSct(din,inode->i_block[i])))<0)goto ERR;
		write_size=(rest_size>nsize-begin)?nsize-begin:rest_size;
		if((rest=writeIndirect[i-INODE_DATA_INDIRECT1](din,buf,table,write_size,begin,file_size,i-INODE_DATA_INDIRECT1))<0)goto ERR;
		if((rest=write_cache(din,table,sectors,blkToSct(din,inode->i_block[i])))<0)goto ERR;
		if((rest_size-=write_size)==0)goto END;
		buf=(char*)buf + write_size;
		begin=0;
		file_size-=write_size;
		nsize*=nblock;
	}
EMPTY_INDIRECT:
	/* ̤ƥ֥å */
	for(;;++i)
	{
		if((rest=getBlock(din))<0)goto ERR;
		inode->i_block[i]=rest;
		memset(table,0,block_size);
		write_size=(rest_size>nsize-begin)?nsize-begin:rest_size;
		if((rest=writeIndirect[i-INODE_DATA_INDIRECT1](din,buf,table,write_size,0,0,i-INODE_DATA_INDIRECT1))<0)goto ERR;
		if((rest=write_cache(din,table,sectors,blkToSct(din,inode->i_block[i])))<0)goto ERR;
		if((rest_size-=write_size)==0)break;
		buf=(char*)buf + write_size;
		nsize*=nblock;
	}
END:
	kfree(table);

	return size;
ERR:
	kfree(table);

	return rest;
}


/*====================================================================================================*
 *                                          misc                                                      *
 *====================================================================================================*/

/* PRIVATE. */
static BITMAP inodeBm[MAX_DEVICE_OPEN];


/*
 * PUBLIC
 * inodeֹ椫顢inodeΥ֥åХåեɤ߹ࡣ
 * parameters : device inode,inode number,block buffer
 * return : inode addres or error number;
 */
static inline int readInode(int din,uint inumber,INODE *sector_buf)
{
	int gdesc;
	int index;
	uint sector;


	gdesc=(inumber-1)/superBlock[din]->s_inodes_per_group;
	index=(inumber-1)%superBlock[din]->s_inodes_per_group;
	sector=blkToSct(din,groupDesc[din][gdesc].bg_inode_table)+index*sizeof(INODE)/superBlock[din]->sectorSize;
	if(read_cache(din,sector_buf,1,sector)!=1)return -EIO;

	return (int)sector_buf+index*sizeof(INODE)%superBlock[din]->sectorSize;
}


/*
 * PUBLIC
 * inodeǥ˽񤭹ࡣ
 * parameters : device inode,inode number,block buffer
 * return : 0 or error number
 */
static inline int writeInode(int din,uint inumber,INODE *sector_buf)
{
	int gdesc;
	int index;
	uint sector;


	gdesc=(inumber-1)/superBlock[din]->s_inodes_per_group;
	index=(inumber-1)%superBlock[din]->s_inodes_per_group;
	sector=blkToSct(din,groupDesc[din][gdesc].bg_inode_table)+index*sizeof(INODE)/superBlock[din]->sectorSize;
	if(write_cache(din,sector_buf,1,sector)!=1)return -EIO;

	return 0;
}


/*
 * PUBLIC
 * inodeֹ롣
 * parameters : device inode
 * return : inode number or error=0
 */
static uint getNewInode(int din)
{
	GROUP_DESC *gd=groupDesc[din];
	SUPER_BLOCK *sb=superBlock[din];
	uint inode;
	int i,last;


	/* ߤΥ롼פ˶inodeʤ */
	if(gd[inodeBm[din].currentGd].bg_free_inodes_count==0)
	{
		int sectors=sb->sectors;

		/* ߤΥӥåȥޥåפ᤹ */
		if(write_cache(din,inodeBm[din].bitmap,sectors,blkToSct(din,gd[inodeBm[din].currentGd].bg_inode_bitmap))!=sectors)
			return 0;

		for(i=inodeBm[din].currentGd+1;;++i)
		{
			if(gd[i].bg_inode_bitmap==0)i=0;
			if(i==inodeBm[din].currentGd)return 0;
			if(gd[i].bg_free_inodes_count!=0)break;
		}
		if(read_cache(din,inodeBm[din].bitmap,sectors,blkToSct(din,gd[i].bg_inode_bitmap))!=sectors)return 0;
		inodeBm[din].currentGd=i;
	}

	/* ӥåȥޥåפθ */
	i=inodeBm[din].currentByte;
	last=sb->s_inodes_per_group/BYTE_BIT;
	while(inodeBm[din].bitmap[i]==0xff)
		if(++i>=last)i=0;
	inodeBm[din].currentByte=i;
	inode=i*BYTE_BIT+emptyBit[inodeBm[din].bitmap[i]];
	inodeBm[din].bitmap[i]|=1<<emptyBit[inodeBm[din].bitmap[i]];

	/* 롼ץǥץȥѡ֥å򹹿롣 */
	--gd[inodeBm[din].currentGd].bg_free_inodes_count;
	--sb->s_free_inodes_count;

	return inode+inodeBm[din].currentGd*sb->s_inodes_per_group+1;
}


/*
 * PUBLIC
 * inodeΰ롣
 * parameters : device inode,inode number
 * return : groupe desc index or error number
 */
static int releaseInodeArea(int din,int inumber)
{
	uchar *bitmap;
	int gd_index;
	SUPER_BLOCK *sb=superBlock[din];
	int sectors=sb->sectors;
	int a,rest;


	/* ֥åӥåȥޥåפθ */
	gd_index=(inumber-1)/(sb->s_inodes_per_group);
	if(gd_index!=inodeBm[din].currentGd)
	{
		if((bitmap=(uchar*)kmalloc(sb->blockSize))==NULL)return -ENOMEM;
		if((rest=read_cache(din,bitmap,sectors,blkToSct(din,groupDesc[din][gd_index].bg_inode_bitmap)))!=sectors)
			return rest;
	}
	else bitmap=inodeBm[din].bitmap;

	/* ӥåȥޥåפ˽񤭹ࡣ */
	a=inumber-gd_index*sb->s_inodes_per_group-1;
	bitmap[a/BYTE_BIT]&=~(1<<a%BYTE_BIT);
	if(gd_index!=inodeBm[din].currentGd)
	{
		/* ߤΥ֥å롼פȰפ뤫 */
		if(gd_index==blockBm[din].currentGd)
		{
			uchar *buf;

			buf=inodeBm[din].bitmap;
			inodeBm[din].bitmap=bitmap;
			bitmap=buf;
			inodeBm[din].currentGd=gd_index;
			inodeBm[din].currentByte=a/BYTE_BIT;
		}
		if((rest=write_cache(din,bitmap,sectors,blkToSct(din,groupDesc[din][gd_index].bg_inode_bitmap)))!=sectors)
			return rest;
		kfree(bitmap);
	}

	/* ֥åȤ䤹 */
	++groupDesc[din][gd_index].bg_free_inodes_count;
	++sb->s_free_inodes_count;

	return gd_index;
}


/*
 * PUBLIC
 * 롼ץǥץλѥǥ쥯ȥ롣
 * parameters : device inode
 */
static inline void addGrpUsedDir(int din)
{
	++groupDesc[din][inodeBm[din].currentGd].bg_used_dirs_count;
}

static inline void delGrpUsedDir(int din)
{
	--groupDesc[din][inodeBm[din].currentGd].bg_used_dirs_count;
}


/*
 * PUBLIC
 * inode롣
 * parameters : device inode,inode number
 * return : 0 or error number
 */
static int releaseInode(int din,INODE *inode)
{
	INODE *sector_buf;
	int rest;


	/* inodeΰ롣 */
	if((rest=releaseInodeArea(din,inode->i_number))<0)return rest;

	/* ǥ쥯ȥʤǥ쥯ȥꥫȤ򸺤餹 */
	if(inode->i_mode&EXT2_S_IFDIR)
		--groupDesc[din][rest].bg_used_dirs_count;

	/* inodeΥ󥯥Ȥ򥯥ꥢ롣 */
	inode->i_links_count=0;
	inode->i_dtime=(uint)sys_time(NULL);
	if((sector_buf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((rest=readInode(din,inode->i_number,sector_buf))<0)return rest;
	memcpy((void*)rest,inode,sizeof(INODE));
	if((rest=writeInode(din,inode->i_number,sector_buf))<0)return rest;
	kfree(sector_buf);

	return 0;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode,򳫻Ϥ֥å롼ץʥС
 * return : 0 or error=-1
 */
static int initInode(int din,int begin_gr)
{
/*	void *buf=NULL;
*/	int rest;
	GROUP_DESC *gd=groupDesc[din];
	SUPER_BLOCK *sb=superBlock[din];
	INODE *buf;
	int i;


	/* inodeӥåȥޥåפɤ߹ࡣ */
	if((inodeBm[din].bitmap=(uchar*)kmalloc(sb->blockSize))==NULL)return -1;
	inodeBm[din].currentGd=begin_gr;
	inodeBm[din].currentByte=0;

	i=begin_gr;
	do
	{
		if(gd[i].bg_free_inodes_count>0)
		{
			if(read_direct(din,inodeBm[din].bitmap,sb->sectors,blkToSct(din,gd[i].bg_inode_bitmap))!=sb->sectors)return -1;
			inodeBm[din].currentGd=i;
			for(i=0;inodeBm[din].bitmap[i]==0xff;++i);
			inodeBm[din].currentByte=i;
			break;
		}
		if(gd[++i].bg_inode_bitmap==0)i=0;
	}while(i!=begin_gr);

	/* 롼inodeν */
	if((buf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((rest=readInode(din,EXT2_ROOT_INODE_INDEX,buf))<0)goto ERR;
	((INODE*)rest)->i_number=EXT2_ROOT_INODE_INDEX;
	if(writeInode(din,EXT2_ROOT_INODE_INDEX,buf)<0)goto ERR;
	kfree(buf);

	return 0;

ERR:
	kfree(buf);

	return -1;
}


/*
 * PUBLIC
 * ޥȻν
 * parameters : device inode
 */
static int umountInode(int din)
{
	/* inodeӥåȥޥåפǥ˽᤹ */
	write_cache(din,inodeBm[din].bitmap,superBlock[din]->sectors,blkToSct(din,groupDesc[din][inodeBm[din].currentGd].bg_inode_bitmap));
	kfree(inodeBm[din].bitmap);

	return 0;
}


/*====================================================================================================*
 *                                       release                                                      *
 *====================================================================================================*/

typedef int (*RELEASE_DATA)(int,uint*,int,int);


static int releaseDirect(int,uint*,int,int);
static int releaseIndirect(int,uint*,int,int);
RELEASE_DATA releaseData[]={releaseDirect,releaseIndirect,releaseIndirect,releaseIndirect};


/*
 * PRIVATE
 * ľܥ֥åơ֥Υ֥å롣
 * parameters : device inode,block table,number of blocks,node=0
 * return : 0
 */
static int releaseDirect(int din,uint *table,int blk_num,int node)
{
	int i;


	for(i=0;i<blk_num;++i)releaseBlock(din,table[i]);

	return 0;
}


/*
 * PRIVATE
 * ܥ֥åơ֥Υ֥å롣
 * parameters : device inode,block table,number of blocks,node
 * return : 0 or error number
 */
static int releaseIndirect(int din,uint *table,int blk_num,int node)
{
	uint *buf;
	int sectors=superBlock[din]->sectors;
	int nblk=1<<((EXT2_BLOCK_BASE_LOG+superBlock[din]->s_log_block_size-INT_LOG)*node);
	int rest;
	int i;


	if((buf=(uint*)kmalloc(superBlock[din]->blockSize))==NULL)return -ENOMEM;

	i=0;
	for(;blk_num>nblk;blk_num-=nblk)
	{
		if((rest=read_cache(din,buf,sectors,blkToSct(din,table[i])))<0)goto END;
		if((rest=releaseData[node-1](din,buf,nblk,node-1))<0)goto END;
		releaseBlock(din,table[i++]);
	}
	if((rest=read_cache(din,buf,sectors,blkToSct(din,table[i])))<0)goto END;
	if((rest=releaseData[node-1](din,buf,blk_num,node-1))<0)goto END;
	releaseBlock(din,table[i]);

	rest=0;
END:
	kfree(buf);

	return rest;
}


/*
 * PUBLIC
 * ǡ֥å롣
 * parameters : device inode,inode address
 * return : 0 or error number
 */
static int releaseDataBlock(int din,INODE *inode)
{
	uint *buf;
	int sectors=superBlock[din]->sectors;
	int block_size=superBlock[din]->blockSize;
	int blk_num=ROUNDUP_DIV(inode->i_size,block_size);
	int nblk;
	int rest;
	int blk,i;


	/* ľܥ֥å */
	blk=(blk_num>=INODE_DATA_INDIRECT1)?INODE_DATA_INDIRECT1:blk_num;
	releaseDirect(din,inode->i_block,blk,0);
	for(i=0;i<blk;++i)inode->i_block[i]=0;
	if((blk_num-=blk)==0)goto END;

	/* ܥ֥å */
	if((buf=(uint*)kmalloc(block_size))==NULL)return -ENOMEM;
	blk=nblk=block_size/sizeof(uint);
	for(i=INODE_DATA_INDIRECT1;;++i)
	{
		if((rest=read_cache(din,buf,sectors,blkToSct(din,inode->i_block[i])))<0)goto ERR;
		if((rest=releaseData[i-INODE_DATA_INDIRECT1](din,buf,(blk_num>blk)?blk:blk_num,i-INODE_DATA_INDIRECT1))<0)goto ERR;
		releaseBlock(din,inode->i_block[i]);
		inode->i_block[i]=0;
		if((blk_num-=blk)<=0)break;
		blk*=nblk;
	}
	kfree(buf);
END:
	inode->i_size=0;

	return 0;
ERR:
	kfree(buf);

	return rest;
}


/********************************************************************************************************
 *
 * Directory
 *
 ********************************************************************************************************/

enum{
	EXT2_NAME_LEN=255,

	/* directory file types. */
	EXT2_FT_UNKNOWN=0,
	EXT2_FT_REG_FILE,
	EXT2_FT_DIR,
	EXT2_FT_CHRDEV,
	EXT2_FT_BLKDEV,
	EXT2_FT_FIFO,
	EXT2_FT_SOCK,
	EXT2_FT_SYMLINK,
	EXT2_FT_MAX,
};


typedef struct{
	uint inode;				/* inode number. */
	ushort rec_len;			/* directory entry length. */
	uchar name_len;			/* name length. */
	uchar file_type;		/* file type. */
	char name[0];			/* File name. */
}DIR_ENTRY;

typedef struct{
	const char *path;
	DIR_ENTRY *dirBuf;
	uint blkNumber;
}DIR_PARAM;

typedef int (*READ_DIR)(int,uint*,int,DIR_PARAM*,int,int (**READ_DIR)());


#define ROUNDUP4(a) (((a+3)>>2)<<2)		/* 4Ƿ夲 */


/*
 * PRIVATE
 * search entry.
 * parameters : device inode,path,entry buffer
 * return : directory entry address or failed=NULL
 */
static inline DIR_ENTRY *searchEntry(int din,const char *path,DIR_ENTRY *entry)
{
	DIR_ENTRY *last;


	last=(DIR_ENTRY*)((char*)entry+superBlock[din]->blockSize);
	if (entry->inode == EXT2_FT_UNKNOWN)
		entry=(DIR_ENTRY*)((char*)entry + entry->rec_len);

	for(;entry<last;entry=(DIR_ENTRY*)((char*)entry + entry->rec_len))
		if(cmpPathLength(entry->name,path,entry->name_len)==0)return entry;

	return NULL;
}


/*
 * PRIVATE
 * ȥ꡼õ
 * parameters : device inode,directory block,directory parameters struct
 * return : directory entry address or failed=NULL
 */
static inline DIR_ENTRY *searchEntrySpace(int din,DIR_PARAM *dparam)
{
	int add_len,ent_len;
	DIR_ENTRY *entry,*last,*new;


	add_len=strlen(dparam->path)+sizeof(DIR_ENTRY);
	entry=dparam->dirBuf;
	last=(DIR_ENTRY*)((char*)dparam->dirBuf+superBlock[din]->blockSize);

	if(entry->inode==EXT2_FT_UNKNOWN)
	{
		if(entry->rec_len>=add_len)
		{
			entry->name_len=add_len-sizeof(DIR_ENTRY);
			return entry;
		}
		else entry=(DIR_ENTRY*)((char*)entry + entry->rec_len);
	}

	for(;entry<last;entry=(DIR_ENTRY*)((char*)entry + entry->rec_len))
	{
		ent_len=ROUNDUP4(entry->name_len+sizeof(DIR_ENTRY));
		if((entry->rec_len-ent_len)>=add_len)
		{
			new=(DIR_ENTRY*)((char*)entry+ent_len);
			new->rec_len=entry->rec_len-ent_len;		/* ȥ꡼ */
			new->name_len=add_len-sizeof(DIR_ENTRY);	/* ͡ॵ */
			entry->rec_len=ent_len;
			return new;
		}
	}

	return NULL;
}


/*
 * PRIVATE
 * ȥ꡼뤫å롣
 * parameters : device inode,directory block,directory parameters struct
 * return : 0 or ȥ꡼=-1
 */
static inline int checkEntry(int din,DIR_PARAM *dparam)
{
	DIR_ENTRY *entry;


	entry=dparam->dirBuf;

	if(entry->inode==EXT2_FT_UNKNOWN)
	{
		if(entry->rec_len==superBlock[din]->blockSize)return 0;
	}
	else
	{
		if(cmpPathLength(entry->name,".",entry->name_len)==0)
		{
			entry=(DIR_ENTRY*)((char*)entry + entry->rec_len);
			if(entry->rec_len==superBlock[din]->blockSize-(sizeof(DIR_ENTRY)+sizeof(int)))return 0;
		}
	}

	return -1;
}


/*
 * PRIVATE
 * ȥ꡼롣
 * parameters : delete entry,directory buffer
 */
static inline void delEntry(DIR_ENTRY *entry,DIR_ENTRY *dir_buf)
{
	DIR_ENTRY *p;


	if(entry==dir_buf)entry->inode=EXT2_FT_UNKNOWN;
	else
	{
		for(p=dir_buf;(char*)p+p->rec_len != (char*)entry;p=(DIR_ENTRY*)((char*)p + p->rec_len));
		p->rec_len+=entry->rec_len;
	}
}


/*
 * PRIVATE
 * ľܥ֥åɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : directory entry address or error number
 */
static int readDirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int sectors=superBlock[din]->sectors;
	DIR_ENTRY *entry;
	int i;


	for(i=0;i<blk_num;++i)
	{
		if(read_cache(din,dparam->dirBuf,sectors,blkToSct(din,table[i]))!=sectors)return -EIO;

		if((entry=searchEntry(din,dparam->path,dparam->dirBuf))!=NULL)
		{
			dparam->blkNumber=table[i];
			return (int)entry;
		}
	}

	return -ENOENT;
}


/*
 * PRIVATE
 * ľܥ֥åɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : directory entry address or error number
 */
static int searchDirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int sectors=superBlock[din]->sectors;
	DIR_ENTRY *entry;
	int i;


	for(i=0;i<blk_num;++i)
	{
		if(read_cache(din,dparam->dirBuf,sectors,blkToSct(din,table[i]))!=sectors)return -EIO;
		if((entry=searchEntrySpace(din,dparam))!=NULL)
		{
			dparam->blkNumber=table[i];
			return (int)entry;
		}
	}

	return -ENOENT;
}


/*
 * PRIVATE
 * ľܥ֥åɹߡ
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : ʤ=-ENOENT or error number
 */
static int checkDirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int sectors=superBlock[din]->sectors;
	int i;


	for(i=0;i<blk_num;++i)
	{
		if(read_cache(din,dparam->dirBuf,sectors,blkToSct(din,table[i]))!=sectors)return -EIO;
		if(checkEntry(din,dparam)==-1)return -EEXIST;
	}

	return -ENOENT;
}


/*
 * PRIVATE
 * Υǥ쥯ȥ֥åõ
 * DIR_PARAMblkNumber0ʤǽΥ֥å0ʾʤblkNumberμΥ֥å֤
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : next block or error number
 */
static int searchNextBlock(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int i;


	if(dparam->blkNumber==0)return *table;
	else
		for(i=0;i<blk_num;++i)
			if(table[i]==dparam->blkNumber)
			{
				if(++i<blk_num)return table[i];
				dparam->blkNumber=0;
				break;
			}

	return -ENOENT;
}


/*
 * PRIVATE
 * ɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node,READ_DIR struct
 * return : directory entry address of error number
 */
static int readIndirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	uint *buf;
	int sectors=superBlock[din]->sectors;
	int nblk=1<<((EXT2_BLOCK_BASE_LOG+superBlock[din]->s_log_block_size-INT_LOG)*node);
	int rest;
	int i;


	if((buf=(uint*)kmalloc(superBlock[din]->blockSize))==NULL)return -ENOMEM;

	for(i=0;blk_num>nblk;blk_num-=nblk,++i)
	{
		if((rest=read_cache(din,buf,sectors,blkToSct(din,table[i])))<0)goto END;
		if((rest=readDirBlock[node-1](din,buf,nblk,dparam,node-1,readDirBlock))!=-ENOENT)goto END;
	}
	if((rest=read_cache(din,buf,sectors,blkToSct(din,table[i])))<0)goto END;
	rest=readDirBlock[node-1](din,buf,blk_num,dparam,node-1,readDirBlock);

END:
	kfree(buf);

	return rest;
}


/*
 * PRIVATE
 * read directory.
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : path,device inode,inode,directory parameters struct,READ_DIR struct
 * return : directory address or error number
 */
static int readDir(int din,INODE *inode,DIR_PARAM *dparam,READ_DIR *readDirBlock)
{
	uint *buf;
	int sectors=superBlock[din]->sectors;
	int block_size=superBlock[din]->blockSize;
	int blk_num=ROUNDUP_DIV(inode->i_size,block_size);
	int nblk;
	int rest;
	int blk,i;


	/* ľܥ֥å */
	blk=(blk_num>=INODE_DATA_INDIRECT1)?INODE_DATA_INDIRECT1:blk_num;
	if((rest=readDirBlock[0](din,inode->i_block,blk,dparam,0,readDirBlock))!=-ENOENT)return rest;
	if((blk_num-=blk)==0)return -ENOENT;

	/* ܥ֥å */
	if((buf=(uint*)kmalloc(block_size))==NULL)return -ENOMEM;
	nblk=block_size/sizeof(uint);
	blk=nblk;
	for(i=INODE_DATA_INDIRECT1;;++i)
	{
		if((rest=read_cache(din,buf,sectors,blkToSct(din,inode->i_block[i])))<0)goto END;
		if((rest=readDirBlock[i-INODE_DATA_INDIRECT1](din,buf,(blk_num>blk)?blk:blk_num,dparam,i-INODE_DATA_INDIRECT1,readDirBlock))!=-ENOENT)
			goto END;
		if((blk_num-=blk)<=0)break;
		blk*=nblk;
	}
END:
	kfree(buf);

	return rest;
}


static READ_DIR readDirBlock[]={readDirectDir,readIndirectDir,readIndirectDir,readIndirectDir};
static READ_DIR addDirBlock[]={searchDirectDir,readIndirectDir,readIndirectDir,readIndirectDir};
static READ_DIR checkDirBlock[]={checkDirectDir,readIndirectDir,readIndirectDir,readIndirectDir};
static READ_DIR searchDirBlock[]={searchNextBlock,readIndirectDir,readIndirectDir,readIndirectDir};


/*
 * PRIVATE
 * ǥ쥯ȥ˥ȥ꡼ĤäƤ뤫ǧ롣
 * parameters : device inode,directory inode
 * return : ȥ꡼ʤ=-ENOENT or error number
 */
static int inline checkDir(int din,INODE *inode)
{
	int rest;
	DIR_PARAM param;


	if((param.dirBuf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)return -ENOMEM;
	rest=readDir(din,inode,&param,checkDirBlock);
	kfree(param.dirBuf);

	return rest;
}


/*
 * PUBLIC
 * ƥǥ쥯ȥޤǸơparam˼Υѥȥǥ쥯ȥinode롣
 * parameters : device inode,INODE_PARAM struct,directory block buffer
 * return : 0 or error number
 */
static int searchParent(int din,INODE_PARAM *param,DIR_ENTRY *dir_buf)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;


	dparam.dirBuf=dir_buf;

	for(dparam.path=param->path;isLastPath(dparam.path)<0;)
	{
		/* ǥ쥯ȥꥨȥ꡼õ */
		if((rest=readDir(din,param->dirInode,&dparam,readDirBlock))<0)return rest;
		entry=(DIR_ENTRY*)rest;

		if(entry->file_type!=EXT2_FT_DIR)return -ENOTDIR;

		/* inode롣 */
		if((rest=readInode(din,entry->inode,param->inodeBuf))<0)return rest;
		param->dirInode=(INODE*)rest;
		param->dirInode->i_number=entry->inode;

		/* '/'μ˥ѥʤ롣 */
		for(dparam.path+=entry->name_len+1;*dparam.path=='/';++dparam.path);
	}
	param->path=dparam.path;

	return 0;
}


/*
 * PUBLIC
 * ƥǥ쥯ȥ꤫ѥΥȥ꡼롣
 * parameters : device inode,INODE_PARAM struct,directory block buffer
 * return : 0 or error number
 */
static int getInode(int din,INODE_PARAM *param,DIR_ENTRY *dir_buf)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;


	dparam.dirBuf=dir_buf;
	dparam.path=param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if((rest=readDir(din,param->dirInode,&dparam,readDirBlock))<0)return rest;
	entry=(DIR_ENTRY*)rest;

    /* inode롣 */
	if((rest=readInode(din,entry->inode,param->inodeBuf))<0)return rest;
	param->dirInode=(INODE*)rest;
	param->dirInode->i_number=entry->inode;

	return 0;
}


/*
 * PUBLIC
 * ƥǥ쥯ȥ˥ȥ꡼ä롣
 * parameters : device inode,file type,inode number,INODE_PARAM struct,directory block buffer
 * return : 0 or error number
 */
static int addDirEnt(int din,int type,uint inumber,INODE_PARAM *param,DIR_ENTRY *dir_buf)
{
	int rest;
	int block_size=superBlock[din]->blockSize;
	int sectors=superBlock[din]->sectors;
	DIR_ENTRY *new_ent;
	DIR_PARAM dparam;


	dparam.dirBuf=dir_buf;
	dparam.path=param->path;

	/* ѥ¸ߤ뤫ǧ */
	if(readDir(din,param->dirInode,&dparam,readDirBlock)>0)return -EEXIST;

	/* ǥ쥯ȥ꤫ȥ꡼õ */
	if((rest=readDir(din,param->dirInode,&dparam,addDirBlock))==-ENOENT)
	{
		/* ˥ǥ쥯ȥ˥֥åƤ롣 */
		if(superBlock[din]->s_free_blocks_count>0)
		{
			new_ent=dparam.dirBuf;
			new_ent->rec_len=block_size;
			new_ent->name_len=strlen(dparam.path);
			memcpy(new_ent->name,param->path,new_ent->name_len);
			new_ent->inode=inumber;
			new_ent->file_type=type;

			if((rest=writeData(din,param->dirInode,new_ent,block_size,param->dirInode->i_size))>0)
			{
				param->dirInode->i_size+=block_size;
				param->dirInode->i_blocks=ROUNDUP_DIV(param->dirInode->i_size,INODE_BLOCK_SIZE);
				param->dirInode->i_atime=param->dirInode->i_mtime=(uint)sys_time(NULL);
			}
			else return rest;
		}
		else return -ENOSPC;
	}
	else if(rest>0)
	{
		/* ȥ꡼롣 */
		new_ent=(DIR_ENTRY*)rest;
		memcpy(new_ent->name,param->path,new_ent->name_len);
		new_ent->inode=inumber;
		new_ent->file_type=type;
		if((rest=write_cache(din,dparam.dirBuf,sectors,blkToSct(din,dparam.blkNumber)))<0)return rest;
	}
	else return rest;

	return 0;
}


/*
 * PUBLIC
 * ƥǥ쥯ȥ꤫ե륨ȥ꡼롣
 * ȥ꡼inodeϰparam롣
 * parameters : device inode,INODE_PARAM struct,directory block buffer
 * return 0 or error number
 */
static int delFileEntry(int din,INODE_PARAM *param,DIR_ENTRY *dir_buf)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;


	dparam.dirBuf=dir_buf;
	dparam.path=param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if((rest=readDir(din,param->dirInode,&dparam,readDirBlock))<0)return rest;
	entry=(DIR_ENTRY*)rest;

	if(entry->file_type==EXT2_FT_REG_FILE)
	{
		if((rest=readInode(din,entry->inode,param->inodeBuf))<0)return rest;
		param->dirInode=(INODE*)rest;
		param->dirInode->i_number=entry->inode;
		delEntry(entry,dparam.dirBuf);
		if((rest=write_cache(din,dparam.dirBuf,superBlock[din]->sectors,blkToSct(din,dparam.blkNumber)))<0)return rest;
	}
	else return -ENOENT;

	return 0;
}


/*
 * PUBLIC
 * ƥǥ쥯ȥ꤫ǥ쥯ȥꥨȥ꡼롣
 * ȥ꡼inodeϰparam롣
 * parameters : name,device inode,INODE_PARAM struct,directory block buffer
 * return 0 or error number
 */
static int delDirEntry(int din,INODE_PARAM *param,DIR_ENTRY *dir_buf)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;
	INODE *parent_inode;
	INODE *parent_ibuf;


	dparam.dirBuf=dir_buf;
	dparam.path=param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if((rest=readDir(din,param->dirInode,&dparam,readDirBlock))<0)return rest;
	entry=(DIR_ENTRY*)rest;

	if(entry->file_type==EXT2_FT_DIR)
	{
		parent_inode=param->dirInode;
		parent_ibuf=param->inodeBuf;

		if((param->inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;

		if((rest=readInode(din,entry->inode,param->inodeBuf))<0)goto ERR;
		param->dirInode=(INODE*)rest;
		param->dirInode->i_number=entry->inode;
		if((rest=checkDir(din,param->dirInode))!=-ENOENT)goto ERR;
		delEntry(entry,dparam.dirBuf);
		if((rest=write_cache(din,dparam.dirBuf,superBlock[din]->sectors,blkToSct(din,dparam.blkNumber)))<0)goto ERR;
		--parent_inode->i_links_count;
		if((rest=writeInode(din,parent_inode->i_number,parent_ibuf))<0)goto ERR;

		kfree(parent_ibuf);
	}
	else return -ENOENT;

	return 0;
ERR:
	kfree(parent_ibuf);

	return rest;
}


/*
 * PUBLIC
 * ȥ꡼inodeoldnew촹롣
 * parameters : device inode,old inode parameter,new inode parameter,old block buffer,new block buffer
 *              old path,new path
 * return : 0 or error number
 */
static int changeInode(int din,INODE_PARAM *old_param,INODE_PARAM *new_param,DIR_ENTRY *old_dbuf,DIR_ENTRY *new_dbuf,
	const char *old_path,const char *new_path)
{
	int rest;
	DIR_ENTRY *old_ent,*new_ent;
	DIR_PARAM old_dparam,new_dparam;


	/* oldΥȥ꡼õ */
	old_dparam.dirBuf=old_dbuf;
	old_dparam.path=old_param->path;
	if((rest=readDir(din,old_param->dirInode,&old_dparam,readDirBlock))<0)return rest;
	old_ent=(DIR_ENTRY*)rest;

	/* newΥȥ꡼õ */
	new_dparam.dirBuf=new_dbuf;
	new_dparam.path=new_param->path;
	rest=readDir(din,new_param->dirInode,&new_dparam,readDirBlock);

	if(rest>0)				/* newΥȥ꡼¸ߤ롣 */
	{
		new_ent=(DIR_ENTRY*)rest;
		if(old_ent->file_type!=new_ent->file_type)
			return (old_ent->file_type==EXT2_FT_REG_FILE)?-EISDIR:-ENOTDIR;

		/* pathͭγǧ */
		if(old_ent->file_type==EXT2_FT_DIR)
			if(old_path[cmpString(old_path,new_path)]=='\0')return -EINVAL;

		/* newΥǡinode롣 */
		if((rest=readInode(din,new_ent->inode,new_param->inodeBuf))<0)return rest;
		new_param->dirInode=(INODE*)rest;
		if(new_ent->file_type==EXT2_FT_DIR)
			if((rest=checkDir(din,new_param->dirInode))!=-ENOENT)return rest;
		if((rest=releaseDataBlock(din,new_param->dirInode))<0)return rest;
		new_param->dirInode->i_number=new_ent->inode;
		if((rest=releaseInode(din,new_param->dirInode))<0)return rest;

		/* ȥ꡼inodeʥС촹롣 */
		new_ent->inode=old_ent->inode;
		if((rest=write_cache(din,new_dparam.dirBuf,superBlock[din]->sectors,blkToSct(din,new_dparam.blkNumber)))<0)
			return rest;
	}
	else if(rest==-ENOENT)	/* newΥȥ꡼¸ߤʤ */
	{
		/* ե̾γǧ */
		if(strlen(new_param->path)>EXT2_NAME_LEN)return -ENAMETOOLONG;

		/* newȥ꡼ */
		if((rest=addDirEnt(din,old_ent->file_type,old_ent->inode,new_param,new_dbuf))<0)return rest;
	}
	else return rest;

	/* oldȥ꡼κ */
	if((rest=readDir(din,old_param->dirInode,&old_dparam,readDirBlock))<0)return rest;	/* ɤ߹ߡ */
	delEntry(old_ent,old_dparam.dirBuf);
	if((rest=write_cache(din,old_dparam.dirBuf,superBlock[din]->sectors,blkToSct(din,old_dparam.blkNumber)))<0)return rest;

	return 0;
}


/*
 * PUBLIC
 * ̾μΥȥ꡼̾ơΥȥ꡼ΥեåȤ֤
 * Υȥ꡼̵0֤
 * parameters : device inode,directory block,entry index,name
 * return : next index or 0 or error number
 */
static int getNextEntry(int din,uint block,int index,char *name)
{
	int rest;
	int block_size=superBlock[din]->blockSize;
	DIR_ENTRY *buf,*entry;


	if(index==block_size)return 0;

	if((buf=kmalloc(block_size))==NULL)return -ENOMEM;
	if((rest=read_cache(din,buf,superBlock[din]->sectors,blkToSct(din,block)))<0)goto END;

	entry=(DIR_ENTRY*)((char*)buf+index);
	if(entry->inode==EXT2_FT_UNKNOWN)
		entry=(DIR_ENTRY*)((char*)entry+entry->rec_len);

	if((char*)entry==(char*)buf+block_size)rest=0;
	else
	{
		memcpy(name,entry->name,entry->name_len);
		name[entry->name_len]='\0';
		rest=(uint)entry-(uint)buf+entry->rec_len;
	}
END:
	kfree(buf);

	return rest;
}


/*
 * PUBLIC
 * Υǥ쥯ȥ֥å롣Υ֥åʤ-ENOENT֤
 * parameters : device inode,inode,directory block
 * retur : next block or error number
 */
static int inline getNextBlock(int din,INODE *inode,uint block)
{
	DIR_PARAM dparam;


	dparam.blkNumber=block;
	return readDir(din,inode,&dparam,searchDirBlock);
}


/*
 * PUBLIC
 * ǥ쥯ȥΥǥ쥯ȥ֥åν
 * parameters : device inode,block number,inode number,parent inode number
 * return : 0 or error number
 */
static int initDirBlock(int din,uint block,uint inumber,uint parent_inumber)
{
	int error=0;
	DIR_ENTRY *dent,*dent2;


	if((dent=(DIR_ENTRY*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	memset(dent,0,superBlock[din]->sectorSize);			/* ꥢʤȥǥ쥯ȥǷٹθˤʤ롣 */

	/* "."ǥ쥯ȥɲá */
	dent->inode=inumber;
	dent->rec_len=sizeof(DIR_ENTRY)+sizeof(int);
	dent->name_len=1;
	dent->file_type=EXT2_FT_DIR;
	dent->name[0]='.';

	dent2=(DIR_ENTRY*)((char*)dent+dent->rec_len);

	/* ".."ǥ쥯ȥɲá */
	dent2->inode=parent_inumber;
	dent2->rec_len=superBlock[din]->blockSize-dent->rec_len;
	dent2->name_len=2;
	dent2->file_type=EXT2_FT_DIR;
	dent2->name[0]='.';
	dent2->name[1]='.';

	if(write_cache(din,dent,1,blkToSct(din,block))!=1)error=-EIO;

	kfree(dent);

	return error;
}


/***********************************************************************
 *
 * < system call >
 *
 ***********************************************************************/

/*
 * return : root inode address in memory.
 */
static uint mount(int din)
{
	void *buf;
	DEV_STAT dev_stat;
	int sectors;
	int gd;
	int a;


	/* Ǥ˥ޥȤƤ뤫ǧ */
	if(superBlock[din]!=NULL)return -1;

	/* ѡ֥åɤ߹ࡣ */
	if(get_devstat(din,&dev_stat)==-1)return -1;
	if((buf=kmalloc(PAGE_SIZE))==NULL)return -1;
	if(read_direct(din,buf,PAGE_SIZE/dev_stat.sect_size,0)!=PAGE_SIZE/dev_stat.sect_size)
		{kfree(buf);return -1;}
	if((superBlock[din]=(SUPER_BLOCK*)kmalloc(sizeof(SUPER_BLOCK)))==NULL)
		{kfree(buf);return -1;}
	memcpy(superBlock[din],(char*)buf+EXT2_SUPER_BLK_POSITION,sizeof(SUPER_BLOCK));
	kfree(buf);

	/* ޥåʥСγǧ */
	if(superBlock[din]->s_magic!=EXT2_SUPER_MAGIC)goto ERR;

	/* ȼѡ֥å */
	superBlock[din]->blockSize=EXT2_BLOCK_BASE<<superBlock[din]->s_log_block_size;
	superBlock[din]->sectorSize=dev_stat.sect_size;
	superBlock[din]->sectors=superBlock[din]->blockSize/superBlock[din]->sectorSize;

	sectors=superBlock[din]->sectors;

	/* 롼ץǥץɤ߹ࡣ */
	a=ROUNDUP_DIV(superBlock[din]->s_blocks_count,superBlock[din]->s_blocks_per_group)*sizeof(GROUP_DESC);
	groupDescSize[din]=ROUNDUP_DIV(a,superBlock[din]->blockSize);
	if((groupDesc[din]=(GROUP_DESC*)kmalloc(superBlock[din]->blockSize*groupDescSize[din]))==NULL)goto ERR;
	if(read_direct(din,groupDesc[din],sectors*groupDescSize[din],blkToSct(din,EXT2_GROUP_DESC_BLK+superBlock[din]->s_first_data_block))!=
		sectors*groupDescSize[din])goto ERR;

	/* ֥åطν */
	if((gd=initBlock(din))==-1)goto ERR;

	/* inodeطν */
	if(initInode(din,gd)==-1)goto ERR;

	return EXT2_ROOT_INODE_INDEX;

ERR:
	kfree(groupDesc[din]);
	groupDesc[din]=NULL;
	kfree(superBlock[din]);
	superBlock[din]=NULL;

	return -1;
}


static int umount(int din)
{
	umountInode(din);
	umountBlock(din);
	writeBackSbGd(din);

	kfree(groupDesc[din]);
	groupDesc[din]=NULL;

	kfree(superBlock[din]);
	superBlock[din]=NULL;

	return 0;
}


/*
 * NULLѥԲġ
 * parameters : path,device inode,directory inode address,truncate flag,inode buffer
 * return : file format or error number
 */
static int open(const char *path,int din,uint dir_inode,int flag,uint *inode)
{
	int rest;
	INODE *inode_buf;
	DIR_ENTRY *dir_buf;
	INODE_PARAM param;


	if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((dir_buf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path=path;
	if((rest=readInode(din,dir_inode,param.inodeBuf))<0)goto END;
	param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&param,dir_buf))<0)goto END;
	if((rest=getInode(din,&param,dir_buf))<0)goto END;
	if(param.dirInode->i_mode&EXT2_S_IFDIR)
	{
		*inode=param.dirInode->i_number;
		rest=DIRECTORY;
		goto END;
	}

	/* truncate. */
	if(flag!=0)
		if((rest=releaseDataBlock(din,param.dirInode))<0)goto END;

	if((inode_buf=(INODE*)kmalloc(sizeof(INODE)))!=NULL)
		memcpy(inode_buf,param.dirInode,sizeof(INODE));
	else inode_buf=(INODE*)-ENOMEM;

	*inode=(uint)inode_buf;

	switch (inode_buf->i_mode & EXT2_S_IFMT)
	{
		case EXT2_S_IFREG:
			rest = NORMAL_FILE;
			break;
		case EXT2_S_IFIFO:
			rest = FIFO_FILE;
			break;
		case EXT2_S_IFLNK:
			rest = SYMBOL_LINK;
			break;
		case EXT2_S_IFBLK:
			rest = DEVICE_FILE;
			break;
		case EXT2_S_IFCHR:
			rest = DEVICE_FILE;
			break;
		default:
			rest = -ENOENT;
	}
END:
	kfree(dir_buf);
	kfree(param.inodeBuf);

	return rest;
}


static int close(int din,uint inode,int type)
{
	int rest;
	INODE *sector_buf;


	if(type==NORMAL_FILE)
	{
		if((sector_buf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
		if((rest=readInode(din,((INODE*)inode)->i_number,sector_buf))<0)goto ERR;
		memcpy((void*)rest,(void*)inode,sizeof(INODE));
		if((rest=writeInode(din,((INODE*)inode)->i_number,sector_buf))<0)goto ERR;
		kfree(sector_buf);
		kfree((void*)inode);
	}

	return 0;
ERR:
	kfree(sector_buf);
	kfree((void*)inode);

	return rest;
}


static int read(int din,uint inode,void *buf,size_t size,size_t begin)
{
	int rest;


	/* γǧ */
	if(begin>=((INODE*)inode)->i_size)return 0;
	if(begin+size>=((INODE*)inode)->i_size)size=((INODE*)inode)->i_size-begin;

	/* ե뤫ɤ߹ࡣ */
	if((rest=readData(din,(INODE*)inode,buf,size,begin))<0)return rest;

	/* դι */
	((INODE*)inode)->i_atime=(uint)sys_time(NULL);

	return size;
}


static int write(int din,uint inode,void *buf,size_t size,size_t begin)
{
	int rest;


	/* γǧ */
	if(begin+size>((INODE*)inode)->i_size)
		if((begin+size-((INODE*)inode)->i_size)/superBlock[din]->blockSize>=superBlock[din]->s_free_blocks_count)
			return -ENOSPC;

	/* եؽ񤭹ߡ */
	if((rest=writeData(din,(INODE*)inode,buf,size,begin))<0)
		return rest;


	/* ι */
	if (begin+size > ((INODE*)inode)->i_size)
	{
		((INODE*)inode)->i_size=begin+size;

		{
			int bsize=superBlock[din]->blockSize;
			int innum=bsize/sizeof(uint32_t);
			int bnum=ROUNDUP_DIV(begin+size,bsize);

			if(bnum>INODE_DATA_INDIRECT1)
			{
				if(bnum>INODE_DATA_INDIRECT1+innum)
				{
					if(bnum>INODE_DATA_INDIRECT1+(innum+1)*innum)
					{
						bnum+=ROUNDUP_DIV(bnum-INODE_DATA_INDIRECT1-innum,innum*innum);
					}
					bnum+=ROUNDUP_DIV(bnum-INODE_DATA_INDIRECT1,innum);
				}
				bnum+=1;
			}
			((INODE*)inode)->i_blocks = bnum*(bsize/INODE_BLOCK_SIZE);
		}
	}

	/* դι */
	((INODE*)inode)->i_atime = ((INODE*)inode)->i_mtime = (uint)sys_time(NULL);

	return size;
}


/*
 * NULLѥԲġ
 * return : inode buffer address or error number
 */
static int creat(const char *path,int din,uint dir_inode,int mode,uint *inode_value)
{
	uint inumber;
	int rest;
	INODE *inode,*inode_buf;
	DIR_ENTRY *dir_buf;
	INODE_PARAM param;
	PROC *proc;


	/* ѥθ */
	if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((dir_buf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path=path;
	if((rest=readInode(din,dir_inode,param.inodeBuf))<0)goto ERR1;
	param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&param,dir_buf))<0)goto ERR1;

	/* ե̾Υå */
	if(strlen(param.path)>EXT2_NAME_LEN)
	{
		rest=-ENAMETOOLONG;
		goto ERR1;
	}

	/* inode롣 */
	if((inumber=getNewInode(din))==0)
	{
		rest=-ENOSPC;
		goto ERR1;
	}

	/* ǥ쥯ȥꥨȥ꡼ɲä롣 */
	if((rest=addDirEnt(din,EXT2_FT_REG_FILE,inumber,&param,dir_buf))<0)goto ERR2;

	/* inode롣 */
	if((rest=readInode(din,inumber,param.inodeBuf))<0)goto ERR2;
	inode=(INODE*)rest;
	memset(inode,0,sizeof(INODE));
	proc=get_current_task();
	inode->i_mode=mode;
	inode->i_uid=proc->uid;
	inode->i_gid=proc->gid;
	inode->i_atime=inode->i_ctime=inode->i_mtime=(uint)sys_time(NULL);
	inode->i_links_count=1;
	inode->i_number=inumber;
	if((rest=writeInode(din,inumber,param.inodeBuf))<0)goto ERR2;

	/* inodeinodeХåե˥ԡ롣 */
	if((inode_buf=(INODE*)kmalloc(sizeof(INODE)))==NULL)
	{
		rest=-ENOMEM;
		goto ERR2;
	}
	else memcpy(inode_buf,inode,sizeof(INODE));

	kfree(dir_buf);
	kfree(param.inodeBuf);

	*inode_value=(uint)inode_buf;

	switch (inode_buf->i_mode & EXT2_S_IFMT)
	{
		case EXT2_S_IFREG:
			return NORMAL_FILE;
			break;
		case EXT2_S_IFIFO:
			return FIFO_FILE;
			break;
		case EXT2_S_IFLNK:
			return SYMBOL_LINK;
			break;
		case EXT2_S_IFBLK:
			return DEVICE_FILE;
			break;
		case EXT2_S_IFCHR:
			return DEVICE_FILE;
			break;
		default:
			return -ENOENT;
	}

ERR2:
	releaseInodeArea(din,inumber);
ERR1:
	kfree(dir_buf);
	kfree(param.inodeBuf);

	return rest;
}


/*
 * NULLѥԲġ
 */
static int mkdir(const char *path,int din,uint dir_inode,int mode)
{
	uint inumber,parent_inum;
	uint block;
	int rest,errrest;
	INODE *inode;
	DIR_ENTRY *dir_buf;
	INODE_PARAM param;
	PROC *proc;


	/* ѥθ */
	if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((dir_buf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path=path;
	if((rest=readInode(din,dir_inode,param.inodeBuf))<0)goto ERR1;
	param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&param,dir_buf))<0)goto ERR1;
	parent_inum=param.dirInode->i_number;

	/* ե̾Υå */
	if(strlen(param.path)>EXT2_NAME_LEN)
	{
		rest=-ENAMETOOLONG;
		goto ERR1;
	}

	/* ƥǥ쥯ȥΥ󥯿1䤹 */
	++param.dirInode->i_links_count;
	if((rest=writeInode(din,parent_inum,param.inodeBuf))<0)goto ERR1;

	/* inode롣 */
	if((inumber=getNewInode(din))==0)
	{
		rest=-ENOSPC;
		goto ERR2;
	}

	/* GRDλѥǥ쥯ȥ䤹 */
	addGrpUsedDir(din);

	/* ǥ쥯ȥꥨȥ꡼ɲä롣 */
	if((rest=addDirEnt(din,EXT2_FT_DIR,inumber,&param,dir_buf))!=0)goto ERR3;

	/* ǥ쥯ȥ롣 */
	if((block=getBlock(din))==0)
	{
		rest=-ENOSPC;
		goto ERR3;
	}

	if((rest=initDirBlock(din,block,inumber,param.dirInode->i_number))<0)goto ERR4;

	/* inode롣 */
	if((rest=readInode(din,inumber,param.inodeBuf))<0)goto ERR4;
	inode=(INODE*)rest;
	memset(inode,0,sizeof(INODE));
	proc=get_current_task();
	inode->i_mode=mode|EXT2_S_IFDIR;
	inode->i_size=superBlock[din]->blockSize;
	inode->i_uid=proc->uid;
	inode->i_gid=proc->gid;
	inode->i_atime=inode->i_ctime=inode->i_mtime=(uint)sys_time(NULL);
	inode->i_links_count=2;												/* "."".."Σġ */
	inode->i_blocks=inode->i_size/INODE_BLOCK_SIZE;
	inode->i_block[0]=block;
	if((rest=writeInode(din,inumber,param.inodeBuf))<0)goto ERR4;

	kfree(dir_buf);
	kfree(param.inodeBuf);

	return 0;

ERR4:
	releaseBlock(din,block);
ERR3:
	releaseInodeArea(din,inumber);
	delGrpUsedDir(din);
ERR2:
	if((errrest=readInode(din,parent_inum,param.inodeBuf))>0)
	{
		--((INODE*)errrest)->i_links_count;
		writeInode(din,parent_inum,param.inodeBuf);
	}
ERR1:
	kfree(dir_buf);
	kfree(param.inodeBuf);

	return rest;
}


/*
 * NULLѥԲġ
 */
static int delete(const char *path,int din,uint dir_inode,int type)
{
	int rest;
	DIR_ENTRY *dir_buf;
	INODE_PARAM param;


	/* ƥǥ쥯ȥθ */
	if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((dir_buf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path=path;
	if((rest=readInode(din,dir_inode,param.inodeBuf))<0)goto END;
	param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&param,dir_buf))<0)goto END;

	/* ƥǥ쥯ȥ꤫ǥ쥯ȥꥨȥ꡼롣 */
	if(type==NORMAL_FILE)
	{
		if((rest=delFileEntry(din,&param,dir_buf))<0)goto END;
	}
	else
	{
		if((rest=delDirEntry(din,&param,dir_buf))<0)goto END;
	}

	/* ǡ֥å롣 */
	if((rest=releaseDataBlock(din,param.dirInode))<0)goto END;

	/* inode롣 */
	if((rest=releaseInode(din,param.dirInode))<0)goto END;

	rest=0;
END:
	kfree(dir_buf);
	kfree(param.inodeBuf);

	return rest;
}


/*
 * NULLѥġ
 */
static int opendir(const char *path,int din,uint dir_inode,uint *inode,uint *block)
{
	int rest;
/*	INODE *inode_buf;
*/	DIR_ENTRY *dir_buf;
	INODE_PARAM param;


	/* ѥθ */
	if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((dir_buf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	if((rest=readInode(din,dir_inode,param.inodeBuf))<0)goto ERR;
	if(*path=='\0')
	{
		*block=((INODE*)rest)->i_block[0];
		return dir_inode;
	}
	param.path=path;
	param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&param,dir_buf))<0)goto ERR;
	if((rest=getInode(din,&param,dir_buf))<0)goto ERR;
	if((param.dirInode->i_mode&EXT2_S_IFDIR)==0)
	{
		rest=-ENOTDIR;
		goto ERR;
	}

	*block=param.dirInode->i_block[0];
	*inode=param.dirInode->i_number;

	rest=0;
ERR:
	kfree(dir_buf);
	kfree(param.inodeBuf);

	return rest;
}


static int readdir(int din,uint inode,uint *block,uint *index,char *name)
{
	int rest;
	INODE_PARAM param;


	for(;;)
	{
		if((rest=getNextEntry(din,*block,*index,name))>0)
		{
			*index=rest;
			rest=0;
			break;
		}
		else if(rest<0)break;

		if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)
			return -ENOMEM;
		if((rest=readInode(din,inode,param.inodeBuf))<0)
		{
			kfree(param.inodeBuf);
			break;
		}
		rest=getNextBlock(din,(INODE*)rest,*block);
		kfree(param.inodeBuf);
		if(rest==-ENOENT)		/* Υ֥å̵ */
		{
			*name='\0';
			rest=0;
			break;
		}
		else if(rest<0)break;	/* error */

		*block=rest;
		*index=0;
	}

	return rest;
}


/*
 * pathNULLǤʤpathõ
 */
static int stat(int din,uint ind,int type,FSTAT *fstat)
{
	int rest;
	INODE *buf=NULL,*inode;


	if(type==DIRECTORY)
	{
		if((buf=kmalloc(superBlock[din]->sectorSize))==NULL)
			return -ENOMEM;
		if((rest=readInode(din,ind,buf))<0)
		{
			kfree(buf);
			return rest;
		}
		inode=(INODE*)rest;
	}
	else inode=(INODE*)ind;

	fstat->st_mode=   inode->i_mode;
	fstat->st_nlink=  inode->i_links_count;
	fstat->st_uid=    inode->i_uid;
	fstat->st_gid=    inode->i_gid;
	fstat->st_size=   inode->i_size;
	fstat->st_atime=  inode->i_atime;
	fstat->st_mtime=  inode->i_mtime;
	fstat->st_ctime=  inode->i_ctime;
	fstat->st_blksize=1024<<superBlock[din]->s_log_block_size;
	fstat->st_ino=    inode->i_number;									/* Ȥꤢinode롣 */
	fstat->st_blocks= ROUNDUP(fstat->st_size,fstat->st_blksize)/512;	/* ѤƤ֥å(512Хñ) */
	fstat->st_rdev=   0;

	kfree(buf);

	return 0;
}


static int rename(int din,uint old_dinode,const char *old_path,uint new_dinode,const char *new_path)
{
	int rest;
	DIR_ENTRY *dir_buf1,*dir_buf2;
	INODE_PARAM old_param,new_param;


	new_param.inodeBuf=NULL;
	dir_buf2=NULL;

	/* old pathõ */
	if((dir_buf1=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)return -ENOMEM;
	if((old_param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)
	{
		rest=-ENOMEM;
		goto END;
	}
	old_param.path=old_path;
	if((rest=readInode(din,old_dinode,old_param.inodeBuf))<0)goto END;
	old_param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&old_param,dir_buf1))<0)goto END;

	/* new pathõ */
	if((new_param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)
	{
		rest=-ENOMEM;
		goto END;
	}
	new_param.path=new_path;
	if((rest=readInode(din,new_dinode,new_param.inodeBuf))<0)goto END;
	new_param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&new_param,dir_buf1))<0)goto END;

	/* old inodenew inode촹롣 */
	if((dir_buf2=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		rest=-ENOMEM;
		goto END;
	}
	if((rest=changeInode(din,&old_param,&new_param,dir_buf1,dir_buf2,old_path,new_path))<0)goto END;

	rest=0;
END:
	kfree(old_param.inodeBuf);
	kfree(new_param.inodeBuf);
	kfree(dir_buf1);
	kfree(dir_buf2);

	return rest;
}


static int chattr(int din,uint ind,int type,uint mode,uint uid,uint gid,uint atime,uint mtime)
{
	int rest;
	INODE *buf=NULL,*inode;


	if(type==DIRECTORY)
	{
		if((buf=kmalloc(superBlock[din]->sectorSize))==NULL)
			return -ENOMEM;
		if((rest=readInode(din,ind,buf))<0)
		{
			kfree(buf);
			return rest;
		}
		inode=(INODE*)rest;
	}
	else inode=(INODE*)ind;

	inode->i_mode=(((INODE*)inode)->i_mode&~0x1ff)|(mode&0x1ff);
	inode->i_uid=uid;
	inode->i_gid=gid;
	inode->i_atime=atime;
	inode->i_mtime=mtime;

	if(buf!=NULL)
	{
		rest=writeInode(din,ind,buf);
		kfree(buf);
		return rest;
	}

	return 0;
}


static int mklink(int din,uint oldinode,uint newind,const char *newpath)
{
	uint inumber;
	int rest;
	DIR_ENTRY *dir_buf;
	INODE_PARAM param;


	if((param.inodeBuf=(INODE*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	if((dir_buf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)
	{
		kfree(param.inodeBuf);
		return -ENOMEM;
	}

	/* newpathinodeΰõ */
	param.path=newpath;
	if((rest=readInode(din,newind,param.inodeBuf))<0)goto ERR1;
	param.dirInode=(INODE*)rest;
	if((rest=searchParent(din,&param,dir_buf))<0)goto ERR1;

	/* newpathե̾Υå */
	if(strlen(param.path)>EXT2_NAME_LEN)
	{
		rest=-ENAMETOOLONG;
		goto ERR1;
	}

	/* inode롣 */
	if((inumber=getNewInode(din))==0)
	{
		rest=-ENOSPC;
		goto ERR1;
	}

	/* ǥ쥯ȥꥨȥ꡼ɲä롣 */
	if((rest=addDirEnt(din,EXT2_FT_REG_FILE,inumber,&param,dir_buf))<0)goto ERR2;

	/* oldpathinodenewpathinode˥ԡ롣 */
	if((rest=readInode(din,inumber,param.inodeBuf))<0)goto ERR2;
	memcpy((void*)rest,(void*)oldinode,sizeof(INODE));
	if((rest=writeInode(din,inumber,param.inodeBuf))<0)goto ERR2;

	kfree(dir_buf);
	kfree(param.inodeBuf);

	return 0;

ERR2:
	releaseInodeArea(din,inumber);
ERR1:
	kfree(dir_buf);
	kfree(param.inodeBuf);

	return rest;
}


static int ioctl(int din,uint inode,int prm_num,int cmd,...)
{
	int rest;


	switch(cmd)
	{
		case I_TRUNC:
			/* truncate. */
			if((rest=releaseDataBlock(din,(INODE*)inode))<0)
				return rest;
	}

	return 0;
}


static int select(uint inode,int flag,int polling)
{
	return 1;
}


static int fsstat(int din,struct statfs *buf)
{
	SUPER_BLOCK *sp=superBlock[din];


	buf->f_type=  0;						/* filesystem type number */
	buf->f_flag=  0;						/* Bit mask of f_flag values */
	buf->f_bsize= sp->blockSize;			/*  File system block size. */
	buf->f_frsize=sp->blockSize;			/* fundamental file system block size */
	buf->f_iosize=sp->blockSize;			/* optimal transfer block size */
	buf->f_blocks=sp->s_blocks_count;		/* total data blocks in file system */
	buf->f_bfree= sp->s_free_blocks_count;	/* free blocks in fs */
	buf->f_bavail=sp->s_free_blocks_count;	/* free blocks avail to non-superuser */
	buf->f_files= sp->s_inodes_count;		/* total file nodes in file system */
	buf->f_ffree= sp->s_free_inodes_count;	/* free file nodes in fs */
	buf->f_favail=0;						/* Number of file serial numbers available to non-privileged process. */
	buf->f_fsid=  0;						/* file system id */
	buf->f_owner= 0;						/* user that mounted the filesystem */
	buf->f_namemax=EXT2_NAME_LEN;			/* Maximum filename length. */
	memcpy(buf->f_fstypename,"ext2",5);		/* fs type name */

	return 0;
}


/***********************************************************************
 *
 * < Init class >
 *
 ***********************************************************************/

static FS ext2Fs={
	"ext2",
	mount,
	umount,
	open,
	close,
	read,
	write,
	ioctl,
	rename,
	creat,
	opendir,
	mkdir,
	stat,
	delete,
	readdir,
	chattr,
	mklink,
	select,
	fsstat,
};


/*
 * GLOBAL
 */
int initExt2tFs()
{
	int i;


	for(i=0;i<MAX_DEVICE_OPEN;++i)
	{
		groupDesc[i]=NULL;
		superBlock[i]=NULL;
	}

	return regist_fs(&ext2Fs);
}
/**************************************************************************************/
int test_ext2(int din)
{
	static char buf[512];

/*	read_cache(din,buf,1,0x10204);
	printk("0x10204=%x %x ",*(uint*)buf,*(uint*)(buf+0x18));
	read_cache(din,buf,1,0x10);
	printk("0x2000=%x ",*(uint*)buf);
*/	read_cache(din,buf,1,0x6be);
	printk("0x6be=%x\n",*(uint*)(buf+0x18));

	return 0;
}
/**************************************************************************************/
