/*-
 * Copyright (c) 1990, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * 2004: modified by minoru murashma.
 */


#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include "../vm.h"
#include "../nlist.h"
#include "../vis.h"
#include "../tzfile.h"
#include "../err.h"
#include "../command.h"
#include "ps.h"


void printheader()
{
	VAR *v;
	struct varent *vent;


	for (vent = vhead; vent; vent = vent->next)
	{
		v = vent->var;
		if (v->flag & LJUST)
		{
			if (vent->next == NULL)	/* last one */
				printf("%s", v->header);
			else
				printf("%-*s", v->width, v->header);
		}
		else
			printf("%*s", v->width, v->header);
		if (vent->next != NULL)
			putchar(' ');
	}
	putchar('\n');
}


void command(KINFO *k,VARENT *ve)
{
	VAR *v;
	int left;
	char *cp, *vis_env, *vis_args;


	if ((vis_args = malloc(strlen(k->ki_args) * 4 + 1)) == NULL)
		err(1, NULL);
	strvis(vis_args, k->ki_args, VIS_TAB | VIS_NL | VIS_NOSLASH);
	if (k->ki_env)
	{
		if ((vis_env = malloc(strlen(k->ki_env) * 4 + 1)) == NULL)
			err(1, NULL);
		strvis(vis_env, k->ki_env, VIS_TAB | VIS_NL | VIS_NOSLASH);
	}
	else vis_env = NULL;

	v = ve->var;
	if (ve->next == NULL)
	{
		/* last field */
		if (termwidth == UNLIMITED)
		{
			if (vis_env)
				printf("%s ", vis_env);
			printf("%s", vis_args);
		}
		else
		{
			left = termwidth - (totwidth - v->width);
			if (left < 1) /* already wrapped, just use std width */
				left = v->width;
			if ((cp = vis_env) != NULL)
			{
				while (--left >= 0 && *cp)
					putchar(*cp++);
				if (--left >= 0)
					putchar(' ');
			}
			for (cp = vis_args; --left >= 0 && *cp != '\0';)
				putchar(*cp++);
		}
	}
	else
		printf("%-*.*s", v->width, v->width, vis_args);	/* XXX env? */

	free(vis_args);
	if (vis_env != NULL)
		free(vis_env);
}


void ucomm(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%-*s", v->width, KI_PROC(k)->p_comm);
}


void logname(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%-*s", v->width, KI_EPROC(k)->e_login);
}


void state(KINFO *k,VARENT *ve)
{
	struct proc *p;
	int flag;
	char *cp;
	VAR *v;
	char buf[16];


	v = ve->var;
	p = KI_PROC(k);
	flag = p->p_flag;
	cp = buf;

	switch (p->p_stat)
	{
		case SSTOP:
			*cp = 'T';
			break;
		case SSLEEP:
			if (flag & P_SINTR)	/* interuptable (long) */
				*cp = p->p_slptime >= MAXSLP ? 'I' : 'S';
			else
				*cp = 'D';
			break;
		case SRUN:
		case SIDL:
			*cp = 'R';
			break;
		case SZOMB:
			*cp = 'Z';
			break;
		default:
			*cp = '?';
	}
	cp++;
	if (flag & P_INMEM)*cp++ = 'W';
	if (p->p_nice < NZERO)*cp++ = '<';
	else if (p->p_nice > NZERO)*cp++ = 'N';
	if (flag & P_TRACED)*cp++ = 'X';
	if (flag & P_WEXIT && p->p_stat != SZOMB)
		*cp++ = 'E';
	if (flag & P_PPWAIT)
		*cp++ = 'V';
	if (flag & (P_SYSTEM | P_NOSWAP | P_PHYSIO))
		*cp++ = 'L';
	if (KI_EPROC(k)->e_flag & EPROC_SLEADER)
		*cp++ = 's';
	if ((flag & P_CONTROLT) && KI_EPROC(k)->e_pgid == KI_EPROC(k)->e_tpgid)
		*cp++ = '+';
	*cp = '\0';
	printf("%-*s", v->width, buf);
}


void pri(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*d", v->width, KI_PROC(k)->p_priority - PZERO);
}


void uname(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%-*s",(int)v->width, user_from_uid(KI_EPROC(k)->e_ucred.cr_uid, 0));
}


void runame(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%-*s",(int)v->width, user_from_uid(KI_EPROC(k)->e_pcred.p_ruid, 0));
}


void tdev(KINFO *k,VARENT *ve)
{
	VAR *v;
	dev_t dev;
	char buff[16];


	v = ve->var;
	dev = KI_EPROC(k)->e_tdev;
	if (dev == NODEV)
		printf("%*s", v->width, "??");
	else
	{
		snprintf(buff, sizeof(buff),"%d/%d", major(dev), minor(dev));
		printf("%*s", v->width, buff);
	}
}


void tname(KINFO *k,VARENT *ve)
{
	VAR *v;
	dev_t dev;
	char *ttname;


	v = ve->var;
	dev = KI_EPROC(k)->e_tdev;
	if (dev == NODEV || (ttname = devname(dev, S_IFCHR)) == NULL)
		printf("%-*s", v->width, "??");
	else
	{
		if (strncmp(ttname, "tty", 3) == 0)
			ttname += 3;
		printf("%*.*s%c", v->width-1, v->width-1, ttname,
			KI_EPROC(k)->e_flag & EPROC_CTTY ? ' ' : '-');
	}
}


void longtname(KINFO *k,VARENT *ve)
{
	VAR *v;
	dev_t dev;
	char *ttname;


	v = ve->var;
	dev = KI_EPROC(k)->e_tdev;
	if (dev == NODEV || (ttname = devname(dev, S_IFCHR)) == NULL)
		printf("%-*s", v->width, "??");
	else
		printf("%-*s", v->width, ttname);
}


void started(KINFO *k,VARENT *ve)
{
	VAR *v;
	static time_t now;
	struct tm *tp;
	char buf[100];


	v = ve->var;
	if (!k->ki_u.u_valid)
	{
		printf("%-*s", v->width, "-");
		return;
	}

	tp = localtime(&k->ki_u.u_start.tv_sec);
	if (!now)time(&now);
	if (now - k->ki_u.u_start.tv_sec < 24 * SECSPERHOUR)
	{
		/* I *hate* SCCS... */
		static char fmt[] = __CONCAT("%l:%", "M%p");
		strftime(buf, sizeof(buf) - 1, fmt, tp);
	}
	else if (now - k->ki_u.u_start.tv_sec < 7 * SECSPERDAY)
	{
		/* I *hate* SCCS... */
		static char fmt[] = __CONCAT("%a%", "I%p");
		strftime(buf, sizeof(buf) - 1, fmt, tp);
	}
	else
		strftime(buf, sizeof(buf) - 1, "%e%b%y", tp);
	printf("%-*s", v->width, buf);
}


void lstarted(KINFO *k,VARENT *ve)
{
	VAR *v;
	char buf[100];


	v = ve->var;
	if (!k->ki_u.u_valid)
	{
		printf("%-*s", v->width, "-");
		return;
	}
	strftime(buf, sizeof(buf) -1, "%C",localtime(&k->ki_u.u_start.tv_sec));
	printf("%-*s", v->width, buf);
}


void wchan(KINFO *k,VARENT *ve)
{
	VAR *v;

	v = ve->var;
	if (KI_PROC(k)->p_wchan) {
		if (KI_PROC(k)->p_wmesg)
			printf("%-*.*s", v->width, v->width,
				      KI_EPROC(k)->e_wmesg);
		else
			printf("%-*x", v->width,
			    (int)KI_PROC(k)->p_wchan &~ KERNBASE);
	} else
		printf("%-*s", v->width, "-");
}


#define pgtok(a)        (((a)*NBPG)/1024)

void vsize(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*d", v->width,pgtok(KI_EPROC(k)->e_vm.vm_dsize + KI_EPROC(k)->e_vm.vm_ssize + KI_EPROC(k)->e_vm.vm_tsize));
}


void rssize(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	/* XXX don't have info about shared */
	printf("%*d", v->width, pgtok(KI_EPROC(k)->e_vm.vm_rssize));
}


void p_rssize(KINFO *k,VARENT *ve)		/* doesn't account for text */
{
	VAR *v;


	v = ve->var;
	printf("%*d", v->width, pgtok(KI_EPROC(k)->e_vm.vm_rssize));
}


void cputime(KINFO *k,VARENT *ve)
{
	VAR *v;
	long secs;
	long psecs;	/* "parts" of a second. first micro, then centi */
	char obuff[128];


	v = ve->var;
	if (KI_PROC(k)->p_stat == SZOMB || !k->ki_u.u_valid)
	{
		secs = 0;
		psecs = 0;
	}
	else
	{
		/*
		 * This counts time spent handling interrupts.  We could
		 * fix this, but it is not 100% trivial (and interrupt
		 * time fractions only work on the sparc anyway).	XXX
		 */
		secs = KI_PROC(k)->p_rtime.tv_sec;
		psecs = KI_PROC(k)->p_rtime.tv_usec;
		if (sumrusage)
		{
			secs += k->ki_u.u_cru.ru_utime.tv_sec + k->ki_u.u_cru.ru_stime.tv_sec;
			psecs += k->ki_u.u_cru.ru_utime.tv_usec + k->ki_u.u_cru.ru_stime.tv_usec;
		}
		/*
		 * round and scale to 100's
		 */
		psecs = (psecs + 5000) / 10000;
		secs += psecs / 100;
		psecs = psecs % 100;
	}
	snprintf(obuff, sizeof(obuff),"%3ld:%02ld.%02ld", secs/60, secs%60, psecs);
	printf("%*s", v->width, obuff);
}


double getpcpu(KINFO *k)
{
	struct proc *p;
	static int failure;


	if (!nlistread)
		failure = donlist();
	if (failure)return (0.0);

	p = KI_PROC(k);
#define	fxtofl(fixpt) ((double)(fixpt) / fscale)

	/* XXX - I don't like this */
	if (p->p_swtime == 0 || (p->p_flag & P_INMEM) == 0)
		return (0.0);
	if (rawcpu)
		return (100.0 * fxtofl(p->p_pctcpu));

	return (100.0 * fxtofl(p->p_pctcpu) / (1.0 - exp(p->p_swtime * log(fxtofl(ccpu)))));
}


void pcpu(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*.1f", v->width, getpcpu(k));
}


double getpmem(KINFO *k)
{
	static int failure;
	struct proc *p;
	struct eproc *e;
	double fracmem;
	int szptudot;


	if (!nlistread)
		failure = donlist();
	if (failure)return (0.0);

	p = KI_PROC(k);
	e = KI_EPROC(k);
	if ((p->p_flag & P_INMEM) == 0)
		return (0.0);
	/* XXX want pmap ptpages, segtab, etc. (per architecture) */
	szptudot = UPAGES;
	/* XXX don't have info about shared */
	fracmem = ((float)e->e_vm.vm_rssize + szptudot)/CLSIZE/mempages;

	return (100.0 * fracmem);
}


void pmem(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*.1f", v->width, getpmem(k));
}


void pagein(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*d", v->width,k->ki_u.u_valid ? k->ki_u.u_ru.ru_majflt : 0);
}


void maxrss(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*s", v->width, "-");
}


void tsize(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printf("%*d", v->width, pgtok(KI_EPROC(k)->e_vm.vm_tsize));
}


/*
 * Generic output routines.  Print fields from various prototype
 * structures.
 */
static void printval(char *bp,VAR *v)
{
	static char ofmt[32] = "%";
	char *fcp, *cp;


	cp = ofmt + 1;
	fcp = v->fmt;
	if (v->flag & LJUST)*cp++ = '-';
	*cp++ = '*';
	while (*cp++ = *fcp++);

	switch (v->type)
	{
		case CHAR:
			printf(ofmt, v->width, *(char *)bp);
			break;
		case UCHAR:
			printf(ofmt, v->width, *(u_char *)bp);
			break;
		case SHORT:
			printf(ofmt, v->width, *(short *)bp);
			break;
		case USHORT:
			printf(ofmt, v->width, *(u_short *)bp);
			break;
		case LONG:
			printf(ofmt, v->width, *(long *)bp);
			break;
		case ULONG:
			printf(ofmt, v->width, *(u_long *)bp);
			break;
		case KPTR:
			printf(ofmt, v->width, *(u_long *)bp &~ KERNBASE);
			break;
		default:
			errx(1, "unknown type %d", v->type);
	}
}


void pvar(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printval((char *)((char *)KI_PROC(k) + v->off), v);
}


void evar(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	printval((char *)((char *)KI_EPROC(k) + v->off), v);
}


void uvar(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	if (k->ki_u.u_valid)
		printval((char *)((char *)&k->ki_u + v->off), v);
	else
		printf("%*s", v->width, "-");
}


void rvar(KINFO *k,VARENT *ve)
{
	VAR *v;


	v = ve->var;
	if (k->ki_u.u_valid)
		printval((char *)((char *)(&k->ki_u.u_ru) + v->off), v);
	else
		printf("%*s", v->width, "-");
}
