/*-
 * Copyright (c) 1980, 1987, 1988, 1991, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * 2004: modified by minoru murashma.
 */

/*
 * login [ name ]
 * login -h hostname	(for telnetd, etc.)
 * login -f name	(for pre-authenticated login: datakit, xterm, etc.)
 */


#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include <time.h>
#include <syslog.h>
#include "../tzfile.h"
#include "ttyent.h"
#include "pathnames.h"
#include "utmp.h"


#define	TTYGRPNAME	"tty"		/* name of group to own ttys */
#define	NBUFSIZ		(UT_NAMESIZE + 1)


/*
 * This bounds the time given to login.  Not a define so it can
 * be patched on machines where it's too small.
 */
u_int	timeout = 300;

struct	passwd *pwd;
int	failures;
char	term[64], *envinit[1], *hostname, *username, *tty;


extern int	 ttyslot(void);
extern char *getpass(const char *prompt);
extern char *crypt(const char *key, const char *salt);


static int getdtablesize()
{
	return sysconf(_SC_OPEN_MAX);
}


void login(struct utmp *ut)
{
	int fd;
	int tty;


	tty = ttyslot();
	if (tty > 0 && (fd = open(_PATH_UTMP, O_WRONLY|O_CREAT, 0644)) >= 0)
	{
		lseek(fd, (off_t)(tty * sizeof(struct utmp)), SEEK_SET);
		write(fd, ut, sizeof(struct utmp));
		close(fd);
	}
	if ((fd = open(_PATH_WTMP, O_WRONLY|O_APPEND, 0)) >= 0) {
		write(fd, ut, sizeof(struct utmp));
		close(fd);
	}
}


void sleepexit(int eval)
{

	sleep(5);
	exit(eval);
}


#undef	UNKNOWN
#define	UNKNOWN	"su"

char *stypeof(char *ttyid)
{
	struct ttyent *t;


	return (ttyid && (t = getttynam(ttyid)) ? t->ty_type : UNKNOWN);
}


int rootterm(char *ttyn)
{
	struct ttyent *t;


	return ((t = getttynam(ttyn)) && t->ty_status & TTY_SECURE);
}


jmp_buf motdinterrupt;


/* ARGSUSED */
void sigint(int signo)
{

	longjmp(motdinterrupt, 1);
}


/* ARGSUSED */
void timedout(int signo)
{
	fprintf(stderr, "Login timed out after %d seconds\n", timeout);
	exit(0);
}

void checknologin()
{
	int fd, nchars;
	char tbuf[8192];


	if ((fd = open(_PATH_NOLOGIN, O_RDONLY, 0)) >= 0)
	{
		while ((nchars = read(fd, tbuf, sizeof(tbuf))) > 0)
			write(fileno(stdout), tbuf, nchars);
		sleepexit(0);
	}
}


void badlogin(char *name)
{

	if (failures == 0)return;

	if (hostname)
	{
		syslog(LOG_NOTICE, "%d LOGIN FAILURE%s FROM %s",failures, failures > 1 ? "S" : "", hostname);
		syslog(LOG_AUTHPRIV|LOG_NOTICE,
			"%d LOGIN FAILURE%s FROM %s, %s",
		    failures, failures > 1 ? "S" : "",
		    hostname, name);
	}
	else
	{
		syslog(LOG_NOTICE, "%d LOGIN FAILURE%s ON %s",failures, failures > 1 ? "S" : "", tty);
		syslog(LOG_AUTHPRIV|LOG_NOTICE,
		    "%d LOGIN FAILURE%s ON %s, %s",
		    failures, failures > 1 ? "S" : "",
		    tty, name);
	}
}


void dolastlog(int quiet)
{
	struct lastlog ll;
	int fd;


	if ((fd = open(_PATH_LASTLOG, O_RDWR, 0)) >= 0)
	{
		lseek(fd, (off_t)pwd->pw_uid * sizeof(ll), SEEK_SET);
		if (!quiet)
		{
			if (read(fd, (char *)&ll, sizeof(ll)) == sizeof(ll) &&ll.ll_time != 0)
			{
				printf("Last login: %.*s ",24-5, (char *)ctime(&ll.ll_time));
				if (*ll.ll_host != '\0')
					printf("from %.*s\n",(int)sizeof(ll.ll_host),ll.ll_host);
				else
					printf("on %.*s\n",(int)sizeof(ll.ll_line),ll.ll_line);
			}
			lseek(fd, (off_t)pwd->pw_uid * sizeof(ll), SEEK_SET);
		}
		memset((void *)&ll, 0, sizeof(ll));
		time(&ll.ll_time);
		strncpy(ll.ll_line, tty, sizeof(ll.ll_line));
		if (hostname)
			strncpy(ll.ll_host, hostname, sizeof(ll.ll_host));
		write(fd, (char *)&ll, sizeof(ll));
		close(fd);
	}
}


void motd()
{
	int fd, nchars;
	sig_t oldint;
	char tbuf[8192];


	if ((fd = open(_PATH_MOTDFILE, O_RDONLY, 0)) < 0)return;

	oldint = signal(SIGINT, sigint);
	if (setjmp(motdinterrupt) == 0)
		while ((nchars = read(fd, tbuf, sizeof(tbuf))) > 0)
			write(fileno(stdout), tbuf, nchars);
	signal(SIGINT, oldint);
	close(fd);
}


void getloginname()
{
	int ch;
	char *p;
	static char nbuf[NBUFSIZ];


	for (;;)
	{
		printf("login: ");
		for (p = nbuf; (ch = getchar()) != '\n'; )
		{
			if (ch == EOF)
			{
				badlogin(username);
				exit(0);
			}
			if (p < nbuf + (NBUFSIZ - 1))
				*p++ = ch;
		}
		if (p > nbuf)
		{
			if (nbuf[0] == '-')
				fprintf(stderr,"login names may not start with '-'.\n");
			else
			{
				*p = '\0';
				username = nbuf;
				break;
			}
		}
	}
}


int main(int argc,char *argv[])
{
	extern char **environ;
	struct group *gr;
	struct stat st;
	struct timeval tp;
	struct utmp utmp;
	int ask, ch, cnt, fflag, hflag, pflag, quietlog, rootlogin, rval=0;
	uid_t uid;
	char *domain, *p, *salt, *ttyn;
	char tbuf[PATH_MAX + 2], tname[sizeof(_PATH_TTY) + 10];
	char localhost[HOST_NAME_MAX];


	signal(SIGALRM, timedout);
	alarm(timeout);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	setpriority(PRIO_PROCESS, 0, 0);

	openlog("login", LOG_ODELAY, LOG_AUTH);

	/*
	 * -p is used by getty to tell login not to destroy the environment
	 * -f is used to skip a second login authentication
	 * -h is used by other servers to pass the name of the remote
	 *    host to login so that it may be placed in utmp and wtmp
	 */
	domain = NULL;
	if (gethostname(localhost, sizeof(localhost)) < 0)
		syslog(LOG_ERR, "couldn't get local hostname: %m");
	else domain = strchr(localhost, '.');

	fflag = hflag = pflag = 0;
	uid = getuid();
	while ((ch = getopt(argc, argv, "fh:p")) != EOF)
		switch (ch)
		{
			case 'f':
				fflag = 1;
				break;
			case 'h':
				if (uid)
					errx(1, "-h option: %s", strerror(EPERM));
				hflag = 1;
				if (domain && (p = strchr(optarg, '.')) &&
				    strcasecmp(p, domain) == 0)
					*p = 0;
				hostname = optarg;
				break;
			case 'p':
				pflag = 1;
				break;
			case '?':
			default:
				if (!uid)
					syslog(LOG_ERR, "invalid flag %c", ch);
				fprintf(stderr,"usage: login [-fp] [-h hostname] [username]\n");
				exit(1);
		}
	argc -= optind;
	argv += optind;

	if (*argv)
	{
		username = *argv;
		ask = 0;
	}
	else ask = 1;

	for (cnt = getdtablesize(); cnt > 2; cnt--)
		close(cnt);

	ttyn = ttyname(STDIN_FILENO);
	if (ttyn == NULL || *ttyn == '\0')
	{
		snprintf(tname, sizeof(tname), "%s??", _PATH_TTY);
		ttyn = tname;
	}
	if ((tty = strrchr(ttyn, '/'))!=NULL)++tty;
	else tty = ttyn;

	for (cnt = 0;; ask = 1)
	{
		if (ask)
		{
			fflag = 0;
			getloginname();
		}
		rootlogin = 0;

		if (strlen(username) > UT_NAMESIZE)
			username[UT_NAMESIZE] = '\0';

		/*
		 * Note if trying multiple user names; log failures for
		 * previous user name, but don't bother logging one failure
		 * for nonexistent name (mistyped username).
		 */
		if (failures && strcmp(tbuf, username))
		{
			if (failures > (pwd ? 0 : 1))
				badlogin(tbuf);
			failures = 0;
		}
		strcpy(tbuf, username);

		if ((pwd = getpwnam(username))!=NULL)
			salt = pwd->pw_passwd;
		else salt = "xx";

		/*
		 * if we have a valid account name, and it doesn't have a
		 * password, or the -f option was specified and the caller
		 * is root or the caller isn't changing their uid, don't
		 * authenticate.
		 */
		if (pwd && ((*pwd->pw_passwd == '\0') || (fflag && ((uid == 0) || (uid == pwd->pw_uid)))))
			break;
		fflag = 0;
		if (pwd && pwd->pw_uid == 0)
			rootlogin = 1;

		setpriority(PRIO_PROCESS, 0, -4);

		p = getpass("Password:");

		if (pwd)
			rval = strcmp(crypt(p, salt), pwd->pw_passwd);								/* <=================== */
		memset(p, 0, strlen(p));

		setpriority(PRIO_PROCESS, 0, 0);

		/*
		 * If trying to log in as root without Kerberos,
		 * but with insecure terminal, refuse the login attempt.
		 */
		if (pwd && rootlogin && !rootterm(tty))
		{
			fprintf(stderr,"%s login refused on this terminal.\n",pwd->pw_name);
			if (hostname)
				syslog(LOG_NOTICE,"LOGIN %s REFUSED FROM %s ON TTY %s",pwd->pw_name, hostname, tty);
			else
				syslog(LOG_NOTICE,"LOGIN %s REFUSED ON TTY %s",pwd->pw_name, tty);
			continue;
		}

		if (pwd && !rval)break;

		printf("Login incorrect\n");
		failures++;
		/* we allow 10 tries, but after 3 we start backing off */
		if (++cnt > 3)
		{
			if (cnt >= 10)
			{
				badlogin(username);
				sleepexit(1);
			}
			sleep((u_int)((cnt - 3) * 5));
		}
	}

	/* committed to login -- turn off timeout */
	alarm((u_int)0);

/*	endpwent();
*/
	/* if user not super-user, check for disabled logins */
	if (!rootlogin)
		checknologin();

	if (chdir(pwd->pw_dir) < 0)
	{
		printf("No home directory %s!\n", pwd->pw_dir);
		if (chdir("/"))exit(0);
		pwd->pw_dir = "/";
		printf("Logging in with home = \"/\".\n");
	}

	quietlog = access(_PATH_HUSHLOGIN, F_OK) == 0;

	if (pwd->pw_change || pwd->pw_expire)
		gettimeofday(&tp, (struct timezone *)NULL);
	if (pwd->pw_change)
	{
		if (tp.tv_sec >= pwd->pw_change)
		{
			printf("Sorry -- your password has expired.\n");
			sleepexit(1);
		}
		else if (((pwd->pw_change - tp.tv_sec) < (2 * DAYSPERWEEK * SECSPERDAY)) && !quietlog)
			printf("Warning: your password expires on %s",ctime(&pwd->pw_change));
	}
	if (pwd->pw_expire)
	{
		if (tp.tv_sec >= pwd->pw_expire)
		{
			printf("Sorry -- your account has expired.\n");
			sleepexit(1);
		}
		else if (((pwd->pw_expire - tp.tv_sec) < (2 * DAYSPERWEEK * SECSPERDAY)) && !quietlog)
			printf("Warning: your account expires on %s",ctime(&pwd->pw_expire));
	}

	/* Nothing else left to fail -- really log in. */
	memset((void *)&utmp, 0, sizeof(utmp));
	time((time_t*)&utmp.ut_time);
	strncpy(utmp.ut_name, username, sizeof(utmp.ut_name));
	if (hostname)
		strncpy(utmp.ut_host, hostname, sizeof(utmp.ut_host));
	strncpy(utmp.ut_line, tty, sizeof(utmp.ut_line));
	login(&utmp);

	dolastlog(quietlog);

	chown(ttyn, pwd->pw_uid,(gr = getgrnam(TTYGRPNAME)) ? gr->gr_gid : pwd->pw_gid);
	setgid(pwd->pw_gid);

/*	initgroups(username, pwd->pw_gid);
*/
	if (*pwd->pw_shell == '\0')
		pwd->pw_shell = _PATH_BSHELL;

	/* Destroy environment unless user has requested its preservation. */
	if (!pflag)environ = envinit;
	setenv("HOME", pwd->pw_dir, 1);
	setenv("SHELL", pwd->pw_shell, 1);
	if (term[0] == '\0')
		strncpy(term, stypeof(tty), sizeof(term));
	setenv("TERM", term, 0);
	setenv("LOGNAME", pwd->pw_name, 1);
	setenv("USER", pwd->pw_name, 1);
	setenv("PATH", _PATH_DEFPATH, 0);


	if (tty[sizeof("tty")-1] == 'd')
		syslog(LOG_INFO, "DIALUP %s, %s", tty, pwd->pw_name);

	/* If fflag is on, assume caller/authenticator has logged root login. */
	if (rootlogin && fflag == 0)
	{
		if (hostname)
			syslog(LOG_NOTICE, "ROOT LOGIN (%s) ON %s FROM %s",username, tty, hostname);
		else
			syslog(LOG_NOTICE, "ROOT LOGIN (%s) ON %s", username, tty);
	}

	if (!quietlog)
	{
		printf("%s\n\t%s  %s\n\n",
	    	"Copyright (c) 1980, 1983, 1986, 1988, 1990, 1991, 1993, 1994",
		    "The Regents of the University of California. ",
		    "All rights reserved.");
		motd();
		snprintf(tbuf,sizeof(tbuf), "%s/%s", _PATH_MAILDIR, pwd->pw_name);
		if (stat(tbuf, &st) == 0 && st.st_size != 0)
			printf("You have %smail.\n",(st.st_mtime > st.st_atime) ? "new " : "");
	}

	signal(SIGALRM, SIG_DFL);
	signal(SIGQUIT, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGTSTP, SIG_IGN);

	tbuf[0] = '-';
	strcpy(tbuf + 1, (p = strrchr(pwd->pw_shell, '/')) ? p + 1 : pwd->pw_shell);

/*	if (setlogin(pwd->pw_name) < 0)
		syslog(LOG_ERR, "setlogin() failure: %m");
*/
	/* Discard permissions last so can't get killed and drop core. */
	if (rootlogin)setuid(0);
	else setuid(pwd->pw_uid);

	execlp(pwd->pw_shell, tbuf, 0);
	err(1, "%s", pwd->pw_shell);

	return 0;
}
