/*
 * proc.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ץط
 */


#include<config.h>
#include<types.h>
#include<mm.h>
#include<fs.h>
#include<errno.h>
#include<interrupt.h>
#include<lock.h>
#include<segment.h>
#include<time.h>
#include<elf.h>
#include<proc.h>
#include<except.h>
#include<signal.h>
#include<console.h>
#include<share/times.h>
#include<net/net.h>
#include<term.h>
#include<test.h>


/* tssǼ¤(104Х) */
typedef struct{
	uint link,esp0,ss0,esp1,ss1,esp2,ss2,cr3,eip,eflags,eax,ecx,
		edx,ebx,esp,ebp,esi,edi,es,cs,ss,ds,fs,gs,ldts,tflag_iomap;
}TSS;


CPU cputask[MAX_CPU];		/* Task infomations for each cpu */

/* TSS󥰻Υååɬ */
static TSS kernel_tss={
	0,
	KERNEL_ESP_BEG,		/* esp0 */
	KERNEL_DATA_DES,	/* ss0 */
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};
static PROC *idle_proc[MAX_CPU];	/* Idle process */


/******************************************************************************************************
 *
 * < 塼륯饹 >
 *
 *******************************************************************************************************/

/*
 * GLOBAL
 * Add task to shedule.
 * NOTE!
 * 	¾ץƤФ롣
 * parameters : Process,wait flag
 */
void add_to_schedule(PROC *proc,int flag)
{
	int cpu=proc->cpu;
	PROC *p;


	enter_spinlock(&cputask[cpu].gate);
	{
		/* ȥե饰γǧ */
		if(proc->state&flag)
		{
			proc->state=TASK_RUNNING;

			if(cputask[cpu].current_task->next==idle_proc[cpu])
			{
				proc->next=proc->prev=proc;
				cputask[cpu].current_task->next=proc;
			}
			else
			{
				/*
				 * ȥץɲåץä⡢󥯤Ǥ褦ˤ롣
				 */
				p=cputask[cpu].current_task->prev->next;
				proc->prev=p;
				proc->next=p->next;
				p->next->prev=proc;
				p->next=proc;
			}
		}
	}
	exit_spinlock(&cputask[cpu].gate);
}


/*
 * GLOBAL
 * Delete process.
 * NOTE!
 *  ץƤ֡
 * parameters : wait flag
 */
void del_from_schedule(int flag)
{
	int cpu=get_current_cpu();
	PROC *proc=cputask[cpu].current_task;


	enter_spinlock(&cputask[cpu].gate);
	{
		/* Delete current task */
		if(proc->next==proc)
		{
			idle_proc[cpu]->next=idle_proc[cpu]->prev=idle_proc[cpu];
			proc->next=idle_proc[cpu];
		}
		else
		{
			proc->next->prev=proc->prev;
			proc->prev->next=proc->next;
		}

		proc->state=flag;
	}
	exit_spinlock(&cputask[cpu].gate);
}


/*
 * GLOBAL
 * Process switch
 * parameters : γ߻Υ֥åȥåesp,γ߻espͤ¸륹å
 * returns : Page directory
 */
uint *switch_task(uint esp,volatile uint before_esp)
{
	uint volatile *pbefore_esp=&before_esp;
	uint volatile *pesp=&esp;
	int cpu;
	PROC *current;


	cpu=get_current_cpu();
	current=cputask[cpu].current_task;
	*pbefore_esp=current->esp;
	current->esp=esp;

	current=current->next;
	*pesp=(uint)&current->esp;
	cputask[cpu].current_task=current;

	return getPageDir();
}


/******************************************************************************************************
 *
 * < ץꥹȥ饹 >
 *
 *******************************************************************************************************/

enum{
	PROC_HASH_NUM=29,	/* ץϥåꥹȺ */
};


/* GLOBAL */
PROC_LIST procHash[PROC_HASH_NUM];		/* ץϥåꥹȡ */


/*
 * PRIVATE
 * ץꥹͤץɥ쥹롣
 * parameters : process link address
 * return : process address
 */
static inline PROC *procLinkToProc(PROC_LIST *p)
{
	return (PROC*)((uint)p-(uint)&((PROC*)0)->proc_next);
}


/*
 * PRIVATE
 * ϥåͤ롣
 * parameters : pid
 * return : hash value
 */
static inline int getProcHash(int pid)
{
	return pid%PROC_HASH_NUM;
}


/*
 * PUBLIC
 * Add to process list.
 * parameters : process address
 */
static inline void addProcList(PROC *proc)
{
	PROC_LIST *hash=&procHash[getProcHash(proc->pid)];
	PROC_LIST *p=(PROC_LIST*)&proc->proc_next;


	enter_spinlock(&hash->lockGate);
	{
		p->prev=hash;
		p->next=hash->next;
		hash->next->prev=p;
		hash->next=p;
	}
	exit_spinlock(&hash->lockGate);
}


/*
 * PUBLIC
 * Delete from process list.
 * parameters : process address
 */
static inline void delProcList(PROC *proc)
{
	PROC_LIST *hash=&procHash[getProcHash(proc->pid)];
	PROC_LIST *p=(PROC_LIST*)&proc->proc_next;


	enter_spinlock(&hash->lockGate);
	{
		p->next->prev=p->prev;
		p->prev->next=p->next;
	}
	exit_spinlock(&hash->lockGate);
}


/*
 * PUBLIC
 * Init process list;
 */
static inline void initProclist()
{
	int i;


	/* ץϥåꥹȽ */
	for(i=0;i<PROC_HASH_NUM;++i)
	{
		procHash[i].next=procHash[i].prev=&procHash[i];
		procHash[i].lockGate=0;
	}
}


/*
 * GLOBAL
 * Search process from process list.
 * parameters : pid
 * return : process address or failed=NULL
 */
PROC *searchProcList(int pid)
{
	PROC_LIST *hash=&procHash[getProcHash(pid)];
	PROC_LIST *p;


	for(p=hash->next;p!=hash;p=p->next)
		if(procLinkToProc(p)->pid==pid)return procLinkToProc(p);

	return NULL;
}


/******************************************************************************************************
 *
 * < ץ饹 >
 *
 *******************************************************************************************************/

extern int *active_cpu;		/* ư CPU */

/* PUBLIC */
static int procLockGate=0;		/* ¾Υץ¤λȻΥåȡ */
static PROC *proc1;				/* pid1ץ¤Ρ */


/*
 * PRIVATE
 * ֥ξʤCPU롣
 * retuuns : cpu number
 */
static int GetFewTaskCpu()
{
	int cpu=0;
	uint num;
	int i;


	num=cputask[0].proc_num;
	for(i = 1; i < *active_cpu; ++i)
		if(cputask[i].proc_num<num)
		{
			num=cputask[i].proc_num;
			cpu=i;
		}

	return cpu;
}


/*
 * PRIVATE
 * λҥץξ֤𤹤롣
 * parameters : process,state address
 * return : 0 or not exit or stop=-1
 */
static void setState(PROC *proc,int *stat_loc)
{
	int tmp=0;


	if(stat_loc==NULL)return;

	if(proc->state==TASK_EXIT)
	{
		if(proc->signum==0)
		{
			*stat_loc|=1<<WIFEXITED_SHIFT;
			tmp=proc->exit_state;
			*stat_loc|=tmp<<WEXITSTATUS_SHIFT;
		}
		else
		{
			*stat_loc|=1<<WIFSIGNALED_SHIFT;
			tmp=proc->signum;
			*stat_loc|=tmp<<WTERMSIG_SHIFT;
		}
	}
	else
	{
		*stat_loc|=1<<WIFSTOPPED_SHIFT;
		tmp=proc->signum;
		*stat_loc|=tmp<<WSTOPSI_SHIFT;
	}
}


/*
 * PRIVATE
 * exitҥץ¤Τ롣
 * parameters : current process
 * return : ҥץ󥯤κǸΥץ
 */
static PROC *releaseChild(PROC *proc)
{
	PROC *p,*q;


	q=(PROC*)((int)proc+((int)&proc->child-(int)&proc->brother));
	for(p=proc->child;p!=NULL;p=p->brother)
	{
		if(p->state==TASK_EXIT)
		{
			q->brother=p->brother;
			kfree(p);
			p=q;
		}
		q=p;
	}

	return q;
}


/*
 * PUBLIC
 * ץ¹Բǽˤ
 * parameters : process
 */
static void addNewProc(PROC *proc)
{
	proc->cpu=GetFewTaskCpu();
	proc->state=TASK_WAIT;
	add_to_schedule(proc,TASK_WAIT);
	++cputask[proc->cpu].proc_num;
}


/*
 * PUBLIC
 * exit()ǥץ򥹥塼뤫롣
 * parameters : process
 */
static void exitFromSchedule(PROC *proc)
{
	static PROC tmp_proc[MAX_CPU];
	int cpu=get_current_cpu();


	del_from_schedule(TASK_EXIT);
	tmp_proc[cpu].next=proc->next;
	tmp_proc[cpu].prev=proc->prev;
	cputask[cpu].current_task=&tmp_proc[cpu];
	--cputask[cpu].proc_num;
}


/*
 * PUBLIC
 * ƥץ¹ԡ
 * 1 exitҥץγ
 * 2 Ĥäҥץpid0ץϤ
 * parameters : current process
 */
static void releaseChildProc(PROC *proc)
{
	PROC *p,*q;


	enter_spinlock(&procLockGate);
	{
		for(p=proc->child;p!=NULL;p=p->brother)
			p->parent=proc1;
		q=releaseChild(proc);
		q->brother=proc1->child;
		proc1->child=proc->child;
	}
	exit_spinlock(&procLockGate);
}


/*
 * PUBLIC
 * 1 exitҥץ롣
 * 2 λҥץξ֤𤹤롣
 * parameters : process structure,search pid,search pgid,task state,state address
 * return : pid or 0 or error number
 */
static int searchExitState(PROC *proc,pid_t pid,pid_t pgid,int state,int *stat_loc)
{
	int rest=-ECHILD;
	PROC *p;


	if(stat_loc!=NULL)*stat_loc=0;

	if(pid!=0)
	{
		for(p=proc->child;p!=NULL;p=p->brother)
			if(p->pid==pid)
			{
				rest=0;
				if(p->state&state)
				{
					setState(p,stat_loc);
					rest=pid;
				}
				break;
			}
	}
	else if(pgid!=0)
	{
		for(p=proc->child;p!=NULL;p=p->brother)
			if(p->pgid==pgid)
			{
				rest=0;
				if(p->state&state)
				{
					setState(p,stat_loc);
					rest=p->pid;
					break;
				}
			}
	}
	else if(proc->child!=NULL)
	{
		rest=0;
		for(p=proc->child;p!=NULL;p=p->brother)
			if(p->state&state)
			{
				setState(p,stat_loc);
				rest=p->pid;
				break;
			}
	}

	/* exitҥץγ */
	releaseChild(proc);

	return rest;
}


/*
 * GLOBAL
 * proc1wait
 */
void waitProc1()
{
	for(;;)
	{
		/* Ԥ */
		del_from_schedule(TASK_SIGNAL_WAIT);
		wait_task();

		enter_spinlock(&procLockGate);
		{
			releaseChild(proc1);
		}
		exit_spinlock(&procLockGate);
	}
	idle();
}


/*
 * GLOBAL
 * Set/release wait terminal flag.
 */
void addNestCnt()
{
	get_current_task()->nest_count+=1;
}

void delNestCnt()
{
	get_current_task()->nest_count-=1;
}


/******************************************************************************************************
 *
 * < ץID롼ID >
 *
 *******************************************************************************************************/

/*
 * PRIVATE
 * ץ롼פθ
 * parameters : search start process,pgid
 * return : process or Failed=NULL
 */
static PROC *searchPg(PROC *proc,pid_t pgid)
{
	PROC *rest;


	do
	{
		if(proc->state!=TASK_EXIT)
		{
			if(proc->pgid==pgid)return proc;
			if(proc->child!=NULL)
				if((rest=searchPg(proc->child,pgid))!=NULL)return rest;
		}
	}while((proc=proc->brother)!=NULL);

	return NULL;
}


/*
 * PUBLIC
 * Get process ID.
 * ա
 *  ĹϢ³Ư硢Сեˤʤǽꡣ
 * return : process ID.
 */
static inline int getPid(PROC *proc)
{
	static int pid=1;

	return pid++;
}


/*
 * GLOBAL
 * Ϳ줿ץ롼פΥץ롣
 * parameters : search start process or NULL=proc1,groupe ID
 * return : prosecc or Failed=NULL
 */
PROC *getNextProc(PROC *proc,pid_t pgid)
{
	/* proc1õ */
	if(proc==NULL)proc=proc1;

	/* μν֤Υץõ */
	if(proc->child!=NULL)return searchPg(proc->child,pgid);
	else
	{
		do
		{
			if(proc->brother!=NULL)return searchPg(proc->brother,pgid);
			if((proc=proc->parent)==NULL)break;
		}while(proc->pgid==pgid);
	}

	return NULL;
}


/*
 * GLOBAL
 * Ϳ줿ץμΥץ롣
 * parameters : search start process or NULL=proc1
 * return : prosecc or Failed=NULL
 */
PROC *getAllProc(PROC *proc)
{
	/* proc1õ */
	if(proc==NULL)proc=proc1;

	/* μν֤Υץ */
	if(proc->child!=NULL)return proc->child;
	else
	{
		do
		{
			if(proc->brother!=NULL)return proc->brother;
		}while((proc=proc->parent)!=NULL);
	}

	return NULL;
}


/***************************************************************************************************
 *
 * System time
 *
 ***************************************************************************************************/


enum{
	CLOCK_TICK=10,		/* åñ ms */
};


/* PRIVATE */
static uint64 beforTime[MAX_CPU];	/* ľΥ桼ȥƥबڤؤä֡ */
static PROC *beforProc[MAX_CPU];	/* ľΥץ */
static uchar which[MAX_CPU];		/* ߤμ¹Ծ 0:system 1:user. */


/*
 * PUBLIC
 * CPU֤ƥץγΰ­
 */
static void addCpuTimeToParent()
{
	PROC *proc,*parent;


	proc=get_current_task();
	parent=proc->parent;
	parent->ctime[0]+=(uint)proc->cpu_time[0]/clock_1m/CLOCK_TICK;
	parent->ctime[1]+=(uint)proc->cpu_time[1]/clock_1m/CLOCK_TICK;
}


/*
 * GLOBAL
 * Calculate CPU time.
 */
void calcCpuTimeInSys()
{
	int cpu;
	uint64 crt_time;
	PROC *proc;


	cpu=get_current_cpu();
	crt_time=rdtsc();
	proc=cputask[cpu].current_task;
	proc->cpu_time[which[cpu]]+=crt_time-beforTime[cpu];
	beforTime[cpu]=crt_time;
	which[cpu]=0;
	beforProc[cpu]=proc;
}

void calcCpuTimeTo(uint volatile ds)
{
	int cpu;
	uint64 crt_time;


	cpu=get_current_cpu();
	crt_time=rdtsc();
	beforProc[cpu]->cpu_time[which[cpu]]+=crt_time-beforTime[cpu];
	beforTime[cpu]=crt_time;
	which[cpu]=(ds>>4)-1;		/* KERNEL_DATA_DES=0x10,USER_DATA_DES=0x23. */
}

void calcCpuTimeToUser()
{
	int cpu;
	uint64 crt_time;


	cpu=get_current_cpu();
	crt_time=rdtsc();
	cputask[cpu].current_task->cpu_time[which[cpu]]+=crt_time-beforTime[cpu];
	beforTime[cpu]=crt_time;
	which[cpu]=1;
}


/******************************************************************************************************
 *
 * ƥॳ
 *
 *******************************************************************************************************/

/*
 * return : ƥץʤҥץID,ҥץʤ0,error=error number
 * todoꥹ
 *  ǥ쥯ȥꥹȥ꡼򥳥ԡ롣
 */
int sys_fork()
{
	void *stack;
	uint esp;
	PROC *parent,*child;


	parent = get_current_task();
	if (parent->count_child >= MAX_CHILD)
		return -EAGAIN;

	/* PROC¤ΤƤ */
	if ((child = kmalloc(sizeof(PROC))) == NULL)
		return -ENOMEM;

	/* 0 */
	memset(&child->PROC_ZERO_TOP,0,(uint)&child->PROC_ZERO_END-(uint)&child->PROC_ZERO_TOP);
	child->parent=parent;
	child->brother=parent->child;
	parent->child=child;
	child->nest_count=parent->nest_count;

	/* ID. */
	child->pid=getPid(child);
	child->pgid=parent->pgid;
	child->uid=0;
	child->gid=parent->gid;

	/* å󥢥ɥ쥹(ü¤Υɥ쥹)򥳥ԡ */
	child->ctlterm=parent->ctlterm;

	/* եط¤Τγơ */
	if((child->file_struct=cpy_file_struct(parent->file_struct))==NULL)goto ERR;

	/* Page tableƤ */
	if((child->mm_struct=forkPage(parent->mm_struct,&stack))==NULL)goto ERR;

	/* ʥꤹ롣 */
	if((child->signal_struct=setSignal(parent->signal_struct))==NULL)goto ERR;

	/* ޡν */
	setTimerStruct(child);

	/* ҥץstack˥ƥȤꤹ */
	esp = setContext((uint)stack);
	if (esp == 0)
		return 0;		/* ҥץ */

	child->esp=esp;

	/* ץꥹȤ˲ä롣 */
	addProcList(child);

	/* Add child process to schedule queue */
	addNewProc(child);

	++parent->count_child;

	return child->pid;
ERR:
	kfree(child);

	return -ENOMEM;
}


int sys_exec(const char *path,char **argv,char **envp,volatile SYSCALL4_FRAME frame)
{
	enum{PARAM=sizeof(frame.para)};	/* ƥॳΰΥ */

	uint entry_addr,stack_esp;
	int rest;
	OPEN_F open_f;
	SYS_INFO *sys_info;


	if((rest=exec_open(path,&open_f))<0)return rest;

	/* ȴĶѿ桼å˥ԡ롣 */
	if((stack_esp=setArgAndEnv(argv,envp))==0)return -1;

	/* 桼ڡ롣 */
	releaseUserTextDataPage(get_current_task()->mm_struct);

	/* ELFХʥΥɡ */
	if((entry_addr=loadElf(&open_f))==0)return -1;

	/* å˥ƥꡣ */
	sys_info=(SYS_INFO*)(USER_ESP_BEG-sizeof(SYS_INFO));
	sys_info->lastAddr=getLastLinearAddr();

	/* ʥ륢ν */
	resetSignal(get_current_task()->signal_struct);

	/* ե빽¤Τν */
	initExecFs();

	/* ƥॳե졼ꡣ */
	frame.es=USER_DATA_DES;
	frame.ds=USER_DATA_DES;
	frame.eip=entry_addr;
	frame.cs=USER_CODE_DES;
	frame.user_esp=stack_esp-PARAM;
	frame.ss=USER_DATA_DES;

	return 0;
}


/*
 * GLOBAL
 * ڡǥ쥯ȥγ̥ץԤ
 * parameters : ƥץϤ
 */
void exit(int state,int signum)
{
	void *mm_struct;
	PROC *proc=get_current_task();
	PROC *parent;


	/* üץγ */
	releaseTerm(proc);

	/* ץꥹȤ롣 */
	delProcList(proc);

	/* ҥץγ */
	releaseChildProc(proc);

	parent=proc->parent;

	/* ץ󥯤롣 */
/*	if((p=parent->child)==proc)parent->child=proc->brother;
	else
	{
		for(;p->brother!=proc;p=p->brother);
		p->brother=proc->brother;
	}
*/
	--parent->count_child;

	/* եǥץγ */
	releaseFileStruct(proc->file_struct);

	/* ޡγ */
	releaseTimer(&proc->timer,&proc->alarm);

	/* ʥγ */
	releaseSignal(proc->signal_struct);

	mm_struct=proc->mm_struct;
	proc->exit_state=state&0xff;
	proc->signum=signum&0xff;

	/* CPUƥץ­ */
	addCpuTimeToParent();

	/* 塼뤫waitƤƤƤӽФ */
	cli();
	enter_spinlock(&procLockGate);
	{
		exitFromSchedule(proc);
		if((isForegrouond(proc))&&(parent->state&TASK_CHILD_WAIT))
			add_to_schedule(parent,TASK_SIGNAL_WAIT);
	}
	exit_spinlock(&procLockGate);

	/* Υ˥åơڡǥ쥯ȥ롣 */
	returnExit(mm_struct);
}


int sys_exit(int state)
{
	exit(state,0);

	return 0;
}


int sys_wait(int pid,int *stat_loc,int options)
{
	int rest;
	int state;
	PROC *proc=get_current_task();


	/* Ĵ٤ץ֤ꡣ */
	if(options&WUNTRACED)state=TASK_WAIT|TASK_DEVICE_WAIT|TASK_SIGNAL_WAIT|TASK_TRACE|TASK_EXIT;
	else state=TASK_TRACE|TASK_EXIT;

	/* ǽλҥץ */
	if(pid>0)rest=searchExitState(proc,pid,0,state,stat_loc);
	else if(pid==0)rest=searchExitState(proc,0,proc->pgid,state,stat_loc);
	else if(pid==-1)rest=searchExitState(proc,0,0,state,stat_loc);
	else rest=searchExitState(proc,0,-pid,state,stat_loc);

	if(rest>0)return rest;
	if((rest<0)&&(options&WNOHANG))return rest;

	/* Ԥ */
	del_from_schedule(TASK_SIGNAL_WAIT|TASK_CHILD_WAIT);
	wait_task();

	/* ҥץ */
	if(pid>0)rest=searchExitState(proc,pid,0,state,stat_loc);
	else if(pid==0)rest=searchExitState(proc,0,proc->pgid,state,stat_loc);
	else if(pid==-1)rest=searchExitState(proc,0,0,state,stat_loc);
	else rest=searchExitState(proc,0,-pid,state,stat_loc);

	return rest;
}


/*
 * Get process ID.
 */
int sys_get_procid(pid_t pid,PROC_ID *pd)
{
	PROC *proc;


	if(pid==0)
		proc=get_current_task();
	else
	{
		if((proc=searchProcList(pid))==NULL)return -ESRCH;
		if(proc->ctlterm!=get_current_task()->ctlterm)return -EPERM;
	}

	pd->pid=proc->pid;
	pd->pgid=proc->pgid;
	pd->ppid=proc->parent->pid;
	pd->uid=proc->uid;
	pd->gid=proc->gid;

	return 0;
}


/*
 * Set process ID.
 */
int sys_set_procid(pid_t pid,PROC_ID *pd)
{
	PROC *proc;


	if(pid==0)
		proc=get_current_task();
	else
	{
		if((proc=searchProcList(pid))==NULL)return -ESRCH;
	}

	if(pd->pgid>=0)
	{
		/* ץå꡼ */
		if(isCtlproc(proc))return -EPERM;

		/* ȥץλҥץǤʡ */
		if(proc->pgid!=get_current_task()->pgid)return -EPERM;

		if(pd->pgid==0)proc->pgid=proc->pid;
		else proc->pgid=pd->pgid;
	}
	else if(pd->uid>=0)
	{
		if(proc->uid==0)proc->uid=pd->uid;
		else return -EPERM;
	}
	else if(pd->gid>=0)
	{
		if(proc->uid==0)proc->gid=pd->gid;
		else return -EPERM;
	}

	return 0;
}


int sys_times(struct tms *tms)
{
	PROC *proc;


	if(checkMem(tms)==-1)return -EFAULT;

	proc=get_current_task();
	tms->tms_stime=proc->cpu_time[0]/clock_1m/CLOCK_TICK;
	tms->tms_utime=proc->cpu_time[1]/clock_1m/CLOCK_TICK;
	tms->tms_cstime=proc->ctime[0];
	tms->tms_cutime=proc->ctime[1];

	return 0;
}


/*
 * ü򥯥ꥢ롣
 *
 */
int sys_setsid()
{
	get_current_task()->ctlterm=NULL;
	return 0;
}


/******************************************************************************************************
 *
 * < 饹 >
 *
 *******************************************************************************************************/

/*
 * GLOBAL
 * Init taskcpu
 * cpu˥ꥢꤹɬפ
 * parameters : number of cpu
 * returns : NOERROR
 */
int init_cputask(int cpu)
{
	cputask[cpu].cpu=cpu;
	cputask[cpu].current_task=NULL;
	cputask[cpu].proc_num=0;
	cputask[cpu].gate=0;

	return 0;
}


/*
 * GLOBAL
 * TSSǥץꤹ
 *  : ǥץ͡TSSɥ쥹
 */
void set_tss(int des)
{
	set_gdt(des,(uint)&kernel_tss,104,TYPE_TSS);

	asm volatile(
		"movl	%0,%%eax\n"
		"ltr	%%ax"
		::"m"(des)
	);
}


/*
 * GLOBAL
 * Set idle process table
 * parameters : Page directory
 * returns : 0 or Error number
 */
int set_idle_proc(int cpu)
{
	/* Set process table */
	idle_proc[cpu] = kmalloc(sizeof(PROC));
	if (idle_proc[cpu] == NULL)
		return -ENOMEM;
	memset(idle_proc[cpu],0,sizeof(PROC));
	idle_proc[cpu]->next=idle_proc[cpu]->prev=idle_proc[cpu];
	idle_proc[cpu]->cpu=cpu;
	idle_proc[cpu]->state=TASK_RUNNING;

	/* init paging */
	if((idle_proc[cpu]->mm_struct=initPaging())==NULL)return -1;

	/* init file descriptor. */
	if(init_file_struct(idle_proc[cpu])==-1)return -ENOMEM;

	/* init signal. */
	if((idle_proc[cpu]->signal_struct=initSignal())==NULL)return -ENOMEM;

	/* add to schedule queue */
	cputask[cpu].current_task=idle_proc[cpu];

	/* ץꥹȤν */
	initProclist();

	return 0;
}


/*
 * GLOBAL
 * ǽΥץꡣ
 * return : 0 or error number
 */
int initProc1()
{
	proc1=get_current_task();
	proc1->parent=NULL;

	return initProc1Fs();
}

/*************************************************************************************************/
void test_proc()
{
	printk("4000616c=%x\n",*(uint*)0x400058f2);
}
/**************************************************************************************************/
