/*
 * console.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include<types.h>
#include<lib.h>
#include<lock.h>
#include<device.h>
#include<wait.h>
#include<interrupt.h>
#include<signal.h>
#include<keymap.h>
#include<term.h>
#include<errno.h>
#include<mm.h>
#include<console.h>


/***************************************************************************************
 *
 * ü
 *
 ***************************************************************************************/


/* GLOBAL */
TERM_CTL termCtl;


/*
 * PUBLIC
 * ҥץƱGIDΥץ˥ʥ롣
 * parameters : begin process,groupe ID,signal
 */
static void sendSignalToGrp(PROC *parent,pid_t pgid,int signal)
{
	PROC *p;


	for(p=parent;(p=getNextProc(p,pgid))!=NULL;)
		sendSignal(p,signal);
}


/*
 * PUBLIC
 * Console termios struct.
 */
static struct termios termio={
	ICRNL|IXOFF|IXON,	/* c_iflag. */
	OPOST,				/* c_oflag. */
	CS8|CREAD,			/* c_cflag. */
	ISIG|ICANON|ECHO,	/* c_lflag. */
};


/*
 * PUBLIC
 * Init terminal.
 */
static inline void initTerm()
{
	/* termiosꡣ */
	termio.c_cc[VINTR]=	CINTR;
	termio.c_cc[VQUIT]=	CQUIT;
	termio.c_cc[VSTART]=CSTART;
	termio.c_cc[VSTOP]=	CSTOP;
	termio.c_cc[VSUSP]=	CDSUSP;
	termio.c_cc[VEOF]=	CEOF;
	termio.c_cc[VEOL]=	CEOL;
	termio.c_cc[VERASE]=CERASE;
	termio.c_cc[VKILL]=	CKILL;
}


/*
 * PUBLIC
 * ɤ߽񤭥ץե饦ɥ롼פå
 */
static inline void isForeground()
{
	if(get_current_task()->pgid!=termCtl.foregrnd)
		del_from_schedule(TASK_SIGNAL_WAIT);
}


/*
 * GLOBAL
 * Release term top process.
 * parameters : process
 */
void releaseTerm(PROC *proc)
{
	if(termCtl.ctlProc==proc)termCtl.ctlProc=NULL;
}


/***************************************************************************************
 *
 * ǥץ쥤
 *
 * ǽ
 * ʸΰ
 * ΰư
 * 륢åס
 *
 ***************************************************************************************/

enum{
	/* VGA Video ram. */
	VRAM_BASE_ADDR=0xb8000,	/* Start address of video memory */
	VRAM_SIZE=0x8000,		/* Size of video memory */

	/* CRT IO registers. */
	CTL_ADDR=0x3d4,			/* Addres register */
	CTL_DATA=0x3d5,			/* Data IO register */
	START_ADDR_HIGH=0xc,	/* High byte of print start address */
	START_ADDR_LOW=0xd,		/* Low  byte of print start address  */
	CURSOR_ADDR_HIGH=0xe,	/* High byte of cursor address */
	CURSOR_ADDR_LOW=0xf,	/* Low  byte of cursor address */

	SCR_LINES=25,			/* Number of lines. */
	SCR_COLUMNS=80,			/* Number of columns. */
	SCR_LAST=				/* Video ram last address. */
		VRAM_SIZE/2-SCR_LINES*SCR_COLUMNS-SCR_COLUMNS*3,

	TAB=8,					/* Tab characters. */
};


typedef struct{
	int viewTop;	/* ӥǥ꡼β̤γϰ֡ */
	int start;		/* ϰ֡ */
}SCREEN;


/* PRIVATE */
static char *vram=(char*)VRAM_BASE_ADDR;
static SCREEN screen;


/*
 * PRIVATE
 * Update cursor position.
 */
static inline void updateCursor()
{
	outb(CTL_ADDR,CURSOR_ADDR_HIGH);
	outb(CTL_DATA,screen.start>>8);
	outb(CTL_ADDR,CURSOR_ADDR_LOW);
	outb(CTL_DATA,screen.start);
}


/*****************************************************
static void testPrint(int pos,int uvalue)
{
	char tmp_buf[12];
	int i;


	for(i=0;;++i)
	{
		tmp_buf[i]=uvalue%10+'0';
		if(!(uvalue/=10))break;
	}
	for(;i>=0;--i,++pos)vram[pos*2]=tmp_buf[i];
}
*******************************************************/


/*
 * PRIVATE
 * Hard scroll up.
 */
static void scrollup()
{
	char *p;
	int i,last;


	screen.viewTop=ROUNDUP(screen.start-SCR_COLUMNS*SCR_LINES+1,SCR_COLUMNS);

	if(screen.viewTop>=SCR_LAST)
	{
		/* Video ramƬ˰ư */
		p=&vram[screen.viewTop*2];
		last=(screen.start-screen.viewTop)*2;
		for(i=0;i<last;i+=2)vram[i]=p[i];

		/* Video ram0ꥢ */
		for(;i<(SCR_LAST+SCR_COLUMNS*SCR_LINES)*2;i+=2)vram[i]=0;

		screen.start-=screen.viewTop;
		screen.viewTop=0;
	}

	/* Hard scroll up */
	outb(CTL_ADDR,START_ADDR_HIGH);
	outb(CTL_DATA,screen.viewTop>>8);
	outb(CTL_ADDR,START_ADDR_LOW);
	outb(CTL_DATA,screen.viewTop);
}


/*
 * PUBLIC
 * 1ʸ
 * parameters : buffer,size
 */
static inline void putIntoScreen(char c)
{
	vram[screen.start*2]=c;
	++screen.start;
}


/*
 * PUBLIC
 * ʣʸ
 * parameters : buffer,size
 */
static inline void putNumIntoScreen(char *buf,int size)
{
	int i,last;


	for(i=screen.start*2,last=i+size*2;i<last;i+=2)vram[i]=*buf++;
}


/*
 * PUBLIC
 * 
 */
/* Left cursor. */
static inline void leftCursor(int n)
{
	screen.start-=n;
	updateCursor();
}

/* Right cursor. */
static inline void rightCursor(int n)
{
	screen.start+=n;
	if(screen.start-screen.viewTop>=SCR_COLUMNS*SCR_LINES)
		scrollup();
	updateCursor();
}

/* Carriage retun. */
static inline void carriageRetun()
{
	screen.start=ROUNDDOWN(screen.start,SCR_COLUMNS);
	updateCursor();
}

/* New line. */
static inline void newLine()
{
	screen.start=ROUNDUP(screen.start+1,SCR_COLUMNS);
	if(screen.start-screen.viewTop>=SCR_COLUMNS*SCR_LINES)
		scrollup();
	updateCursor();
}

/* Tab. */
static inline void tab()
{
	screen.start=ROUNDUP(screen.start+1,TAB);
	if(screen.start-screen.viewTop>=SCR_COLUMNS*SCR_LINES)
		scrollup();
	updateCursor();
}


/*
 * PUBLIC
 * 
 */
static void initScreen()
{
	short *p;


	/* Start position 0 */
	outb(CTL_ADDR,START_ADDR_HIGH);
	outb(CTL_DATA,0);
	outb(CTL_ADDR,START_ADDR_LOW);
	outb(CTL_DATA,0);

	/* Clear screen */
	for(p=(short*)VRAM_BASE_ADDR;p<(short*)(VRAM_BASE_ADDR+VRAM_SIZE);++p)*p=0x700;
}


/***************************************************************************************
 *
 * 󥽡Хåե
 *
 ***************************************************************************************/

enum{
	CONSOLE_BUF_SIZE=90,	/* Line buffer size. */
	CONSOLE_EOF=0xff,		/* EOF character. */
};


typedef struct{
	int crt;					/* ȥݥ */
	int last;					/* 饹ȥݥ */
	int lineTop;				/* 饤ȥåץݥ */
	char buf[CONSOLE_BUF_SIZE];	/* 󥽡Хåե */
}CONSOL_BUF;


/* PUBLIC */
static CONSOL_BUF lbuf;


/*
 * PRIVATE
 * ꡼Ǹ夫饳ԡ
 * parameters : destination buffer,source buffer
 */
static inline void memcpyBack(char *dst,char *src,int n)
{
	while(--n>=0)dst[n]=src[n];
}


/*
 * PUBLIC
 * 󥽡Хåե
 */
/* 1ʸɲá */
static inline void putToBuf(char c)
{
	if(lbuf.crt==CONSOLE_BUF_SIZE)return;

	/* Ȱʹߤʸ˰ư */
	if(lbuf.last==CONSOLE_BUF_SIZE)--lbuf.last;
	memcpyBack(lbuf.buf+lbuf.crt+1,lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
	++lbuf.last;

	/* ʸ */
	lbuf.buf[lbuf.crt]=c;
	putNumIntoScreen(lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
	rightCursor(1);
	++lbuf.crt;
}

/* Delete. */
static inline void deleteBuf()
{
	/* +1ʹߤʸ˰ưǸ˶ */
	if(lbuf.last>lbuf.crt)
	{
		memcpy(lbuf.buf+lbuf.crt,lbuf.buf+lbuf.crt+1,lbuf.last-lbuf.crt);
		lbuf.buf[lbuf.last-1]='\0';
		putNumIntoScreen(lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
		--lbuf.last;
	}

}

/* Back space. */
static inline void backSpaceBuf()
{
	/* Ȱʹߤʸ˰ưǸ˶ */
	if(lbuf.crt>lbuf.lineTop)
	{
		memcpy(lbuf.buf+lbuf.crt-1,lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
		lbuf.buf[lbuf.last-1]='\0';
		--lbuf.crt;
		leftCursor(1);
		putNumIntoScreen(lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
		--lbuf.last;
	}
}

/* Left cursor. */
static inline void leftCursorBuf()
{
	if(lbuf.crt>lbuf.lineTop)
	{
		leftCursor(1);
		--lbuf.crt;
	}
}

/* Right cursor. */
static inline void rightCursorBuf()
{
	if(lbuf.crt<lbuf.last)
	{
		rightCursor(1);
		++lbuf.crt;
	}
}

/* Erase. */
static inline void eraseLine()
{
	memset(lbuf.buf+lbuf.lineTop,0,lbuf.last-lbuf.lineTop);
	leftCursor(lbuf.crt-lbuf.lineTop);
	putNumIntoScreen(lbuf.buf+lbuf.lineTop,lbuf.last-lbuf.lineTop);
	lbuf.crt=lbuf.last=lbuf.lineTop;
}

/* New line. */
static inline void newLineBuf()
{
	newLine();
	if(lbuf.last<CONSOLE_BUF_SIZE)
		lbuf.buf[lbuf.last++]='\n';
	lbuf.lineTop=lbuf.crt=lbuf.last;
}

/*
 * EOF.
 * return : ʸʤ=1 or 0
 */
static int Eof()
{
	if(lbuf.lineTop==lbuf.last)	/* ʸʤ */
	{
		lbuf.buf[lbuf.last++]=CONSOLE_EOF;
		lbuf.lineTop=lbuf.crt=lbuf.last;
		return 1;
	}

	return 0;
}


/*
 * PUBLIC
 * Check console buffer.
 * parameters : user buffer,max copy size
 * return : ϥХȿ or EOF=0 or Ϥʤ=-1
 */
static int readConsoleBuf(uchar *user_buf,size_t size)
{
	int i;


	if(termio.c_lflag&ICANON)	/* Canonical mode. */
	{
		if(lbuf.last==0)return -1;		/* Ϥʤ */

		if((uchar)*lbuf.buf==CONSOLE_EOF)
		{
			*lbuf.buf='\0';
			lbuf.lineTop=lbuf.crt=lbuf.last=0;

			return 0;
		}

		i=0;
		while (lbuf.buf[i++]!='\n')
			if (i >= lbuf.last)
				return -1;
		size=(size>i)?i:size;
		memcpy(user_buf,lbuf.buf,size);
		memcpy(lbuf.buf,lbuf.buf+i,lbuf.last-i);
		lbuf.crt-=i;
		lbuf.last-=i;
		lbuf.lineTop-=i;
	}
	else	/* Non-canonical mode. */
	{
/************************ NON CANONICAL MODE ̤ ***************************/
	}

	return size;
}


/*
 * PUBLIC
 * 󥽡Хåեɤ߹߲ǽ
 * return : no=0 or yes=1
 */
static int checkConsoleBuf()
{
	int i;


	if((uchar)*lbuf.buf==CONSOLE_EOF)return 1;
	for(i=0;i<lbuf.last;++i)
		if(lbuf.buf[i]=='\n')return 1;

	return 0;
}


/***************************************************************************************
 *
 * ܡ
 *
 ***************************************************************************************/

enum{
	IRQ_NO=1,		/* IRQ number */
};


/* PUBLIC */
static PROC *waitProc;	/* Ԥץ */
static int waitLock;	/* ѥå */


/*
 * PRIVATE
 * Do control command in canonical mode.
 * parameters : character
 * return : 1=ƤӽФȤϽλ of 2=readԤ򵯤 or 0
 */
static inline int doCanonTermCtr(uchar ch)
{
	uchar *cc;


	if(termio.c_lflag&ICANON)
	{
		cc=termio.c_cc;
		if(ch==cc[VINTR])			/* Send SIGINT. */
		{
			/* ե饦ȥ롼פSIGINT롣 */
			if(termCtl.ctlProc->child!=NULL)
				sendSignalToGrp(termCtl.ctlProc,termCtl.foregrnd,SIGINT);
			return 1;
		}
		else if(ch==cc[VQUIT])		/* Send SIGQUIT. */
		{
			return 0;
		}
		else if(ch==cc[VSTART])		/* Send SIGCONT.  */
		{
			return 0;
		}
		else if(ch==cc[VSTOP])		/* Send SIGSTOP. */
		{
			return 0;
		}
		else if(ch==cc[VSUSP])		/* Send SIGTSTP. */
		{
			/* ե饦ȥ롼פSIGTSTP롣 */
			if(termCtl.ctlProc->child!=NULL)
				sendSignalToGrp(termCtl.ctlProc,termCtl.foregrnd,SIGTSTP);
			return 0;
		}
		else if(ch==cc[VEOF])		/* եνλ */
		{
			if(Eof())return 2;
			return 0;
		}
		else if(ch==cc[VEOL])		/* Ԥνλ */
		{
			newLineBuf();
			return 1;
		}
		else if(ch==cc[VERASE])		/* ʸõ */
		{
			backSpaceBuf();
			return 1;
		}
		else if(ch==cc[VKILL])		/* á */
		{
			eraseLine();
			return 1;
		}
	}

	return 0;
}


/*
 * PRIVATE
 * Turn on/off LED.
 */
static inline void switchLed(uchar value)
{
	outb(KEYBOARD_OUTPUT_REG,0xed);		/* Command turn on/of LED. */
	while(inb(KEYBOARD_STAT_REG)&2);
	outb(KEYBOARD_OUTPUT_REG,value);	/* LED on/off. */
}


/*
 * PRIVATE
 * keyboard main handler.
 * parameters : scan key coad
 * return : task switch on=1 or off=0
 */
static int keyboardHnadler(uchar key_coad)
{
	enum{
		KEY_BREAK=0x80,	/* key break bit. */

		/* Add key bit. */
		SHT_BIT=1<<0,
		CTR_BIT=1<<1,

		/* LED bit. */
		LED_SCRL=1<<0,	/* ScrollLock LED. */
		LED_NMLC=1<<1,	/* NumLock LED. */
		LED_CPSL=1<<2,	/* CapsLock LED. */
	};

	static int add_key=0;				/* normal=0,add shift=1,add ctrl=2. */
	static int func_key=0;				/* ǽե饰(󥳡0xe0) */
	static int shift_key=0;				/* Shift key flag. */
	static int led=0;					/* LED flag, */
	static int ctr_key=0;				/* Ctrl key. */
	static uchar add_map[]={0,1,2,2};	/* ղåѴޥåס */

	uchar map_coad;


	/* ǽν */
	if(func_key)
	{
		if((key_coad>=VMAP_START_COAD)&&(key_coad<=VMAP_END_COAD))
			key_coad+=VMAP_ADD;
		func_key=0;
	}
	else if(key_coad==KEY_FUNC)
	{
		func_key=1;
		return 0;
	}

	/* ޥåץɼ */
	if((map_coad=keymap[key_coad&~KEY_BREAK][add_key])==0)return 0;

	/* ֥졼ɤθCtrl ShiftʳϼΤƤ롣 */
	if(key_coad&KEY_BREAK)
	{
		switch(map_coad)
		{
			case MAP_SHT:
				shift_key=0;
				break;
			case MAP_CTR:
				ctr_key=0;
				break;
			default:
				return 0;
		}
		add_key=add_map[(shift_key|ctr_key)^(led>>2)];

		return 0;
	}

	if(termio.c_lflag&ICANON)		/* Canonical mode. */
	{
		/* ȥ륳ޥɤθ */
		if(map_coad<' ')
		{
			switch(doCanonTermCtr(map_coad))
			{
				case 1:return 0;
				case 2:return 1;
			}
			switch(map_coad)
			{
				case '\r':
					if(termio.c_iflag&IGNCR)break;
				case '\n':
					newLineBuf();
					return 1;
			}
		}
		else
		{
			if(map_coad<MIN_MAPKEY)putToBuf(map_coad);
			else
				switch(map_coad)
				{
					case MAP_BS:
						backSpaceBuf();
						break;
					case MAP_DEL:
						deleteBuf();
						break;
					case MAP_LFT:
						leftCursorBuf();
						break;
					case MAP_RHT:
						rightCursorBuf();
						break;
					case MAP_SHT:
						shift_key=SHT_BIT;
						add_key=add_map[(shift_key|ctr_key)^(led>>2)];
						break;
					case MAP_CPSL:
						switchLed(led^=LED_CPSL);
						add_key=add_map[(shift_key|ctr_key)^(led>>2)];
						break;
					case MAP_CTR:
						ctr_key=CTR_BIT;
						add_key=add_map[(shift_key|ctr_key)^(led>>2)];
						break;
				}
		}
	}
	else						/* Non-canonical mode. */
	{
		/************************ ̤ ***************************/
	}

	return 0;
}


/*
 * GLOBAL
 * keyboard interrupt handler.
 * return : task switch on off
 */
static int keyboardIntrHandler()
{
	int rest=0;


	if(keyboardHnadler(inb(KEYBOARD_OUTPUT_REG)))
	{
		enter_spinlock(&waitLock);
		{
			if(waitProc!=NULL)
			{
				forceSendSignal(waitProc,SIGCONT);
				waitProc=NULL;
				rest=1;
			}
		}
		exit_spinlock(&waitLock);
	}

	return rest;
}


/***************************************************************************************
 *
 * 񤭹
 *
 ***************************************************************************************/


/*
 * ͥޤϥƥ뤫ƤФ롣
 * parameters : baffer,baffer size,dummy
 * return : write size
 */
int writeConsole(void *buf,size_t size,size_t start)
{
	char *str=(char*)buf;
	int i;


	for(i=0;i<size;++i)
	{
		if(str[i]>=' ')
			putIntoScreen(str[i]);
    	else if(str[i]=='\n')
    		newLine();
    	else if(str[i]=='\r')
    		carriageRetun();
    	else if(str[i]=='\t')
    		tab();
    }
    rightCursor(0);

	return i;
}


/***************************************************************************************
 *
 * ƥॳ
 *
 ***************************************************************************************/


static int open()
{
	return 0;
}

/* size=žǽХȿ */
static int read(void *buf,size_t size,size_t start)
{
	int len;


	/* ե饦ɥ롼פ? */
	while(isForegrouond(get_current_task())==0)
	{
		del_from_schedule(TASK_SIGNAL_WAIT);
		wait_task();
	}

	/* Ϥ뤫󥽡Хåեǧ */
	enter_spinlock(&waitLock);
	if((len=readConsoleBuf(buf,size))==-1)
	{
		waitProc=get_current_task();			/* Ԥ */
		exit_spinlock(&waitLock);
		sigWait();								/* ȡ */

		len=readConsoleBuf(buf,size);
	}
	else exit_spinlock(&waitLock);

	return len;
}


static int write(void *buf,size_t count,size_t x)
{
	/* ե饦ɥ롼פ? */
	while(isForegrouond(get_current_task())==0)
	{
		del_from_schedule(TASK_SIGNAL_WAIT);
		wait_task();
	}

	return writeConsole(buf,count,0);
}


int ioctl(int cmd,...)
{
	uint *arg=(uint*)&cmd+1;
	PROC *proc,*grp_proc;
	pid_t pgid;
	struct winsize *win;


	switch(cmd)
	{
		case TIOCSCTTY:
			if(termCtl.ctlProc==NULL)
			{
				termCtl.ctlProc=get_current_task();
				termCtl.foregrnd=termCtl.ctlProc->pgid;
				termCtl.ctlProc->ctlterm=&termCtl;
			}
			else return -EPERM;
			break;
		case TIOSPGRP:
			pgid=(gid_t)*arg;
			proc=get_current_task();
			if(proc!=termCtl.ctlProc)return -ENOTTY;
			if(proc->pgid!=pgid)
			{
				/* Ϳ줿롼פƱåΤΤ */
				if((grp_proc=getNextProc(proc,pgid))==NULL)return -EINVAL;
				if((grp_proc->ctlterm==NULL)||(((TERM_CTL*)grp_proc->ctlterm)->ctlProc!=proc))return -EPERM;
			}
			termCtl.foregrnd=pgid;

			break;
		case TIOGPGRP:
			if(get_current_task()->ctlterm!=&termCtl)return -ENOTTY;
			return termCtl.foregrnd;
		case TIOCGWINSZ:
			win=(struct winsize*)*arg;
			if(checkMem(win)==-1)return -EFAULT;
			win->ws_row=SCR_LINES;
			win->ws_col=SCR_COLUMNS;
			break;
	}

	return 0;
}


/* return : 0 or 1=Ѳ */
static int select(int flag,int polling)
{
	int rest;


	if(flag==FD_WRITE)return 1;

	/* polling mode. */
	if(polling)
		return checkConsoleBuf();

	enter_spinlock(&waitLock);
	{
		if((rest=checkConsoleBuf())==0)
			waitProc=get_current_task();	/* Ԥ */
	}
	exit_spinlock(&waitLock);

	return rest;
}


/***************************************************************************************
 *
 * 
 *
 ***************************************************************************************/

/* PRIVATE,Device interface */
static DEV_INFO devInfo={
	"console",0,0,0,open,read,write,ioctl,select
};


/*
 * CLOBAL
 * ǥХϿ򤹤롣
 */
int registConsole()
{
	return regist_device(&devInfo);
}


/*
 * GLOBAL
 * Init console
 */
int initConsole()
{
	/* ꡼ν */
	initScreen();

	/* üǽν */
	initTerm();

	return 0;
}


/*
 * GLOBAL
 * init keyboard driver
 */
void initKeyboard()
{
	/* Turn off LED. */
	switchLed(0);

	/* ߤ */
	irq_entry[IRQ1]=keyboardIntrHandler;

	release_irq_mask(IRQ_NO);

	/* clear data register */
	inb(KEYBOARD_OUTPUT_REG);
}
