<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * 入力値の検証を行うためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Validator
{
    /** @var object Moony_Requestのインスタンス */
    var $request;

    /** @var string エラー時に表示するテンプレートファイル名 */
    var $template;

    /** @var boolean 妥当かどうかの検証結果 */
    var $is_valid;

    /** @var array エラーメッセージの連想配列 */
    var $errors;

    /**
     * コンストラクタです。
     * 通常はgetInstanceを使用してください。
     *
     * @access private
     */
    function Moony_Validator()
    {
        $this->template = null;
        $this->is_valid = true;
        $this->errors = array();
    }

    /**
     * このクラスのシングルトンインスタンスを取得します。
     *
     * @access public
     * @param object $request Moony_Requestのインスタンス
     * @return object シングルトンインスタンス
     */
    function &getInstance(&$request)
    {
        static $instance = null;
        if (is_null($instance)) {
            $instance =& new Moony_Validator();
        }
        $instance->request = &$request;
        return $instance;
    }

    /**
     * 必須項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setRequired($name, $message)
    {
        $value = $this->request->get($name);
        if (Moony_Checker::isPresent($value)) {
            return true;
        }
        $this->addError($name, $message);
        $this->is_valid = false;
        return false;
    }

    /**
     * 正規表現を用いたフォーマットの検証設定を行います。
     * 内部でpreg_matchが使用されます。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param string $pattern 正規表現パターン
     * @return boolean 対象項目の検証結果
     */
    function setRegex($name, $message, $pattern)
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isRegex'), $pattern);
    }

    /**
     * 正規表現を用いたフォーマットの検証設定を行います。
     * 内部でmb_eregが使用されます（マルチバイト対応）。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param string $pattern 正規表現パターン
     * @return boolean 対象項目の検証結果
     */
    function setRegexMb($name, $message, $pattern)
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isRegexMb'), $pattern);
    }

    /**
     * 数値項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setNumeric($name, $message)
    {
        return $this->setFunc($name, $message, 'is_numeric');
    }

    /**
     * 英字項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setAlpha($name, $message)
    {
        return $this->setFunc($name, $message, 'ctype_alpha');
    }

    /**
     * 英数字項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setAlnum($name, $message)
    {
        return $this->setFunc($name, $message, 'ctype_alnum');
    }

    /**
     * 数字項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setDigit($name, $message)
    {
        return $this->setFunc($name, $message, 'ctype_digit');
    }

    /**
     * カタカナ項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setKatakana($name, $message)
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isKatakana'));
    }

    /**
     * ひらがな項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setHiragana($name, $message)
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isHiragana'));
    }

    /**
     * Eメールアドレス項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setEmail($name, $message)
    {
        return $this->setRegex($name, $message, '/^[a-z0-9\-\._]+@[a-z0-9]([0-9a-z\-]*[a-z0-9]\.){1,}[a-z]{1,4}$/i');
    }

    /**
     * URL項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @return boolean 対象項目の検証結果
     */
    function setUrl($name, $message)
    {
        return $this->setRegex($name, $message, '/^https?:\/\/[-_.!~*\'()a-z0-9;\/?:\@&=+\$,%#]+$/i');
    }

    /**
     * 入力桁数に関する検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param integer $min 最小長
     * @param integer $max 最大長
     * @param boolean $inclusive 最小長/最大長と同値を許容するかどうか
     * @return boolean 対象項目の検証結果
     */
    function setLength($name, $message, $min, $max, $inclusive = true)
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isLengthBetween'), $min, $max, $inclusive);
    }

    /**
     * 入力値の範囲に関する検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param mixed $start 最小値
     * @param mixed $end 最大値
     * @param boolean $inclusive 最小値/最大値と同値を許容するかどうか
     * @return boolean 対象項目の検証結果
     */
    function setRange($name, $message, $start, $end, $inclusive = true)
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isBetween'), $start, $end, $inclusive);
    }

    /**
     * 日付項目の検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param string $format sscanf関数で日付文字列を「年、月、日」の順番で解析できるようなフォーマット
     * @return boolean 対象項目の検証結果
     */
    function setDate($name, $message, $format = '%d-%d-%d')
    {
        return $this->setFunc($name, $message, array('Moony_Checker', 'isDate'), $format);
    }

    /**
     * ユーザ関数を用いた検証設定を行います。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param callback $callback 検証に用いるユーザ関数
     * @param array $params 関数の2番目以降の引数（1つ目の引数は入力値）
     * @return boolean 対象項目の検証結果
     */
    function setFunc($name, $message, $callback, $params = null)
    {
        $value = $this->request->get($name);
        if (Moony_Checker::isValid($value, $callback, $params)) {
            return true;
        }
        $this->addError($name, $message);
        $this->is_valid = false;
        return false;
    }

    /**
     * エラー時に使用する
     * テンプレートファイル名を設定します。
     * 拡張子を除いた名称を設定してください。
     *
     * @access public
     * @param string $template テンプレートファイル名
     */
    function setTemplate($template)
    {
        $this->template = $template;
    }

    /**
     * エラー時に使用する
     * テンプレートファイル名を取得します。
     *
     * @access public
     * @return string テンプレートファイル名
     */
    function getTemplate()
    {
        return $this->template;
    }

    /**
     * エラーメッセージを追加します。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     */
    function addError($name, $message)
    {
        $this->errors[$name][] = $message;
    }

    /**
     * エラーが存在するかどうかを返します。
     *
     * @access public
     * @return boolean 存在すればtrue
     */
    function hasError()
    {
        return !$this->is_valid || (count($this->errors) > 0);
    }

    /**
     * 登録された全てのエラーメッセージを
     * 連想配列として取得します。
     *
     * @access public
     * @return array 全てのエラーメッセージ
     */
    function getErrors()
    {
        return $this->errors;
    }
}
?>
