<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * ファイルのアップロードを行うためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Uploader
{
    /** @var string アップロード対象のファイル項目名 */
    var $name;

    /** @var array アップロードされるファイルに関する情報 */
    var $file;

    /** @var string アップロードファイル名に用いるエンコーディング */
    var $encoding;

    /**
     * コンストラクタです。
     * 初期化処理を行います。
     *
     * @access public
     */
    function Moony_Uploader()
    {
        $this->name = null;
        $this->file = null;
        $this->encoding = MOONY_INTERNAL_ENCODING;
    }

    /**
     * アップロードの処理対象を設定します。
     *
     * @access public
     * @param object $request Moony_Requestのインスタンス
     * @param string $name アップロード対象の処理項目名
     * @return boolean 正しく設定されればtrue
     */
    function setTarget(&$request, $name)
    {
        $this->name = $name;
        $this->file = $request->getFile($name);
        if (is_null($this->file)) {
            return false;
        }
        return true;
    }

    /**
     * アップロードファイル名に用いるエンコーディングを設定します。
     * ファイル名に日本語が含まれる可能性がある場合、
     * 環境に適したエンコーディングを必ず指定してください。
     * 明示的に指定されない場合、Moonyの内部エンコーディングが使用されます。
     *
     * @access public
     * @param string $encoding エンコーディング
     */
    function setEncoding($encoding)
    {
        $this->encoding = $encoding;
    }

    /**
     * ファイルのアップロード処理を行います。
     * 何かエラーが発生している場合、処理を一切行わずにfalseを返します。
     * 保存先のファイル名はアップロードされるファイル名と同一です。
     *
     * @access public
     * @param string $save_dir 保存先のディレクトリ名
     * @param string|array $save_name 保存先のファイル名（配列を処理する場合、配列で指定）
     * @param integer $file_mode 保存したファイルのパーミッション（8進数で指定）
     * @param boolean $on_error_remove 処理途中でエラーが発生した場合、アップロード済みのファイルを削除するかどうか
     * @return boolean 全てのアップロードが成功した場合true、そうでない場合false
     */
    function upload($save_dir, $save_name, $file_mode = 0644, $on_error_remove = false)
    {
        // 対象が確定されているか
        if (is_null($this->file)) {
            Moony_Logger::warn('Unable to specify the target file',  __FILE__, __LINE__);
            return false;
        }
        // アップロード時のエラーがないか
        if ($this->hasError()) {
            Moony_Logger::warn('Upload preparation failed',  __FILE__, __LINE__);
            return false;
        }
        // 保存先ディレクトリが存在するか
        clearstatcache();
        if (!file_exists($save_dir)) {
            // 存在しない場合、作成を試みる
            if (mkdir($save_dir) === false) {
                Moony_Logger::warn('Unable to create directory: ' . $save_dir,  __FILE__, __LINE__);
                return false;
            }
        }
        // ファイル名の指定が正しいかどうか
        if (is_array($this->file['name'])) {
            if (!is_array($save_name) && count($save_name) < count($this->file['name'])) {
                Moony_Logger::warn('File name not set: ' . $save_name,  __FILE__, __LINE__);
                return false;
            }
        } else {
            $save_name = is_array($save_name) ? $save_name[0] : $save_name;
        }
        // ファイル名エンコーディングの変換
        if ($this->encoding != MOONY_INTERNAL_ENCODING) {
            if (is_array($this->file['name'])) { 
                mb_convert_variables($this->encoding, MOONY_INTERNAL_ENCODING, &$this->file['name']);
            } else {
                $this->file['name'] = mb_convert_encoding($this->file['name'], $this->encoding, MOONY_INTERNAL_ENCODING);
            }
        }
        // アップロード処理
        if (is_array($this->file['name'])) {
            $paths = array();
            $i = 0;
            foreach ($this->file['name'] as $key => $name) {
                if ($this->file['error'][$key] == UPLOAD_ERR_OK) {
                    $save_path = Moony_Utils::buildPath($save_dir, $save_name[$i]);
                    if (move_uploaded_file($this->file['tmp_name'][$key], $save_path)) {
                        chmod($save_path, $file_mode);
                        $paths[] = $save_path;
                    } else {
                        Moony_Logger::warn('File upload failed: ' . $this->name,  __FILE__, __LINE__);
                        if ($on_error_remove) {
                            foreach ($paths as $path) {
                                @unlink($path);
                            }
                        }
                        return false;
                    }
                }
                $i++;
            }
        } else {
            if ($this->file['error'] == UPLOAD_ERR_OK) {
                $save_path = Moony_Utils::buildPath($save_dir, $save_name);
                if (move_uploaded_file($this->file['tmp_name'], $save_path)) {
                    chmod($save_path, $file_mode);
                } else {
                    Moony_Logger::warn('File upload failed: ' . $this->name,  __FILE__, __LINE__);
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 対象のファイルが間違いなく
     * アップロード準備されているかどうか調べます。
     *
     * @access public
     * @return boolean 何かエラーがある場合true
     */
    function hasError()
    {
        $error_code = Moony_Utils::getArrayValue('error', $this->file);
        if (is_array($error_code)) {
            foreach ($error_code as $code) {
                if ($code != UPLOAD_ERR_OK && $code != UPLOAD_ERR_NO_FILE) {
                    return true;
                }
            }
        } else {
            if ($error_code != UPLOAD_ERR_OK && $error_code != UPLOAD_ERR_NO_FILE) {
                return true;
            }
        }
        return false;
    }

    /**
     * アップロードファイルの元の名称を取得します。
     *
     * @access public
     * @return string|array アップロードファイルの元の名称
     */
    function getOriginalName()
    {
        return $this->file['name'];
    }
}
?>
